/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/model/formula/cellref'
], function (Address, CellRef) {

    'use strict';

    // class CellRef ==========================================================

    describe('Spreadsheet class CellRef', function () {

        it('should exist', function () {
            expect(CellRef).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // convenience shortcuts
        var a = Address.parse;

        // create a CellRef instance on-the-fly from $1.$1 notation (column/row offsets)
        function cell(text) {
            var m = /^(\$)?(-?\d+)\.(\$)?(-?\d+)$/i.exec(text);
            return new CellRef(parseInt(m[2], 10), parseInt(m[4], 10), !!m[1], !!m[3]);
        }

        // dummy spreadsheet document model
        var docModel = {
            isValidAddress: function (address) {
                return (address[0] >= 0) && (address[0] <= this.getMaxCol()) &&
                    (address[1] >= 0) && (address[1] <= this.getMaxRow());
            },
            getMaxCol: _.constant(9),
            getMaxRow: _.constant(19)
        };

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a cell reference', function () {
                var r1 = new CellRef(1, 2, false, true);
                expect(r1).to.have.property('col', 1);
                expect(r1).to.have.property('row', 2);
                expect(r1).to.have.property('absCol', false);
                expect(r1).to.have.property('absRow', true);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('toString');
            });
            it('should return an internal string representation', function () {
                expect(new CellRef(1, 2, false, false).toString()).to.equal('B3');
                expect(new CellRef(1, 2, false, true).toString()).to.equal('B$3');
                expect(new CellRef(1, 2, true, false).toString()).to.equal('$B3');
                expect(new CellRef(1, 2, true, true).toString()).to.equal('$B$3');
                // helper self test
                expect(cell('1.2')).to.stringifyTo('B3');
                expect(cell('1.$2')).to.stringifyTo('B$3');
                expect(cell('$1.2')).to.stringifyTo('$B3');
                expect(cell('$1.$2')).to.stringifyTo('$B$3');
                expect(cell('$-1.$2').col).to.equal(-1);
                expect(cell('$1.$-2').row).to.equal(-2);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(CellRef).itself.to.respondTo('create');
            });
            it('should create a cell reference', function () {
                var r1 = CellRef.create(a('B3'), false, true);
                expect(r1).to.be.an['instanceof'](CellRef);
                expect(r1).to.have.property('col', 1);
                expect(r1).to.have.property('row', 2);
                expect(r1).to.have.property('absCol', false);
                expect(r1).to.have.property('absRow', true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('clone');
            });
            it('should create a clone', function () {
                var r1 = cell('1.$2'), r2 = r1.clone();
                expect(r2).to.be.an['instanceof'](CellRef);
                expect(r2).to.not.equal(r1);
                expect(r2).to.stringifyTo('B$3');
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('equals');
            });
            var r1 = cell('1.$2');
            it('should return true for equal cell references', function () {
                expect(r1.equals(cell('1.$2'))).to.equal(true);
            });
            it('should return false for different cell references', function () {
                expect(r1.equals(cell('2.$2'))).to.equal(false);
                expect(r1.equals(cell('$1.$2'))).to.equal(false);
                expect(r1.equals(cell('1.2'))).to.equal(false);
                expect(r1.equals(cell('1.$1'))).to.equal(false);
            });
        });

        describe('method "toAddress"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('toAddress');
            });
            it('should return the address', function () {
                expect(cell('1.2').toAddress()).to.be.an['instanceof'](Address);
                expect(cell('1.2').toAddress()).to.stringifyTo('B3');
                expect(cell('$1.2').toAddress()).to.stringifyTo('B3');
                expect(cell('1.$2').toAddress()).to.stringifyTo('B3');
                expect(cell('$1.$2').toAddress()).to.stringifyTo('B3');
            });
        });

        describe('method "relocate"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('relocate');
            });
            it('should relocate relative components without wrapping', function () {
                var r1 = cell('1.2');
                expect(r1.relocate(docModel, a('A2'), a('C5'), false)).to.equal(true);
                expect(r1).to.stringifyTo('D6');
                expect(r1.relocate(docModel, a('C5'), a('A2'), false)).to.equal(true);
                expect(r1).to.stringifyTo('B3');
                expect(cell('0.0').relocate(docModel, a('B2'), a('A2'), false)).to.equal(false);
                expect(cell('0.0').relocate(docModel, a('B2'), a('B1'), false)).to.equal(false);
                expect(cell('9.19').relocate(docModel, a('A1'), a('B1'), false)).to.equal(false);
                expect(cell('9.19').relocate(docModel, a('A1'), a('A2'), false)).to.equal(false);
            });
            it('should relocate relative components with wrapping', function () {
                var r1 = cell('1.2');
                expect(r1.relocate(docModel, a('A2'), a('C5'), true)).to.equal(true);
                expect(r1).to.stringifyTo('D6');
                expect(r1.relocate(docModel, a('C5'), a('A2'), true)).to.equal(true);
                expect(r1).to.stringifyTo('B3');
                expect(r1.relocate(docModel, a('E6'), a('B2'), true)).to.equal(true);
                expect(r1).to.stringifyTo('I19');
                expect(r1.relocate(docModel, a('B2'), a('E6'), true)).to.equal(true);
                expect(r1).to.stringifyTo('B3');
            });
            it('should not relocate absolute components', function () {
                var r1 = cell('$1.$2');
                expect(r1.relocate(docModel, a('A2'), a('C5'), false)).to.equal(true);
                expect(r1).to.stringifyTo('$B$3');
                var r2 = cell('$1.$2');
                expect(r2.relocate(docModel, a('A2'), a('C5'), true)).to.equal(true);
                expect(r2).to.stringifyTo('$B$3');
            });
        });

        describe('method "colText"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('colText');
            });
            it('should return the column text', function () {
                expect(cell('1.$2').colText()).to.equal('B');
                expect(cell('$1.$2').colText()).to.equal('$B');
            });
        });

        describe('method "rowText"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('rowText');
            });
            it('should return the row text', function () {
                expect(cell('1.$2').rowText()).to.equal('$3');
                expect(cell('1.2').rowText()).to.equal('3');
            });
        });

        describe('method "refText"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('refText');
            });
            it('should return the reference text', function () {
                expect(cell('$1.$2').refText()).to.equal('$B$3');
                expect(cell('1.$2').refText()).to.equal('B$3');
                expect(cell('$1.2').refText()).to.equal('$B3');
                expect(cell('1.2').refText()).to.equal('B3');
            });
        });

        describe('method "colTextRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('colTextRC');
            });
            it('should return the column text for R1C1', function () {
                expect(cell('$1.$2').colTextRC('RC')).to.equal('C2');
                expect(cell('$1.$2').colTextRC('ZS')).to.equal('S2');
                expect(cell('1.$2').colTextRC('RC')).to.equal('C[1]');
                expect(cell('0.$2').colTextRC('RC')).to.equal('C');
                expect(cell('-1.$2').colTextRC('RC')).to.equal('C[-1]');
            });
        });

        describe('method "rowTextRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('rowTextRC');
            });
            it('should return the column text for R1C1', function () {
                expect(cell('$1.$2').rowTextRC('RC')).to.equal('R3');
                expect(cell('$1.$2').rowTextRC('ZS')).to.equal('Z3');
                expect(cell('$1.2').rowTextRC('RC')).to.equal('R[2]');
                expect(cell('$1.0').rowTextRC('RC')).to.equal('R');
                expect(cell('$1.-2').rowTextRC('RC')).to.equal('R[-2]');
            });
        });

        describe('method "refTextRC"', function () {
            it('should exist', function () {
                expect(CellRef).to.respondTo('refTextRC');
            });
            it('should return the column text for R1C1', function () {
                expect(cell('$1.$2').refTextRC('RC')).to.equal('R3C2');
                expect(cell('$1.$2').refTextRC('ZS')).to.equal('Z3S2');
                expect(cell('1.$2').refTextRC('RC')).to.equal('R3C[1]');
                expect(cell('0.$2').refTextRC('RC')).to.equal('R3C');
                expect(cell('-1.$2').refTextRC('RC')).to.equal('R3C[-1]');
                expect(cell('$1.2').refTextRC('RC')).to.equal('R[2]C2');
                expect(cell('$1.0').refTextRC('RC')).to.equal('RC2');
                expect(cell('$1.-2').refTextRC('RC')).to.equal('R[-2]C2');
                expect(cell('1.2').refTextRC('RC')).to.equal('R[2]C[1]');
                expect(cell('0.0').refTextRC('RC')).to.equal('RC');
                expect(cell('-1.-2').refTextRC('RC')).to.equal('R[-2]C[-1]');
            });
        });
    });

    // ========================================================================
});
