/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/captionmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms'
], function (Utils, Forms) {

    'use strict';

    // mix-in class CaptionMixin ==============================================

    /**
     * Extends a Group object with additional methods to manipulate a caption
     * (icon and/or label text) of a form control element.
     *
     * Note: This is a mix-in class supposed to extend an existing instance of
     * the class Group or one of its derived classes. Expects the symbol 'this'
     * to be bound to an instance of Group.
     *
     * @constructor
     *
     * @param {HTMLElement|jQuery} controlNode
     *  The form control element used as container for the caption nodes.
     * @param {Object} [initOptions]
     *  Optional parameters. The following options are supported:
     *  @param {Object} [initOptions.smallerVersion]
     *      If specified, the caption will known, how it should act, if there
     *      will be not enough free place to show normal view
     */
    function CaptionMixin(controlNode, initOptions) {

        var // smallerVersion options
            smallerVersion = Utils.getOption(initOptions, 'smallerVersion'),
            // saves the (old) styles in case of switching to small version
            saveStyles = {};

        // methods ------------------------------------------------------------

        /**
         * Returns the caption container node of the wrapped control element.
         *
         * @returns {jQuery}
         *  The caption container node of the wrapped control element.
         */
        this.getCaptionNode = function () {
            return Forms.getCaptionNode(controlNode);
        };

        /**
         * Changes the icon of the wrapped control element.
         *
         * @param {String} icon
         *  The CSS class name of the new icon.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  Forms.setCaptionIcon().
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.setIcon = function (icon, options) {
            Forms.setCaptionIcon(controlNode, icon, options);
            this.layout();
            return this;
        };

        /**
         * Removes the icon of the wrapped control element.
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.removeIcon = function () {
            Forms.removeCaptionIcon(controlNode);
            this.layout();
            return this;
        };

        /**
         * Returns the current label text of the wrapped control element.
         *
         * @returns {String}
         *  The current text label.
         */
        this.getLabel = function () {
            return Forms.getCaptionText(controlNode);
        };

        /**
         * Changes the label text of the wrapped control element.
         *
         * @param {String} text
         *  The new label text.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  Forms.setCaptionText().
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.setLabel = function (text, options) {
            Forms.setCaptionText(controlNode, text, options);
            this.layout();
            return this;
        };

        /**
         * Removes the text label from the wrapped control element.
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.removeLabel = function () {
            Forms.removeCaptionText(controlNode);
            this.layout();
            return this;
        };

        /**
         * Inserts an icon and a text label into the wrapped control element,
         * and removes the old caption (icon and text label) from the control.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method Forms.setCaption().
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.setCaption = function (options) {
            Forms.setCaption(controlNode, options);
            this.layout();
            return this;
        };

        /**
         * Sets a tool tip text at the wrapped control element.
         *
         * @param {String} text
         *  The tool tip text to be set at the control.
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.setToolTip = function (text) {
            Forms.setToolTip(controlNode, { tooltip: text });
            return this;
        };

        /**
         * removes the tool tip text from the wrapped control element.
         *
         * @returns {CaptionMixin}
         *  A reference to this instance.
         */
        this.removeToolTip = function () {
            Forms.setToolTip(controlNode, null);
            return this;
        };

        /**
         * Overwrites the base-methods (group.js) to
         * activate/deactivate the small version of the button
         */
        this.activateSmallVersion = function () {
            // if smallerVersion isn't empty
            if (!_.isEmpty(smallerVersion)) {
                // change label
                if (_.has(smallerVersion, 'label')) {
                    saveStyles.label = saveStyles.label || this.getLabel();
                    this.setLabel(smallerVersion.label);
                }

                // icon
                if (_.has(smallerVersion, 'icon')) {
                    this.setIcon(smallerVersion.icon);
                }

                // hide label
                if (_.has(smallerVersion, 'hideLabel') && smallerVersion.hideLabel === true) {
                    saveStyles.label = saveStyles.label || this.getLabel();
                    this.removeLabel();
                }
            }
        };
        this.deactivateSmallVersion = function () {
            // if smallerVersion isn't empty and there are saved styles
            if (!_.isEmpty(smallerVersion) && !_.isEmpty(saveStyles)) {
                // reset label
                if (_.has(saveStyles, 'label')) {
                    this.setLabel(saveStyles.label);
                    delete saveStyles.label;
                }

                // remove icon
                if (_.has(smallerVersion, 'icon')) {
                    this.removeIcon();
                }

                // show old label
                if (_.has(saveStyles, 'hideLabel') && smallerVersion.hideLabel === true) {
                    this.setLabel(saveStyles.label);
                    delete saveStyles.label;
                }
            }
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            controlNode = null;
        });

    } // class CaptionMixin

    // exports ================================================================

    return CaptionMixin;

});
