/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/textframework/utils/keycodeutils', [
    'io.ox/office/tk/keycodes'
], function (KeyCodes) {

    'use strict';

    // static class KeyCodeUtils ================================================

    var KeyCodeUtils = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns true, if the passed keyboard event is ctrl+v, meta+v or shift+insert.
     * Attention: Comparison with KeyCodes.INSERT only allowed in processKeyDown, not in processKeyPressed
     * Info: Use keyCode only keyDown, charCode only in keyPress
     *
     * @param {jQuery.Event} event
     *  A jQuery keyboard event object.
     */
    KeyCodeUtils.isPasteKeyEventKeyDown = function (event) {
        return KeyCodes.matchKeyCode(event, 'V', { ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'INSERT', { shift: true });
    };

    /**
     * Returns true, if the passed keyboard event is ctrl+c, meta+c or ctrl+insert.
     * Attention: Comparison with KeyCodes.INSERT only allowed in processKeyDown, not in processKeyPressed
     *
     * @param {jQuery.Event} event
     *  A jQuery keyboard event object.
     */
    KeyCodeUtils.isCopyKeyEventKeyDown = function (event) {
        return KeyCodes.matchKeyCode(event, 'C', { ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'INSERT', { ctrlOrMeta: true });
    };

    /**
     * Returns true, if the passed keyboard event is ctrl+x, meta+x or shift+delete.
     * Attention: Comparison with KeyCodes.DELETE only allowed in processKeyDown, not in processKeyPressed
     *
     * @param {jQuery.Event} event
     *  A jQuery keyboard event object.
     */
    KeyCodeUtils.isCutKeyEventKeyDown = function (event) {
        return KeyCodes.matchKeyCode(event, 'X', { ctrlOrMeta: true }) || KeyCodes.matchKeyCode(event, 'DELETE', { shift: true });
    };

    /**
     * Returns true, if the passed keyboard event is ctrl+a.
     *
     * @param {jQuery.Event} event
     *  A jQuery keyboard event object.
     */
    KeyCodeUtils.isSelectAllEvent = function (event) {
        return KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true });
    };

    /**
     * Returns true, if the passed keyboard event is the global F6 accessibility key event
     */
    KeyCodeUtils.isF6AcessibilityKeyEvent = function (event) {
        // ignore all modifier keys
        return event.keyCode === KeyCodes.F6;
    };

    /**
     * Returns true, if the passed keyboard event is ctrl+f, meta+f, ctrl+g, meta+g, ctrl+shift+g or meta+shift+g.
     *
     * @param {jQuery.Event} event
     *  A jQuery keyboard event object.
     */
    KeyCodeUtils.isSearchKeyEvent = function (event) {
        return (KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true, shift: null }) && (event.charCode === 102 || event.keyCode === KeyCodes.F || event.charCode === 103 || event.keyCode === KeyCodes.G));
    };

    /**
     * Returns true, if the passed keyboard event is a common browser keyboard shortcut
     * that should be handled by the browser itself.
     *
     * @param {jQuery.event} event
     *  A jQuery keyboard event object.
     *
     * @returns {Boolean}
     *  Whether the passed event is a browser shortcut.
     */
    KeyCodeUtils.isBrowserShortcutKeyEvent = function (event) {

        var // Ctrl
            ctrl = KeyCodes.matchModifierKeys(event, { ctrl: true }),
            // Ctrl and Shift
            ctrlShift = KeyCodes.matchModifierKeys(event, { ctrl: true, shift: true }),
            // Meta and Alt
            metaAlt = KeyCodes.matchModifierKeys(event, { meta: true, alt: true }),
            // Meta and Shift
            metaShift = KeyCodes.matchModifierKeys(event, { meta: true, shift: true }),
            // Ctrl and Meta
            ctrlMeta = KeyCodes.matchModifierKeys(event, { ctrl: true, meta: true }),
            // Ctrl or Meta
            ctrlOrMeta = KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true }),
            // Ctrl or Meta and Shift
            ctrlOrMetaShift = KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true, shift: true }),
            // Ctrl or Meta and Alt
            ctrlOrMetaAlt = KeyCodes.matchModifierKeys(event, { ctrlOrMeta: true, alt: true });

        if (event.type === 'keypress') {
            // check charCode for keypress event

            // Switch to the specified/last tab - Ctrl + 1...8, Cmd + 1...8 / Ctrl + 9, Cmd + 9
            if (ctrlOrMeta && event.charCode >= 49 && event.charCode <= 57) {
                return true;
            }

            // Open a new tab             - Ctrl + T, Cmd + T
            // Reopen the last closed tab - Ctrl + Shift + T, Cmd + Shift + T
            if ((ctrlOrMeta || ctrlOrMetaShift) && (event.charCode === 116 || event.charCode === 84)) {
                return true;
            }

            // Close the current tab     - Ctrl + W, Ctrl + F4, Cmd + W
            // Open a new browser window - Ctrl + N, Cmd + N
            if (ctrlOrMeta && (event.charCode === 119 || event.charCode === 110)) {
                return true;
            }

            // Close the current window - Ctrl + Shift + W, Cmd + Shift + W
            if (ctrlOrMetaShift && (event.charCode === 87 || event.charCode === 119)) {
                return true;
            }

            // Open a new window in incognito mode - Ctrl + Shift + N, Cmd + Shift + N, Ctrl + Shift + P, Cmd + Shift + P
            if (ctrlOrMetaShift && (event.charCode === 78 || event.charCode === 80)) {
                return true;
            }

            // Minimize the current window - Ctrl + M, Cmd + M
            if (ctrlOrMeta && event.charCode === 109) {
                return true;
            }

            // Hide browser - Ctrl + H, Cmd + H
            // Hide all other windows - Ctrl + Alt + H, Cmd + Alt + H
            if ((ctrlOrMeta || ctrlOrMetaAlt) && (event.charCode === 104 || event.charCode === 170)) {
                return true;
            }

            // Quit browser - Ctrl + Q, Cmd + Q
            if (ctrlOrMeta && event.charCode === 113) {
                return true;
            }

            // Zoom in on the page - Ctrl + '+', Cmd + '+'
            // Zoom out on the page - Ctrl + '-', Cmd + '-'
            // Reset zoom level - Ctrl + 0, Cmd + 0
            if (ctrlOrMeta && (event.charCode === 43 || event.charCode === 45 || event.charCode === 48)) {
                return true;
            }

            // Full-screen mode - Ctrl + Cmd + F
            if (ctrlMeta && (event.charCode === 102 || event.charCode === 6)) { //6 is for safari
                return true;
            }

            // Open the browsing history - Ctrl + H, Cmd + Shift + H
            if ((ctrl || metaShift) && (event.charCode === 72 || event.charCode === 104)) {
                return true;
            }

            // Open the download history - Ctrl + J, Cmd + J, Cmd + Shift + J, Cmd + Alt + 2
            if (((ctrlOrMeta || metaShift) && (event.charCode === 106 || event.charCode === 74)) || (metaAlt && event.charCode === 8220)) {
                return true;
            }

            // Bookmark the current web site - Ctrl + D, Cmd + D
            if (ctrlOrMeta && event.charCode === 100) {
                return true;
            }

            // Toggles the bookmarks bar                                               - Ctrl + Shift + B, Cmd + Shift + B
            // Save all open pages in your current window as bookmarks in a new folder - Ctrl + Shift + D, Cmd + Shift + D
            if (ctrlOrMetaShift && (event.charCode === 66 || event.charCode === 68)) {
                return true;
            }

            // Open the Bookmarks window - Ctrl + Shift + B, Cmd + Shift + B, Cmd + Alt + B
            if ((ctrlOrMetaShift || metaAlt) && (event.charCode === 66 || event.charCode === 8747)) {
                return true;
            }

            // Open Developer Tools - Ctrl + Shift + (I,J,K,S), Cmd + Alt + (I,J,K,S)
            if ((ctrlShift || metaAlt) && (event.charCode === 8260 || event.charCode === 73 ||
                                           event.charCode === 186  || event.charCode === 74 ||
                                           event.charCode === 8710 || event.charCode === 75 ||
                                           event.charCode === 8218 || event.charCode === 83)) {
                return true;
            }

            // For function keys Firefox sends a keydown event with corresponding keyCode as expected,
            // but also sends a keypress event with charCode 0 and a corresponding keyCode.
            if (_.browser.Firefox && (event.charCode === 0)) {

                // Full-screen mode - F11, Open Firebug - F12
                if (event.keyCode === KeyCodes.F11 || event.keyCode === KeyCodes.F12) {
                    return true;
                }
            }

        } else {
            // check keyCode for keyup and keydown events

            // Switch to the specified/last tab - Ctrl + 1...8, Cmd + 1...8 / Ctrl + 9, Cmd + 9
            if (ctrlOrMeta && ((event.keyCode >= KeyCodes['1'] && event.keyCode <= KeyCodes['9']))) {
                return true;
            }

            // Switch to the next/previous tab - Ctrl + Tab / Ctrl + Shift + Tab
            if ((KeyCodes.matchKeyCode(event, KeyCodes.TAB, { ctrl: true }) || KeyCodes.matchKeyCode(event, KeyCodes.TAB, { ctrl: true, shift: true }))) {
                return true;
            }

            // Open a new tab             - Ctrl + T, Cmd + T
            // Reopen the last closed tab - Ctrl + Shift + T, Cmd + Shift + T
            if ((ctrlOrMeta || ctrlOrMetaShift) && event.keyCode === KeyCodes.T) {
                return true;
            }

            // Close the current tab     - Ctrl + W, Ctrl + F4, Cmd + W
            // Open a new browser window - Ctrl + N, Cmd + N
            if (ctrlOrMeta && (event.keyCode === KeyCodes.W || event.keyCode === KeyCodes.F4 || event.keyCode === KeyCodes.N)) {
                return true;
            }

            // Close the current window - Alt + F4, Ctrl + Shift + W, Cmd + Shift + W
            if ((KeyCodes.matchModifierKeys(event, { alt: true }) && event.keyCode === KeyCodes.F4) || (ctrlOrMetaShift && event.keyCode === KeyCodes.W)) {
                return true;
            }

            // Open a new window in incognito mode - Ctrl + Shift + N, Cmd + Shift + N, Ctrl + Shift + P, Cmd + Shift + P
            if (ctrlOrMetaShift && (event.keyCode === KeyCodes.N || event.keyCode === KeyCodes.P)) {
                return true;
            }

            // Minimize the current window - Ctrl + M, Cmd + M
            if (ctrlOrMeta && event.keyCode === KeyCodes.M) {
                return true;
            }

            // Hide browser - Ctrl + H, Cmd + H
            // Hide all other windows - Ctrl + Alt + H, Cmd + Alt + H
            if ((ctrlOrMeta || ctrlOrMetaAlt) && event.keyCode === KeyCodes.H) {
                return true;
            }

            // Quit browser - Ctrl + Q, Cmd + Q
            if (ctrlOrMeta && event.keyCode === KeyCodes.Q) {
                return true;
            }

            // Zoom in on the page - Ctrl + '+', Cmd + '+'
            // Zoom out on the page - Ctrl + '-', Cmd + '-'
            // Reset zoom level - Ctrl + 0, Cmd + 0
            if (ctrlOrMeta && (event.keyCode === KeyCodes.NUM_PLUS  || event.keyCode === 187 /*Safari*/ || event.keyCode === 171 /*Firefox*/ ||
                               event.keyCode === KeyCodes.NUM_MINUS || event.keyCode === 189 /*Safari*/ || event.keyCode === 173 /*Firefox*/ ||
                               event.keyCode === KeyCodes.NUM_0     || event.keyCode === KeyCodes['0'])) {
                return true;
            }

            // Full-screen mode - F11, Ctrl + Cmd + F
            if ((event.keyCode === KeyCodes.F11) || (ctrlMeta && event.keyCode === KeyCodes.F)) {
                return true;
            }

            // Open the browsing history - Ctrl + H, Cmd + Shift + H
            if ((ctrl || metaShift) && event.keyCode === KeyCodes.H) {
                return true;
            }

            // Open the download history - Ctrl + J, Cmd + J, Cmd + Shift + J, Cmd + Alt + 2
            if (((ctrlOrMeta || metaShift) && event.keyCode === KeyCodes.J) || (metaAlt && event.keyCode === KeyCodes['2'])) {
                return true;
            }

            // Bookmark the current web site - Ctrl + D, Cmd + D
            if (ctrlOrMeta && event.keyCode === KeyCodes.D) {
                return true;
            }

            // Toggles the bookmarks bar                                               - Ctrl + Shift + B, Cmd + Shift + B
            // Save all open pages in your current window as bookmarks in a new folder - Ctrl + Shift + D, Cmd + Shift + D
            if (ctrlOrMetaShift && (event.keyCode === KeyCodes.B || event.keyCode === KeyCodes.D)) {
                return true;
            }

            // Open the Bookmarks window - Ctrl + Shift + B, Cmd + Shift + B, Cmd + Alt + B
            if ((ctrlOrMetaShift || metaAlt) && event.keyCode === KeyCodes.B) {
                return true;
            }

            // Open Firebug - F12
            if (event.keyCode === KeyCodes.F12) {
                return true;
            }

            // Open Developer Tools - Ctrl + Shift + (I,J,K,S), Cmd + Alt + (I,J,K,S)
            if ((ctrlShift || metaAlt) && (event.keyCode === KeyCodes.I ||
                                           event.keyCode === KeyCodes.J ||
                                           event.keyCode === KeyCodes.K ||
                                           event.keyCode === KeyCodes.S)) {
                return true;
            }
        }

        return false;
    };

    /**
     * Getting the printable character for a specified char code.
     *
     * @param {Number} charCode
     *  The char code for that the corresponding printable char needs to be determined.
     *
     * @returns {String}
     *  The character from the specified char code. Or undefined, if it cannot be determined.
     */
    KeyCodeUtils.getPrintableCharFromCharCode = function (charCode) {
        return (_.isNumber(charCode) && (charCode >= 32)) ? String.fromCharCode(charCode) : undefined;
    };

    /**
     * Getting the printable character for a specified event evaluating the event properties
     * 'charCode' and 'which'.
     *
     * @param {jQuery.Event} event
     *  A jQuery keyboard event object.
     *
     * @returns {String}
     *  The character from the specified event. Or an empty string, if not character could be determined.
     */
    KeyCodeUtils.getPrintableChar = function (event) {
        return KeyCodeUtils.getPrintableCharFromCharCode(event.charCode) || KeyCodeUtils.getPrintableCharFromCharCode(event.which) || '';
    };

    // exports ================================================================

    return KeyCodeUtils;

});
