/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/view/view', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/baseframework/view/popup/compoundmenu',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/view/editview',
    'io.ox/office/editframework/view/popup/attributestooltip',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/view/labels',
    'io.ox/office/textframework/view/controls',
    'io.ox/office/textframework/view/textdialogs',
    'io.ox/office/textframework/view/popup/universalcontextmenu',
    'gettext!io.ox/office/text/main',
    'less!io.ox/office/text/view/style',
    'io.ox/core/tk/datepicker'
], function (DateUtils, Utils, Forms, CompoundMenu, AttributeUtils, EditView, AttributesToolTip, Config, Position, DOM, Labels, Controls, Dialogs, UniversalContextMenu, gt) {

    'use strict';

    var // predefined zoom factors
        ZOOM_FACTORS = Utils.SMALL_DEVICE ? [100, 150, 200] : [35, 50, 75, 100, 150, 200, 300, 400, 600, 800];

    // class TextView =========================================================

    /**
     * The text editor view.
     *
     * Triggers the events supported by the base class EditView, and the
     * following additional events:
     * - 'change:zoom': When the zoom level for the displayed document was changed.
     * - 'change:pageWidth': When the width of the page or the inner margin of the
     *      page was changed. This happens for example on orientation changes on
     *      small devices.
     *
     * @constructor
     *
     * @extends EditView
     *
     * @param {TextApplication} app
     *  The application containing this view instance.
     *
     * @param {TextModel} docModel
     *  The document model created by the passed application.
     */
    function TextView(app, docModel) {

        var // self reference
            self = this,

            // the root node of the entire application pane
            appPaneNode = null,

            // the scrollable document content node
            contentRootNode = null,

            // the page node
            pageNode = null,

            //page content node
            pageContentNode = null,

            // scroll position of the application pane
            scrollPosition = { left: 0, top: 0 },

            // debug operations for replay
            replayOperations = null,

            // maximum osn for debug operations for replay
            replayOperationsOSN = null,

            // current zoom type (percentage or keyword)
            zoomType = (Utils.COMPACT_DEVICE && Utils.isPortrait()) ? 'width' : 100,

            // the current effective zoom factor, in percent
            zoomFactor = 100,

            // outer width of page node, used for zoom calculations
            pageNodeWidth = 0,

            // outer height of page node, used for zoom calculations
            pageNodeHeight = 0,

            // the generic content menu for all types of contents
            contextMenu = null,

            // change track pop-up and controls
            changeTrackPopup = null,
            changeTrackBadge = new Controls.ChangeTrackBadge(this),
            changeTrackAcceptButton = new Controls.Button({ icon: 'fa-check', value: 'accept', tooltip: gt('Accept this change in the document') }),
            changeTrackRejectButton = new Controls.Button({ icon: 'fa-times', value: 'reject', tooltip: gt('Reject this change in the document') }),
            changeTrackShowButton = new Controls.Button({ icon: 'fa-eye', value: 'show', tooltip: gt('Show this change in the document') }),

            // change track pop-up timeout promise
            changeTrackPopupTimeout = null,

            // instance of change field format popup
            fieldFormatPopup = null,
            fieldFormatList = null,
            descriptionNode = null,
            autoUpdateCheckbox = null,
            setToButton = null,
            toggleDatePickerBtn = null,
            datePickerNode = $('<div>').addClass('date-picker-node'),
            currentCxNode = null,
            autoDate = null,
            currentFormatOfFieldPopup = null,

            // button in popup to remove field
            //removeFieldButton = new Controls.Button({ label: gt('Remove'), value: 'remove', tooltip: gt('Remove this field from document') }),

            // the top visible paragraph set during fast loading the document
            topVisibleParagraph = null,

            // storage for last know touch position
            // (to open context menu on the correct position)
            touchX = null,
            touchY = null;

        // TODO: Use the app-content node defined in baseview

        // base constructor ---------------------------------------------------

        EditView.call(this, app, docModel, searchHandler, {
            initHandler: initHandler,
            initDebugHandler: initDebugHandler,
            initGuiHandler: initGuiHandler,
            initDebugGuiHandler: initDebugGuiHandler,
            grabFocusHandler: grabFocusHandler,
            contentScrollable: true,
            contentMargin: 30,
            enablePageSettings: true,
            userLayerShowOwnColor: true
        });

        // private methods ----------------------------------------------------

        /**
         * Moves the browser focus focus to the editor root node.
         */
        function grabFocusHandler(options) {
            if (Utils.TOUCHDEVICE) {
                var changeEvent = Utils.getOption(options, 'changeEvent', { sourceEventType: 'click' });
                var softKeyboardOpen = Utils.isSoftKeyboardOpen();
                var sourceType = changeEvent.sourceEventType;
                var node = docModel.getNode();

                if (sourceType === 'tap' && !softKeyboardOpen) {
                    Utils.focusHiddenNode();
                } else {
                    Utils.trigger('change:focus', node, node, null);
                    node.focus();
                }
            } else {
                docModel.getNode().focus();
            }
        }

        /**
         * Caches the current scroll position of the application pane.
         */
        function saveScrollPosition() {
            scrollPosition = { top: contentRootNode.scrollTop(), left: contentRootNode.scrollLeft() };
        }

        /**
         * Restores the scroll position of the application pane according to
         * the cached scroll position.
         */
        function restoreScrollPosition() {
            contentRootNode.scrollTop(scrollPosition.top).scrollLeft(scrollPosition.left);
        }

        /**
         * Scrolls the application pane to the focus position of the text
         * selection.
         */
        function scrollToSelection() {

            var // the boundaries of the text cursor
                boundRect = docModel.getSelection().getFocusPositionBoundRect();

            // make cursor position visible in application pane
            if (boundRect) {
                self.scrollToPageRectangle(boundRect);
            }
        }

        /**
         * Performance: Register in the selection, whether the current operation is an insertText
         * operation. In this case the debounced 'scrollToSelection' function can use the saved
         * DOM node instead of using Position.getDOMPosition in 'getPointForTextPosition'. This
         * makes the calculation of scrolling requirements faster.
         */
        function registerInsertText(options) {
            docModel.getSelection().registerInsertText(options);

            //fixing the problem with an upcoming soft-keyboard
            if (Utils.TOUCHDEVICE) { scrollToSelection(); }
        }

        /**
         * Gets the anchor (start) node of the given selection.
         *
         * @returns {Node | null}
         *  the anchor node of the given selection, otherwise null if its not found.
         */
        function getChangeTrackPopupAnchorNode() {
            if (!docModel || !docModel.getSelection()) { return null; }
            var selectionEndPoint = Position.getDOMPosition(docModel.getCurrentRootNode(), docModel.getSelection().getEndPosition());
            if (!selectionEndPoint) { return docModel.getNode(); }
            return selectionEndPoint.node.parentNode;
        }
        /**
         * Returns objects required for creation of the change track popup, containing:
         * - relevant change track info containing type, author, and date of the change,
         * - plus additional info containing info/accept/reject button texts.
         *
         * @returns {Object | Null}
         *  returns an object containing required meta information for the popup, and null if current selection is not a change
         *  tracked node (changeTrackInfo is null)
         *
         */
        function getChangeTrackPopupMeta() {

            // get change tracking information from model
            var changeTrackInfo = _.clone(docModel.getChangeTrack().getRelevantTrackingInfo());
            if (!changeTrackInfo) { return null; }

            // create a GUI label for the action type
            changeTrackInfo.actionText = Labels.CHANGE_TRACK_LABELS[changeTrackInfo.type];
            if (!changeTrackInfo.actionText) {
                Utils.error('TextView.getChangeTrackPopupMeta(): missing label for change tracking type "' + changeTrackInfo.type + '"');
                return null;
            }
            changeTrackInfo.actionText += ': ' + Utils.capitalize(changeTrackInfo.nodeType);

            return changeTrackInfo;
        }

        /**
         * Custom Change track date formatter.
         *
         * @param {Number} timestamp
         *  timestamp in milliseconds
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.toLocal=true]
         *      timestamp are converted to local time zone per default.
         *
         * @returns {String}
         *  The formatted date string.
         */
        function formatChangeTrackDate(timestamp, options) {

            if (!_.isNumber(timestamp)) { return gt('Unknown'); }

            var toLocal = Utils.getBooleanOption(options, 'toLocal', true),
                DateConstructor = toLocal ? DateUtils.getLocalDate : DateUtils.getUTCDate,
                now = new DateConstructor(),
                timeStampDate = new DateConstructor(timestamp);

            function getTimeString() { return DateUtils.format(timeStampDate, DateUtils.getTimeFormat()); }

            function getDateString() {
                // get locale country default date format
                var formatString = DateUtils.getDateFormat();
                // hide year if the document is modified in this year
                if (timeStampDate.getYear() === now.getYear()) { formatString = DateUtils.getDateFormat(); }
                // show full day name if timestamp is still in the previous 7 days
                if (timestamp > (_.now() - DateUtils.getWeekInMS())) { formatString = DateUtils.getWeekDayFormat(); }
                return DateUtils.format(timeStampDate, formatString);
            }

            function isToday(date) {
                return date.getDate() === now.getDate() &&
                    date.getMonth() === now.getMonth() &&
                    date.getYear() === now.getYear();
            }

            function isYesterday(date) {
                return date.getDate() === (now.getDate() - 1) &&
                    date.getMonth() === now.getMonth() &&
                    date.getYear() === now.getYear();
            }

            // TODO: better use today & yesterday with moment.js !!!
            // show yesterday and only time if the timestamp is from today
            if (isYesterday(timeStampDate)) { return gt('Yesterday'); }
            if (isToday(timeStampDate)) { return getTimeString(); }

            // return date string for all other cases
            return getDateString();
        }

        /**
         * Initiates and shows change track popup where applicable.
         *
         * @param {Object} options
         *  @param {jQuery Event} options.browserEvent
         *   The original jQuery Event object from a user selection. This object
         *   is used to filter out selections that are triggered by the keyboard.
         */
        function handleChangeTrackPopup(options) {

            // do not show popup for odf files (not supported yet)
            if (app.isODF()) { return; }

            if (Utils.IOS) {
                docModel.getChangeTrack().setHighlightToCurrentChangeTrack();
            }

            if (self.panesCombined()) {
                return;
            }

            // make sure old popup (if any) is hidden
            changeTrackPopup.hide();

            var // the event of the browser
                browserEvent = Utils.getObjectOption(options, 'browserEvent', null),
                // array of event types, that will trigger change track popup
                relevantEvents = ['mousedown', 'mouseup', 'touchstart', 'touchend'];

            // quit early if no listened events are found
            if (!browserEvent || !_.contains(relevantEvents, browserEvent.type) || (browserEvent.type === 'mousedown' && browserEvent.button === 2)) {
                if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }
                return;
            }

            // clear any existing proprietary change track selection (e.q. table column selections)
            docModel.getChangeTrack().clearChangeTrackSelection();

            // abort existing old timeout promise
            if (changeTrackPopupTimeout) { changeTrackPopupTimeout.abort(); }

            changeTrackPopupTimeout = self.executeDelayed(function () {

                // hyperlink popup has higher priority. Don't show change track popup if its visible.
                if (pageNode.find('.inline-popup.hyperlink').is(':visible')) { return; }

                // try show changetrack popup directly if check spelling is off
                // TODO: spell checker popups must have priority
                // -> check if spell checker is enabled or popup is not visible
                self.showChangeTrackPopup();

            }, 1000, 'Text: View: changeTrackPopupTimeout');
        }

        /**
         * Initializes the change track popup.
         */
        function initChangeTrackPopup() {
            var ctRoot = self.getContentRootNode();

            // create and prepare controls of the popup
            changeTrackPopup = new CompoundMenu(self, {
                classes: 'change-track-popup f6-target',
                autoFocus: false,
                autoClose: false,
                autoLayout: false,
                rootContainerNode: ctRoot
            });
            changeTrackAcceptButton.setLabel(gt('Accept'));
            changeTrackRejectButton.setLabel(gt('Reject'));
            changeTrackShowButton.setLabel(gt('Show'));
            changeTrackPopup
                .addGroup(null, changeTrackBadge)
                .addSeparator()
                .addGroup('acceptSelectedChangeTracking', changeTrackAcceptButton, { inline: true })
                .addGroup('rejectSelectedChangeTracking', changeTrackRejectButton, { inline: true })
                .addGroup('changetrackPopupShow', changeTrackShowButton, { inline: true });

            // disable inherited contenteditable attribute from page node to prevent text selection
            changeTrackPopup.getNode().attr('contenteditable', false);

            // ARIA
            changeTrackPopup.getNode().attr({ tabindex: '-1', role: 'dialog', 'aria-label': gt('Change tracking') });

            // manual popup sizing and positioning
            changeTrackPopup.on('popup:beforelayout', function () {
                var anchorNode = $(getChangeTrackPopupAnchorNode()),
                    anchorPosition = Utils.getChildNodePositionInNode(ctRoot, anchorNode),
                    contentNode = this.getContentNode(),
                    contentSize = Utils.getExactNodeSize(contentNode),
                    zoom = self.getZoomFactor() / 100;
                // calculate positioning based on the anchor node position relative to the content root node
                //anchorPosition.left = anchorPosition.left;
                anchorPosition.top += anchorNode.height() * zoom;
                anchorPosition.width = contentSize.width;
                anchorPosition.height = contentSize.height;
                // apply positioning and dimensions
                this.getNode().css(anchorPosition);
            });
        }

        /**
         * Gets the anchor (start) node of the given field.
         *
         * @returns {jQuery | null}
         *  The anchor node of the given field, otherwise null if its not found.
         */
        function getFieldAnchorNode() {
            return docModel.getFieldManager().getSelectedFieldNode();
        }

        /**
         * Initialize field format change popup.
         *  @param {Object} fieldinstruction
         *  @param {Object} options
         */
        function initFieldFormatPopup(fieldinstruction, options) {
            var contentRootNode = self.getContentRootNode(),
                fieldFormatPopupNode = null;

            fieldFormatList = new Controls.FieldFormatList(self);
            descriptionNode = $('<div>').addClass('field-format-description').text(gt('Edit field'));
            autoUpdateCheckbox = new Controls.CheckBox({ label: gt('Update automatically'), boxed: true, tooltip: gt('Update this date field automatically on document load, download and mail') });
            setToButton = new Controls.Button({ label: gt('Set to today'), tooltip: /*#. Set to today's date */ gt('Set to today\'s date') }, { inline: true });
            setToButton.getNode().addClass('set-today-btn');
            toggleDatePickerBtn = new Controls.Button({ icon: 'fa fa-calendar', tooltip: /*#. Pick date from calendar */ gt('Pick date') }, { inline: true });
            toggleDatePickerBtn.getNode().addClass('toggle-datepicker-btn');

            fieldFormatPopup = new CompoundMenu(self, {
                classes: 'field-format-popup f6-target',
                autoFocus: true,
                autoClose: false,
                autoLayout: false,
                rootContainerNode: contentRootNode
            });

            fieldFormatPopupNode = fieldFormatPopup.getNode();
            fieldFormatPopupNode.find('.view-component').prepend(descriptionNode);

            fieldFormatPopup
                .addSectionLabel(gt('Select format'))
                .addGroup('document/formatfield', fieldFormatList);

            if (fieldinstruction.type && (/^DATE|^TIME/i).test(fieldinstruction.type)) {
                if (options && options.autoDate) {
                    autoDate = true;
                    autoUpdateCheckbox.setValue(true);
                    setToButton.getNode().addClass('disable');
                    toggleDatePickerBtn.getNode().addClass('disable');
                } else {
                    autoDate = false;
                    autoUpdateCheckbox.setValue(false);
                    setToButton.getNode().removeClass('disable');
                    toggleDatePickerBtn.getNode().removeClass('disable');
                }
                currentFormatOfFieldPopup = fieldinstruction.instruction;

                fieldFormatPopup
                    .addSectionLabel(gt('Change date'))
                    .addGroup(null, autoUpdateCheckbox)
                    .addGroup(null, setToButton)
                    .addGroup('document/toggledatepicker', toggleDatePickerBtn);
                    //.addGroup('removeField', removeFieldButton, { inline: true });
            }

            datePickerNode.datepicker({
                calendarWeeks: false,
                todayBtn: false
            }).hide();

            //fieldFormatPopup.getNode().find('.view-component').prepend(descriptionNode.append(listTitleNode, groupTitleNode)).append(datePickerNode, fieldPopupDateInput, fieldPopupTimeInput);
            fieldFormatPopupNode.find('.view-component').append(datePickerNode);

            // disable inherited contenteditable attribute from page node to prevent text selection
            fieldFormatPopupNode.attr('contenteditable', false);

            $(setToButton.getNode().add(toggleDatePickerBtn.getNode())).wrapAll($('<div>').addClass('today-datepicker-wrapper'));

            // ARIA
            fieldFormatPopupNode.attr({ tabindex: '-1', role: 'dialog', 'aria-label': gt('Field formatting') });

            // manual popup sizing and positioning
            fieldFormatPopup.on('popup:beforelayout', function () {
                var anchorNode = $(getFieldAnchorNode()),
                    anchorPosition,
                    contentNode = this.getContentNode(),
                    contentSize = Utils.getExactNodeSize(contentNode),
                    zoom = app.getView().getZoomFactor() / 100;

                if (!anchorNode.length) {
                    docModel.getFieldManager().destroyFieldPopup();
                    return;
                }
                anchorPosition = Utils.getChildNodePositionInNode(contentRootNode, anchorNode);
                // calculate positioning based on the anchor node position relative to the content root node
                //anchorPosition.left = anchorPosition.left;
                anchorPosition.top += anchorNode.height() * zoom;
                anchorPosition.width = contentSize.width;
                anchorPosition.height = contentSize.height;
                // apply positioning and dimensions
                this.getNode().css(anchorPosition);
            });

            fieldFormatPopup.on('popup:hide', function () {
                docModel.getSelection().restoreBrowserSelection();
            });

            datePickerNode.on('changeDate', function () {
                var date = datePickerNode.data().datepicker.viewDate;
                var pickedDate = self.formatDateWithNumberFormatter(date, currentFormatOfFieldPopup);
                var standardizedDate = self.formatDateWithNumberFormatter(date, 'yyyy-mm-dd\Thh:mm:ss.00');

                currentCxNode.data('datepickerValue', datePickerNode.data().datepicker.getDate());
                self.trigger('fielddatepopup:change', { value: pickedDate, standard: standardizedDate });
            });

            autoUpdateCheckbox.on('group:change', function () {
                self.autoFieldCheckboxUpdate();
            });

            setToButton.on('group:change', function () {
                self.setFieldToTodayDate();
            });
        }

        /**
         * Finding the first paragraph that is inside the visible area
         */
        function setFirstVisibleParagraph() {

            var // all paragraph elements in the document
                allParagraphs = docModel.getNode().find(DOM.PARAGRAPH_NODE_SELECTOR),
                // top offset for the scroll node
                topOffset = Utils.round(contentRootNode.offset().top, 1);

            // iterating over all collected change tracked nodes
            Utils.iterateArray(allParagraphs, function (paragraph) {

                if (Utils.round($(paragraph).offset().top, 1) > topOffset) {
                    topVisibleParagraph = paragraph;
                    return Utils.BREAK;
                }

            });
        }

        /**
         * Handles the 'contextmenu' event for the complete browser.
         */
        function globalContextMenuHandler(event) {
            // quit if there were no contextmenu event
            if (_.isUndefined(event)) { return false; }

            // quit if the contextmenu event was triggered on an input-/textfield
            if ($(event.target).is('input, textarea')) { return; }

            var // was the event triggered by keyboard?
                isGlobalTrigger = (_.browser.IE) ? $(event.target).is('.clipboard') : $(event.target).is('html'),
                // node, on which the new event should be triggered
                triggerNode     = null,

                // search inside this node for the allowed targets
                windowNode      = app.getWindowNode(),
                // parent elements on which 'contextmenu' is allowed (inside the app node)
                winTargetAreas  = ['.commentlayer', '.page', '.textdrawinglayer', '.popup-content'],
                // global target areas
                globalTargetAreas  = ['.popup-container > .popup-content'],
                // was one of these targets clicked
                targetClicked   = false;

            // check if the event target is inside of one of the allowed containers
            _.each(winTargetAreas, function (selector) {
                var parentNode = windowNode.find(selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });
            _.each(globalTargetAreas, function (selector) {
                var parentNode = $('body > ' + selector);

                if (parentNode.length > 0) {
                    if (Utils.containsNode(parentNode, event.target) === true || Utils.isElementNode(event.target, selector)) {
                        targetClicked = true;
                    }
                }
            });

            // get out here, if no allowed target was clicked, or the globla trigger
            // was fired (via keyboard)
            if (!targetClicked && !isGlobalTrigger) { return false; }

            // if a drawing is selected
            if (docModel.isDrawingSelected()) {
                if ($(event.target).is('.tracker')) { return false; }
                // get the (first) selected drawing as trigger node
                triggerNode = $(docModel.getSelection().getSelectedDrawing()[0]);

            } else {
                var windowSelection = window.getSelection();

                // get the current focus node to trigger on it
                _.each($('.popup-content'), function (popupContentNode) {
                    if (Utils.containsNode(popupContentNode, event.target)) { triggerNode = $(event.target); }
                });

                if (_.isNull(triggerNode) && windowSelection.type !== 'None') {
                    triggerNode = $(window.getSelection().focusNode.parentNode);
                } else {
                    triggerNode = $(event.target);
                }

                if (_.isNull(triggerNode)) {
                    Utils.warn('TextView.globalContextMenuHandler(): No "triggerNode" found. Contextmenu will not be open!');
                    return;
                }
            }

            // if we have a last known touch position, use it
            if (touchX && touchY) {
                event.pageX = touchX;
                event.pageY = touchY;

            // in case of global trigger (via keyboard), or x/y are both "0"
            // locate a meaningful position
            } else if (isGlobalTrigger || (event.pageX === 0 && event.pageY === 0) || (_.isUndefined(event.pageX) && _.isUndefined(event.pageY))) {
                // find the position of the trigger node
                var xy = Position.getPixelPositionToRootNodeOffset($('html'), triggerNode);
                // and set pageX/pageY manually
                event.pageX = (xy.x + (triggerNode.width() / 2));
                event.pageY = (xy.y + (triggerNode.height() / 2));
            }
            // close eventually open field popup, see #42478
            docModel.getFieldManager().destroyFieldPopup();

            // trigger our own contextmennu event on the found trigger node
            triggerNode.trigger(new $.Event('documents:contextmenu', { sourceEvent: event }));

            return false;
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // deferred and debounced scroll handler, registering insertText operations immediately
                scrollToSelectionDebounced = self.createDebouncedMethod(registerInsertText, scrollToSelection, { delay: 50, maxDelay: 500, infoString: 'Text: View: scrollToSelection' });

            // register the global event handler for 'contextmenu' event
            if (Utils.COMPACT_DEVICE) {
                app.registerGlobalEventHandler($(document), 'taphold', globalContextMenuHandler);
                app.registerGlobalEventHandler($(document), 'contextmenu', function () { return false; });
            } else {
                app.registerGlobalEventHandler($(document), 'contextmenu', globalContextMenuHandler);
            }

            // to save the position from 'touchstart', we have to register some global eventhandler
            // we need this position to open the contextmenu on the correct location
            if (Utils.IOS || _.browser.Android) {
                app.registerGlobalEventHandler($(document), 'touchstart', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = event.originalEvent.changedTouches[0].pageX;
                    touchY = event.originalEvent.changedTouches[0].pageY;
                });
                app.registerGlobalEventHandler($(document), 'touchend', function (event) {
                    if (_.isUndefined(event)) { return false; }
                    touchX = touchY = null;
                });
            }

            // initialize other instance fields
            appPaneNode = self.getAppPaneNode();
            contentRootNode = self.getContentRootNode();

            // insert the editor root node into the application pane
            self.insertContentNode(docModel.getNode());

            // Page node, used for zooming
            pageNode = docModel.getNode();

            //page content node, used for zooming on small devices
            pageContentNode = $(pageNode).children('div.pagecontent');

            // handle scroll position when hiding/showing the entire application
            self.listenTo(app.getWindow(), {
                beforehide: function () {
                    saveScrollPosition();
                    appPaneNode.css('opacity', 0);
                },
                show: function () {
                    restoreScrollPosition();
                    // restore visibility deferred, needed to restore IE scroll position first
                    self.executeDelayed(function () { appPaneNode.css('opacity', 1); }, undefined, 'Text: View: Setting opacity');
                }
            });

            // disable drag&drop for the content root node to prevent dropping images which are
            // loaded by the browser
            self.listenTo(self.getContentRootNode(), 'drop dragstart dragenter dragexit dragover dragleave', false);

            // Registering scroll handler
            self.listenTo(self.getContentRootNode(), 'scroll', function () {
                // handler for updating change track markers in side bar after scrolling
                // -> the handler contains only functionality, if the document contains change tracks
                docModel.updateChangeTracksDebouncedScroll();
            });

            // handle selection change events
            self.listenTo(docModel, 'selection', function (event, selection, options) {
                // disable internal browser edit handles (browser may re-enable them at any time)
                self.disableBrowserEditControls();
                // scroll to text cursor (but not, if only a remote selection was updated (39904))
                if (!Utils.getBooleanOption(options, 'updatingRemoteSelection', false)) { scrollToSelectionDebounced(options); }

                var userData = {
                    selection: {
                        type: selection.getSelectionType(),
                        start: selection.getStartPosition(),
                        end: selection.getEndPosition()
                    },
                    target: docModel.getActiveTarget() || undefined
                };
                if (docModel.isHeaderFooterEditState()) {
                    userData.targetIndex = docModel.getPageLayout().getTargetsIndexInDocument(docModel.getCurrentRootNode(), docModel.getActiveTarget());
                }
                // send user data to server
                app.updateUserData(userData);
                // manage change track popups, but do not update, if 'keepChangeTrackPopup' is set to true
                if (!Utils.getBooleanOption(options, 'keepChangeTrackPopup', false)) {
                    handleChangeTrackPopup(options);
                }
            });

            // process page break event triggered by the model
            self.listenTo(docModel, 'pageBreak:before', function () {
                // scrolled downwards -> restore scroll position after inserting page breaks
                if (contentRootNode.scrollTop() > 0) { setFirstVisibleParagraph(); }
            });

            // process page break event triggered by the model
            self.listenTo(docModel, 'pageBreak:after', function () {
                var selection = docModel.getSelection();
                if (!selection.isUndefinedSelection() && !selection.isTopPositionSelected()) {
                    // scrolling to an existing selection made by the user.
                    // 'topVisibleParagraph' should be used also in this case (user makes selection and then scrolls away
                    // without making further selection), but scrolling to 'topVisibleParagraph' does not work reliable,
                    // if there is a selection set.
                    scrollToSelection();
                } else if (topVisibleParagraph) {
                    // scrolling to the scroll position set by the user (if it was saved in 'pageBreak:before')
                    self.scrollToChildNode(topVisibleParagraph, { forceToTop: true });
                }
                topVisibleParagraph = null;
            });

            // store the values of page width and height with paddings, after loading is finished
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();
            if (Utils.SMALL_DEVICE) {
                contentRootNode.addClass('draft-mode');
            }

            // initialize change track popup
            initChangeTrackPopup();
        }

        /**
         * Additional debug initialization after construction.
         */
        function initDebugHandler(operationsPane, clipboardPane) {

            var // pending operations
                pendingOperations = [],
                // delayed recording operations active
                delayedRecordingActive = false,
                // checking the operation length, so that operations removed from the
                // 'optimizeOperationHandler' will not collected twice in the recorder.
                oplCounter = 1;

            // collect operations
            function collectOperations(event, operations, external) {
                if (docModel.isRecordingOperations()) {
                    pendingOperations = pendingOperations.concat(_(operations).map(function (operation) {
                        return { operation: operation, external: external };
                    }));

                    if (!delayedRecordingActive) {
                        recordOperations();
                    }
                }
            }

            // record operations
            function recordOperations() {

                if (delayedRecordingActive) { return; }

                delayedRecordingActive = true;
                self.executeDelayed(function () {
                    var // full JSON string
                        fullJSONString = replayOperations.getFieldValue(),
                        // json part string to be appended
                        jsonPartString = '',
                        // current part to be processed
                        operations = pendingOperations.splice(0, 20),
                        // first entry
                        firstEntry = (fullJSONString.length === 0);

                    _(operations).each(function (entry) {

                        // skipping operations, that were 'optimized' in 'sendActions' in the
                        // optimizeOperationHandler. Otherwise 'merged operations' are still
                        // followed by the operations, that will not be sent to
                        // the server, because they were removed by the 'optimizeOperationHandler'.
                        var skipOperation = false,
                            opl = ('opl' in entry.operation) ? entry.operation.opl : '';

                        // checking if this operation was merged by the 'optimizeOperationHandler'. This can
                        // only happen for internal operations (task 29601)
                        if (!entry.external) {
                            if (_.isNumber(opl) && (opl > 1)) {
                                oplCounter = opl;
                            } else if (oplCounter > 1) {
                                oplCounter--;
                                skipOperation = true;
                            }
                        }

                        if (!skipOperation) {
                            if (!firstEntry) {
                                jsonPartString += ',';
                            }
                            jsonPartString += JSON.stringify(entry.operation);
                            firstEntry = false;
                        }
                    });

                    if (fullJSONString.length >= 2) {
                        if (fullJSONString.charAt(0) === '[') {
                            fullJSONString = fullJSONString.substr(1, fullJSONString.length - 2);
                        }
                    }
                    fullJSONString = '[' + fullJSONString + jsonPartString + ']';
                    replayOperations.setValue(fullJSONString);

                    delayedRecordingActive = false;
                    if (pendingOperations.length > 0) {
                        recordOperations();
                    }
                }, 50, 'Text: View: Record operations');
            }

            // create the output nodes in the operations pane
            operationsPane
                .addDebugInfoHeader('selection', 'Current selection')
                .addDebugInfoNode('selection', 'type', 'Type of the selection')
                .addDebugInfoNode('selection', 'start', 'Start position')
                .addDebugInfoNode('selection', 'end', 'End position')
                .addDebugInfoNode('selection', 'target', 'Target position')
                .addDebugInfoNode('selection', 'dir', 'Direction')
                .addDebugInfoNode('selection', 'attrs1', 'Explicit attributes of cursor position')
                .addDebugInfoNode('selection', 'attrs2', 'Explicit attributes (parent level)')
                .addDebugInfoNode('selection', 'style', 'Paragraph style');

            // log all selection events, and new formatting after operations
            self.listenTo(docModel, 'selection operations:success', function () {

                function debugAttrs(position, ignoreDepth, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    position = _.initial(position, ignoreDepth);
                    var element = (position && position.length) ? Position.getDOMPosition(docModel.getCurrentRootNode(), position, true) : null;
                    if (element && element.node) {
                        var attrs = AttributeUtils.getExplicitAttributes(element.node);
                        markup = '<span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>' + Utils.escapeHTML(Utils.stringifyForDebug(attrs)) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                function debugStyle(family, target) {
                    if (!operationsPane.isVisible()) { return; }

                    var markup = '';

                    var styleId = docModel.getAttributes(family).styleId;
                    if (styleId) {
                        var attrs = docModel.getStyleCollection(family).getStyleAttributeSet(styleId);
                        markup = '<span>family=' + family + '</span> <span ' + AttributesToolTip.createAttributeMarkup(attrs) + '>id=' + Utils.escapeHTML(styleId) + '</span>';
                    }

                    operationsPane.setDebugInfoMarkup('selection', target, markup);
                }

                var selection = docModel.getSelection();
                operationsPane
                    .setDebugInfoText('selection', 'type', selection.getSelectionType())
                    .setDebugInfoText('selection', 'start', selection.getStartPosition().join(', '))
                    .setDebugInfoText('selection', 'end', selection.getEndPosition().join(', '))
                    .setDebugInfoText('selection', 'target', selection.getRootNodeTarget())
                    .setDebugInfoText('selection', 'dir', selection.isTextCursor() ? 'cursor' : selection.isBackwards() ? 'backwards' : 'forwards');

                debugAttrs(selection.getStartPosition(), 0, 'attrs1');
                debugAttrs(selection.getStartPosition(), 1, 'attrs2');
                debugStyle('paragraph', 'style');
            });

            // operations to replay
            replayOperations = new Controls.TextField({ tooltip: _.noI18n('Paste operations here'), select: true });

            // operations to replay
            replayOperationsOSN = new Controls.TextField({ tooltip: _.noI18n('Define OSN for last executed operation (optional)'), width: 50 });

            // record operations after importing the document
            self.waitForImport(function () {
                self.listenTo(docModel, 'operations:after', collectOperations);
            });

            // process debug events triggered by the model
            self.listenTo(docModel, 'debug:clipboard', function (event, content) {
                clipboardPane.setClipboardContent(content);
            });
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function initGuiHandler(viewMenuGroup) {

            var // ODF does not support inner borders at paragraphs -> special option only for this case
                displayInnerBorders = !app.isODF(),

                fontToolBar             = null,
                fontStyleToolBar        = null,
                fontColorToolBar        = null,
                alignmentToolBar        = null,
                fillBorderToolBar       = null,
                paragraphStylesToolBar  = null,
                listSettingsToolBar     = null,
                insertToolBar           = null,
                tableToolBar            = null,
                tableBorderToolBar      = null,
                tableStyleToolBar       = null,
                drawingToolBar          = null,
                drawingAnchorToolBar    = null,
                drawingFillToolBar      = null,
                drawingLineToolBar      = null,
                reviewToolBar           = null,
                changeTrackingToolBar   = null,
                commentToolBar          = null,

                btnInsertTextframe      = null,
                btnInsertComment        = null,

                pickFontFamily          = new Controls.FontFamilyPicker(self),
                pickFontSize            = new Controls.FontSizePicker(),
                pickLineHeight          = new Controls.LineHeightPicker(),
                pickParagraphSpacing    = new Controls.ParagraphSpacingPicker(),
                pickBulletListStyle     = new Controls.BulletListStylePicker(self, docModel),
                pickNumberedListStyle   = new Controls.NumberedListStylePicker(self, docModel),
                pickTableSize           = new Controls.TableSizePicker({ label: /*#. a table in a text document */ gt.pgettext('text-doc', 'Table'), tooltip: /*#. insert a table in a text document */ gt.pgettext('text-doc', 'Insert a table'), maxCols: Config.MAX_TABLE_COLUMNS, maxRows: Config.MAX_TABLE_ROWS, smallerVersion: { css: { width: 50 } } }),
                pickLanguage            = new Controls.LanguagePicker({ width: 200 }),
                pickFillColorParagraph  = new Controls.FillColorPicker(self, { icon: 'docs-para-fill-color', tooltip: gt('Paragraph fill color'), title: gt('Paragraph fill color'), dropDownVersion: { label: gt('Paragraph fill color') } }),
                pickBorderModeParagraph = new Controls.BorderModePicker({ tooltip: gt('Paragraph borders'), showInsideHor: displayInnerBorders, dropDownVersion: { label: gt('Paragraph borders') } }),
                pickTextColor           = new Controls.TextColorPicker(self),
                pickTextFillColor       = new Controls.FillColorPicker(self, { icon: 'docs-font-fill-color', tooltip: /*#. fill color behind single characters (instead of entire paragraph/cell) */ gt('Text highlight color'), title: gt('Text highlight color') }),
                insertField             = new Controls.InsertFieldPicker(self),

                btnBold                 = new Controls.Button(Labels.BOLD_BUTTON_OPTIONS),
                btnItalic               = new Controls.Button(Labels.ITALIC_BUTTON_OPTIONS),
                btnUnderline            = new Controls.Button(Labels.UNDERLINE_BUTTON_OPTIONS),
                btnStrike               = new Controls.Button(Labels.STRIKEOUT_BUTTON_OPTIONS),
                btnReset                = new Controls.Button(Labels.CLEAR_FORMAT_BUTTON_OPTIONS),
                btnDecindent            = new Controls.Button({ icon: 'docs-list-dec-level', tooltip: /*#. indentation of lists (one list level up) */ gt('Demote one level'), dropDownVersion: { label: /*#. indentation of lists (one list level up) */ gt('Demote one level') } }),
                btnIncindent            = new Controls.Button({ icon: 'docs-list-inc-level', tooltip: /*#. indentation of lists (one list level down) */ gt('Promote one level'), dropDownVersion: { label: /*#. indentation of lists (one list level down) */ gt('Promote one level') } }),
                btnInsertImage          = new Controls.ImagePicker(self),
                btnInsertHyperlink      = new Controls.Button(Labels.HYPERLINK_BUTTON_OPTIONS),
                btnInsertRow            = new Controls.Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }),
                btnInsertColumn         = new Controls.Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }),
                btnDeleteRow            = new Controls.Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }),
                btnDeleteColumn         = new Controls.Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }),
                btnDeleteDrawing        = new Controls.Button(Labels.DELETE_DRAWING_BUTTON_OPTIONS),
                btnSpelling             = new Controls.Button({ icon: 'docs-online-spelling', tooltip: gt('Check spelling permanently'), toggle: true, dropDownVersion: { label: gt('Check spelling') } }),
                btnInsertCommentReview  = new Controls.Button({ icon: 'docs-comment-add', label: gt('Comment'), tooltip: /*#. insert a comment into the text */ gt('Insert a comment'), smallerVersion: { hideLabel: true } }),
                btnDeleteAllComments    = new Controls.Button({ icon: 'docs-comment-remove', label: gt('Delete all'), tooltip: /*#. delete all comments in the text */ gt('Delete all comments'), smallerVersion: { hideLabel: true } }),
                btnSelectPrevComment    = new Controls.Button({ label: /*#. select the previous comment */ gt('Previous'), icon: 'docs-comment-back', tooltip: gt('Select the previous comment in the document'), smallerVersion: { hideLabel: true } }),
                btnSelectNextComment    = new Controls.Button({ label: /*#. select the next comment */ gt('Next'), icon: 'docs-comment-next', tooltip: gt('Select the next comment in the document'), smallerVersion: { hideLabel: true } }),

                grpVertAlign            = new Controls.TextPositionGroup(),

                alignmentList           = new Controls.ParagraphAlignmentPicker({ dropDownVersion: { visible: true } }),
                alignmentGroup          = new Controls.ParagraphAlignmentGroup({ smallerVersion: { pendant: alignmentList } });

            // -----------------------------------------------------
            // TABS
            //      prepare all tabs (for normal or combined panes)
            // -----------------------------------------------------
            if (!self.panesCombined()) {
                self.createToolBarTab('format',  { label: Labels.FORMAT_HEADER_LABEL, visibleKey: 'document/editable/text',       priority: 1 });
                self.createToolBarTab('insert',  { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/text',       priority: 1 });
                self.createToolBarTab('table',   { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/show/tabletab' });
                self.createToolBarTab('drawing', { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawing', priority: 2 });
                self.createToolBarTab('review',  { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'document/editable',            priority: 1, sticky: true });

            } else {
                self.createToolBarTab('format',    { label: Labels.FONT_HEADER_LABEL,   visibleKey: 'document/editable/text' });
                self.createToolBarTab('paragraph', { label: Labels.PARAGRAPH_LABEL,     visibleKey: 'document/editable/text' });
                self.createToolBarTab('insert',    { label: Labels.INSERT_HEADER_LABEL, visibleKey: 'document/editable/text' });
                self.createToolBarTab('table',     { label: Labels.TABLE_HEADER_LABEL,  visibleKey: 'document/show/tabletab' });
                self.createToolBarTab('drawing',   { labelKey: 'drawing/type/label',    visibleKey: 'document/editable/anydrawing', priority: 2 });
                if (Config.SPELLING_ENABLED) {
                    self.createToolBarTab('review',    { label: Labels.REVIEW_HEADER_LABEL, visibleKey: 'document/editable',            priority: 3 });
                }
                if (!app.isODF()) {
                    self.createToolBarTab('changetracking', { label: Labels.CHANGETRACKING_HEADER_LABEL, visibleKey: 'document/editable', sticky: true });
                }
                self.createToolBarTab('comment', { label: gt.pgettext('menu-title', 'Comments'), visibleKey: 'document/editable' });
            }

            // -----------------------------------------------------
            // TOOLBARS
            //      prepare all toolbars (for normal or combined panes)
            // -----------------------------------------------------
            fontToolBar                 = self.createToolBar('format', { priority: 1, prepareShrink: true, icon: 'fa-font', tooltip: Labels.FONT_HEADER_LABEL });
            fontStyleToolBar            = self.createToolBar('format', { priority: 2, prepareShrink: true, icon: 'docs-font-bold', caretTooltip: gt('More font styles'), splitBtn: { key: 'character/bold', options: Labels.BOLD_BUTTON_OPTIONS } });
            fontColorToolBar            = self.createToolBar('format', { priority: 3, prepareShrink: true, icon: 'docs-color', tooltip: gt('Colors') });

            if (!self.panesCombined()) {
                alignmentToolBar        = self.createToolBar('format', { priority: 4 });
                fillBorderToolBar       = self.createToolBar('format', { priority: 4, hideable: true, prepareShrink: true, icon: 'docs-border-h', tooltip: gt('Paragraph formatting') });
                paragraphStylesToolBar  = self.createToolBar('format', { priority: 5, hideable: true, icon: 'fa-paragraph', tooltip: Labels.PARAGRAPH_STYLES_LABEL });
                listSettingsToolBar     = self.createToolBar('format', { priority: 5, prepareShrink: true, icon: 'docs-lists', tooltip: Labels.LIST_SETTINGS_LABEL });
            } else {
                alignmentToolBar        = self.createToolBar('paragraph', { priority: 1 });
                fillBorderToolBar       = self.createToolBar('paragraph', { priority: 2, prepareShrink: true, icon: 'docs-border-h', tooltip: gt('Paragraph formatting') });
                listSettingsToolBar     = self.createToolBar('paragraph', { priority: 3, prepareShrink: true, icon: 'docs-lists', tooltip: Labels.LIST_SETTINGS_LABEL });
            }

            insertToolBar               = self.createToolBar('insert');

            tableToolBar                = self.createToolBar('table');
            tableBorderToolBar          = self.createToolBar('table', { priority: 1, hideable: true, classes: 'noSeparator' });
            tableStyleToolBar           = self.createToolBar('table', { priority: 2, hideable: true });

            drawingToolBar              = self.createToolBar('drawing');

            if (!self.panesCombined()) {
                drawingAnchorToolBar    = self.createToolBar('drawing', { visibleKey: 'document/editable/anydrawing' });
                drawingLineToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/line' });
                drawingFillToolBar      = self.createToolBar('drawing', { visibleKey: 'drawing/shapes/support/fill', hideable: true });
            }

            if (!self.panesCombined()) {
                reviewToolBar           = self.createToolBar('review', { visibleKey: 'document/spelling/available', icon: 'docs-online-spelling', caretTooltip: /*#. tooltip for a drop-down menu with functions for reviewing the document (spelling, change tracking) */ gt('More review actions'), splitBtn: { key: 'document/onlinespelling', options: { toggle: true, tooltip: gt('Check spelling') } }, priority: 2, prepareShrink: true });
                changeTrackingToolBar   = self.createToolBar('review');
            } else {
                if (Config.SPELLING_ENABLED) {
                    reviewToolBar           = self.createToolBar('review', { visibleKey: 'document/spelling/available', icon: 'docs-online-spelling', caretTooltip: /*#. tooltip for a drop-down menu with functions for reviewing the document (spelling, change tracking) */ gt('More review actions'), splitBtn: { key: 'document/onlinespelling', options: { toggle: true, tooltip: gt('Check spelling') } } });
                }
                changeTrackingToolBar   = self.createToolBar('changetracking');
                commentToolBar          = self.createToolBar('comment');
            }

            // -----------------------------------------------------
            // CONTROLS
            //      add all controls
            // -----------------------------------------------------

            // FONT ------------------------------------------

            fontToolBar
                .addGroup('character/fontname', pickFontFamily)
                .addGap()
                .addGroup('character/fontsize', pickFontSize);

            fontStyleToolBar
                .addGroup('character/bold', btnBold)
                .addGroup('character/italic', btnItalic)
                .addGroup('character/underline', btnUnderline)
                .addSeparator({ classes: 'noVerticalSeparator' })
                .addGroup('character/format',
                    new Controls.CompoundButton(self, {
                        icon: 'docs-font-format',
                        tooltip: gt('More font styles'),
                        dropDownVersion: { visible: false }
                    })
                    .addGroup('character/strike', btnStrike)
                    .addSeparator()
                    .addGroup('character/vertalign', grpVertAlign)
                    .addSeparator()
                    .addGroup('character/reset', btnReset)
                )
                .addGroup('character/strike', btnStrike.clone({ dropDownVersion: { visible: true } }))
                .addSeparator({ classes: 'hidden' })
                .addGroup('character/vertalign', grpVertAlign.clone({ dropDownVersion: { visible: true } }))
                .addSeparator({ classes: 'hidden' })
                .addGroup('character/reset', btnReset.clone({ dropDownVersion: { visible: true } }));

            fontColorToolBar
                .addGroup('character/color',     pickTextColor)
                .addGroup('character/fillcolor', pickTextFillColor);

            // PARAGRAPH ------------------------------------------

            alignmentToolBar
                .addGroup('paragraph/alignment', alignmentGroup)
                .addGroup('paragraph/alignment', alignmentList);

            fillBorderToolBar
                .addGroup(null,
                    new Controls.CompoundButton(self, {
                        icon: 'docs-para-line-spacing-100',
                        tooltip: Labels.PARAGRAPH_AND_LINE_SPACING_LABEL,
                        dropDownVersion: { label: Labels.PARAGRAPH_AND_LINE_SPACING_LABEL }
                    })
                    .addSectionLabel(Labels.LINE_SPACING_LABEL)
                    .addGroup('paragraph/lineheight', pickLineHeight)

                    .addSectionLabel(Labels.PARAGRAPH_SPACING_LABEL)
                    .addGroup('paragraph/spacing', pickParagraphSpacing)
                )
                .addGap()
                .addGroup('paragraph/fillcolor', pickFillColorParagraph)
                .addGap()
                .addGroup('paragraph/borders', pickBorderModeParagraph);

            if (!self.panesCombined()) {
                paragraphStylesToolBar.addGroup('paragraph/stylesheet', new Controls.ParagraphStylePicker(self));
            }

            listSettingsToolBar
                .addGroup('paragraph/list/bullet', pickBulletListStyle)
                .addGap()
                .addGroup('paragraph/list/numbered', pickNumberedListStyle)
                .addGap()
                .addGroup('paragraph/list/decindent', btnDecindent)
                .addGroup('paragraph/list/incindent', btnIncindent);

            // INSERT ------------------------------------------

            insertToolBar
                .addGroup('table/insert', pickTableSize, { visibleKey: 'table/insert/available' })
                .addSeparator()
                .addGroup('image/insert/dialog', btnInsertImage)
                .addSeparator();

            if (!self.panesCombined()) {
                btnInsertTextframe = new Controls.Button(Labels.INSERT_TEXTFRAME_BUTTON_OPTIONS);

                insertToolBar
                    .addGroup('textframe/insert', btnInsertTextframe)
                    .addSeparator();
            }

            btnInsertComment   = new Controls.Button(_.extend({ icon: 'docs-comment-add', smallerVersion: { hideLabel: true } }, Labels.INSERT_COMMENT_OPTIONS));
            insertToolBar
                .addGroup('comment/insert', btnInsertComment)
                .addSeparator();

            insertToolBar.addGroup('character/hyperlink/dialog', btnInsertHyperlink);

            if (!self.panesCombined()) {
                var btnInsertTab         = new Controls.Button({ icon: 'docs-insert-tab', label: gt('Tab stop'), tooltip: /*#. insert a horizontal tab stop into the text */ gt('Insert tab stop'), smallerVersion: { hideLabel: true } }),
                    btnInsertBreak       = new Controls.Button({ icon: 'docs-insert-linebreak', label: gt('Line break'),  tooltip: /*#. insert a manual line break into the text */ gt('Insert line break'), smallerVersion: { hideLabel: true } }),
                    btnInsertPagebreak   = new Controls.Button({ icon: 'docs-page-break', label: gt('Page break'),  tooltip: /*#. insert a manual page break into the text */ gt('Insert page break'), smallerVersion: { hideLabel: true } }),

                    btnSameInDocumentHF  = { label: /*#. header and footer: same across entire document */gt('Same across entire document') },
                    btnDifferentFirstHF  = { label: /*#. header and footer: different first page */gt('Different first page') },
                    btnDiferentEvenOddHF = { label: /*#. header and footer: different even and odd pages */gt('Different even & odd pages') },
                    btnDifferentAllHF    = { label: /*#. header and footer: different first, even and odd pages */gt('Different first, even & odd pages') },
                    btnRemoveHF          = new Controls.Button({ label: /*#. header and footer: remove all headers and footers from document*/gt('Remove all headers and footers') });

                insertToolBar
                    .addGroup('character/insert/tab',   btnInsertTab)
                    .addGroup('character/insert/break', btnInsertBreak)
                    .addGroup('character/insert/pagebreak', btnInsertPagebreak)
                    .addGroup('document/insert/headerfooter',
                        new Controls.CompoundSplitButton(self, {
                            icon: 'docs-header-footer',
                            label: /*#. insert header and footer in document */ gt('Header & footer'),
                            tooltip: gt('Insert header & footer in document'),
                            smallerVersion: { hideLabel: true },
                            splitValue: 'goto',
                            updateCaptionMode: 'none'
                        })
                        .addGroup('document/insert/headerfooter',  new Controls.RadioGroup()
                            .createOptionButton('default',  btnSameInDocumentHF)
                            .createOptionButton('first',    btnDifferentFirstHF)
                            .createOptionButton('evenodd',  btnDiferentEvenOddHF)
                            .createOptionButton('all',      btnDifferentAllHF)
                        )
                        .addSeparator()
                        .addGroup('document/insert/headerfooter/remove', btnRemoveHF)
                    )
                    .addGroup('document/insertfield', insertField);
            }

            // TABLE ---------------------------------

            tableToolBar
                .addGroup('table/insert/row', btnInsertRow)
                .addGroup('table/delete/row', btnDeleteRow)
                .addGap()
                .addGroup('table/insert/column', btnInsertColumn)
                .addGroup('table/delete/column', btnDeleteColumn)
                .addGap();

            if (!app.isODF() && !self.panesCombined()) {

                var btnTableSplit       = new Controls.Button({ icon: 'docs-table-split', tooltip: /*#. split current table */ gt('Split table') }),
                    pickFillColorTable  = new Controls.FillColorPicker(self, { icon: 'docs-table-fill-color', tooltip: gt('Cell fill color') }),
                    pickBorderModeTable = new Controls.BorderModePicker({ tooltip: Labels.CELL_BORDERS_LABEL, showInsideHor: true, showInsideVert: true }),
                    pickBorderWidth     = new Controls.BorderWidthPicker({ tooltip: gt('Cell border width') }),
                    pickTableStyle      = new Controls.TableStylePicker(self);

                tableToolBar
                    .addGroup('table/split', btnTableSplit)
                    .addSeparator()
                    .addGroup('table/fillcolor', pickFillColorTable)
                    .addGap()
                    .addGroup('table/cellborder', pickBorderModeTable);
                tableBorderToolBar
                    .addGroup('table/borderwidth', pickBorderWidth);
                tableStyleToolBar
                    .addGroup('table/stylesheet', pickTableStyle);
            }

            // DRAWING ------------------------------------------

            drawingToolBar
                .addGroup('drawing/delete', btnDeleteDrawing);

            if (!self.panesCombined()) {

                var pickAnchorage           = new Controls.AnchoragePicker(),
                    drawingOrder            = new Controls.DrawingOrder(self, { icon: 'docs-z-order' }),
                    pickDrawingPosition     = new Controls.DrawingPositionPicker(),
                    cBoxToggleAutoFit       = new Controls.CheckBox({ label: gt('Autofit'), boxed: true, tooltip: gt('Turn automatic height resizing of text frames on or off') }),
                    pickBorderPresetStyle   = new Controls.BorderPresetStylePicker(app.isODF() ? Labels.BORDER_ODF_PRESET_STYLES : Labels.BORDER_OOXML_PRESET_STYLES),
                    pickBorderColor         = new Controls.BorderColorPicker(self, { label: gt('Border color'), smallerVersion: { hideLabel: true } }),
                    pickFillColorDrawing    = new Controls.FillColorPicker(self, { label: gt('Background color'),  icon: 'docs-drawing-fill-color', smallerVersion: { hideLabel: true } });

                drawingAnchorToolBar
                    .addGap()
                    .addGroup('drawing/anchorTo', pickAnchorage)
                    .addGroup('drawing/position', pickDrawingPosition)
                    .addGroup('debug/textframeautofit', cBoxToggleAutoFit)
                    .addGroup('drawing/order', drawingOrder);

                drawingLineToolBar
                    .addGroup('drawing/border/style', pickBorderPresetStyle)
                    .addGroup('drawing/border/color', pickBorderColor);

                drawingFillToolBar
                    .addGroup('drawing/fill/color', pickFillColorDrawing);
            }

            // REVIEW ------------------------------------------

            if (Config.SPELLING_ENABLED) {
                reviewToolBar
                    .addGroup('document/onlinespelling', btnSpelling)
                    .addSeparator()
                    .addGroup('character/language', pickLanguage);
            }

            if (!app.isODF()) {

                var cBoxToggleCT        = new Controls.CheckBox(_.extend({ boxed: true }, Labels.TRACK_CHANGES_OPTIONS)),
                    btnSelectPrevCT     = new Controls.Button({ label: /*#. change tracking: select the previous change */ gt('Previous'), icon: 'fa-chevron-left', tooltip: gt('Select the previous change in the document'), smallerVersion: { hideLabel: true } }),
                    btnSelectNextCT     = new Controls.Button({ label: /*#. change tracking: select the next change */ gt('Next'), icon: 'fa-chevron-right', tooltip: gt('Select the next change in the document'), smallerVersion: { hideLabel: true } });

                if (!self.panesCombined()) {

                    var btnAcceptMoveSelectCT   = new Controls.Button(Labels.ACCEPT_AND_NEXT_OPTIONS),
                        btnAcceptSelectCT       = new Controls.Button(Labels.ACCEPT_CURRENT_OPTIONS),
                        btnAcceptCT             = new Controls.Button(Labels.ACCEPT_ALL_OPTIONS),
                        btnRejectMoveSelectCT   = new Controls.Button(Labels.REJECT_AND_NEXT_OPTIONS),
                        btnRejectSelectCT       = new Controls.Button(Labels.REJECT_CURRENT_OPTIONS),
                        btnRejectCT             = new Controls.Button(Labels.REJECT_ALL_OPTIONS);

                    changeTrackingToolBar
                        .addGroup('toggleChangeTracking', cBoxToggleCT)
                        .addSeparator()
                        .addGroup('acceptMoveSelectedChangeTracking',       new Controls.CompoundSplitButton(self, { icon: 'fa-check', label: /*#. change tracking function: accept the current document change */ gt('Accept'), tooltip: gt('Accept the current change and select the next change'), smallerVersion: { hideLabel: true } })
                            .addGroup('acceptMoveSelectedChangeTracking',   btnAcceptMoveSelectCT)
                            .addGroup('acceptSelectedChangeTracking',       btnAcceptSelectCT)
                            .addGroup('acceptChangeTracking',               btnAcceptCT)
                        )
                        .addGroup('rejectMoveSelectedChangeTracking',       new Controls.CompoundSplitButton(self, { icon: 'fa-times', label: /*#. change tracking function: reject the current document change */ gt('Reject'), tooltip: gt('Reject the current change and select the next change'), smallerVersion: { hideLabel: true } })
                            .addGroup('rejectMoveSelectedChangeTracking',   btnRejectMoveSelectCT)
                            .addGroup('rejectSelectedChangeTracking',       btnRejectSelectCT)
                            .addGroup('rejectChangeTracking',               btnRejectCT)
                        )
                        .addGroup('selectPrevChangeTracking',               btnSelectPrevCT)
                        .addGroup('selectNextChangeTracking',               btnSelectNextCT);
                } else {
                    changeTrackingToolBar
                        .addGroup('toggleChangeTracking',           cBoxToggleCT.clone({ label: /*#. change tracking: switch change tracking on/off */ gt('On') }))
                        .addSeparator()
                        .addGroup('acceptSelectedChangeTracking',   new Controls.Button({ icon: 'fa-check', label: gt('Accept'), tooltip: gt('Accept the current change'), smallerVersion: { hideLabel: true } }))
                        .addGroup('rejectSelectedChangeTracking',   new Controls.Button({ icon: 'fa-times', label: gt('Reject'), tooltip: gt('Reject the current change'), smallerVersion: { hideLabel: true } }))
                        .addGroup('selectPrevChangeTracking',       btnSelectPrevCT)
                        .addGroup('selectNextChangeTracking',       btnSelectNextCT);
                }
            }

            if (!self.panesCombined()) {
                changeTrackingToolBar
                    .addSeparator()
                    .addGroup('comment/insert', btnInsertCommentReview)
                    .addGap()
                    .addGroup('comment/displayModeParent', new Controls.CommentDisplayModePicker(self))
                    .addGap()
                    .addGroup('comment/prevComment', btnSelectPrevComment)
                    .addGroup('comment/nextComment', btnSelectNextComment)
                    .addGroup('comment/deleteAll', btnDeleteAllComments);
            } else {
                commentToolBar
                    .addGroup('comment/insert', btnInsertCommentReview)
                    .addGap()
                    .addGroup('comment/displayModeParent', new Controls.CommentDisplayModePicker(self, { smallDevice: true }))
                    .addGap()
                    .addGroup('comment/prevComment', btnSelectPrevComment)
                    .addGroup('comment/nextComment', btnSelectNextComment)
                    .addGroup('comment/deleteAll', btnDeleteAllComments);
            }

            // the 'View' drop-down menu
            viewMenuGroup
                .addSectionLabel(Labels.ZOOM_LABEL)
                .addGroup('view/zoom/dec', new Controls.Button(Labels.ZOOMOUT_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom/inc', new Controls.Button(Labels.ZOOMIN_BUTTON_OPTIONS), { inline: true, sticky: true })
                .addGroup('view/zoom', new Controls.PercentageLabel(), { inline: true })
                .addGroup('view/zoom/type', new Controls.Button({ label: Labels.ZOOM_SCREEN_WIDTH_LABEL, value: 'width' }))
                .addSectionLabel(Labels.OPTIONS_LABEL)
                // note: we do not set aria role to 'menuitemcheckbox' or 'button' due to Safari just working correctly with 'checkox'. CheckBox constructor defaults aria role to 'checkbox'
                .addGroup('view/toolbars/show', new Controls.CheckBox(Labels.SHOW_TOOLBARS_CHECKBOX_OPTIONS))
                .addGroup('document/users', new Controls.CheckBox(Labels.SHOW_COLLABORATORS_CHECKBOX_OPTIONS));

            // event handling

            self.on('refresh:layout', function () {

                var // the app content node
                    appContentRootNode = self.getContentRootNode(),
                    // the page attributes contain the width of the page in hmm
                    pageAttributeWidth = docModel.getPageLayout().getPageAttribute('width'),
                    // whether the page padding need to be reduced
                    reducePagePadding = (Utils.COMPACT_DEVICE && pageAttributeWidth > Utils.convertLengthToHmm(appContentRootNode.width(), 'px')),
                    // the remote selection object
                    remoteSelection = docModel.getRemoteSelection(),
                    // temporary page node width value
                    tempPageNodeWidth = pageNode.outerWidth(),
                    // whether the applications width has changed
                    // -> the value for appWidthChanged must be true, if the width of the page node was modified
                    // or if the class 'small-device' was added or removed (41160)
                    appWidthChanged = (pageNodeWidth !== tempPageNodeWidth) || (appContentRootNode.hasClass('small-device') !== reducePagePadding);

                // check, if there is sufficient space for left and right padding on the page
                appContentRootNode.toggleClass('small-device', reducePagePadding);

                // fix for #33829 - if padding is reduced, dependent variables have to be updated
                pageNodeWidth = tempPageNodeWidth;
                pageNodeHeight = pageNode.outerHeight();

                // rotating device screen on tablets needs margin recalculated, except for < 7" devices (there are no margins and page breaks)
                // also changing page properties from user dialog, needs page layout repaint
                if (appWidthChanged && self.isImportFinished()) {
                    if (!Utils.SMALL_DEVICE) {
                        if (docModel.isHeaderFooterEditState()) {
                            docModel.getPageLayout().leaveHeaderFooterEditMode();
                            docModel.getSelection().setNewRootNode(docModel.getNode()); // restore original rootNode
                            docModel.getSelection().setTextSelection(docModel.getSelection().getFirstDocumentPosition());
                        }
                        // INFO: calling insertPageBreaks from model is absolutely necessary!
                        // It is debounced method in model, and direct method in pageLayout.
                        // Debounced method has little start delay, but direct model can be interupted by calls from other places,
                        // which can lead to unpredicted page sizes!
                        docModel.insertPageBreaks();
                        self.recalculateDocumentMargin({ keepScrollPosition: true });
                    } else {
                        docModel.trigger('update:absoluteElements');   // updating comments and absolutely positioned drawings (41160)
                    }
                    if (Utils.COMPACT_DEVICE) {
                        if (reducePagePadding) {
                            self.executeDelayed(function () { self.setZoomType('width'); }, undefined, 'Text: View: Setting zoom type');
                        } else {
                            self.setZoomType(100);
                        }
                    }
                }

                // updating the change track side bar, if necessary
                docModel.updateChangeTracksDebounced();

                if (zoomType === 'width') {
                    self.executeDelayed(function () { self.setZoomType('width'); }, undefined, 'Text: View: Setting zoom type');
                }

                // clean and redraw remote selections
                remoteSelection.renderCollaborativeSelections();
            });
        }

        /**
         * Additional initialization of debug GUI after importing the document.
         */
        function initDebugGuiHandler(viewMenuGroup) {

            viewMenuGroup
                .addGroup('debug/pagebreaks/toggle', new Controls.CheckBox({ label: _.noI18n('Show page breaks') }))
                .addGroup('debug/draftmode/toggle', new Controls.CheckBox({ label: _.noI18n('View in draft mode') }));

            self.createToolBar('debug')
                .addGroup('document/cut',   new Controls.Button({ icon: 'fa-cut',   tooltip: _.noI18n('Cut to clipboard') }))
                .addGroup('document/copy',  new Controls.Button({ icon: 'fa-copy',  tooltip: _.noI18n('Copy to clipboard') }))
                .addGroup('document/paste', new Controls.Button({ icon: 'fa-paste', tooltip: _.noI18n('Paste from clipboard') }));

            self.createToolBar('debug')
                .addGroup('debug/recordoperations', new Controls.Button({ icon: 'fa-dot-circle-o', iconStyle: 'color:red;', toggle: true, tooltip: _.noI18n('Record operations') }))
                .addGroup('debug/replayoperations', new Controls.Button({ icon: 'fa-play-circle-o', tooltip: _.noI18n('Replay operations') }))
                .addGroup(null, replayOperations)
                .addGroup(null, replayOperationsOSN);

            self.createToolBar('debug')
                .addGroup('debug/commentView', new Controls.CommentDisplayViewPicker())
                .addGap()
                .addGroup('debug/uselocalstorage', new Controls.Button({ icon: 'fa-rocket', tooltip: _.noI18n('Use local storage'), toggle: true }))
                .addGroup('debug/useFastLoad', new Controls.Button({ icon: 'fa-fighter-jet', tooltip: _.noI18n('Use fast load'), toggle: true }))
                //important for uitests!
                .addGroup('paragraph/list/bullet', new Controls.BulletListStylePicker(self, docModel))
                .addSeparator()
                .addGroup('debug/deleteHeaderFooter', new Controls.Button({ icon: 'fa-eraser', tooltip: _.noI18n('Delete Header/Footer') }));
        }

        /**
         * Executes a search/replace operation, as requested by the base class
         * EditView.
         */
        function searchHandler(command, settings) {
            var search = docModel.getSearchHandler();
            switch (command) {
                case 'search:start':
                    return search.quickSearch(settings.query, 0);
                case 'search:next':
                    return search.selectNextSearchResult();
                case 'search:prev':
                    return search.selectPreviousSearchResult();
                case 'search:end':
                    return search.removeHighlighting();
                case 'replace:next':
                    return search.replaceSelectedSearchResult(settings.replace);
                case 'replace:all':
                    return search.searchAndReplaceAll(settings.query, settings.replace);
                default:
                    Utils.error('TextView.searchHandler(): unsupported search command "' + command + '"');
                    return 'internal';
            }
        }

        function calculateOptimalDocumentWidth() {
            var
                documentWidth = pageNode.outerWidth() + (pageNode.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageNode.data(DOM.COMMENTMARGIN_CLASS) : 0),
                parentWidth = appPaneNode.width(),
                appContent = appPaneNode.find('.app-content'),
                outerDocumentMargin = 3 + Utils.getElementCssLength(appContent, 'margin-left') + Utils.getElementCssLength(appContent, 'margin-right'), // "3" because of the rounding issues
                optimalLevel = ((parentWidth * parentWidth) / (documentWidth * (parentWidth + Utils.SCROLLBAR_WIDTH + 4 + outerDocumentMargin))) * 100;

            return Math.floor(optimalLevel);
        }

        function setAdditiveMargin() {
            if (Utils.IOS) {
                //extra special code for softkeyboard behavior, we nee more space to scroll to the end of the textfile
                var dispHeight = Math.max(screen.height, screen.width);

                pageNode.css('margin-bottom', (Utils.getElementCssLength(pageNode, 'margin-bottom') + dispHeight / 3) + 'px');

                //workaround for clipped top of the first page (20)
                pageNode.css('margin-top', (Utils.getElementCssLength(pageNode, 'margin-top') + 20) + 'px');
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the position of the passed node in the entire scroll area of
         * the application pane.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element whose dimensions will be calculated. Must be
         *  contained somewhere in the application pane. If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @returns {Object}
         *  An object with numeric attributes representing the position and
         *  size of the passed node relative to the entire scroll area of the
         *  application pane. See method Utils.getChildNodePositionInNode() for
         *  details.
         */
        this.getChildNodePositionInNode = function (node) {
            return Utils.getChildNodePositionInNode(contentRootNode, node);
        };

        /**
         * Scrolls the application pane to make the passed node visible.
         *
         * @param {HTMLElement|jQuery} node
         *  The DOM element that will be made visible by scrolling the
         *  application pane. Must be contained somewhere in the application
         *  pane. If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Object} options
         *  @param {Boolean} [options.forceToTop=false]
         *   Whether the specified node shall be at the top border of the
         *   visible area of the scrolling node.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToChildNode = function (node, options) {
            if (Utils.getBooleanOption(options, 'regardSoftkeyboard', false)) {
                //fix for Bug 39979 on Ipad text is sometimes under the keyboard
                if (!Utils.IOS || !Utils.isSoftKeyboardOpen()) { delete options.regardSoftkeyboard; }
            }
            Utils.scrollToChildNode(contentRootNode, node, _.extend(options || {}, { padding: 15 }));
            return this;
        };

        /**
         * Scrolls the application pane to make the passed document page
         * rectangle visible.
         *
         * @param {Object} pageRect
         *  The page rectangle that will be made visible by scrolling the
         *  application pane. Must provide the properties 'left', 'top',
         *  'width', and 'height' in pixels. The properties 'left' and 'top'
         *  are interpreted relatively to the entire document page.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.scrollToPageRectangle = function (pageRect) {
            var options = { padding: 15 };

            Utils.scrollToPageRectangle(contentRootNode, pageRect, options);
            return this;
        };

        /**
         * Provides the debug operations as string for the debugging feature
         * replay operations.
         *
         * @returns {String}
         *  The content of the debug operations field, or null if debug mode is
         *  not active.
         */
        this.getDebugOperationsContent = function () {
            var // the operations in the required json format
                json = null;

            if (replayOperations) {
                json = replayOperations.getFieldValue();

                try {
                    json = json.length ? JSON.parse(json) : null;
                    if (_.isObject(json) && json.operations) {
                        json = json.operations;
                    }
                    return json;
                } catch (e) {
                    // error handling outside this function
                }
            }
            return null;
        };

        /**
         * Setting the maximum OSN for the operations from the operation player, that will
         * be executed.
         *
         * @returns {Number|Null}
         *  The OSN of the last operation that will be executed from the operation player.
         */
        this.getDebugOperationsOSN = function () {

            var // the operations in the required json format
                number = null;

            if (replayOperationsOSN) {

                number = replayOperationsOSN.getFieldValue();

                number = parseInt(number, 10);

                if (!(_.isNumber(number) && _.isFinite(number))) {
                    number = null;
                    replayOperationsOSN.setValue('');
                }
            }

            return number;
        };

        /**
         * Returns the current zoom type.
         *
         * @returns {Number|String}
         *  The current zoom type, either as fixed percentage, or as one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         */
        this.getZoomType = function () {
            return zoomType;
        };

        /**
         * Returns the current effective zoom factor in percent.
         *
         * @returns {Number}
         *  The current zoom factor in percent.
         */
        this.getZoomFactor = function () {
            return zoomFactor;
        };

        /**
         * Returns the minimum zoom factor in percent.
         *
         * @returns {Number}
         *  The minimum zoom factor in percent.
         */
        this.getMinZoomFactor = function () {
            return ZOOM_FACTORS[0];
        };

        /**
         * Returns the maximum zoom factor in percent.
         *
         * @returns {Number}
         *  The maximum zoom factor in percent.
         */
        this.getMaxZoomFactor = function () {
            return _.last(ZOOM_FACTORS);
        };

        /**
         * Changes the current zoom settings.
         *
         * @param {Number|String} newZoomType
         *  The new zoom type. Either a fixed percentage, or one of the
         *  keywords 'width' (page width is aligned to width of the visible
         *  area), or 'page' (page size is aligned to visible area).
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.setZoomType = function (newZoomType) {
            var optimalLevel = calculateOptimalDocumentWidth();

            if (zoomType !== newZoomType && _.isNumber(newZoomType)) {
                if (zoomType < newZoomType) {
                    this.increaseZoomLevel(newZoomType);
                } else {
                    this.decreaseZoomLevel(newZoomType);
                }
            } else if (newZoomType === 'width') {
                if (Utils.SMALL_DEVICE || docModel.isDraftMode()) {
                    optimalLevel = 100;
                }
                if (Math.abs(zoomFactor - optimalLevel) > 1.1) {
                    this.increaseZoomLevel(optimalLevel);
                }
            }
            zoomType = newZoomType;
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and decreases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.decreaseZoomLevel = function (newZoomFactor) {

            var // find last entry in ZOOM_FACTORS with a factor less than current zoom
                prevZoomFactor = Utils.findLast(ZOOM_FACTORS, function (factor) { return factor < zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                // the parent node of the page
                appContentRoot = $(appPaneNode).find('.app-content');

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(prevZoomFactor) ? prevZoomFactor : this.getMinZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if (Utils.SMALL_DEVICE || docModel.isDraftMode()) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                if (zoomInsertion < 100 && docModel.isDraftMode()) {
                    pageContentNode.css({ width: '100%' });
                } else {
                    pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                }
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    appContentRoot.css('overflow', 'hidden');
                } else {
                    appContentRoot.css({ overflow: '', paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                // pageNode.css({
                //    'zoom': zoomInsertion / 100
                // });

                zoomType = zoomFactor;
                scrollToSelection();
            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Switches to 'fixed' zoom mode, and increases the current zoom level
         * by one according to the current effective zoom factor, and updates
         * the view.
         *
         * @param {Number} newZoomFactor
         *  The new zoom type. If passed as argument, use this zoom factor,
         *  otherwise use one of predefined
         *
         * @param {Boolean} option.clean
         *  Switching between draft and normal mode requires zoom cleanup
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.increaseZoomLevel = function (newZoomFactor, clean) {

            var // find first entry in ZOOM_FACTORS with a factor greater than current zoom
                nextZoomFactor = _(ZOOM_FACTORS).find(function (factor) { return factor > zoomFactor; }),
                zoomInsertion = newZoomFactor || 0,
                appContentRoot = $(appPaneNode).find('.app-content');

            if (zoomInsertion === 0) {
                zoomInsertion = _.isNumber(nextZoomFactor) ? nextZoomFactor : this.getMaxZoomFactor();
            }
            zoomFactor = zoomInsertion;

            if ((Utils.SMALL_DEVICE || docModel.isDraftMode()) && !clean) {
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform', 'scale(' + (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageContentNode, 'transform-origin', 'top left');
                pageContentNode.css({ width: ((100 / zoomInsertion) * 100) + '%', backgroundColor: '#fff' });
                zoomType = zoomFactor;
            } else {

                Utils.setCssAttributeWithPrefixes(pageNode, 'transform', 'scale(' +  (zoomInsertion / 100) + ')');
                Utils.setCssAttributeWithPrefixes(pageNode, 'transform-origin', 'top');

                if (_.browser.Firefox) {
                    pageNode.css({ opacity: (zoomInsertion > 100) ? 0.99 : 1 });
                }

                self.recalculateDocumentMargin();

                if (zoomInsertion <= 99) {
                    appContentRoot.css('overflow', 'hidden');
                } else {
                    appContentRoot.css({ overflow: '', paddingLeft: '', paddingRight: '' });
                }

                // Use zoom as soon as all browsers support it.
                //  pageNode.css({
                //      'zoom': zoomInsertion / 100
                //  });

                zoomType = zoomFactor;
                scrollToSelection();

            }
            self.trigger('change:zoom');
            return this;
        };

        /**
         * Creates necessary margin values when using transform scale css property
         *
         */
        this.recalculateDocumentMargin = function (options) {
            var
                marginAddition = zoomFactor < 99 ? 3 : 0,
                zoomFactorScaled = zoomFactor / 100,
                recalculatedMargin = (pageNodeWidth * (zoomFactorScaled - 1)) / 2 + marginAddition,
                // whether the (vertical) scroll position needs to be restored
                keepScrollPosition = Utils.getBooleanOption(options, 'keepScrollPosition', false),
                // an additional right margin used for comments that might be set at the page node
                commentMargin = pageNode.hasClass(DOM.COMMENTMARGIN_CLASS) ? pageNode.data(DOM.COMMENTMARGIN_CLASS) * zoomFactorScaled : 0;

            if (zoomFactor !== 100) {
                // #34735 page width and height values have to be updated
                pageNodeWidth = pageNode.outerWidth();
                pageNodeHeight = pageNode.outerHeight();
            }

            if (_.browser.Chrome && zoomFactor > 100) {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (30 / zoomFactorScaled) + 'px ' + recalculatedMargin + 'px'
                });
            } else {
                pageNode.css({
                    margin:  marginAddition + 'px ' + (recalculatedMargin + commentMargin) + 'px ' + (pageNodeHeight * (zoomFactorScaled - 1) + marginAddition) + 'px ' + recalculatedMargin + 'px'
                });
            }

            setAdditiveMargin();

            // in the end scroll to cursor position/selection (but not after event 'refresh:layout', that uses the option 'keepScrollPosition')
            if (!keepScrollPosition) { scrollToSelection(); }
        };

        /**
         * Rejects an attempt to edit the text document (e.g. due to reaching
         * the table size limits).
         *
         * @param {String} cause
         *  The cause of the rejection.
         *
         * @returns {TextView}
         *  A reference to this instance.
         */
        this.rejectEditTextAttempt = function (cause) {

            switch (cause) {
                case 'tablesizerow':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further rows.') });
                    break;

                case 'tablesizecolumn':
                    this.yell({ type: 'info', message: gt('The table reached its maximum size. You cannot insert further columns.') });
                    break;
            }

            return this;
        };

        // returns the change track popup object
        this.getChangeTrackPopup = function () {
            return changeTrackPopup;
        };

        // show changes handler of the changetracking popup
        this.changeTrackPopupSelect = function () {
            docModel.getChangeTrack().showChangeTrackGroup();
            self.getChangeTrackPopup().show();
        };

        /**
         * Custom Change track date formatter. This function makes the private
         * date formatter 'formatChangeTrackDate' publicly available. Therefore
         * it supports the same parameter.
         *
         * @param {Number} timeInMilliSeconds
         *  Timestamp in milliseconds
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.toLocal=true]
         *      Timestamp is converted to local time zone per default.
         *
         * @returns {String}
         *  The formatted date string
         */
        this.getDisplayDateString = function (timeInMilliSeconds, options) {
            return formatChangeTrackDate(timeInMilliSeconds, options);
        };

        this.showPageSettingsDialog = function () {
            return new Dialogs.PageSettingsDialog(this).show();
        };

        /**
         * Shows the change track popup consisting updated change track data.
         */
        this.showChangeTrackPopup = function () {
            var popupMeta = getChangeTrackPopupMeta();
            // create show popup only on change track nodes
            if (popupMeta) {
                // update popup anchor node
                changeTrackPopup.setAnchor(getChangeTrackPopupAnchorNode());
                // update popup contents
                changeTrackBadge.update({
                    author: popupMeta.author || gt('Unknown'),
                    authorColorIndex: app.getAuthorColorIndex(popupMeta.author),
                    authorUserId: popupMeta.uid,
                    date: formatChangeTrackDate(Date.parse(popupMeta.date), { toLocal: false }),
                    action: popupMeta.actionText || gt('Unknown')
                });
                changeTrackPopup.show();
            }
        };

        /**
         * Returns the timer (promise) which opens the changeTrack-popup
         * after a specific time. It's abortable.
         *
         * @returns {promise}
         *  The changeTrack-popup-opening-timer
         */
        this.getChangeTrackPopupTimeout = function () {
            return changeTrackPopupTimeout;
        };

        /**
         * Removes the drawinganchor marker of all paragraphs.
         */
        this.clearVisibleDrawingAnchor = function () {
            docModel.getCurrentRootNode().find('.visibleAnchor').removeClass('visibleAnchor').find('.anchorIcon').remove();
        };

        /**
         * Sets the marker for the anchor
         *
         * @param {HTMLElement|jQuery} drawing
         *  The DOM element for that the anchor will be made
         *  visible.
         *
         * @param {Object} attributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         */
        this.setVisibleDrawingAnchor = function (drawing, attributes) {
            var oxoPosition = null,
                paragraph = null;

            // get out here, no attributes given
            if (_.isEmpty(attributes)) { return; }

            // remove all visible anchors first
            self.clearVisibleDrawingAnchor();

            // inline drawing get no visible anchor
            if (!_.has(attributes, 'drawing') || !_.has(attributes.drawing, 'inline') || attributes.drawing.inline) { return; }

            oxoPosition = Position.getOxoPosition(docModel.getCurrentRootNode(), drawing, 0);
            paragraph = $(Position.getParagraphElement(docModel.getCurrentRootNode(), _.initial(oxoPosition)));

            $(paragraph).addClass('visibleAnchor').prepend('<i class="docs-position-anchor anchorIcon helper">');
        };

        /**
         * Public accesor to show the field format popup.
         *
         * @param {Object} fieldinstruction
         *  Instruction of the field for which the group with format codes will be updated.
         * @param {jQuery} node
         *  Field node.
         * @param {Object} options
         *  @param {Boolean} options.autoDate
         */
        this.showFieldFormatPopup = function (fieldinstruction, node, options) {
            var nodeHelper,
                datePickerVal = '',
                fieldText = '';

            if (!docModel.getEditMode() || docModel.getSelection().hasRange() || !fieldinstruction) { return; }
            initFieldFormatPopup(fieldinstruction, options);
            fieldFormatPopup.setAnchor(getFieldAnchorNode());

            if (fieldinstruction.formats && fieldinstruction.type && (/^DATE|^TIME/i).test(fieldinstruction.type)) {
                //self.show();
                datePickerNode.hide();
                if (node) {
                    currentCxNode = $(node);

                    if ($(node).data('datepickerValue')) {
                        datePickerVal = node.data('datepickerValue');
                        datePickerNode.datepicker('update', datePickerVal); // reset previously picked date
                    } else {
                        nodeHelper = $(node).next();
                        fieldText = nodeHelper.text();
                        while (nodeHelper.length && nodeHelper.is('span')) {
                            nodeHelper = nodeHelper.next();
                            fieldText += nodeHelper.text();
                        }
                        datePickerVal = self.getDocModel().getNumberFormatter().parseLeadingDate(fieldText, { complete: true });
                        if (!datePickerVal || !moment(datePickerVal.date).isValid() || _.isUndefined(datePickerVal.Y) || _.isUndefined(datePickerVal.M) || _.isUndefined(datePickerVal.D)) {
                            datePickerNode.datepicker('update', ''); // reset previously picked date
                        } else {
                            datePickerNode.datepicker('update', new Date(datePickerVal.Y, datePickerVal.M, datePickerVal.D));
                        }
                    }
                }
            }

            fieldFormatList.update(fieldinstruction);
            fieldFormatPopup.show();
        };

        /**
         * Public accesor to hide and destroy the field format popup.
         */
        this.hideFieldFormatPopup = function () {
            _.defer(function () {
                if (fieldFormatPopup) {
                    fieldFormatPopup.destroy();
                    fieldFormatPopup = null;

                    datePickerNode.empty();
                }
            });
        };

        /**
         * Clicking on button from field format popup, field will get current date.
         */
        this.setFieldToTodayDate = function () {
            var date = new Date();
            var standardizedDate = self.formatDateWithNumberFormatter(date, 'yyyy-mm-dd\Thh:mm:ss.00');
            currentCxNode.data('datepickerValue', date);
            self.trigger('fielddatepopup:change', { value: null, format: currentFormatOfFieldPopup, standard: standardizedDate }); // send default - which is current date
        };

        /**
         * Show or hide datepicker in field popup, and refresh popup.
         */
        this.toggleDatePicker = function () {
            datePickerNode.toggle();
            fieldFormatPopup.hide();
            fieldFormatPopup.show();
        };

        /**
         * Callback function for clicking checkbox in popup to update date field automatically.
         */
        this.autoFieldCheckboxUpdate = function () {
            datePickerNode.hide();

            autoDate = !autoDate;

            if (autoDate) {
                setToButton.getNode().addClass('disable');
                toggleDatePickerBtn.getNode().addClass('disable');
            } else {
                setToButton.getNode().removeClass('disable');
                toggleDatePickerBtn.getNode().removeClass('disable');
            }

            fieldFormatPopup.hide();
            fieldFormatPopup.show();

            autoUpdateCheckbox.setValue(autoDate);
            // trigger event for setting field to fixed
            self.trigger('fielddatepopup:autoupdate', autoDate);
        };

        /**
         * Formats default passed date to given format.
         *
         * @param {String} date
         * @param {String} format
         * @returns {String}
         */
        this.formatDateWithNumberFormatter = function (date, format) {
            return docModel.getNumberFormatter().formatValue(docModel.getNumberFormatter().convertDateToNumber(date), format);
        };

        // initialization -----------------------------------------------------

        // create the context menu
        contextMenu = new UniversalContextMenu(this, { delay: (_.browser.IE) ? 1 : 200 });

        // store the values of page width and height with paddings, after loading is finished
        this.waitForImportSuccess(function () {
            pageNodeWidth = pageNode.outerWidth();
            pageNodeHeight = pageNode.outerHeight();

            if (Utils.COMPACT_DEVICE && !Utils.SMALL_DEVICE) {
                self.executeDelayed(function () {
                    var optimalDocWidth = calculateOptimalDocumentWidth();
                    if (optimalDocWidth < 100) {
                        zoomType = 'width';
                    }
                }, undefined, 'Text: View: Setting zoom type, compact device');
            }

            if (Utils.IOS) {
                var appContent = pageNode.parent();
                appContent.attr('contenteditable', true);

                var listenerList = docModel.getListenerList();
                pageNode.off(listenerList);
                appContent.on(listenerList);

                //we dont want the focus in the document on the beginning!
                Utils.focusHiddenNode();
            }

            // quick fix for bug 40053
            if (_.browser.Safari) { Forms.addDeviceMarkers(app.getWindowNode()); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            contextMenu.destroy();
            if (changeTrackPopup) { changeTrackPopup.destroy(); } // still null, if launching application fails
            if (fieldFormatPopup) { fieldFormatPopup.destroy(); }
            app = self = replayOperations = replayOperationsOSN = docModel = null;
            appPaneNode = contentRootNode = pageNode = pageContentNode = contextMenu = null;
            changeTrackPopup = changeTrackBadge = changeTrackAcceptButton = changeTrackRejectButton = changeTrackShowButton = null;
            fieldFormatPopup = fieldFormatList = datePickerNode = descriptionNode = autoUpdateCheckbox = setToButton = toggleDatePickerBtn = null;
        });

    } // class TextView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: TextView });

});
