/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/spreadsheet/model/numberformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/locale/formatter',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/complex'
], function (Utils, LocaleData, Parser, Formatter, AppObjectMixin, SheetUtils, Complex) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;

    // default built-in formats used in OOXML
    var BUILTIN_FORMAT_MAP = {
        0: 'GENERAL',
        1: '0',
        2: '0.00',
        3: '#,##0',
        4: '#,##0.00',
        9: '0%',
        10: '0.00%',
        11: '0.00E+00',
        12: '# ?/?',
        13: '# ??/??',
        14: LocaleData.SHORT_DATE,
        48: '0.00E+0',
        49: '@'
    };

    // all built-in number format identifiers, mapped by format code
    var INV_BUILTIN_FORMAT_MAP = null;

    // predefined number formats mapped by categories
    var CATEGORY_FORMAT_MAP = null;

    // load the localized number format definitions
    var numberFormatsPromise = LocaleData.loadResource('io.ox/office/spreadsheet/resource/numberformats').done(function (data) {
        CATEGORY_FORMAT_MAP = data.categories;
        _.extend(BUILTIN_FORMAT_MAP, data.builtInFormats);
        INV_BUILTIN_FORMAT_MAP = Utils.mapProperties(_.invert(BUILTIN_FORMAT_MAP), function (id) { return parseInt(id, 10); });
    });

    // class NumberFormatter ==================================================

    /**
     * A number formatter for a spreadsheet document. Supports the format code
     * syntax used in document operations, and allows to deal with all data
     * types of a spreadsheet document: numbers, strings, Boolean values, and
     * error codes.
     *
     * @constructor
     *
     * @extends Formatter
     * @extends AppObjectMixin
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    var NumberFormatter = Formatter.extend({ constructor: function (docModel) {

        var // self reference
            self = this,

            // access to localized formula labels
            grammarConfig = null,

            // lazy initialized helper parsing the date
            unformatDate = null,

            // lazy initialized helper parsing the time
            unformatTime = null;

        // base constructors --------------------------------------------------

        Formatter.call(this);
        AppObjectMixin.call(this, docModel.getApp());

        // private methods ----------------------------------------------------

        /**
         * create helper object for converting in-place-formated values back to cell-format
         * takes the editCodesDatabase' date & time
         * parses the special chars between the alphanumercis ('mm.dd.yy' -> '.') and create a regexp for "x.x.x" & "x.x" formats
         * gets the index of the chars in the format ('mm.dd.yy -> m=0 d=1 y=2)
         */
        function prepareUnformat(source/*, key0, key1, key2*/) {
            var args = arguments;
            var splitFinder = /\w+((?=[\S])\W)\w+/;
            var splitSymbol = splitFinder.exec(source);

            if (splitSymbol) {
                var result = {};

                result.splitSymbol = splitSymbol[1];
                result.finder = new RegExp('\\d+\\' + result.splitSymbol + '\\d+\\' + result.splitSymbol + '\\d+|\\d+\\' + result.splitSymbol + '\\d+');

                var possKeys = source.split(new RegExp('\\' + result.splitSymbol, 'g'));
                _.each(possKeys, function (value, index) {
                    for (var i = 1; i < args.length; i++) {
                        var argKey = args[i];
                        if (value && (value[0].toLowerCase() === argKey)) {
                            result[argKey] = index;
                            break;
                        }
                    }
                });
                return result;
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Converts the passed complex number to a string. The real and
         * imaginary coefficients will be formatted with the 'General' number
         * format using the passed maximum number of digits.
         *
         * @param {Complex} complex
         *  The complex number to be converted to a string.
         *
         * @param {Number} maxNumLength
         *  The maximum number of characters allowed for the absolute part of
         *  a coefficient, including the decimal separator and the complete
         *  exponent in scientific notation, but without the sign character.
         *  MUST be positive.
         *
         * @returns {String}
         *  The converted complex number.
         */
        this.convertComplexToString = function (complex, maxNumLength) {

            var // the real coefficient, as string
                real = this.formatStandardNumber(complex.real, maxNumLength),
                // the imaginary coefficient, as string
                imag = this.formatStandardNumber(complex.imag, maxNumLength),
                // fall-back to unit 'i'
                unit = complex.unit || 'i';

            // leave out imaginary part if zero
            if (complex.imag === 0) { return real; }

            // leave out real part if missing; do not add single '1' for imaginary part
            return ((complex.real === 0) ? '' : real) + (((complex.real !== 0) && (complex.imag > 0)) ? '+' : '') + (/^-?1$/.test(imag) ? '' : imag) + unit;
        };

        /**
         * Converts the passed text to a complex number.
         *
         * @param {String} text
         *  The string to be converted to a complex number.
         *
         * @returns {Complex|Null}
         *  The complex number represented by the passed string; or null, if
         *  the string cannot be parsed to a complex number.
         */
        this.convertStringToComplex = function (text) {

            var // the matches of a regular expression
                matches = null,
                // the parse result for a floating-point number from formatter
                parseResult = null,
                // the parsed real coefficient
                real = 0;

            // do not accept empty strings
            if (text.length === 0) { return null; }

            // string may be a single imaginary unit without coefficients: i, +i, -i (same for j)
            if ((matches = /^([-+]?)([ij])$/.exec(text))) {
                return new Complex(0, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull leading floating-point number from the string
            if (!(parseResult = Parser.parseLeadingNumber(text, { sign: true }))) {
                return null;
            }

            // check for simple floating-point number without imaginary coefficient: a, +a, -a
            real = parseResult.number;
            text = parseResult.remaining;
            if (text === '') {
                return new Complex(real, 0);
            }

            // check for imaginary number without real coefficient: bi, +bi, -bi
            if ((text === 'i') || (text === 'j')) {
                return new Complex(0, real, text);
            }

            // check for following imaginary unit without coefficients, but with sign: a+i, a-i
            if ((matches = /^([-+])([ij])$/.exec(text))) {
                return new Complex(real, (matches[1] === '-') ? -1 : 1, matches[2]);
            }

            // pull trailing floating-point number from the string: a+bi, a-bi (sign is required here, something like 'abi' is not valid)
            if (!(parseResult = Parser.parseLeadingNumber(text, { sign: true })) || (parseResult.sign.length === 0)) {
                return null;
            }

            // remaining text must be the imaginary unit
            text = parseResult.remaining;
            if ((text === 'i') || (text === 'j')) {
                return new Complex(real, parseResult.number, text);
            }

            return null;
        };

        /**
         * Returns all available number format codes for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {Array|Null}
         *  The format codes for the specified category, or null if no format
         *  codes have been defined for the category. Each element in the
         *  returned array is an object with the following properties:
         *  - {String} value
         *      The actual format code.
         *  - {String} label
         *      A caption label with a formatted example number, intended to be
         *      used in GUI form controls.
         *  - {Boolean} [red=false]
         *      If set to true, the format code contains a [RED] color modifier
         *      for negative numbers.
         */
        this.getCategoryCodes = function (category) {
            var categoryFormats = CATEGORY_FORMAT_MAP[category.toLocaleLowerCase()];
            return _.isArray(categoryFormats) ? _.copy(categoryFormats, true) : null;
        };

        /**
         * Returns the default number format code for the specified format
         * category and current UI language.
         *
         * @param {String} category
         *  The identifier of the number format category.
         *
         * @returns {String|Null}
         *  The default format code for the specified category, or null if no
         *  format code has been found.
         */
        this.getCategoryDefaultCode = function (category) {
            var categoryCodes = CATEGORY_FORMAT_MAP[category.toLocaleLowerCase()];
            return (_.isArray(categoryCodes) && _.isObject(categoryCodes[0])) ? categoryCodes[0].value : null;
        };

        /**
         * Returns the predefined number format code for the specified format
         * code identifier.
         *
         * @param {Number} formatId
         *  The format code identifier.
         *
         * @returns {String|Null}
         *  The number format code for the specified identifier if existing,
         *  otherwise null.
         */
        this.getBuiltInFormatCode = function (formatId) {
            return (formatId in BUILTIN_FORMAT_MAP) ? BUILTIN_FORMAT_MAP[formatId] : null;
        };

        /**
         * Returns the predefined number format code for the specified format
         * code identifier.
         *
         * @param {String} formatCode
         *  The number format code to be converted to a built-in identifier.
         *
         * @returns {Number|Null}
         *  The built-in identifier of the passed format code if available,
         *  otherwise null.
         */
        this.getBuiltInFormatId = function (formatCode) {
            return (formatCode in INV_BUILTIN_FORMAT_MAP) ? INV_BUILTIN_FORMAT_MAP[formatCode] : null;
        };

        /**
         * Returns the format code of the passed value of a 'numberFormat'
         * formatting attribute.
         *
         * @param {Object} [numberFormat]
         *  The value of a 'numberFormat' formatting attribute. Must contain at
         *  least one of the properties 'id' (numeric identifier of a built-in
         *  format code), and 'code' (explicit format code, as string). If an
         *  identifier and an explicit format code exists, the format code will
         *  be ignored, and the built-in format code of the identifier will be
         *  returned. If omitted, the 'General' format code will be returned.
         *
         * @returns {String}
         *  The effective format code of the passed attribute value.
         */
        this.resolveFormatCode = function (numberFormat) {

            var // the built-in identifier
                id = Utils.getIntegerOption(numberFormat, 'id'),
                // explicit format code
                code = Utils.getStringOption(numberFormat, 'code');

            // prefer built-in identifier over format code
            if (_.isNumber(id) && (id in BUILTIN_FORMAT_MAP)) { return BUILTIN_FORMAT_MAP[id]; }

            // return existing explicit format code
            if (_.isString(code) && (code.length > 0)) { return code; }

            // warn if an invalid number format has been passed
            if (numberFormat) {
                Utils.warn('NumberFormatter.resolveFormatCode(): invalid number format: ' + JSON.stringify(numberFormat));
            }

            return 'General';
        };

        /**
         * Returns the parsed number format descriptor of the passed value of a
         * 'numberFormat' formatting attribute.
         *
         * @param {Object} numberFormat
         *  The value of a 'numberFormat' formatting attribute. Must contain at
         *  least one of the properties 'id' (numeric identifier of a built-in
         *  format code), and 'code' (explicit format code, as string). If an
         *  identifier and an explicit format code exists, the format code will
         *  be ignored, and the built-in format code of the identifier will be
         *  returned.
         *
         * @returns {ParsedFormat}
         *  The parsed number format descriptor of the passed attribute value.
         */
        this.parseNumberFormat = function (numberFormat) {
            return Parser.parseFormatCode(this.resolveFormatCode(numberFormat));
        };

        /**
         * Converts the passed format code to a value that can be used as
         * 'numberFormat' formatting attribute. Tries to resolve the identifier
         * of a built-in number format if possible.
         *
         * @param {String} formatCode
         *  A format code.
         *
         * @returns {Object}
         *  A number format attribute value. The object will contain the
         *  property 'id', if the format code represents a built-in number
         *  format; otherwise the property 'code' with the passed format code.
         */
        this.createNumberFormat = function (formatCode) {
            // replace format codes of built-in number formats with identifier
            var id = this.getBuiltInFormatId(formatCode);
            return _.isNumber(id) ? { id: id } : { code: formatCode };
        };

        /**
         * Tries to convert the passed text to a cell result literal (either a
         * floating-point number, a Boolean value, an error code, or a string).
         *
         * @param {String} text
         *  The string to be converted to a cell result value.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.keepApos=false]
         *      If set to true, a leading apostrophe in the passed text (used
         *      to protect a string from conversion to other data types) will
         *      be kept in the result. by default, the result will be a string
         *      without the leading apostrophe.
         *
         * @returns {Number|Boolean|String|ErrorCode}
         *  The resulting converted result value (the original string, if the
         *  passed text cannot be converted to another data type).
         */
        this.parseResultValue = function (text, options) {

            // try to parse boolean literals
            var boolValue = grammarConfig.getBooleanValue(text);
            if (_.isBoolean(boolValue)) { return boolValue; }

            // try to parse error code literals
            var errorCode = grammarConfig.getErrorCode(text);
            if (errorCode) { return errorCode; }

            // try to convert passed text to a floating-point number
            var parseResult = this.parseFormattedNumber(text);
            if (parseResult) { return parseResult.number; }

            // everything else is a regular string
            return Utils.getBooleanOption(options, 'keepApos', false) ? text : text.replace(/^'/, '');
        };

        /**
         * Returns the passed cell value, formatted with the specified number
         * format.
         *
         * @param {Number|String|Boolean|ErrorCode} value
         *  The typed cell value to be formatted.
         *
         * @param {String|ParsedFormat} [formatCode]
         *  The format code used to format the passed value as string, or as
         *  parsed format (instance of ParsedFormat as returned by the method
         *  Parser.parseFormatCode()). If omitted, numbers will be formatted
         *  according to the 'General' format code, and strings and booleans
         *  will not be formatted.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed format code is invalid,
         *  or if the value cannot be formatted with the format code (for
         *  example, a number is too large to be formatted as date/time).
         */
        this.formatValue = _.wrap(this.formatValue, function (formatValueFunc, value, formatCode) {

            // error codes will never be formatted
            if (value instanceof ErrorCode) {
                return grammarConfig.getErrorName(value);
            }

            // format boolean values like strings
            if (_.isBoolean(value)) {
                value = grammarConfig.getBooleanName(value);
            }

            // fall back to standard format code
            if (!formatCode) { formatCode = 'General'; }

            // use base class method to format numbers and strings
            return formatValueFunc.call(this, value, formatCode, { generalLength: SheetUtils.MAX_LENGTH_STANDARD_CELL });
        });

        /**
         * converts the assigned formated numbers to the source-value
         * with help of the unformat-data & the assigned category
         */
        this.parseEditString = function (value, category) {

            switch (category) {

                case 'percent':
                    if (_.isString(value)) {
                        value = Parser.stringToNumber(value);
                        if (_.last(value) === '%') {
                            value /= 100;
                        }
                    }
                    break;

                case 'date':
                case 'time':
                case 'datetime':
                    if (unformatDate) {
                        var date = new Date(0);

                        var dateInfo = unformatDate.finder.exec(value);
                        if (dateInfo) {
                            dateInfo = dateInfo[0].split(/\W/g);

                            date.setUTCDate(parseInt(dateInfo[unformatDate.d], 10));
                            date.setUTCMonth(parseInt(dateInfo[unformatDate.m], 10) - 1);
                            date.setUTCFullYear(parseInt(dateInfo[unformatDate.y], 10));
                        }

                        var timeInfo = unformatTime.finder.exec(value);
                        if (timeInfo) {
                            timeInfo = timeInfo[0].split(/\W/g);

                            date.setUTCSeconds(parseInt(timeInfo[unformatTime.s], 10));
                            date.setUTCMinutes(parseInt(timeInfo[unformatTime.m], 10));
                            date.setUTCHours(parseInt(timeInfo[unformatTime.h], 10));
                        }

                        if (dateInfo || timeInfo) {
                            value = self.convertDateToNumber(date);
                        }
                    }

                    break;
            }
            return value;
        };

        // initialization -----------------------------------------------------

        // initialize document-dependent settings
        this.waitForImportStart(function () {
            // TODO: disable negative dates handling for OOXML when formatting numbers locally
            this.configure({ negativeDates: true }); // { negativeDates: docModel.getApp().isODF() }
        }, this);

        // initialize locale-dependent settings
        docModel.onInitFormulaResource(function () {
            grammarConfig = docModel.getGrammarConfig('ui');
        });

        unformatDate = prepareUnformat(LocaleData.SHORT_DATE, 'd', 'm', 'y');
        unformatTime = prepareUnformat(LocaleData.LONG_TIME, 'h', 'm', 's');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = null;
        });

    } }); // class NumberFormatter

    // exports ================================================================

    return numberFormatsPromise.then(_.constant(NumberFormatter));

});
