/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/drawingmodelmixin', [
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (SheetUtils) {

    'use strict';

    var // convenience shortcuts
        Range = SheetUtils.Range;

    // class DrawingModelMixin ================================================

    /**
     * A mix-in class that extends the public API of any drawing model object
     * that has been inserted into a sheet of a spreadsheet document.
     *
     * @constructor
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class DrawingModel or any of its sub classes. Expects the symbol 'this'
     *  to be bound to an instance of DrawingModel.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model containing the drawing collection with this drawing
     *  model object.
     *
     * @param {Function} cloneConstructor
     *  A callback function invoked from the public method clone() which has to
     *  create and return a new instance cloned from this drawing model.
     */
    function DrawingModelMixin(sheetModel, cloneConstructor) {

        var // the drawing collection containing this drawing model
            drawingCollection = sheetModel.getDrawingCollection();

        // protected methods --------------------------------------------------

        /**
         * Creates and returns a cloned instance of this drawing object model
         * for the specified sheet.
         *
         * @internal
         *  Used by the class SheetDrawingCollection during clone construction.
         *  DO NOT CALL from external code!
         *
         * @param {SheetModel} targetModel
         *  The model instance of the new cloned sheet that will own the clone
         *  returned by this method.
         *
         * @returns {DrawingModelMixin}
         *  A clone of this drawing model, initialized for ownership by the
         *  passed sheet model.
         */
        this.clone = function (targetModel) {
            return cloneConstructor.call(this, targetModel);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the sheet model instance that contains this drawing model.
         *
         * @returns {SheetModel}
         *  The sheet model instance that contains this drawing model.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the position of this drawing model in the drawing collection
         * of the sheet.
         *
         * @returns {Array<Number>|Null}
         *  The sheet position of the drawing object, if it is still contained
         *  in the drawing collection; otherwise null.
         */
        this.getSheetPosition = function () {
            return drawingCollection.getModelPosition(this, { deep: true });
        };

        /**
         * Returns the complete document position of this drawing model, as
         * used in document operations, including the leading sheet index.
         *
         * @returns {Array<Number>|Null}
         *  The complete position of the drawing object, if it is still
         *  contained in the drawing collection, including the zero-based sheet
         *  index as first array element; otherwise null.
         */
        this.getDocumentPosition = function () {
            var position = this.getSheetPosition();
            return position ? [sheetModel.getIndex()].concat(position) : null;
        };

        /**
         * Returns the address of the cell range covered by this drawing model
         * object.
         *
         * @returns {Range}
         *  The address of the cell range covered by this drawing model.
         */
        this.getRange = function () {
            var anchorAttrs = this.getMergedAttributes().drawing;
            return Range.create(anchorAttrs.startCol, anchorAttrs.startRow, anchorAttrs.endCol, anchorAttrs.endRow);
        };

        /**
         * Returns the sheet rectangle in pixels covered by this drawing
         * object.
         *
         * @returns {Object|Null}
         *  The position of this drawing object in the sheet in pixels,
         *  according to the current sheet zoom factor, in the properties
         *  'left', 'top', 'width', and 'height'; or null if the drawing object
         *  is hidden by itself, or is located entirely in hidden columns or
         *  rows and thus invisible.
         */
        this.getRectangle = function () {
            return drawingCollection.getModelRectangle(this);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            sheetModel = cloneConstructor = drawingCollection = null;
        });

    } // class DrawingModelMixin

    // exports ================================================================

    return DrawingModelMixin;

});
