/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellstylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/model/stylecollection'
], function (Utils, Color, Border, StyleCollection) {

    'use strict';

    // class CellStyleCollection ==============================================

    /**
     * Contains the cell style sheets of a spreadsheet document.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {SpreadsheetModel} docModel
     *  The document model containing this instance.
     */
    function CellStyleCollection(docModel) {

        var // self reference
            self = this,

            // the identifier of the built-in 'Hyperlink' style
            hyperlinkStyleId = null;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'cell', { autoStyleSupport: true, families: 'character apply' });

        // private methods ----------------------------------------------------

        /**
         * Creates a new style sheet, if not existing yet. Updates the category
         * and priority of existing and created style sheets.
         */
        function createStyleSheet(styleId, category, priority, attributes) {

            if (self.containsStyleSheet(styleId)) {
                // update category and priority in existing style sheets
                self.setStyleOptions(styleId, { category: category, priority: priority, builtIn: true });
            } else {
                // create a new (dirty) style sheet, use name as identifier
                self.insertStyleSheet(styleId, styleId, null, attributes, {
                    category: category,
                    priority: priority,
                    hidden: category === 'hidden',
                    dirty: true,
                    builtIn: true
                });
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Creates all missing built-in cell styles.
         *
         * @internal
         *  Not intended to be called from any other code than the post-process
         *  handler of the application.
         *
         * @returns {CellStyleCollection}
         *  A reference to this instance.
         */
        this.createMissingStyles = function () {

            function getRGB(value) { return { type: 'rgb', value: value }; }
            function getScheme(value) { return docModel.createJSONColor({ type: 'scheme', value: value }, 'fill'); }
            function getSchemeTint20(value) { return docModel.createJSONColor({ type: 'scheme', value: value }, 'fill', { type: 'tint', value: 20002 }); }
            function getSchemeTint40(value) { return docModel.createJSONColor({ type: 'scheme', value: value }, 'fill', { type: 'tint', value: 40001 }); }
            function getSchemeTint60(value) { return docModel.createJSONColor({ type: 'scheme', value: value }, 'fill', { type: 'tint', value: 60003 }); }

            var // the merged attribute set of the default style
                attributes = this.getDefaultStyleAttributeSet(),
                // the default font name of the document
                fontName = attributes.character.fontName,

                // predefined border values
                LIGHT_GRAY_BORDER  = { style: 'single', width: 26, color: getRGB('B2B2B2') },
                MED_GRAY_BORDER    = { style: 'single', width: 26, color: getRGB('7F7F7F') },
                DARK_GRAY_BORDER   = { style: 'single', width: 26, color: getRGB('3F3F3F') },
                DOUBLE_GRAY_BORDER = { style: 'double', width: 79, color: getRGB('3F3F3F') },

                // predefined combinations for apply flags
                APPLY_FONT             = { fill: false, border: false, align: false, number: false, protect: false },
                APPLY_FONT_FILL        = { border: false, align: false, number: false, protect: false },
                APPLY_FONT_BORDER      = { fill: false, align: false, number: false, protect: false },
                APPLY_FONT_FILL_BORDER = { align: false, number: false, protect: false },
                APPLY_FILL_BORDER      = { font: false, align: false, number: false, protect: false };

            // standard styles
            createStyleSheet('Standard', 'markup', 1, {});
            createStyleSheet('Positive', 'markup', 2, { character: { fontName: fontName, color: getRGB('006100') }, cell: { fillColor: getRGB('C6EFCE') }, apply: APPLY_FONT_FILL });
            createStyleSheet('Neutral',  'markup', 3, { character: { fontName: fontName, color: getRGB('9C6500') }, cell: { fillColor: getRGB('FFEB9C') }, apply: APPLY_FONT_FILL });
            createStyleSheet('Negative', 'markup', 4, { character: { fontName: fontName, color: getRGB('9C0006') }, cell: { fillColor: getRGB('FFC7CE') }, apply: APPLY_FONT_FILL });
            createStyleSheet('Note',     'markup', 5, { character: { fontName: fontName, color: getScheme('dark1') }, cell: { fillColor: getRGB('FFFFCC'), borderTop: LIGHT_GRAY_BORDER, borderBottom: LIGHT_GRAY_BORDER, borderLeft: LIGHT_GRAY_BORDER, borderRight: LIGHT_GRAY_BORDER }, apply: APPLY_FILL_BORDER });

            // heading styles
            createStyleSheet('Title',     'headings', 1, { character: { bold: true, fontName: fontName, fontSize: 18, color: getScheme('dark2') }, apply: APPLY_FONT });
            createStyleSheet('Heading 1', 'headings', 2, { character: { bold: true, fontName: fontName, fontSize: 15, color: getScheme('dark2') }, cell: { borderBottom: { style: 'single', width: 79, color: getScheme('accent1') } },        apply: APPLY_FONT_BORDER });
            createStyleSheet('Heading 2', 'headings', 3, { character: { bold: true, fontName: fontName, fontSize: 13, color: getScheme('dark2') }, cell: { borderBottom: { style: 'single', width: 79, color: getSchemeTint40('accent1') } }, apply: APPLY_FONT_BORDER });
            createStyleSheet('Heading 3', 'headings', 4, { character: { bold: true, fontName: fontName, fontSize: 11, color: getScheme('dark2') }, cell: { borderBottom: { style: 'single', width: 53, color: getSchemeTint40('accent1') } }, apply: APPLY_FONT_BORDER });
            createStyleSheet('Heading 4', 'headings', 5, { character: { bold: true, fontName: fontName, fontSize: 11, color: getScheme('dark2') }, apply: APPLY_FONT });
            createStyleSheet('Total',     'headings', 9, { character: { bold: true, fontName: fontName, fontSize: 11, color: getScheme('dark1') }, cell: { borderTop: { style: 'single', width: 26, color: getScheme('accent1') }, borderBottom: { style: 'double', width: 79, color: getScheme('accent1') } }, apply: APPLY_FONT_BORDER });

            // other content styles
            createStyleSheet('Output',           'hidden', 1, { character: { fontName: fontName, color: getRGB('3F3F3F'), bold: true }, cell: { fillColor: getRGB('F2F2F2'), borderTop: DARK_GRAY_BORDER, borderBottom: DARK_GRAY_BORDER, borderLeft: DARK_GRAY_BORDER, borderRight: DARK_GRAY_BORDER }, apply: APPLY_FONT_FILL_BORDER });
            createStyleSheet('Calculation',      'hidden', 2, { character: { fontName: fontName, color: getRGB('FA7D00'), bold: true }, cell: { fillColor: getRGB('F2F2F2'), borderTop: MED_GRAY_BORDER,  borderBottom: MED_GRAY_BORDER,  borderLeft: MED_GRAY_BORDER,  borderRight: MED_GRAY_BORDER },  apply: APPLY_FONT_FILL_BORDER });
            createStyleSheet('Input',            'hidden', 3, { character: { fontName: fontName, color: getRGB('3F3F76') },             cell: { fillColor: getRGB('FFCC99'), borderTop: MED_GRAY_BORDER,  borderBottom: MED_GRAY_BORDER,  borderLeft: MED_GRAY_BORDER,  borderRight: MED_GRAY_BORDER },  apply: APPLY_FONT_FILL_BORDER });
            createStyleSheet('Explanatory Text', 'hidden', 4, { character: { fontName: fontName, color: getRGB('7F7F7F'), italic: true }, apply: APPLY_FONT });
            createStyleSheet('Linked Cell',      'hidden', 5, { character: { fontName: fontName, color: getRGB('FA7D00') }, cell: { borderBottom: { style: 'double', width: 17, color: getRGB('FA7D00') } }, apply: APPLY_FONT_BORDER });
            createStyleSheet('Warning Text',     'hidden', 6, { character: { fontName: fontName, color: getRGB('FF0000') }, apply: APPLY_FONT });
            createStyleSheet('Check Cell',       'hidden', 7, { character: { fontName: fontName, color: getScheme('light1'), bold: true }, cell: { fillColor: getRGB('A5A5A5'), borderTop: DOUBLE_GRAY_BORDER, borderBottom: DOUBLE_GRAY_BORDER, borderLeft: DOUBLE_GRAY_BORDER, borderRight: DOUBLE_GRAY_BORDER }, apply: APPLY_FONT_FILL_BORDER });

            // accent styles
            _.times(6, function (index) {
                var styleName = 'Accent ' + (index + 1),
                    colorName = 'accent' + (index + 1);
                createStyleSheet('20% - ' + styleName, 'themes', index,      { character: { fontName: fontName, color: getScheme('dark1') },  cell: { fillColor: getSchemeTint20(colorName) }, apply: APPLY_FONT_FILL });
                createStyleSheet('40% - ' + styleName, 'themes', index + 6,  { character: { fontName: fontName, color: getScheme('dark1') },  cell: { fillColor: getSchemeTint40(colorName) }, apply: APPLY_FONT_FILL });
                createStyleSheet('60% - ' + styleName, 'themes', index + 12, { character: { fontName: fontName, color: getScheme('light1') }, cell: { fillColor: getSchemeTint60(colorName) }, apply: APPLY_FONT_FILL });
                createStyleSheet(styleName,            'themes', index + 18, { character: { fontName: fontName, color: getScheme('light1') }, cell: { fillColor: getScheme(colorName) },        apply: APPLY_FONT_FILL });
            });

            // hyperlink style
            createStyleSheet('Hyperlink', 'hidden', 99, { character: { fontName: fontName, color: docModel.createJSONColor(Color.HYPERLINK, 'text'), underline: true }, apply: APPLY_FONT });
            hyperlinkStyleId = this.getStyleIdByName('Hyperlink');

            return this;
        };

        /**
         * Returns the identifier of the built-in 'Hyperlink' cell style sheet.
         *
         * @returns {String}
         *  The identifier of the built-in 'Hyperlink' cell style sheet.
         */
        this.getHyperlinkStyleId = function () {
            return hyperlinkStyleId;
        };

        // initialization -----------------------------------------------------

        // destroy class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } // class CellStyleCollection

    // exports ================================================================

    // derive this class from class StyleCollection
    return StyleCollection.extend({ constructor: CellStyleCollection });

});
