/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/cellcollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/font',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/object/arraytemplate',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/cellattributesmodel',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (Utils, Font, LocaleData, Parser, ArrayTemplate, TriggerObject, TimerMixin, Border, AttributeUtils, Config, SheetUtils, CellAttributesModel, TokenArray) {

    'use strict';

    var // convenience shortcuts
        ErrorCode = SheetUtils.ErrorCode,
        Address = SheetUtils.Address,
        Range = SheetUtils.Range,
        IntervalArray = SheetUtils.IntervalArray,
        AddressArray = SheetUtils.AddressArray,
        RangeArray = SheetUtils.RangeArray,

        // maximum number of cells contained in a server response
        MAX_CELLS_IN_RESPONSE = Utils.minMax(Math.floor(10 * Utils.PERFORMANCE_LEVEL), 100, 1000),

        // whether to calculate results of all changed formula cells locally
        DEBUG_LOCAL_FORMULAS = Config.getDebugUrlFlag('spreadsheet:local-formulas');

    // class CellDescriptor ===================================================

    /**
     * A simple descriptor object for a cell, returned by the public methods of
     * the class CellCollection.
     *
     * @constructor
     *
     * @property {Address} address
     *  The address of the cell represented by this instance.
     *
     * @property {Boolean} exists
     *  Whether the cell really exists in the cell collection. If set to false,
     *  the cell is undefined (blank), and the formatting attributes have been
     *  taken from the column or row containing the cell.
     *
     * @property {String|Null} display
     *  The formatted display string of the cell. The value null represents a
     *  cell result that cannot be formatted to a valid display string with the
     *  current number format of the cell.
     *
     * @property {Number|String|Boolean|Null} result
     *  The typed result value of the cell, or the formula result. The value
     *  null represents a blank cell (e.g. only formatted).
     *
     * @property {String|Null} formula
     *  The formula expression, if the cell is a formula cell; otherwise the
     *  value null.
     *
     * @property {String|Null} style
     *  The identifier of the auto style containing the character and cell
     *  formatting attributes for all undefined cells in the column/row.
     *
     * @property {Object} explicit
     *  The explicit formatting attribute set, including the cell formatting
     *  attributes for all undefined cells in the column/row.
     *
     * @property {Object} attributes
     *  The merged attribute set, including the cell formatting attributes for
     *  all undefined cells in the column/row.
     *
     * @property {ParsedFormat} format
     *  The parsed number format code of the cell.
     */
    function CellDescriptor(cellModel) {

        // cell address
        this.address = cellModel ? cellModel.address.clone() : null;
        this.exists = !!cellModel;

        // cell contents
        this.display = cellModel ? cellModel.display : '';
        this.result = cellModel ? cellModel.result : null;
        this.formula = cellModel ? cellModel.formula : null;

        // formatting attributes
        this.style = cellModel ? cellModel.getAutoStyleId() : null;
        this.explicit = cellModel ? cellModel.getExplicitAttributes() : null;
        this.attributes = cellModel ? cellModel.getMergedAttributes() : null;
        this.format = cellModel ? cellModel.getParsedFormat() : Parser.parseFormatCode('General');

    } // class CellDescriptor

    // static methods ---------------------------------------------------------

    /**
     * Creates a cell descriptor for a blank undefined cell located in the
     * column and row specified by the passed column and row descriptors. The
     * formatting attributes will be derived from the row, if it contains the
     * 'customFormat' attribute, otherwise from the column.
     *
     * @param {ColRowDescriptor}
     *  The descriptor for the column containing the blank cell.
     *
     * @param {ColRowDescriptor}
     *  The descriptor for the row containing the blank cell.
     *
     * @returns {CellDescriptor}
     *  The new cell descriptor.
     */
    CellDescriptor.createDefault = function (colDesc, rowDesc) {

        // create and initialize the cell descriptor
        var cellDesc = new CellDescriptor();
        cellDesc.address = new Address(colDesc.index, rowDesc.index);

        // Copies the explicit and merged attribute set from a column/row descriptor
        // to the cell descriptor, and deletes the extra column/row attributes. The
        // latter is needed to improve performance. The application controller would
        // detect changed formatting items, if a cell sometimes contains extra
        // column/row attributes, and would trigger a GUI update.
        function copyColRowAttrs(colRowDesc, propName) {
            cellDesc.style = colRowDesc.style;
            cellDesc.explicit = _.clone(colRowDesc.explicit);
            delete cellDesc.explicit[propName];
            cellDesc.attributes = _.clone(colRowDesc.attributes);
            delete cellDesc.attributes[propName];
            cellDesc.format = colRowDesc.format;
        }

        // if the row entry contains attributes, use them (ignore column default attributes)
        if (rowDesc.attributes.row.customFormat) {
            copyColRowAttrs(rowDesc, 'row');
        } else {
            copyColRowAttrs(colDesc, 'column');
        }

        return cellDesc;
    };

    // class HyperlinkModel ===================================================

    /**
     * Simple model representation of a cell range with hyperlink URL.
     *
     * @constructor
     *
     * @property {Range} range
     *  The address of the cell range the URL is attached to.
     *
     * @property {String} url
     *  The URL attached to the cell range.
     */
    function HyperlinkModel(range, url) {

        this.range = range.clone();
        this.url = url;

    } // class HyperlinkModel

    // public methods ---------------------------------------------------------

    /**
     * Returns a clone of this model instance.
     *
     * @returns {HyperlinkModel}
     *  A clone of this model instance.
     */
    HyperlinkModel.prototype.clone = function () {
        return new HyperlinkModel(this.range, this.url);
    };

    // class HyperlinkModelArray ==============================================

    /**
     * A typed array of HyperlinkModel instances.
     *
     * @constructor
     */
    var HyperlinkModelArray = ArrayTemplate.create(HyperlinkModel);

    // class CellCollection ===================================================

    /**
     * Collects cell contents and formatting attributes for a single sheet.
     * To save memory and to improve performance, instances of this class store
     * specific parts of the sheet only. More cell data will be fetched from
     * the server on demand.
     *
     * Triggers the following events:
     *  - 'change:cells'
     *      After the contents or formatting of some cells in this collection
     *      have been changed. Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} ranges
     *          The cell range addresses of all changed cells.
     *      (3) {Object} [options]
     *          Optional parameters:
     *          - {Boolean} [options.direct=false]
     *              If set to true, the cells have been changed locally in this
     *              client, and the event handlers shall process the changed
     *              cells immediately.
     *  - 'change:usedarea'
     *      After the size of the used area in the sheet has been changed.
     *      Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} usedCols
     *          The number of used columns in the sheet.
     *      (3) {Number} usedRows
     *          The number of used rows in the sheet.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends TimerMixin
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function CellCollection(sheetModel) {

        var // self reference
            self = this,

            // the document model, and model containers
            docModel = sheetModel.getDocModel(),
            grammarConfig = docModel.getGrammarConfig('ui'),
            cellStyles = docModel.getCellStyles(),
            numberFormatter = docModel.getNumberFormatter(),

            // the collections of the active sheet
            colCollection = sheetModel.getColCollection(),
            rowCollection = sheetModel.getRowCollection(),
            mergeCollection = sheetModel.getMergeCollection(),

            // token array used to calculate formula results of any cell entries locally
            tokenArray = new TokenArray(sheetModel, { temp: true }),

            // all cell entries mapped by column/row index
            cellMap = {},

            // the number of cell models in this collection (counted separately for performance)
            cellCount = 0,

            // the current bounding ranges (all ranges covered by this collection)
            boundRanges = new RangeArray(),

            // all hyperlink ranges
            linkModels = new HyperlinkModelArray(),

            // number of used columns and rows in the sheet
            usedCols = 0,
            usedRows = 0;

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        TimerMixin.call(this);

        // private class CellModel --------------------------------------------

        /**
         * Represents the collection entries of a CellCollection instance.
         *
         * @constructor
         *
         * @extends CellAttributesModel
         *
         * @property {Address} address
         *  The address of the cell represented by this instance.
         *
         * @property {String|Null} display
         *  The formatted display string of the cell. The value null represents
         *  a cell result that cannot be formatted to a valid display string
         *  with the current number format of the cell.
         *
         * @property {Number|String|Boolean|Null} result
         *  The typed result value of the cell, or the formula result. The
         *  value null represents a blank cell (e.g. only formatted).
         *
         * property {String|Null} formula
         *  The formula expression, if the cell is a formula cell; otherwise
         *  the value null.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Object} [json]
         *  A plain JSON object describing the contents and formatting of the
         *  new cell. If missing, creates a default cell model representing an
         *  empty unformatted cell. May contain the properties 'display',
         *  'result', 'formula', 'style', and 'attrs'.
         */
        var CellModel = CellAttributesModel.extend({ constructor: function (address, json) {

            this.address = address.clone();
            this.display = Utils.getOption(json, 'display', '');
            this.result = convertToResult(Utils.getOption(json, 'result', null));
            this.formula = Utils.getStringOption(json, 'formula', null);

            // base constructor - - - - - - - - - - - - - - - - - - - - - - - -

            // merge auto style identifier from JSON data with explicit attributes
            CellAttributesModel.call(this, sheetModel, _.extend({ styleId: Utils.getStringOption(json, 'style', '') }, Utils.getObjectOption(json, 'attrs')));

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // update current settings, e.g. display string according to formula or number format
            this.updateValue();

        } }); // class CellModel

        // static methods - - - - - - - - - - - - - - - - - - - - - - - - - - -

        CellModel.compare = function (cellModel1, cellModel2) {
            return cellModel1.address.compareTo(cellModel2.address);
        };

        CellModel.clone = function (sourceModel) {

            var newCellModel = new CellModel(sourceModel.address, { style: sourceModel.getAutoStyleId() });

            newCellModel.cloneFrom(sourceModel); // clone auto style and explicit attributes

            newCellModel.display = sourceModel.display;
            newCellModel.result =  sourceModel.result;
            newCellModel.formula =  sourceModel.formula;

            return newCellModel;
        };

        // public methods - - - - - - - - - - - - - - - - - - - - - - - - - - -

        /**
         * Updates the properties of this cell model, according to the current
         * formula and number format.
         */
        CellModel.prototype.updateValue = function () {

            // update formula result locally if specified via debug URL option
            if (DEBUG_LOCAL_FORMULAS && _.isString(this.formula) && (this.formula[0] === '=')) {
                tokenArray.parseFormula('ui', this.formula.substr(1));
                var result = tokenArray.interpretFormula('val', { refAddress: this.address, targetAddress: this.address });
                switch (result.type) {
                    case 'result':
                        this.result = result.value;
                        this.display = getDisplayText(this.result, '');
                        break;
                    case 'warn':
                    case 'error':
                        switch (result.value) {
                            case 'circular':
                                this.result = ErrorCode.VALUE;
                                this.display = grammarConfig.getErrorName(ErrorCode.VALUE);
                                break;
                            case 'unsupported':
                            case 'internal':
                            case 'missing':
                            case 'unexpected':
                                this.result = ErrorCode.NA;
                                this.display = grammarConfig.getErrorName(ErrorCode.NA);
                        }
                }
            }

            // create appropriate display text for 'General' number format (TODO: for other dynamic formats too, or even all formats?)
            if (_.isNumber(this.result) && isFinite(this.result) && (this.getParsedFormat().category === 'standard')) {
                this.display = numberFormatter.formatStandardNumber(this.result, SheetUtils.MAX_LENGTH_STANDARD_CELL);
            } else if (!_.isString(this.display)) {
                this.display = null;
            }
        };

        // private methods ----------------------------------------------------

        var log = SheetUtils.isLoggingActive() ? function (msg) {
            SheetUtils.log.apply(SheetUtils, [sheetModel.getName() + '.CellCollection' + msg].concat(_.toArray(arguments).slice(1)));
        } : _.noop;

        var logUsedArea = SheetUtils.isLoggingActive() ? this.createDebouncedMethod($.noop, function () {
            log(': usedRange=' + self.getUsedRange());
        }) : _.noop;

        var logBoundRanges = SheetUtils.isLoggingActive() ? this.createDebouncedMethod($.noop, function () {
            log(': boundRanges=' + boundRanges);
        }) : _.noop;

        /**
         * Joins the passed cell addresses to an array of range addresses, and
         * triggers a 'change:cells' event, if at least one cell address has
         * been passed to this method.
         *
         * @param {RangeArray|Range|Null} ranges
         *  An array with cell range addresses, or a single cell range address.
         *  Can be set to null to pass cell addresses only via the parameter
         *  'cells' (see below).
         *
         * @param {AddressArray|Null} cells
         *  An array with cell addresses of changed single cells. Can be set to
         *  null to pass cell ranges only via the parameter 'ranges' (see
         *  above).
         *
         * @param {Object} [options]
         *  Optional parameters passed to all event handlers.
         */
        function triggerChangeCellsEvent(ranges, cells, options) {

            var // the changed cell ranges
                changedRanges = ranges ? RangeArray.get(ranges) : new RangeArray(),
                // the changed cells, merged to cell range addresses
                changedCells = cells ? RangeArray.mergeAddresses(cells) : new RangeArray(),
                // the resulting merged cell ranges
                allRanges = changedRanges.empty() ? changedCells : changedRanges.concat(changedCells).merge(),
                // the bounding range of all changed cells
                boundRange = allRanges.boundary(),
                // ratio of bounding range covered by the changed range
                coveredRatio = boundRange ? (allRanges.cells() / boundRange.cells()) : 0;

            // do not trigger, if the passed parameters were missing/empty
            if (!allRanges.empty()) {
                // notify the bounding range, if it is covered by at least 80%
                self.trigger('change:cells', (coveredRatio >= 0.8) ? new RangeArray(boundRange) : allRanges, options);
            }
        }

        /**
         * Sets the number of used columns and rows in the sheet, and triggers
         * a 'change:usedarea' event, if at least one of the counts has been
         * changed.
         *
         * @param {Number} newUsedCols
         *  The number of columns used in the sheet.
         *
         * @param {Number} newUsedRows
         *  The number of rows used in the sheet.
         */
        function setUsedArea(newUsedCols, newUsedRows) {

            // if one count is zero, the other count will be zero too
            if ((newUsedCols === 0) || (newUsedRows === 0)) {
                newUsedCols = newUsedRows = 0;
            }

            // update members and trigger event
            if ((usedCols !== newUsedCols) || (usedRows !== newUsedRows)) {
                usedCols = newUsedCols;
                usedRows = newUsedRows;
                self.trigger('change:usedarea', usedCols, usedRows);
                logUsedArea();
            }
        }

        /**
         * Extends the number of used columns and rows in the sheet, so that
         * the specified cell is included in the used area, and triggers a
         * 'change:usedarea' event, if at least one of the counts has been
         * changed.
         *
         * @param {Address} address
         *  The address of the cell to be included in the used area.
         */
        function extendUsedArea(address) {
            setUsedArea(Math.max(address[0] + 1, usedCols), Math.max(address[1] + 1, usedRows));
        }

        /**
         * Extends the bounding ranges of this collection with the specified
         * cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array with cell range addresses, or a single cell range address,
         *  to be included into the bounding ranges of this collection.
         */
        function extendBoundRanges(ranges) {
            boundRanges = boundRanges.append(ranges).merge();
            logBoundRanges();
        }

        /**
         * Convert error code text to error code objects, and removes leading
         * apostrophes from strings.
         */
        function convertToResult(value) {
            var error = _.isString(value) ? grammarConfig.getErrorCode(value) : null;
            return error ? error : !_.isString(value) ? value : value.replace(/^'/, '');
        }

        /**
         * Converts the passed result value to an intermediate display string.
         *
         * @param {Any} result
         *  The typed result value.
         *
         * @param {String} defText
         *  Default text for missing/unsupported result values.
         *
         * @returns {String}
         *  A best-matching display string for the result value.
         */
        function getDisplayText(result, defText) {
            return _.isString(result) ? result :
                _.isNumber(result) ? numberFormatter.formatStandardNumber(result, SheetUtils.MAX_LENGTH_STANDARD_CELL) :
                _.isBoolean(result) ? grammarConfig.getBooleanName(result) :
                (result instanceof ErrorCode) ? grammarConfig.getErrorName(result) :
                defText;
        }

        /**
         * Tries to detect the type of the passed cell text with a few simple
         * tests, and returns a partial cell descriptor containing the typed
         * value and formula properties.
         *
         * @param {Object} contents
         *  A cell content descriptor, with optional 'value' and 'result'
         *  properties.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.parse=false]
         *      If set to true, the passed cell value must be a string and will
         *      be parsed to determine the data type.
         *
         * @returns {Object|Null}
         *  A cell descriptor containing the properties 'display' with the
         *  passed cell text, 'result' with the converted value (null, number,
         *  boolean, error code, or string), and 'formula' if the passed value
         *  may be a formula. If no value was passed, returns null.
         */
        function parseCellValue(contents, parse) {

            var // the value passed in the contents object
                value = contents.value,
                // the formula result passed in the contents object
                result = Utils.getOption(contents, 'result'),
                // the resulting data object
                cellData = null;

            // parse value if passed
            if (_.isString(value)) {
                cellData = { display: value, result: null, formula: null };
                if (value.length === 0) {
                    cellData.result = null;
                } else if (!_.isUndefined(result)) {
                    // formula with precalculated result
                    cellData.result = convertToResult(result);
                    cellData.display = getDisplayText(cellData.result, CellCollection.PENDING_DISPLAY);
                    cellData.formula = value;
                } else if (/^=./.test(value)) {
                    // formula without precalculated result
                    cellData.result = null;
                    cellData.display = CellCollection.PENDING_DISPLAY;
                    cellData.formula = value;
                } else if (parse) {
                    cellData.result = numberFormatter.parseResultValue(value);
                    if (_.isBoolean(cellData.result) || (cellData.result instanceof ErrorCode)) {
                        cellData.display = cellData.display.toUpperCase();
                    }
                } else {
                    cellData.result = value;
                }
            } else if (!_.isUndefined(value)) {
                cellData = { display: getDisplayText(value, ''), result: value, formula: null };
            }

            return cellData;
        }

        /**
         * Returns the cell model  with the specified cell address from the
         * collection.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {CellModel|Null}
         *  The cell model; or null, if the cell does not exist in this cell
         *  collection.
         */
        function getCellModel(address) {
            var cellModel = cellMap[address.key()];
            return cellModel ? cellModel : null;
        }

        /**
         * Inserts a new cell model into this collection. An existing cell will
         * be deleted.
         *
         * @param {CellModel} cellModel
         *  The new cell model to be inserted into this collection.
         */
        function putCellModel(cellModel) {
            var key = cellModel.address.key();
            if (key in cellMap) { cellMap[key].destroy(); } else { cellCount += 1; }
            cellMap[key] = cellModel;
        }

        /**
         * Removes a cell model from this collection and destroys it.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether an existing cell model at the position has been deleted.
         */
        function deleteCellModel(address) {
            var key = address.key(),
                cellModel = cellMap[key];
            if (cellModel) {
                cellModel.destroy();
                delete cellMap[key];
                cellCount -= 1;
                return true;
            }
            return false;
        }

        /**
         * Moves an cell model in this collection to a new position. An
         * existing cell model at the target position will be deleted.
         *
         * @param {Address} fromAddress
         *  The address of the cell to be moved.
         *
         * @param {Address} toAddress
         *  The address of the target position.
         *
         * @returns {Boolean}
         *  Whether the passed addresses are different, and an existing cell
         *  model has been moved to the new location, or an existing cell model
         *  at the old position has been deleted.
         */
        function moveCellModel(fromAddress, toAddress) {

            var moved = false;

            if (fromAddress.differs(toAddress)) {

                // destroy/remove old cell at target position
                var toKey = toAddress.key();
                if (toKey in cellMap) {
                    cellMap[toKey].destroy();
                    delete cellMap[toKey];
                    cellCount -= 1;
                    moved = true;
                }

                // move existing cell to target position
                var fromKey = fromAddress.key();
                if (fromKey in cellMap) {
                    cellMap[toKey] = cellMap[fromKey];
                    cellMap[toKey].address = toAddress.clone();
                    delete cellMap[fromKey];
                    moved = true;
                }
            }

            return moved;
        }

        /**
         * Creates a new cell model for this collection. An existing cell model
         * will be deleted.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Object} [json]
         *  A plain JSON object describing the contents and formatting of the
         *  new cell. If missing, creates a default cell model representing an
         *  empty unformatted cell. May contain the properties 'display',
         *  'result', 'formula', 'style', and 'attrs'.
         *
         * @returns {CellModel}
         *  A new cell model. Always contains the properties 'display',
         *  'result', 'formula', and 'style'.
         */
        function createCellModel(address, json) {
            var cellModel = new CellModel(address, json);
            putCellModel(cellModel);
            return cellModel;
        }

        /**
         * Returns the cell model at the specified address from the collection
         * for read/write access. If the cell entry is missing, it will be
         * created, and will be initialized with the default formatting of the
         * column or row.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {CellModel}
         *  The cell model corresponding to the passed cell address.
         */
        function getOrCreateCellModel(address) {

            // return an existing cell model
            var cellModel = getCellModel(address);
            if (cellModel) { return cellModel; }

            // no cell found: create a new cell model with auto style of row or column
            var rowDesc = rowCollection.getEntry(address[1]),
                colRowDesc = rowDesc.attributes.row.customFormat ? rowDesc : colCollection.getEntry(address[0]);
            return createCellModel(address, { style: colRowDesc.style, attrs: colRowDesc.explicit });
        }

        /**
         * Updates the value and/or formatting attributes of a cell model.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @param {Object} [parsedValue]
         *  The contents for the cell, as plain JSON object with in the
         *  (optional) properties 'display', 'result', and 'formula'.
         *
         * @param {Object} [attributes]
         *  If specified, additional explicit attributes for the cell.
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the method
         *  CellAttributesModel.setAttributes().
         */
        function updateCellModel(address, parsedValue, attributes, options) {

            // the cell model to be updated
            var cellModel = null;

            // update the cell value
            if (_.isObject(parsedValue)) {
                // do not create a new cell, if the cell will be cleared
                cellModel = _.isNull(parsedValue.result) ? getCellModel(address) : getOrCreateCellModel(address);
                if (cellModel) {
                    // bug 41191: do not use _.extend(cellModel, ...), the variable 'parsedValue'
                    // may contain any data including a (wrong) cell address
                    _.each(CellCollection.DEFAULT_CELL_DATA, function (value, key) {
                        if (key in parsedValue) { cellModel[key] = parsedValue[key]; }
                    });
                    cellModel.updateValue();
                }
            }

            // set auto style
            if (_.isObject(options) && _.isString(options.style)) {
                cellModel = cellModel || getOrCreateCellModel(address);
                cellModel.setAttributes({ styleId: options.style }, options);
            }

            // update explicit formatting attributes
            if (_.isObject(attributes)) {
                cellModel = cellModel || getOrCreateCellModel(address);
                cellModel.setAttributes(attributes, options);
            }
        }

        /**
         * Visits all cell positions in the passed range, regardless of their
         * contents.
         *
         * @param {Range} range
         *  The address of a single cell range.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each cell in the
         *  passed range. Receives the following parameters:
         *  (1) {ColRowDescriptor} colDesc
         *      A descriptor for the column containing the cell.
         *  (2) {ColRowDescriptor} rowDesc
         *      A descriptor for the row containing the cell.
         *  (3) {CellModel|Null} cellModel
         *      The model of the cell, if existing; otherwise null.
         *  (4) {Address} address
         *      The address of the visited cell (will also be set, if parameter
         *      'cellModel' is null).
         *  If the callback function returns the Utils.BREAK object, the
         *  iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the callback function  will be called with this
         *      context (the symbol 'this' will be bound to the context inside
         *      the callback function).
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle cells contained in hidden rows or
         *      columns. Must be one of the following values:
         *      - 'none' (or omitted): Only visible cells will be visited.
         *      - 'all': All visible and hidden cells will be visited.
         *      - 'last': All visible cells, and all cells contained in hidden
         *          columns and/or rows that precede a visible column/row will
         *          be visited.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the cells in the range will be visited in
         *      reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateCellsInRange(range, callback, options) {

            var // the column and row interval covered by the passed range
                colInterval = range.colInterval(),
                rowInterval = range.rowInterval(),
                // the calling context for the callback function
                context = Utils.getOption(options, 'context'),
                // whether to visit hidden cell entries
                hiddenMode = Utils.getStringOption(options, 'hidden', 'none'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false);

            return rowCollection.iterateEntries(rowInterval, function (rowDesc) {
                return colCollection.iterateEntries(colInterval, function (colDesc) {
                    var address = new Address(colDesc.index, rowDesc.index);
                    return callback.call(context, colDesc, rowDesc, getCellModel(address), address);
                }, { hidden: hiddenMode, reverse: reverse });
            }, { hidden: hiddenMode, reverse: reverse });
        }

        /**
         * Visits all collection entries that exist in the passed ranges, in no
         * specific order.
         *
         * @param {RangeArray|Range}
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Function} callback
         *  The callback function that will be invoked for each existing cell
         *  model in the passed cell ranges. Receives the following parameters:
         *  (1) {CellModel} cellModel
         *      The current cell model.
         *  (2) {String} key
         *      The unique key of the cell address in the cell map.
         *  If the callback function returns the Utils.BREAK object, the
         *  iteration process will be stopped immediately.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateExistingCellModels(ranges, callback) {

            // result of the 'some' loops
            var result = false;

            // convert passed ranges to disjunct ranges
            ranges = RangeArray.get(ranges).merge();

            // performance: decide whether to iterate over cell addresses in the ranges, or over the cell map
            if (cellCount < ranges.cells()) {

                // more cells in the ranges than collection entries: process all collection entries
                result = _.some(cellMap, function (cellModel, key) {
                    return ranges.containsAddress(cellModel.address) && (callback.call(self, cellModel, key) === Utils.BREAK);
                });

            } else {

                // more collection entries than cells in the ranges: iterate all cells in the ranges
                result = ranges.some(function (range) {
                    for (var row = range.start[1]; row <= range.end[1]; row += 1) {
                        for (var col = range.start[0]; col <= range.end[0]; col += 1) {
                            var address = new Address(col, row), key = address.key();
                            if ((key in cellMap) && (callback.call(self, cellMap[key], key, address) === Utils.BREAK)) { return true; }
                        }
                    }
                });
            }

            // if result is true, the callback function has returned Utils.BREAK
            return result ? Utils.BREAK : undefined;
        }

        /**
         * Deletes all cell entries that exist in the passed ranges.
         *
         * @param {RangeArray|Range}
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {AddressArray}
         *  An array with the addresses of all existing cells that have been
         *  deleted.
         */
        function deleteEntriesInRanges(ranges) {

            var // the addresses of all deleted cells
                deletedCells = new AddressArray();

            // delete all existing collection entries
            iterateExistingCellModels(ranges, function (cellModel, key) {
                deletedCells.push(cellModel.address);
                cellCount -= 1;
                cellModel.destroy();
                delete cellMap[key];
            });

            return deletedCells;
        }

        /**
         * Moves all cell entries that exist in the passed range, by the
         * specified amount of columns and rows.
         *
         * @param {Range}
         *  The address of the cell range to be moved.
         *
         * @param {Number} moveCols
         *  The number of columns the ranges will be moved. Negative values
         *  will move the ranges to the beginning of the sheet.
         *
         * @param {Number} moveRows
         *  The number of rows the ranges will be moved. Negative values will
         *  move the ranges to the beginning of the sheet.
         *
         * @returns {AddressArray}
         *  An array with the addresses of the old and new positions of all
         *  existing cells that have been moved, and the existing cells in the
         *  target range that have been deleted.
         */
        function moveEntriesInRange(range, moveCols, moveRows) {

            var // the target range to move the cells to
                targetRange = docModel.getCroppedMovedRange(range, moveCols, moveRows),
                // a temporary map used to collect moved entries (prevents overwriting existing entries)
                tempMap = {};

            // delete the parts of the target range not covered by the original range
            var changedCells = deleteEntriesInRanges(new RangeArray(targetRange).difference(range));

            // move all cell entries into the temporary map
            iterateExistingCellModels(range, function (cellModel, key) {

                // store the old cell address
                changedCells.push(cellModel.address.clone());

                // move the cell to the temporary map
                cellModel.address[0] += moveCols;
                cellModel.address[1] += moveRows;
                tempMap[cellModel.address.key()] = cellModel;
                changedCells.push(cellModel.address.clone());
                delete cellMap[key];
            });

            // insert all cell entries back into the map
            _.each(tempMap, function (cellModel, key) {
                cellMap[key] = cellModel;
            });

            return changedCells;
        }

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Range} range
         *  The address of the cell range to be inserted.
         *
         * @param {String} direction
         *  Either 'columns' to move the existing cells horizontally (to the
         *  right or left according to the sheet orientation), or 'rows' to
         *  move the existing cells down.
         *
         * @returns {AddressArray}
         *  An array with the cell addresses of all changed cells (the
         *  addresses of inserted cells with formatting copied from the
         *  preceding column/row, and the old and new addresses of existing
         *  moved cells).
         */
        var insertCells = SheetUtils.profileMethod('CellCollection.insertCells()', function (range, columns) {
            SheetUtils.log('range=' + range + ' columns=' + columns);

            var // the insertion interval
                interval = range.interval(columns),
                // the last available column/row index in the sheet
                maxIndex = docModel.getMaxIndex(columns),
                // number of columns the entries will be moved
                moveCols = columns ? range.cols() : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : range.rows(),
                // the cell range with all cells to be moved
                moveRange = range.clone(),
                // the entire modified range (inserted and moved)
                modifiedRange = range.clone(),
                // the bounding ranges inside the moved range
                movedBoundRanges = null,
                // the addresses of all changed cells
                changedCells = new AddressArray();

            // calculate moved and modified range
            moveRange.setEnd(maxIndex - (columns ? moveCols : moveRows), columns);
            modifiedRange.setEnd(maxIndex, columns);
            if (moveRange.getStart(columns) > moveRange.getEnd(columns)) { moveRange = null; }

            // calculate the moved parts of the bounding ranges
            if (moveRange) {
                movedBoundRanges = boundRanges.intersect(moveRange);
                movedBoundRanges.forEach(function (boundRange, index) {
                    movedBoundRanges[index] = docModel.getCroppedMovedRange(boundRange, moveCols, moveRows);
                });
            }

            // cut the entire modified area (inserted and moved) from the bounding ranges, and
            // add the target position of the existing moved parts of the old bounding ranges
            boundRanges = boundRanges.difference(modifiedRange);
            logBoundRanges();
            if (movedBoundRanges) { extendBoundRanges(movedBoundRanges); }

            // move the existing cell entries
            if (moveRange) {
                changedCells.append(moveEntriesInRange(moveRange, moveCols, moveRows));
            }

            // insert new cell entries according to preceding cells
            if (interval.first > 0) {

                var // the column/row collection in the specified move direction
                    collection = columns ? colCollection : rowCollection,
                    // the range address of the preceding column/row
                    prevRange = range.clone(),
                    // the names of the border attributes in move direction
                    innerBorderName1 = columns ? 'borderLeft' : 'borderTop',
                    innerBorderName2 = columns ? 'borderRight' : 'borderBottom',
                    // the names of the border attributes in the opposite direction
                    outerBorderName1 = columns ? 'borderTop' : 'borderLeft',
                    outerBorderName2 = columns ? 'borderBottom' : 'borderRight',
                    // limit number of cell entries inserted into the collection
                    count = 0;

                // adjust column/row indexes for the range address preceding the inserted range
                prevRange.setBoth(interval.first - 1, columns);

                // process all cell entries existing in the previous column/row
                self.iterateCellsInRanges(boundRanges.intersect(prevRange), function (prevCellDesc) {

                    var // the explicit cell attributes of the preceding cell
                        prevCellAttrs = prevCellDesc.explicit.cell,
                        // the descriptor of the cell after the inserted range
                        nextCellDesc = null,
                        // the explicit cell attributes of the following cell
                        nextCellAttrs = null,
                        // the explicit attributes to be set at the new cells
                        newAttributes = _.copy(prevCellDesc.explicit),
                        // a helper cell address to obtain cell entries from the collection
                        tempAddress = prevCellDesc.address.clone();

                    // returns whether the passed cell attribute map contains two equal border attributes
                    function hasEqualBorders(cellAttributes, borderName1, borderName2) {
                        var border1 = cellAttributes[borderName1], border2 = cellAttributes[borderName2];
                        return _.isObject(border1) && _.isObject(border2) && Border.isEqual(border1, border2);
                    }

                    // returns the border attribute, if it is equal in both passed cell attribute maps
                    function getEqualBorder(cellAttributes1, borderName1, cellAttributes2, borderName2) {
                        var border1 = cellAttributes1[borderName1], border2 = cellAttributes2[borderName2];
                        return (_.isObject(border1) && _.isObject(border2) && Border.isEqual(border1, border2)) ? border1 : null;
                    }

                    // get following cell
                    var nextIndex = range.getEnd(columns) + 1;
                    tempAddress.set(nextIndex, columns);
                    if (nextIndex <= maxIndex) {
                        nextCellDesc = self.getCellEntry(tempAddress);
                        nextCellAttrs = nextCellDesc.explicit.cell;
                    }

                    // prepare explicit border attributes for the new cells
                    if (_.isObject(prevCellAttrs)) {

                        // remove old borders
                        delete newAttributes.cell[innerBorderName1];
                        delete newAttributes.cell[innerBorderName2];
                        delete newAttributes.cell[outerBorderName1];
                        delete newAttributes.cell[outerBorderName2];

                        // copy borders, that are equal in preceding and following cell entries
                        if (_.isObject(nextCellAttrs)) {

                            var border = null;

                            // insert columns: clone top border if equal in preceding and following column
                            // insert rows: clone left border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, outerBorderName1, nextCellAttrs, outerBorderName1))) {
                                newAttributes.cell[outerBorderName1] = _.copy(border, true);
                            }

                            // insert columns: clone bottom border if equal in preceding and following column
                            // insert rows: clone right border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, outerBorderName2, nextCellAttrs, outerBorderName2))) {
                                newAttributes.cell[outerBorderName2] = _.copy(border, true);
                            }

                            // insert columns: clone left/right border if equal in preceding and following column
                            // insert rows: clone top/bottom border if equal in preceding and following row
                            if ((border = getEqualBorder(prevCellAttrs, innerBorderName2, nextCellAttrs, innerBorderName1))) {
                                // ... but only if *both* borders in either preceding or following column/row are equal
                                if (hasEqualBorders(prevCellAttrs, innerBorderName1, innerBorderName2) || hasEqualBorders(nextCellAttrs, innerBorderName1, innerBorderName2)) {
                                    newAttributes.cell[innerBorderName1] = _.copy(border, true);
                                    newAttributes.cell[innerBorderName2] = _.copy(border, true);
                                }
                            }
                        }

                        // remove empty cell attribute map
                        if (_.isEmpty(newAttributes.cell)) {
                            delete newAttributes.cell;
                        }
                    }

                    // no cell entries, if no explicit attributes left
                    if (_.isEmpty(newAttributes)) { return; }

                    // prevent memory overflow or performance problems
                    count += interval.size();
                    if (count > SheetUtils.MAX_FILL_CELL_COUNT) { return Utils.BREAK; }

                    // create equal cell entries inside the insertion interval
                    collection.iterateEntries(interval, function (moveDesc) {
                        tempAddress.set(moveDesc.index, columns);
                        changedCells.push(tempAddress.clone());
                        createCellModel(tempAddress, { attrs: newAttributes });
                    }, { hidden: 'all' });

                }, { type: 'existing', hidden: 'all' });
            }

            return changedCells;
        });

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Range} range
         *  The address of the cell range to be deleted.
         *
         * @param {Boolean} columns
         *  Either true to move the remaining cells horizontally (to the left
         *  or right according to the sheet orientation), or false to move the
         *  remaining cells up.
         *
         * @returns {AddressArray}
         *  An array with the cell addresses of all changed cells (the
         *  addresses of the deleted cells, and the old and new addresses of
         *  existing moved cells).
         */
        var deleteCells = SheetUtils.profileMethod('CellCollection.deleteCells()', function (range, columns) {
            SheetUtils.log('range=' + range + ' columns=' + columns);

            var // the last available column/row index in the sheet
                maxIndex = docModel.getMaxIndex(columns),
                // number of columns the entries will be moved
                moveCols = columns ? -range.cols() : 0,
                // number of rows the entries will be moved
                moveRows = columns ? 0 : -range.rows(),
                // the cell range with all cells to be moved
                moveRange = range.clone(),
                // the entire modified range (deleted and moved)
                modifiedRange = range.clone(),
                // the bounding ranges inside the moved range
                movedBoundRanges = null,
                // the addresses of all changed cells
                changedCells = null;

            // calculate moved and modified range
            moveRange.setStart(range.getEnd(columns) + 1, columns);
            moveRange.setEnd(maxIndex, columns);
            modifiedRange.setEnd(maxIndex, columns);
            if (moveRange.getStart(columns) > moveRange.getEnd(columns)) { moveRange = null; }

            // calculate the target position of the moved parts of the bounding ranges
            if (moveRange) {
                movedBoundRanges = boundRanges.intersect(moveRange);
                movedBoundRanges.forEach(function (boundRange, index) {
                    movedBoundRanges[index] = docModel.getCroppedMovedRange(boundRange, moveCols, moveRows);
                });
            }

            // cut the entire modified area (deleted and moved) from the bounding ranges, and
            // add the target position of the existing moved parts of the old bounding ranges
            boundRanges = boundRanges.difference(modifiedRange);
            logBoundRanges();
            if (movedBoundRanges) { extendBoundRanges(movedBoundRanges); }

            // clear the entire deleted range, then move the following cell entries
            changedCells = deleteEntriesInRanges(range);
            if (moveRange) {
                changedCells.append(moveEntriesInRange(moveRange, moveCols, moveRows));
            }

            return changedCells;
        });

        /**
         * Returns whether the passed attribute set contains any formatting
         * attributes for cells, or a style sheet identifier.
         */
        function containsCellAttributes(attributeSet) {
            return ('styleId' in attributeSet) || cellStyles.getSupportedFamilies().some(function (family) { return family in attributeSet; });
        }

        /**
         * Updates the cell entries after changing the formatting attributes of
         * entire columns or rows.
         */
        function changeIntervalHandler(interval, columns, attributes, options) {

            var // the column/row range covering the passed interval
                range = docModel.makeFullRange(interval, columns),
                // all merged ranges covering the range partly or completely
                mergedRanges = null,
                // special treatment for border attributes applied to entire ranges
                rangeBorders = Utils.getBooleanOption(options, 'rangeBorders', false),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = Utils.getBooleanOption(options, 'visibleBorders', false),
                // additional options for border attributes
                attributeOptions = { visibleBorders: visibleBorders },
                // the row intervals of the bounding ranges covering the modified columns
                rowIntervals = null,
                // the number of modified cells
                count = 0;

            // do nothing if the passed attributes do not modify the cell formatting
            if (!attributes || !containsCellAttributes(attributes)) { return; }

            // all merged ranges covering the range partly or completely
            mergedRanges = mergeCollection.getMergedRanges(range);

            // update all existing cell entries
            self.iterateCellsInRanges(range, function (cellDesc) {

                // do not update any merged ranges here
                if (mergedRanges.containsAddress(cellDesc.address)) { return; }

                // add options for inner/outer border treatment
                if (rangeBorders) {
                    attributeOptions.innerLeft = cellDesc.address[0] > range.start[0];
                    attributeOptions.innerRight = cellDesc.address[0] < range.end[0];
                    attributeOptions.innerTop = cellDesc.address[1] > range.start[1];
                    attributeOptions.innerBottom = cellDesc.address[1] < range.end[1];
                }

                // update the cell model
                updateCellModel(cellDesc.address, undefined, attributes, attributeOptions);

            }, { type: 'existing', hidden: 'all' });

            // update reference cell of all merged ranges located completely inside the passed range
            mergedRanges.forEach(function (mergedRange) {

                // do not handle merged ranges not contained completely
                if (!range.contains(mergedRange)) { return; }

                // add options for inner/outer border treatment
                if (rangeBorders) {
                    attributeOptions.innerLeft = mergedRange.start[0] > range.start[0];
                    attributeOptions.innerRight = mergedRange.end[0] < range.end[0];
                    attributeOptions.innerTop = mergedRange.start[1] > range.start[1];
                    attributeOptions.innerBottom = mergedRange.end[1] < range.end[1];
                }

                // update the cell model
                updateCellModel(mergedRange.start, undefined, attributes, attributeOptions);
            });

            // column formatting: create cell entries for all rows with explicit default formats
            if (!columns) { return; }

            // the merged row intervals of the bounding ranges covering the modified columns
            rowIntervals = boundRanges.intersect(range).rowIntervals().merge();

            // visit all rows with custom formatting attributes
            return rowCollection.iterateEntries(rowIntervals, function (rowDesc) {

                // add options for inner/outer border treatment
                if (rangeBorders) {
                    attributeOptions.innerTop = rowDesc.index > range.start[1];
                    attributeOptions.innerBottom = rowDesc.index < range.end[1];
                }

                // prevent memory overflow or performance problems
                count += interval.size();
                if (count > SheetUtils.MAX_FILL_CELL_COUNT) { return Utils.BREAK; }

                // update cells in all columns of the row
                colCollection.iterateEntries(interval, function (colDesc) {

                    var // the cell address
                        address = new Address(colDesc.index, rowDesc.index);

                    // do not modify existing cell entries (already updated above)
                    if (getCellModel(address)) { return; }

                    // add options for inner/outer border treatment
                    if (rangeBorders) {
                        attributeOptions.innerLeft = colDesc.index > interval.first;
                        attributeOptions.innerRight = colDesc.index < interval.last;
                    }

                    // update the cell model
                    updateCellModel(address, undefined, attributes, attributeOptions);

                }, { hidden: 'all' });
            }, { hidden: 'all', customFormat: true });
        }

        /**
         * Imports the passed cell contents array as received from a server
         * notification.
         *
         * @param {Array} json
         *  The JSON representation of the cell ranges to be imported into this
         *  cell collection, as array of cell descriptors with the following
         *  properties:
         *  - {Array<Number>} start
         *      The address of the first cell in a range with equal contents
         *      and formatting.
         *  - {Array<Number>} [end]
         *      The address of the last cell in a range with equal contents and
         *      formatting. If omitted, a single cell will be addressed.
         *  - {Number|String|Boolean|Null} [result]
         *      The typed value (or formula result) of the cells. Error codes
         *      are encoded as strings with leading hash character. The value
         *      null, or the omitted property represents a blank cell.
         *  - {String|Null} [display]
         *      The formatted display string of the cells. The omitted property
         *      represents the empty string (either blank cells, or an empty
         *      string as formula result, or a number format resulting in an
         *      empty string). The value null represents a cell result that
         *      cannot be formatted to a valid display string with the current
         *      number format of the cell.
         *  - {String} [formula]
         *      The formula expression of formula cells, without the leading
         *      equality sign. If omitted, the cells are not formula cells.
         *  - {String} [style]
         *      The identifier of an auto style containing the formatting of
         *      the cells. If omitted, the cell will be formatted with the
         *      default cell style sheet of the document, without additional
         *      explicit attributes.
         *
         * @returns {RangeArray}
         *  The addresses of all imported cell ranges.
         */
        var importRangeContents = SheetUtils.profileMethod('CellCollection.importRangeContents()', function (json) {

            // the imported cell ranges
            var changedRanges = new RangeArray();

            // process all entries in the passed array
            json.forEach(function (jsonEntry) {

                // try to extract a valid range address
                var range = docModel.createRange(jsonEntry);
                if (!range) { return; }

                // create entries for all cells in the range
                var address = Address.A1.clone();
                for (address[1] = range.start[1]; address[1] <= range.end[1]; address[1] += 1) {
                    for (address[0] = range.start[0]; address[0] <= range.end[0]; address[0] += 1) {
                        createCellModel(address, jsonEntry);
                    }
                }

                // add the range to the result array
                changedRanges.push(range);
            });

            // merge the cell ranges (in most cases, each cell is imported in its own array entry)
            changedRanges = changedRanges.merge();
            SheetUtils.log('imported ranges: ' + changedRanges + ' (' + changedRanges.cells() + ' cells)');

            // process the merged ranges covering the imported ranges
            var mergedRanges = mergeCollection.getMergedRanges(changedRanges);
            if (!mergedRanges.empty()) {

                // collect all merged ranges in a map (start cell address as key)
                var mergedRangeMap = {};
                mergedRanges.forEach(function (mergedRange) {
                    mergedRangeMap[mergedRange.start.key()] = mergedRange;
                });

                // add merged ranges to changed ranges if their reference cell has been updated,
                // and delete cell models for all cells covered/hidden by merged ranges
                iterateExistingCellModels(mergedRanges.intersect(changedRanges), function (cellModel, key) {
                    if (key in mergedRangeMap) {
                        changedRanges.push(mergedRangeMap[key]);
                    } else {
                        deleteCellModel(cellModel.address);
                    }
                });

                // merge the changed ranges again (merged ranges have been appended to the array)
                changedRanges = changedRanges.merge();
            }

            return changedRanges;
        });

        /**
         * Fetches the contents and formatting of the passed cell ranges, and
         * updates the entries of this collection. If the passed ranges are too
         * large, the server may respond with less cells than requested. The
         * response of the promise returned by this method will contain the
         * addresses of the ranges that have actually been imported.
         *
         * @param {RangeArray} ranges
         *  The addresses of the cell ranges to be imported.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the array of range addresses
         *  of all cells that have been returned by the server request.
         */
        function requestRangeContents(ranges) {

            var // the application instance
                app = docModel.getApp(),
                // the server request
                promise = null;

            // do nothing without any ranges, or if application is in quit/error state
            if (ranges.empty()) { return $.when(ranges); }

            // do nothing if application is in quit/error state
            if (app.isInQuit() || app.isInternalError()) { return $.Deferred().reject(); }

            // merge the cell ranges
            ranges = ranges.merge();
            SheetUtils.info('CellCollection.requestRangeContents(): requesting cells: ranges=' + ranges + ' max=' + MAX_CELLS_IN_RESPONSE);

            // wait for pending actions before sending the server request (fix for bug 30044,
            // must wait for the 'insertSheet' operation before requesting data for that sheet)
            promise = app.saveChanges().then(function () {
                return app.sendActionRequest('updateview', {
                    sheet: sheetModel.getIndex(),
                    locale: Config.LOCALE,
                    cells: ranges,
                    maxCells: MAX_CELLS_IN_RESPONSE
                });
            });

            promise.fail(function (response) {
                if (response !== 'abort') {
                    SheetUtils.error('CellCollection.requestRangeContents(): request failed, response:', response);
                }
            });

            // import all cells contained in the response
            promise = promise.then(function (json) {

                // bug 36913: sheet may have been deleted when the response arrives
                // TODO: need a way to bind promise chains to the lifetime of an arbitrary object
                if (!self || self.destroyed) { return; }

                SheetUtils.info('CellCollection.requestRangeContents(): response=', json);

                var // the ranges covered by the server response (default: all requested ranges)
                    responseRanges = ranges,
                    // the deleted cells in the imported ranges
                    changedCells = null,
                    // the changed ranges in the imported ranges
                    changedRanges = null;

                // update the number of used columns and rows
                setUsedArea(
                    Utils.getIntegerOption(json.sheet, 'usedCols', usedCols),
                    Utils.getIntegerOption(json.sheet, 'usedRows', usedRows)
                );

                // extract the range addresses contained in the response
                if (_.isArray(json.cellRanges)) {
                    responseRanges = docModel.createRangeArray(json.cellRanges);
                    if (!responseRanges || (responseRanges.length === 0)) {
                        Utils.error('CellCollection.requestRangeContents(): invalid bounding ranges in response');
                        return $.Deferred().reject();
                    }
                }

                // merge the new ranges into the bounding ranges of this collection
                extendBoundRanges(responseRanges);

                // delete the cell range
                changedCells = deleteEntriesInRanges(responseRanges);

                // import cell contents
                if (_.isArray(json.cells)) {
                    changedRanges = importRangeContents(json.cells);
                }

                // notify all change listeners
                triggerChangeCellsEvent(changedRanges, changedCells);

                // resolve returned promise with the response ranges
                return responseRanges;
            });

            return promise;
        }

        /**
         * Fetches the contents and formatting of all passed cell ranges, and
         * updates the entries of this collection. If the passed ranges are too
         * large, the server may respond with less cells than requested. This
         * method will send further server requests, until all passed ranges
         * have been imported.
         *
         * @param {RangeArray} ranges
         *  The addresses of the cell ranges to be updated.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.missing=false]
         *      If set to true, the passed ranges will be reduced to the ranges
         *      not yet covered by this cell collection (outside the bounding
         *      ranges of this collection). Used to expand this cell collection
         *      without querying known cell contents.
         */
        var fetchAllRangeContents = (function () {

            var // the cell ranges to be queried with the next request
                pendingRanges = new RangeArray(),
                // the cell ranges queried with the current request
                requestRanges = null;

            // direct callback: collect the passed cell range addresses
            function registerRequest(ranges, options) {

                // reduce to unknown cell ranges if specified
                if (Utils.getBooleanOption(options, 'missing', false)) {
                    // remove bounding ranges from the passed ranges
                    ranges = ranges.difference(boundRanges);
                    // remove ranges currently queried from the passed ranges
                    if (requestRanges) {
                        ranges = ranges.difference(requestRanges);
                    }
                }

                // add passed ranges to the array of pending ranges to be queried
                pendingRanges.append(ranges);
            }

            // deferred callback: send single server request for all collected ranges
            function executeRequest() {

                // nothing to do, if pending ranges are still empty, or if requests are already running
                if (requestRanges || pendingRanges.empty()) { return; }

                // immediately clear the pending ranges array (new ranges may be fetched while the server request is running)
                requestRanges = pendingRanges;
                pendingRanges = new RangeArray();

                var // send the server request
                    promise = requestRangeContents(requestRanges);

                // start a new request for ranges not contained in the response
                promise.done(function (responseRanges) {

                    var // missing cell ranges (requested, but not contained in the server response)
                        missingRanges = requestRanges.difference(responseRanges);

                    // reset the array containing the requested ranges for the next server request
                    requestRanges = null;

                    // add the missing ranges to the pending ranges array, and start a new server query
                    // (also for ranges that have been added to the pending ranges in the meantime)
                    fetchAllRangeContents(missingRanges);
                });

                // clean-up after server error
                promise.fail(function () {
                    requestRanges = null;
                    // TODO: Back to pendingRanges in case of a server error? Must be included in transformation handler then.
                });
            }

            // create and return the debounced method fetchAllRangeContents()
            return self.createDebouncedMethod(registerRequest, executeRequest, { delay: 50, maxDelay: 250 });
        }());

        /**
         * Handles 'docs:update:cells' notifications. If the bound ranges of
         * this collection covers any changed cells, they will be queried from
         * the server.
         *
         * @param {Object} json
         *  A map with cell descriptors and range addresses of changed ranges
         *  per sheet.
         */
        function updateCellsHandler(changedData) {

            var // extract the changed data of the own sheet
                sheetData = changedData[sheetModel.getIndex()];

            // nothing to do, if the own sheet did not change at all
            if (!_.isObject(sheetData)) { return; }

            // update the number of used columns and rows
            setUsedArea(
                Utils.getIntegerOption(sheetData, 'usedCols', usedCols),
                Utils.getIntegerOption(sheetData, 'usedRows', usedRows));

            // import all cell contents, and notify all change listeners
            if (_.isArray(sheetData.rangeContents)) {
                log('.updateCellsHandler: rangeContents', sheetData.rangeContents);
                triggerChangeCellsEvent(importRangeContents(sheetData.rangeContents));
            }

            // request cell contents for all other changed ranges (reduce to the bounding ranges)
            if (sheetData.dirtyRanges) {
                log('.updateCellsHandler: dirtyRanges=' + sheetData.dirtyRanges);
                fetchAllRangeContents(sheetData.dirtyRanges.intersect(boundRanges));
            }
        }

        /**
         * Updates this collection, after columns or rows have been inserted
         * into or deleted from the sheet.
         */
        function transformationHandler(interval, insert, columns) {

            var // the range address of the inserted/deleted cells
                range = docModel.makeFullRange(interval, columns);

            if (insert) {
                insertCells(range, columns);
            } else {
                deleteCells(range, columns);
            }

            // transform all hyperlink ranges
            linkModels = linkModels.filter(function (linkModel) {
                linkModel.range = docModel.transformRange(linkModel.range, interval, insert, columns);
                return _.isObject(linkModel.range);
            });
        }

        /**
         * Updates all affected cell entries after merged ranges have been
         * inserted into the sheet.
         */
        function insertMergedHandler(event, ranges, options) {

            // bug 37969: do not move cell contents, or delete covered ranges,
            // when the merged ranges have been moved after a column/row operation
            // (will be handled by the method CellCollection.transformationHandler())
            if (Utils.getBooleanOption(options, 'implicit', false)) { return; }

            // moves the formatting attributes of the reference cell to a column or row
            function moveAttributesToColRow(range, columns) {

                var // the column/row collection to be modified
                    collection = columns ? colCollection : rowCollection,
                    // the column/row interval of the range
                    interval = range.interval(columns),
                    // the address of the reference cell
                    address = columns ? new Address(interval.first, 0) : new Address(0, interval.first),
                    // the model for the reference cell
                    cellModel = getCellModel(address);

                // copy attributes of the cell to the column/row
                if (cellModel) {
                    var attributeSet = { styleId: cellModel.styleId };
                    if (!columns) { attributeSet.row = { customFormat: true }; }
                    collection.setAttributes(interval, attributeSet, { silent: true });
                }
            }

            // add to bounding ranges (all cells have the same format)
            extendBoundRanges(ranges);

            // move value and formatting of first non-empty cell in each range to reference cell
            ranges.forEach(function (range) {

                var // the first non-empty cell will be moved to the reference cell
                    moved = false;

                // delete all covered cells, move first non-empty cell to reference cell
                self.iterateCellsInRanges(range, function (cellDesc) {
                    if (!moved && !CellCollection.isBlank(cellDesc)) {
                        // move first non-empty cell to the reference position
                        moved = true;
                        moveCellModel(cellDesc.address, range.start);
                    } else if (!range.startsAt(cellDesc.address)) {
                        // remove all other cell entries (but the first) from the collection
                        deleteCellModel(cellDesc.address);
                    }
                }, { type: 'existing', hidden: 'all' });

                // special handling for column and row ranges
                if (docModel.isColRange(range)) {
                    moveAttributesToColRow(range, true);
                } else if (docModel.isRowRange(range)) {
                    moveAttributesToColRow(range, false);
                }
            });

            // notify listeners
            triggerChangeCellsEvent(ranges);
        }

        /**
         * Updates all affected cell entries after merged ranges have been
         * deleted from the sheet.
         */
        function deleteMergedHandler(event, ranges, options) {

            // do nothing for merged ranges that have been deleted implicitly while deleting
            // columns or rows (will be handled by the method CellCollection.transformationHandler())
            if (Utils.getBooleanOption(options, 'implicit', false)) { return; }

            var // the parts of the passed ranges covered by the collection
                changedRanges = new RangeArray();

            // copy formatting of reference cell to all other cells formerly hidden by the
            // merged range (but not for entire column/row ranges)
            ranges.forEach(function (range) {

                // do not create cell entries for entire columns/rows
                // TODO: unmerging columns needs cell entries for explicitly formatted rows
                if (docModel.isColRange(range) || docModel.isRowRange(range)) { return; }

                var // the model of the reference cell
                    cellModel = getCellModel(range.start);

                // nothing to do, if the cell has no formatting attributes (TODO: handle default auto style)
                if (!cellModel) {
                    extendBoundRanges(range);
                    changedRanges.push(range);
                    return;
                }

                var // the auto style identifier
                    styleId = cellModel.getAutoStyleId(),
                    // the explicit attributes of the reference cell
                    attributes = cellModel.getExplicitAttributes(),
                    // performance: do not fill more than a specific number of cells
                    fillRanges = new RangeArray(range).shortenTo(SheetUtils.MAX_UNMERGE_CELL_COUNT);

                // add to bounding ranges (all cells have the same format)
                extendBoundRanges(fillRanges);

                // register in changed ranges, add to bounding ranges (all cells have the same format)
                changedRanges.append(fillRanges);

                // create the cell entries
                self.iterateCellsInRanges(fillRanges, function (cellDesc) {
                    updateCellModel(cellDesc.address, undefined, attributes, { style: styleId });
                }, { hidden: 'all' });
            });

            // notify listeners
            triggerChangeCellsEvent(changedRanges);
        }

        /**
         * preformat should be called when cell is changed,
         * if value is assigned or in the attribues is a numberformat-entry
         * the numberformatter is called and assignedValue gets an display-entry
         * if assignedValue is null a new object is created
         *
         * @returns {Object}
         *  assignedValue or a new object
         */
        function preformat(assignValue, value, attributes, cellDesc) {

            var // temporary cell-attributes for the formating
                assignAttrs,
                // temporary cell-result for the formating
                assignResult,
                // the numberformat-entry taken from the operation
                numberFormat = null;

            assignValue = assignValue || {};

            //if the operation has a numberformat-entry then the numberformat was changed, so we want to pre-format
            if (attributes && attributes.cell && attributes.cell.numberFormat) {
                numberFormat = attributes.cell.numberFormat;
            }

            //if the operation has a value-entry then the value was changed, so we want to pre-format
            if (_.isString(value) || _.isNumber(value) || numberFormat) {
                if (numberFormat) {
                    assignAttrs = attributes;
                    assignResult = cellDesc.result;
                } else {
                    assignAttrs = cellDesc.attributes;
                    assignResult = value;

                    // parse only needed if values are changed
                    var unform = numberFormatter.parseEditString(assignResult, cellDesc.format.category);
                    if (_.isString(unform) || _.isNumber(unform)) {
                        assignResult = unform;
                        assignValue.result = unform;
                    }
                }

                var formatCode = numberFormatter.resolveFormatCode(assignAttrs ? assignAttrs.cell.numberFormat : null);
                assignValue.display = _.isNull(assignResult) ? '' : numberFormatter.formatValue(assignResult, formatCode);
            }

            return assignValue;
        }

        // protected methods --------------------------------------------------

        /**
         * Returns the internal contents of this collection, needed for cloning
         * into another collection.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @returns {Object}
         *  The internal contents of this collection.
         */
        this.getCloneData = function () {
            return {
                cellMap: cellMap,
                cellCount: cellCount,
                boundRanges: boundRanges,
                linkModels: linkModels,
                usedCols: usedCols,
                usedRows: usedRows
            };
        };

        /**
         * Clones all contents from the passed collection into this collection.
         *
         * @internal
         *  Used by the class SheetModel during clone construction. DO NOT CALL
         *  from external code!
         *
         * @param {CellCollection} collection
         *  The source collection whose contents will be cloned into this
         *  collection.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.cloneFrom = function (collection) {

            var // the internal contents of the source collection
                cloneData = collection.getCloneData();

            // clone the contents of the source collection
            _.each(cloneData.cellMap, function (cellModel, key) {

                var newCellModel = CellModel.clone(cellModel);
                cellMap[key] = newCellModel;
            });

            cellCount = cloneData.cellCount;
            boundRanges = cloneData.boundRanges.clone(true);
            linkModels = cloneData.linkModels.clone(true);
            usedCols = cloneData.usedCols;
            usedRows = cloneData.usedRows;

            return this;
        };

        /**
         * Callback handler for the document operation 'setCellContents'.
         * Changes the contents and formatting attributes of the cells in this
         * collection, and triggers a 'change:cells' event for all changed
         * cells.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'setCellContents' document operation.
         */
        this.applySetCellContentsOperation = function (context) {

            var // the address of the top-left cell
                start = context.getAddress('start'),
                // the cell contents (2D array)
                contents = context.getArr('contents'),
                // whether to parse the cell contents
                parse = context.getOptStr('parse').length > 0,
                // the maximum column index in the sheet
                maxCol = docModel.getMaxCol(),
                // the maximum number of columns in a changed row
                colCount = 0,
                // the addresses of all changed cells
                changedCells = new AddressArray();

            // check that the 'contents' operation property is a 2D array
            context.ensure(contents.every(_.isArray), 'invalid row in cell contents');

            // check validity of the covered row interval
            context.ensure(start[1] + contents.length - 1 <= docModel.getMaxRow(), 'cell contents array too long');

            // process all rows in the contents array
            contents.forEach(function (rowContents, rowIndex) {

                var // the address of the current cell
                    address = new Address(start[0], start[1] + rowIndex);

                // process all cell entries in the row array
                rowContents.forEach(function (cellContents) {

                    var // number of repetitions for this cell
                        repeat = Utils.getIntegerOption(cellContents, 'repeat', 1);

                    // check validity of the covered column interval
                    context.ensure(address[0] + repeat - 1 <= maxCol, 'row element in cell contents too long');

                    // skip gaps (null value is allowed in operation and leaves cell unmodified)
                    if (_.isNull(cellContents)) {
                        address[0] += repeat;
                        return;
                    }

                    // otherwise, array element must be an object
                    context.ensure(_.isObject(cellContents), 'invalid cell contents element');

                    // update the affected cell entries with the parsed value
                    _.times(repeat, function () {

                        var parsedValue = parseCellValue(cellContents, parse),
                            cellDesc = self.getCellEntry(address);

                        if (_.isObject(parsedValue)) {
                            parsedValue = preformat(parsedValue, parsedValue.result, null, cellDesc);
                        }

                        updateCellModel(address, parsedValue, cellContents.attrs);
                        changedCells.push(address.clone());
                        address[0] += 1;
                    });
                });

                // update the maximum column count
                colCount = Math.max(colCount, address[0] - start[0]);
            });

            // update the used area of the sheet
            if ((colCount > 0) && (contents.length > 0)) {
                extendUsedArea(new Address(start[0] + colCount - 1, start[1] + contents.length - 1));
            }

            // notify all change listeners (process internal operations immediately)
            triggerChangeCellsEvent(null, changedCells, { direct: !context.external });
        };

        /**
         * Callback handler for the document operation 'fillCellRange'. Fills
         * all cells in the specified cell range with the same value and
         * formatting attributes.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'fillCellRange' document operation.
         */
        this.applyFillCellRangeOperation = function (context) {

            var // the address of the cell range to be filled
                range = context.getRange(),
                // the original value
                value = context.getOpt('value'),
                // whether to parse the cell contents
                parse = context.getOptStr('parse').length > 0,
                // the parsed value for all cell in all ranges
                parsedValue = parseCellValue(context.operation, parse),
                // the new attributes passed in the operation
                attributes = context.getOptObj('attrs'),
                // special treatment for border attributes applied to entire ranges
                rangeBorders = context.getOptBool('rangeBorders'),
                // special treatment for border attributes applied to entire ranges
                visibleBorders = context.getOptBool('visibleBorders'),
                // delete value: visit existing cells with a value only (bug 32610)
                iteratorType = (CellCollection.isBlank(parsedValue) && !_.isObject(attributes)) ? 'content' : 'all';

            // nothing to do, if neither value nor formatting will be changed
            if (!parsedValue && !attributes) { return; }

            // update all affected cell entries
            this.iterateCellsInRanges(range, function (cellDesc, origRange, mergedRange) {

                var // additional options for border attributes
                    attributeOptions = { visibleBorders: visibleBorders },
                    // temporary object create from parsedValue, if parsedValue is null and required, an object will be created
                    assignValue = parsedValue;

                // do not update cells hidden by merged ranges
                if (!mergedRange || mergedRange.startsAt(cellDesc.address)) {

                    // add options for inner/outer border treatment
                    mergedRange = mergedRange || new Range(cellDesc.address);
                    attributeOptions.innerLeft = rangeBorders && (mergedRange.start[0] > origRange.start[0]);
                    attributeOptions.innerRight = rangeBorders && (mergedRange.end[0] < origRange.end[0]);
                    attributeOptions.innerTop = rangeBorders && (mergedRange.start[1] > origRange.start[1]);
                    attributeOptions.innerBottom = rangeBorders && (mergedRange.end[1] < origRange.end[1]);

                    assignValue = preformat(assignValue, value, attributes, cellDesc);

                    // update the cell
                    updateCellModel(cellDesc.address, assignValue, attributes, attributeOptions);
                }
            }, { type: iteratorType, hidden: 'all', merged: true });

            // update the bounding ranges and the used area of the sheet
            extendBoundRanges(range);
            extendUsedArea(range.end);

            // notify all change listeners (process internal operations immediately)
            triggerChangeCellsEvent(range, null, { direct: !context.external });
        };

        /**
         * Callback handler for the document operation 'clearCellRange'. Clears
         * values and formatting of all cells in the specified cell range.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'clearCellRange' document operation.
         */
        this.applyClearCellRangeOperation = function (context) {

            var // the address of the cell range to be cleared
                range = context.getRange(),
                // all existing entries that have been deleted
                changedCells = deleteEntriesInRanges(range);

            // update the bounding ranges of the sheet (used area cannot be shrunken locally)
            extendBoundRanges(range);

            // notify all change listeners (process internal operations immediately)
            triggerChangeCellsEvent(null, changedCells, { direct: !context.external });
        };

        /**
         * Callback handler for the document operation 'autoFill'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'autoFill' document operation.
         */
        this.applyAutoFillOperation = function (context) {

            var // the address of the source cell range
                range = context.getRange(),
                // the target cell address from the operation
                target = context.getAddress('target'),
                // the target range to be filled with this operation
                targetRange = range.clone(),
                // whether to fill cells horizontally
                columns = null,
                // whether to fill cells before the source range
                reverse = null,
                // the cell contents to be inserted into the target cells
                contents = { display: CellCollection.PENDING_DISPLAY, result: null },
                // the auto style identifier to be inserted into the target cells
                styleId = null,
                // the cell formatting to be inserted into the target cells
                attributes = null,
                // whether to increase the numbers in the target cells
                incNumbers = false;

            // calculate the target range to be filled
            if (range.end[0] === target[0]) {
                columns = false;
                if (target[1] < range.start[1]) {
                    // fill cells above source range
                    targetRange.start[1] = target[1];
                    targetRange.end[1] = range.start[1] - 1;
                    reverse = true;
                } else if (range.end[1] < target[1]) {
                    // fill cells below source range
                    targetRange.start[1] = range.end[1] + 1;
                    targetRange.end[1] = target[1];
                    reverse = false;
                }
            } else if (range.end[1] === target[1]) {
                columns = true;
                if (target[0] < range.start[0]) {
                    // fill cells left of source range
                    targetRange.start[0] = target[0];
                    targetRange.end[0] = range.start[0] - 1;
                    reverse = true;
                } else if (range.end[0] < target[0]) {
                    // fill cells right of source range
                    targetRange.start[0] = range.end[0] + 1;
                    targetRange.end[0] = target[0];
                    reverse = false;
                }
            }

            // bail out on invalid target address
            context.ensure(_.isBoolean(columns) || _.isBoolean(reverse), 'invalid target address');

            // prepare auto-fill for a single cell as source range
            if (range.single()) {

                // copy all cell contents and formatting to the target range
                contents = this.getCellEntry(range.start);
                styleId = contents.style;
                attributes = contents.explicit;
                delete contents.style;
                delete contents.explicit;
                delete contents.attributes;

                // bug 33439: do not auto fill formula cells
                if (_.isString(contents.formula)) {
                    contents.display = CellCollection.PENDING_DISPLAY;

                // number cell: increase the cell value (bug 31121: not for formatted numbers)
                } else if (CellCollection.isNumber(contents) && (contents.format.category === 'standard')) {
                    incNumbers = true;
                    if (reverse) {
                        contents.result -= (targetRange.cells() + 1);
                    }
                }
            }

            // update all cells in the target range
            if (targetRange.cells() <= SheetUtils.MAX_FILL_CELL_COUNT) {
                this.iterateCellsInRanges(targetRange, function (cellDesc, origRange, mergedRange) {
                    // do not update cells hidden by merged ranges
                    if (!mergedRange || mergedRange.startsAt(cellDesc.address)) {
                        if (incNumbers) {
                            contents.result += 1;
                            contents.display = String(contents.result);
                        }
                        updateCellModel(cellDesc.address, contents, attributes, styleId ? { style: styleId } : null);
                    }
                }, { hidden: 'all', merged: true });
            }

            // update the bounding ranges and the used area of the sheet
            extendBoundRanges(targetRange);
            extendUsedArea(targetRange.end);

            // notify all change listeners (process internal operations immediately)
            triggerChangeCellsEvent(targetRange, null, { direct: !context.external });
        };

        /**
         * Callback handler for the document operation 'insertCells'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertCells' document operation.
         */
        this.applyInsertCellsOperation = function (context) {
            context.error('not implemented'); // TODO
        };

        /**
         * Callback handler for the document operation 'deleteCells'.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteCells' document operation.
         */
        this.applyDeleteCellsOperation = function (context) {
            context.error('not implemented'); // TODO
        };

        /**
         * Callback handler for the document operation 'insertHyperlink' that
         * attaches a URL to a cell range.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertHyperlink' document operation.
         */
        this.applyInsertHyperlinkOperation = function (context) {

            // the new hyperlink model
            var newModel = new HyperlinkModel(context.getRange(), context.getStr('url'));

            // remove all hyperlink ranges completely covered by the new range
            linkModels = linkModels.reject(function (linkModel) {
                return newModel.range.contains(linkModel.range);
            });

            // insert the new hyperlink range
            linkModels.push(newModel);

            // notify all change listeners
            triggerChangeCellsEvent(newModel.range);
        };

        /**
         * Callback handler for the document operation 'deleteHyperlink' that
         * removes all covered hyperlink ranges.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteHyperlink' document operation.
         */
        this.applyDeleteHyperlinkOperation = function (context) {

            var // the range to be cleared
                range = context.getRange(),
                // the changed ranges that will be notified
                deletedRanges = new RangeArray();

            // remove all hyperlink ranges overlapping with the range
            linkModels = linkModels.reject(function (linkModel) {
                if (linkModel.range.overlaps(range)) {
                    deletedRanges.push(linkModel.range);
                    return true;
                }
            });

            // notify all change listeners
            triggerChangeCellsEvent(deletedRanges);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the number of defined cells in the sheet.
         *
         * @returns {Number}
         *  The number of defined cells in the sheet.
         */
        this.getCellCount = function () {
            return cellCount;
        };

        /**
         * Returns the number of columns in the used area of the sheet.
         *
         * @returns {Number}
         *  The number of columns in the used area of the sheet.
         */
        this.getUsedCols = function () {
            return usedCols;
        };

        /**
         * Returns the number of rows in the used area of the sheet.
         *
         * @returns {Number}
         *  The number of rows in the used area of the sheet.
         */
        this.getUsedRows = function () {
            return usedRows;
        };

        /**
         * Returns the number of columns or rows in the used area of the sheet.
         *
         * @param {Boolean} columns
         *  Whether to return the number of used columns (true), or the number
         *  of used rows (false).
         *
         * @returns {Number}
         *  The number of columns/rows in the used area of the sheet.
         */
        this.getUsedCount = function (columns) {
            return columns ? usedCols : usedRows;
        };

        /**
         * Returns the address of the last used cell in the sheet. If the sheet
         * is completely empty, returns the address [0, 0].
         *
         * @returns {Address}
         *  The address of the last used cell in the sheet.
         */
        this.getLastUsedCell = function () {
            return ((usedCols > 0) && (usedRows > 0)) ? new Address(usedCols - 1, usedRows - 1) : Address.A1.clone();
        };

        /**
         * Returns the range address of the used area in the sheet. If the
         * sheet is completely empty, returns a range address covering the cell
         * A1 only.
         *
         * @returns {Range}
         *  The range address of the used area in the sheet.
         */
        this.getUsedRange = function () {
            return new Range(Address.A1.clone(), this.getLastUsedCell());
        };

        /**
         * Returns whether the passed cell address is contained in the bounding
         * ranges of this collection.
         *
         * @param {Address} address
         *  The cell address to be checked.
         *
         * @returns {Boolean}
         *  Whether the bounding ranges of this collection contain the passed
         *  cell address.
         */
        this.containsCell = function (address) {
            return boundRanges.containsAddress(address);
        };

        /**
         * Returns whether the passed ranges are completely contained in the
         * bounding ranges of this collection.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Boolean}
         *  Whether the bounding ranges of this collection contain the passed
         *  cell ranges completely.
         */
        this.containsRanges = function (ranges) {
            return ranges.difference(boundRanges).empty();
        };

        /**
         * Returns whether the specified cell is blank (no result value). A
         * blank cell may contain formatting attributes.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Boolean}
         *  Whether the specified cell is blank.
         */
        this.isBlankCell = function (address) {
            return CellCollection.isBlank(getCellModel(address));
        };

        /**
         * Returns the result value of the cell at the specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {Number|String|Boolean|Null}
         *  The typed result value of the cell, or the formula result.
         */
        this.getCellResult = function (address) {
            var cellModel = getCellModel(address);
            return cellModel ? cellModel.result : null;
        };

        /**
         * Returns the formula string of the cell at the specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {String|Null}
         *  The formula string of the specified cell; or null, if the cell does
         *  not contain a formula.
         */
        this.getCellFormula = function (address) {
            var cellModel = getCellModel(address);
            return cellModel ? cellModel.formula : null;
        };

        /**
         * Returns the URL of a hyperlink at the specified address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @returns {String|Null}
         *  The URL of a hyperlink attached to the specified cell; or null, if
         *  the cell does not contain a hyperlink.
         */
        this.getCellURL = function (address) {
            var linkModel = linkModels.find(function (model) { return model.range.containsAddress(address); });
            return linkModel ? linkModel.url : null;
        };

        /**
         * Returns a descriptor object for the cell at the specified address.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {CellDescriptor}
         *  A descriptor object for the specified cell.
         */
        this.getCellEntry = function (address) {

            // return descriptor of existing cell model
            var cellModel = getCellModel(address);
            if (cellModel) { return new CellDescriptor(cellModel); }

            // build descriptor for undefined cells (use row or column formatting)
            return CellDescriptor.createDefault(colCollection.getEntry(address[0]), rowCollection.getEntry(address[1]));
        };

        /**
         * Returns a descriptor object for a defined cell at the specified
         * address, otherwise null.
         *
         * @param {Address} address
         *  The address of the cell.
         *
         * @returns {CellDescriptor|Null}
         *  The cell descriptor object, if the cell exists in this collection;
         *  otherwise null.
         */
        this.getExistingCellEntry = function (address) {
            // return descriptor of existing cell models only
            var cellModel = getCellModel(address);
            return cellModel ? new CellDescriptor(cellModel) : null;
        };

        /**
         * Invokes the passed callback function for specific cells in the
         * passed cell ranges.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, whose
         *  cells will be iterated. All specified cells will be visited by this
         *  method, regardless of the size of the ranges.
         *
         * @param {Function} callback
         *  The callback function invoked for all matching cells in the passed
         *  ranges. The ranges will be processed independently, cells covered
         *  by several ranges will be visited multiple times. Receives the
         *  following parameters:
         *  (1) {CellDescriptor} cellDesc
         *      The descriptor object for the current cell.
         *  (2) {Range} originalRange
         *      The address of the range containing the current cell (one of
         *      the ranges contained in the 'ranges' parameter passed to this
         *      method).
         *  (3) {Range|Null} mergedRange
         *      If the option 'merged' has been set to true (see below), the
         *      address of the merged range containing the current cell will be
         *      passed. If no merged range exists for the visited cell, this
         *      parameter will be null.
         *  If the callback function returns the Utils.BREAK object, the
         *  iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the callback will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      callback function).
         *  @param {Range} [options.boundRange]
         *      If specified, the cell range address of the bounding range this
         *      method invocation is restricted to. If omitted, the full ranges
         *      will be visited.
         *  @param {String} [options.type='all']
         *      Specifies which type of cells will be visited. Must be one of
         *      the following values:
         *      - 'all': All defined and undefined cells will be visited.
         *      - 'existing': Only cells that exist in this collection will be
         *          visited, regardless of their content value (including blank
         *          but formatted cells).
         *      - 'content': Only non-blank cells (with any content value) will
         *          be visited.
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle cells contained in hidden rows or
         *      columns. Must be one of the following values:
         *      - 'none' (or omitted): Only visible cells will be visited.
         *      - 'all': All visible and hidden cells will be visited.
         *      - 'last': All visible cells, and all cells contained in hidden
         *          columns and/or rows that precede a visible column/row will
         *          be visited.
         *  @param {Boolean} [options.ordered=false]
         *      If set to true, the cells in each cell range will be visited in
         *      order (row-by-row from top to bottom, and left-to-right inside
         *      each of the rows; but in reversed order, if the option
         *      'reverse' is set to true). Otherwise, the order in which the
         *      cells will be visited is undetermined.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the ranges AND the cells in each range will be
         *      visited in reversed order. This option will be ignored, if the
         *      option 'ordered' has not been set to true.
         *  @param {Boolean} [options.merged=false]
         *      If set to true, the 'mergedRange' parameter of the callback
         *      function will be initialized with the address of the merged
         *      range containing the visited cell. To improve iteration
         *      performance, this will not be done by default.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInRanges = function (ranges, callback, options) {

            var // the calling context for the callback function
                context = Utils.getOption(options, 'context'),
                // the bounding range to restrict the visited area
                customBoundRange = Utils.getObjectOption(options, 'boundRange', docModel.getSheetRange()),
                // which cell entries (by type) will be visited
                cellType = Utils.getStringOption(options, 'type', 'all'),
                // whether to visit hidden cell entries
                hiddenMode = Utils.getStringOption(options, 'hidden', 'none'),
                // whether to visit cells in order
                ordered = Utils.getBooleanOption(options, 'ordered', false),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether to pass the addresses of merged ranges to the callback
                findMerged = Utils.getBooleanOption(options, 'merged', false),
                // the merged ranges covering the passed ranges
                mergedRanges = findMerged ? mergeCollection.getMergedRanges(ranges) : null;

            // process all ranges (inside the specified bounding range)
            return Utils.iterateArray(RangeArray.get(ranges), function (origRange) {

                function invokeCallback(cellDesc) {
                    var mergedRange = mergedRanges ? mergedRanges.findByAddress(cellDesc.address) : null;
                    return callback.call(context, cellDesc, origRange, mergedRange);
                }

                // the intersection between current range and bounding range (skip ranges outside)
                var range = origRange.intersect(customBoundRange);
                if (!range) { return; }

                // visit all cells (defined and undefined); simply loop over rows and columns (always in-order)
                if ((cellType !== 'existing') && (cellType !== 'content')) {
                    return iterateCellsInRange(range, function (colDesc, rowDesc, cellModel) {
                        var cellDesc = cellModel ? new CellDescriptor(cellModel) : CellDescriptor.createDefault(colDesc, rowDesc);
                        return invokeCallback(cellDesc);
                    }, options);
                }

                // visit only existing cells: collect existing cell entries (needed for sorting)
                return (function () {

                    var // visible column and row intervals in the current range
                        colIntervals = getVisitedIntervals(colCollection, range.colInterval()),
                        rowIntervals = getVisitedIntervals(rowCollection, range.rowInterval()),
                        // number of visible columns, rows, and cells in the current range
                        colCount = colIntervals.size(),
                        rowCount = rowIntervals.size(),
                        totalCount = colCount * rowCount;

                    // returns the intervals to be actually visited for the specified interval
                    function getVisitedIntervals(collection, interval) {
                        return (hiddenMode === 'all') ? new IntervalArray(interval) : collection.getVisibleIntervals(interval, { lastHidden: (hiddenMode === 'last') ? 1 : 0 });
                    }

                    // returns whether the passed intervals contain a specific index (intervals MUST be sorted)
                    function intervalsContainIndex(intervals, index) {
                        var arrayIndex = _.sortedIndex(intervals, { last: index }, 'last');
                        return (arrayIndex < intervals.length) && (intervals[arrayIndex].first <= index);
                    }

                    // returns whether the existing cell will be visited according to passed cell type
                    function isVisitedCell(cellModel) {
                        return cellModel && ((cellType === 'existing') || !CellCollection.isBlank(cellModel));
                    }

                    // little helper to visit all cell entries in the current range, by iterating the cell map directly
                    function iterateCellMap(callback2) {
                        return _.any(cellMap, function (cellModel) {
                            if (isVisitedCell(cellModel)) {
                                if (intervalsContainIndex(colIntervals, cellModel.address[0]) && intervalsContainIndex(rowIntervals, cellModel.address[1])) {
                                    return callback2.call(self, cellModel) === Utils.BREAK;
                                }
                            }
                        }) ? Utils.BREAK : undefined;
                    }

                    // no visible column and row contained in the range
                    if (totalCount === 0) { return; }

                    // performance: iterate over the entire cell map, if more cells are in the range than collection entries
                    if (cellCount < totalCount) {

                        // if 'ordered' option has not been passed, visit the cells unordered
                        if (!ordered) {
                            return iterateCellMap(function (cellModel) {
                                return invokeCallback(new CellDescriptor(cellModel));
                            });
                        }

                        // collect and sort all cells to be visited
                        var visitedCells = [];
                        iterateCellMap(function (cellModel) { visitedCells.push(cellModel); });
                        visitedCells.sort(CellModel.compare);

                        // invoke callback for all sorted entries
                        return Utils.iterateArray(visitedCells, function (cellModel) {
                            return invokeCallback(new CellDescriptor(cellModel));
                        }, { reverse: reverse });
                    }

                    // more collection entries than cells in the ranges: iterate all cells in the range (always ordered)
                    return iterateCellsInRange(range, function (colDesc, rowDesc, cellModel) {
                        if (isVisitedCell(cellModel)) {
                            return invokeCallback(new CellDescriptor(cellModel));
                        }
                    }, options);
                }());

            }, { reverse: reverse });
        };

        /**
         * Invokes the passed callback function for specific cells in a single
         * column or row, while moving away from the start cell into the
         * specified directions.
         *
         * @param {Address} address
         *  The address of the cell that will be visited first (the option
         *  'skipStartCell' can be used, if iteration shall start with the
         *  nearest available neighbor of the cell, and not with the cell
         *  itself).
         *
         * @param {String} directions
         *  How to move to the next cells while iterating. Supported values are
         *  'up', 'down', 'left', or 'right', or any white-space separated list
         *  of these values. Multiple directions will be processed in the same
         *  order as specified in the parameter.
         *
         * @param {Function} callback
         *  The callback function invoked for all matching cells in the
         *  specified direction(s). Receives the following parameters:
         *  (1) {CellDescriptor} cellDesc
         *      The descriptor object for the current cell.
         *  (2) {Range|Null} mergedRange
         *      If the option 'merged' has been set to true (see below), the
         *      address of the merged range containing the current cell will be
         *      passed. If no merged range exists for the visited cell, this
         *      parameter will be null.
         *  If the callback function returns the Utils.BREAK object, the
         *  iteration process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      If specified, the callback will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      callback function).
         *  @param {Range} [options.boundRange]
         *      If specified, the cell range address of the bounding range this
         *      method invocation is restricted to. If omitted, the entire area
         *      of the sheet will be used.
         *  @param {String} [options.type='all']
         *      Specifies which type of cells will be visited. Must be one of
         *      the following values:
         *      - 'all': All defined and undefined cells will be visited.
         *      - 'existing': Only cells that exist in this collection will be
         *          visited, regardless of their content value (including empty
         *          but formatted cells).
         *      - 'content': Only non-empty cells (with any content value) will
         *          be visited.
         *  @param {String} [options.hidden='none']
         *      Specifies how to handle cells contained in hidden rows or
         *      columns. Must be one of the following values:
         *      - 'none' (or omitted): Only visible cells will be visited.
         *      - 'all': All visible and hidden cells will be visited.
         *      - 'last': All visible cells, and all cells contained in hidden
         *          columns and/or rows that precede a visible column/row will
         *          be visited.
         *  @param {Boolean} [options.skipStartCell=false]
         *      If set to true, iteration will start at the nearest visible
         *      neighbor of the cell specified in the 'address' parameter
         *      instead of that cell.
         *  @param {Boolean} [options.merged=false]
         *      If set to true, the 'mergedRange' parameter of the callback
         *      function will be initialized with the address of the merged
         *      range containing the visited cell.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateCellsInLine = function (address, directions, callback, options) {

            var // number of cells to skip before iteration starts
                skipCount = Utils.getBooleanOption(options, 'skipStartCell', false) ? 1 : 0;

            return Utils.iterateArray(directions.split(/\s+/), function (direction) {

                var // whether to iterate with variable column index
                    columns = /^(left|right)$/.test(direction),
                    // whether to iterate in reversed order (towards first column/row)
                    reverse = /^(up|left)$/.test(direction),
                    // the entire column/row range containing the passed address
                    range = docModel.makeFullRange(address.get(!columns), !columns);

                // check passed direction identifier
                if (!columns && !reverse && (direction !== 'down')) {
                    Utils.error('CellCollection.iterateCellsInLine(): invalid direction "' + direction + '"');
                    return Utils.BREAK;
                }

                // reduce range to leading or trailing part (do nothing, if the range becomes invalid)
                if (reverse) {
                    range.setEnd(address.get(columns) - skipCount, columns);
                } else {
                    range.setStart(address.get(columns) + skipCount, columns);
                }
                if (range.getStart(columns) > range.getEnd(columns)) { return; }

                // visit all cells in the resulting range, according to the passed settings
                return self.iterateCellsInRanges(range, function (cellDesc, origRange, mergedRange) {
                    return callback.call(this, cellDesc, mergedRange);
                }, _.extend({}, options, { reverse: reverse, ordered: true }));
            });
        };

        /**
         * Returns the next available cell of a specific type (any blank cells,
         * defined cells only, or content cells only).
         *
         * @param {Address} address
         *  The address of the cell whose nearest adjacent cell will be
         *  searched.
         *
         * @param {String} direction
         *  The direction to look for the cell. Must be one of the values
         *  'left', 'right', 'up', or 'down'.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  method CellCollection.iterateCellsInLine(), except for the option
         *  'skipStartCell' (always set to true).
         *
         * @returns {CellDescriptor|Null}
         *  The descriptor of the next available cell; or null, if no cell has
         *  been found.
         */
        this.findNearestCell = function (address, direction, options) {

            var // the resulting cell descriptor
                resultDesc = null;

            // iterate to the first available content cell
            this.iterateCellsInLine(address, direction, function (cellDesc) {
                resultDesc = cellDesc;
                return Utils.BREAK;
            }, _.extend({}, options, { skipStartCell: true }));

            return resultDesc;
        };

        /**
         * Returns the content range (the entire range containing consecutive
         * content cells) surrounding the specified cell range.
         *
         * @param {Range} range
         *  The address of the cell range whose content range will be searched.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.directions='up down left right']
         *      All directions the passed range will be expanded to, as
         *      space-separated token list with the tokens 'up', 'down',
         *      'left', and 'right'. If omitted, the range will be expanded to
         *      all four directions.
         *  @param {Boolean} [options.skipHidden=false]
         *      Whether to skip hidden columns and rows when searching for the
         *      content range.
         *
         * @returns {Range}
         *  The address of the content range of the specified cell range.
         */
        this.findContentRange = SheetUtils.profileMethod('CellCollection.findContentRange()', function (range, options) {

            var // column/row collections of the active sheet
                colCollection = sheetModel.getColCollection(),
                rowCollection = sheetModel.getRowCollection(),
                // collection of merged ranges in the active sheet
                mergeCollection = sheetModel.getMergeCollection(),
                // the resulting content range (start by expanding to merged ranges)
                currRange = mergeCollection.expandRangeToMergedRanges(range),
                // the result of the last iteration, to detect exiting the loop
                lastRange = null,
                // all directions the range will be expanded to
                directions = Utils.getTokenListOption(options, 'directions', ['up', 'down', 'left', 'right']),
                // whether to skip hidden columns and rows
                skipHidden = Utils.getBooleanOption(options, 'skipHidden', false),
                // hidden mode as expected by the other methods of CellCollection
                hiddenMode = skipHidden ? 'none' : 'all';

            // returns the next column/row index outside the passed cell range
            function getNextIndex(srcRange, forward, columns) {

                var // column/row collection in specified direction
                    collection = columns ? colCollection : rowCollection,
                    // index of the first column/row outside the range
                    startIndex = forward ? (srcRange.getEnd(columns) + 1) : (srcRange.getStart(columns) - 1);

                // when processing all hidden columns/rows, return the start index if it is valid
                if (!skipHidden) {
                    return ((startIndex < 0) || (startIndex > collection.getMaxIndex())) ? -1 : startIndex;
                }

                // find the descriptor of the next visible column/row
                var nextDesc = forward ? collection.getNextVisibleEntry(startIndex) : collection.getPrevVisibleEntry(startIndex);
                return nextDesc ? nextDesc.index : -1;
            }

            // array iterator function, must be defined outside the do-while loop
            function expandRangeBorder(direction, index) {

                var // whether to move towards the end of the sheet
                    forward = /^(down|right)$/.test(direction),
                    // whether to move through columns in the same row
                    columns = /^(left|right)$/.test(direction),
                    // the descriptor of the next column/row
                    nextIndex = getNextIndex(currRange, forward, columns);

                // do nothing, if the range already reached the sheet borders
                if (nextIndex < 0) {
                    directions.splice(index, 1);
                    return false;
                }

                // build a single column/row range next to the current range
                var boundRange = currRange.clone();
                boundRange.setBoth(nextIndex, columns);

                // expand sideways, unless these directions are not active anymore
                if (_.contains(directions, columns ? 'up' : 'left')) {
                    nextIndex = getNextIndex(boundRange, false, !columns);
                    if (nextIndex >= 0) { boundRange.setStart(nextIndex, !columns); }
                }
                if (_.contains(directions, columns ? 'down' : 'right')) {
                    nextIndex = getNextIndex(boundRange, true, !columns);
                    if (nextIndex >= 0) { boundRange.setEnd(nextIndex, !columns); }
                }

                // find a content cell next to the current range (including hidden columns/rows)
                self.iterateCellsInRanges(boundRange, function (cellDesc1) {

                    var // reference to start or end address in the current range, according to direction
                        currAddress = forward ? currRange.end : currRange.start;

                    // content cell found: expand the current range
                    currAddress.set(cellDesc1.address.get(columns), columns);

                    // performance: iterate into the expanded direction as long as there are adjacent content cells
                    self.iterateCellsInLine(cellDesc1.address, direction, function (cellDesc2) {
                        if (CellCollection.isBlank(cellDesc2)) { return Utils.BREAK; }
                        currAddress.set(cellDesc2.address.get(columns), columns);
                    }, { hidden: hiddenMode, skipStartCell: true });

                    // content cell found and range expanded: exit loop
                    return Utils.BREAK;
                }, { hidden: hiddenMode, type: 'content' });
            }

            // expand in all four directions until the range does not change anymore
            do {

                // store result of last iteration for comparison below
                lastRange = currRange.clone();

                // expand current range into all remaining directions
                Utils.iterateArray(directions, expandRangeBorder, { reverse: true });

                // expand the range to include merged ranges
                currRange = mergeCollection.expandRangeToMergedRanges(currRange);

            } while ((directions.length > 0) && currRange.differs(lastRange));

            // return the resulting range
            return currRange;
        });

        /**
         * Returns whether all passed ranges are blank (no values). Ignores
         * cell formatting attributes.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         *
         * @returns {Boolean}
         *  Whether all cells in the passed ranges are blank.
         */
        this.areRangesBlank = function (ranges) {

            // whether any cell has a value
            var hasContentCell = false;

            // try to find the first content cell
            this.iterateCellsInRanges(ranges, function () {
                hasContentCell = true;
                return Utils.BREAK;
            }, { type: 'content', hidden: 'all' });

            return !hasContentCell;
        };

        /**
         * Returns the address of the first or last cell with the specified
         * formatting attributes in the passed cell ranges, regardless whether
         * the cell is actually defined, or the attribute values have been
         * derived from column or row attributes. Uses an optimized search
         * algorithm for the empty areas between the defined cells, regarding
         * intervals of equally formatted columns and rows, and the position of
         * merged ranges covering these column/row default formatting.
         *
         * @param {RangeArray|Range} ranges
         *  An array of range addresses, or a single cell range address, whose
         *  cells will be searched for the attributes.
         *
         * @param {Object} attributes
         *  The incomplete attribute set to be matched against the formatting
         *  attributes in the passed cell ranges.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, all visible and hidden columns and rows of the
         *      passed cell ranges will be evaluated. Otherwise, only visible
         *      cells will be processed.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the cell ranges, AND the single cells in each
         *      cell range will be searched in reversed order.
         *
         * @returns {CellDescriptor|Null}
         *  The descriptor of the first (or last, depending on direction) cell
         *  with matching formatting attributes; or null, if no such cell
         *  exists in the passed cell ranges.
         */
        this.findCellWithAttributes = SheetUtils.profileMethod('CellCollection.findCellWithAttributes()', function (ranges, attributes, options) {

            var // whether to visit hidden columns and rows
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                hiddenMode = hidden ? 'all' : 'none',
                // whether to find the last matching cell
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // the resulting cell address with matching attributes
                resultAddress = null;

            // Finds the address of the next existing cell in the current range following (or preceding) the
            // current cell.
            function findExistingCell(range, address) {

                // First, search in the remaining cells of the current row.
                var cellDesc = self.findNearestCell(address, reverse ? 'left' : 'right', { boundRange: range, type: 'existing', hidden: hiddenMode });
                if (cellDesc) { return cellDesc.address; }

                // Next, search for a cell in the remaining range without the current row.
                if (range.singleRow()) { return null; }
                var remainRange = range.clone();
                if (reverse) { remainRange.end[1] -= 1; } else { remainRange.start += 1; }
                self.iterateCellsInRanges(remainRange, function (cell) {
                    cellDesc = cell;
                    return Utils.BREAK;
                }, { type: 'existing', hidden: hiddenMode, ordered: true, reverse: reverse });
                return cellDesc ? cellDesc.address : null;
            }

            // Returns the address of the next available blank cell in the passed range with matching
            // attributes. The cell range is assumed to to completely blank and undefined. A blank cell will
            // only be searched by evaluating the column/row formatting of the range, and the merged ranges
            // covering the passed range.
            function findAttributesInBlankRange(range) {

                var // the merged ranges covering the passed range (may cover column/row attributes)
                    mergedRanges = mergeCollection.getMergedRanges(range),
                    // the column intervals with matching attributes (will be initialized on first access)
                    matchColIntervals = null,
                    // the resulting cell address
                    address = null;

                // Iterate through visible and equally formatted row intervals, and search for a blank cell
                // according to the formatting attributes of the rows.
                rowCollection.iterateEntries(range.rowInterval(), function (rowDesc, rowInterval) {

                    // The addresses of all ranges with matching attributes in the row block.
                    var matchRanges = null;

                    // Rows with custom formatting override default column attributes.
                    if (rowDesc.attributes.row.customFormat) {

                        // If the row block does not match, it can be skipped completely. Otherwise, the address of
                        // the entire row block will be used to initialize 'matchingRanges'.
                        if (AttributeUtils.matchesAttributesSet(rowDesc.attributes, attributes)) {
                            matchRanges = new RangeArray(Range.createFromIntervals(range.colInterval(), rowInterval));
                        }

                    } else {

                        // In a row block without own formatting attributes, find the column intervals with matching
                        // attributes. The intervals will be initialized once on demand, and will be reused for all
                        // following row intervals.
                        if (!matchColIntervals) {
                            matchColIntervals = new IntervalArray();
                            colCollection.iterateEntries(range.colInterval(), function (colDesc, colInterval) {
                                if (AttributeUtils.matchesAttributesSet(colDesc.attributes, attributes)) {
                                    matchColIntervals.push(colInterval);
                                }
                            }, { hidden: hidden ? 'all' : 'none', unique: true });
                            matchColIntervals = matchColIntervals.merge();
                        }

                        // Build the range array from all matching column intervals, and the current row interval.
                        matchRanges = RangeArray.createFromIntervals(matchColIntervals, rowInterval);
                    }

                    // Continue with the next row block, if no matching ranges have been found.
                    if (!matchRanges) { return; }

                    // Remove the merged ranges from the resulting ranges (merged ranges always cover the default
                    // formatting attributes of columns or rows). After removing the merged ranges, the remaining ranges
                    // may need to be reduced to their visible areas. The result may be empty (all matching cells in the
                    // row block are covered by merged ranges).
                    matchRanges = matchRanges.difference(mergedRanges);
                    if (!hidden) { matchRanges = RangeArray.map(matchRanges, sheetModel.shrinkRangeToVisible.bind(sheetModel)); }
                    if (matchRanges.empty()) { return; }

                    // The first (or last) cell in the remaining ranges is the resulting cell. Find the smallest start
                    // address (or the largest end address in reverse mode) of all ranges, and exit the entire loop.
                    matchRanges.forEach(function (matchRange) {
                        if (!reverse && (!address || (matchRange.start.compareTo(address) < 0))) {
                            address = matchRange.start;
                        } else if (reverse && (!address || (matchRange.end.compareTo(address) > 0))) {
                            address = matchRange.end;
                        }
                    });
                    return Utils.BREAK;

                }, { hidden: hidden ? 'all' : 'none', reverse: reverse, unique: true });

                return address;
            }

            // Search in the passed cell ranges until a matching cell has been found.
            Utils.iterateArray(RangeArray.get(ranges), function (range) {

                // the current cell address; and the current row index, for convenience
                var address = null, row = null;

                // Iterate as long as there is something left to search for (the body will exit the loop by itself).
                // The range address will be shrunken inside the loop if an entire row has been processed (therefore,
                // a clone of the range will be created).
                range = range.clone();
                while (true) {

                    // Reduce the passed range to the visible area unless the 'hidden' flag is set. Continue with
                    // the next cell range, if no more visible cells are left (the range shrinks to null).
                    if (!hidden && !(range = sheetModel.shrinkRangeToVisible(range))) { return; }

                    // Start with the top-left (or bottom-right in reverse mode) corner of the range, if the address
                    // is not valid (anymore).
                    if (!address || !range.containsAddress(address)) {
                        address = (reverse ? range.end : range.start).clone();
                        row = address[1];
                    }

                    // First, check an existing cell at the current address. If the cell contains the attributes, it
                    // will be the final result of this method. Do not check the cell if it is not defined; in this
                    // case it may be covered by a merged range.
                    var cellModel = getCellModel(address);
                    if (cellModel && AttributeUtils.matchesAttributesSet(cellModel.getMergedAttributes(), attributes)) {
                        resultAddress = address;
                        return Utils.BREAK;
                    }

                    // Find the next existing cell in the cell collection following (or preceding) the current cell.
                    var nextAddress = findExistingCell(range, address);

                    // If the cell at the current address exists, move the address to next cell which is required for
                    // searching in blank gap ranges.
                    if (cellModel) { address[0] += (reverse ? -1 : 1); }

                    // Search for a cell between current address and next existing cell in the current row; or up to
                    // the end (or beginning) of the current row, if it does not contain any other existing cells.
                    // Example 1: Range is A1:I9, current address is D1, next address is H1: This part will search
                    // for blank cells in the gap range E1:G1.
                    // Example 2: Range is A1:I9, current address is D1, next address is C4 (or null): This part will
                    // search for blank cells in the gap range E1:I1 (current row only).
                    var lastCol = (nextAddress && (row === nextAddress[1])) ? (nextAddress[0] + (reverse ? 1 : -1)) : (reverse ? range.start[0] : range.end[0]);
                    if (reverse ? (lastCol <= address[0]) : (address[0] <= lastCol)) {
                        resultAddress = findAttributesInBlankRange(Range.create(address[0], row, lastCol, row), attributes, options);
                        if (resultAddress) { return Utils.BREAK; }
                    }

                    // Process the entire blank row block between current row, and the row with the next existing cell
                    // (or up to the end/beginning of the range, if no other cell exists in the cell collection).
                    // Example 1: Range is A1:I9, current address is D1, next address is H1: This part will not search
                    // for blank cells (next address is in the same row).
                    // Example 2: Range is A1:I9, current address is D1, next address is C4: This part will search for
                    // blank cells in the gap range A2:I3 (entire blank rows between the addresses).
                    // Example 3: Range is A1:I9, current address is D1, next address is null: This part will search
                    // for blank cells in the gap range A2:I9 (all blank rows up to the end of the range).
                    var lastRow = !nextAddress ? (reverse ? range.start[1] : range.end[1]) : (row !== nextAddress[1]) ? (nextAddress[1] + (reverse ? 1 : -1)) : row;
                    if (reverse ? (lastRow < row) : (row < lastRow)) {
                        resultAddress = findAttributesInBlankRange(Range.create(range.start[0], row + (reverse ? -1 : 1), range.end[0], lastRow), attributes, options);
                        if (resultAddress) { return Utils.BREAK; }
                    }

                    // Finally, the blank cells preceding the next existing cell inside its row need to be checked, but
                    // only, if that cell is located in another row.
                    // Example 1: Range is A1:I9, current address is D1, next address is C4: This part will search for
                    // blank cells in the gap range A4:B4 (only row 4 containing the next existing cell).
                    var firstCol = reverse ? range.end[0] : range.start[0];
                    if (nextAddress && (row !== nextAddress[1]) && (reverse ? (nextAddress[0] < firstCol) : (firstCol < nextAddress[0]))) {
                        resultAddress = findAttributesInBlankRange(Range.create(firstCol, nextAddress[1], nextAddress[0] + (reverse ? 1 : -1), nextAddress[1]), attributes, options);
                        if (resultAddress) { return Utils.BREAK; }
                    }

                    // If there is no more existing cell, no cell with matching attributes could be found at all in the
                    // current range. Continue with the next cell range in this case.
                    if (!nextAddress) { return; }

                    // Otherwise, continue with the next existing cell in the cell collection.
                    address = nextAddress;
                    (reverse ? range.end : range.start)[1] = row = address[1];
                }
            }, { reverse: reverse });

            return resultAddress ? this.getCellEntry(resultAddress) : null;
        });

        /**
         * Checks whether the passed cell ranges are covered by the bounding
         * ranges of this cell collection. Missing cells will be queried from
         * the server, and listeners of this collection will be notified with a
         * 'change:cells' event.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address,
         *  to be included in this collection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.visible=false]
         *      If set to true, only the visible parts of the cell ranges will
         *      be requested from the server. Cells contained in hidden columns
         *      or rows will be skipped, except for cells located in the last
         *      hidden columns or rows preceding the next visible column/row.
         *  @param {Boolean} [options.merged=false]
         *      If set to true, the reference cells of merged ranges that
         *      overlap the passed cell ranges will be updated too, if they are
         *      missing in this cell collection.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.fetchMissingCellRanges = SheetUtils.profileMethod('CellCollection.fetchMissingCellRanges()', function (ranges, options) {

            var // the resulting cell range addresses of the visible parts of the passed cell range
                visibleRanges = null;

            // request only the visible parts of the range
            if (Utils.getBooleanOption(options, 'visible', false)) {
                visibleRanges = RangeArray.map(ranges, function (range) {
                    var colIntervals = colCollection.getVisibleIntervals(range.colInterval(), { lastHidden: 5 }),
                        rowIntervals = rowCollection.getVisibleIntervals(range.rowInterval(), { lastHidden: 5 });
                    return RangeArray.createFromIntervals(colIntervals, rowIntervals);
                });
            } else {
                visibleRanges = ranges;
            }

            // add unknown reference cells of merged ranges overlapping with the requested ranges
            if (Utils.getBooleanOption(options, 'merged', false)) {

                var // the merged ranges overlapping with the visible ranges
                    mergedRanges = mergeCollection.getMergedRanges(visibleRanges);

                if (!mergedRanges.empty()) {
                    mergedRanges.forEach(function (mergedRange) {
                        if (!boundRanges.containsAddress(mergedRange.start)) {
                            visibleRanges.push(new Range(mergedRange.start));
                        }
                    });
                    visibleRanges = visibleRanges.merge();
                }
            }

            // request the missing parts of the visible ranges from the server
            fetchAllRangeContents(visibleRanges, { missing: true });

            return this;
        });

        /**
         * Fetches all formula cells existing in this cell collection from the
         * server. Used to validate preliminary contents shown while loading
         * the document.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.fetchAllFormulaCells = SheetUtils.profileMethod('CellCollection.fetchAllFormulaCells()', function () {

            var // the cell addresses of all existing formula cells
                addresses = new AddressArray();

            // collect addresses of all formula cells
            _.each(cellMap, function (cellModel) {
                if (_.isString(cellModel.formula)) {
                    addresses.push(cellModel.address);
                }
            });

            // fetch all formula cells from the server
            if (!addresses.empty()) {
                fetchAllRangeContents(RangeArray.mergeAddresses(addresses));
            }

            return this;
        });

        /**
         * Recalculates the results of all formula cells existing in this cell
         * collection locally.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all formula cells have been
         *  recalculated.
         */
        this.refreshAllFormulaCells = DEBUG_LOCAL_FORMULAS ? SheetUtils.profileAsyncMethod('CellCollection.refreshAllFormulaCells()', function () {

            var // the cell addresses of all existing formula cells
                addresses = new AddressArray();

            return this.iterateObjectSliced(cellMap, function (cellModel) {
                if (_.isString(cellModel.formula)) {
                    addresses.push(cellModel.address);
                    cellModel.updateValue();
                }
            }).done(function () {
                triggerChangeCellsEvent(null, addresses);
            });
        }) : $.when;

        /**
         * Returns the results and display strings of all cells contained in
         * the passed cell ranges. If the cell collection does not cover all
         * passed ranges, the missing data will be requested from the server.
         *
         * @param {Array<RangeArray>} rangesArray
         *  An array of arrays (!) of cell range addresses. Each array element
         *  is treated as an independent range list.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.hidden=false]
         *      If set to true, all cells contained in hidden columns or rows
         *      will be included into the result. By default, only visible
         *      cells will be returned.
         *  @param {Boolean} [options.attributes=false]
         *      If set to true, the cell content objects will contain the
         *      merged formatting attributes of the cells.
         *  @param {Boolean} [options.compressed=false]
         *      If set to true, the returned cell arrays will be optimized:
         *      Consecutive cells with equal contents and formatting will be
         *      represented by a single array element with an additional
         *      property 'count'. Especially useful, if large ranges located in
         *      the unused area of the sheet are queried.
         *  @param {Number} [options.maxCount]
         *      If specified, the maximum number of cells per range list that
         *      will be returned in the result, regardless how large the passed
         *      ranges are (the passed ranges will be shortened before the cell
         *      contents will be requested). Example: The first array element
         *      in the parameter 'rangesArray' covers 2000 cells (in several
         *      ranges), the second array element covers 500 cells. With this
         *      option set to the value 1000, the first range list will be
         *      shortened to 1000 cells, the second range list will be resolved
         *      completely. If omitted, and the ranges are too large (as
         *      defined by the constant SheetUtils.MAX_QUERY_CELL_COUNT), the
         *      entire request will be rejected (in compressed mode, the size
         *      of the compressed result array will be checked against that
         *      constant, not the number of cells in the original ranges).
         *  @param {Number} [options.maxRequests=20]
         *      The maximum number of server requests allowed to be sent. After
         *      the specified number of server requests, the collected
         *      (possibly incomplete) cell data will be returned.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the cell contents referred by
         *  the passed ranges. The result will be an array of arrays of cell
         *  content objects (the length of the outer result array will be equal
         *  to the length of the 'rangesArray' parameter). Each cell object in
         *  the inner arrays will contain the following properties:
         *  - {String|Null} cellData.display
         *      The display string of the cell result (will be the empty string
         *      for blank cells). The value null represents a cell result that
         *      cannot be formatted to a valid display string with the current
         *      number format of the cell.
         *  - {Number|String|Boolean|ErrorCode|Null} cellData.result
         *      The typed result value (will be null for blank cells).
         *  - {Object} [cellData.attrs]
         *      The merged formatting attributes of the cell. Will only be set,
         *      if the option 'attributes' has been set to true (see above).
         *  - {ParsedFormat} [cellData.format]
         *      The parsed number format code of the cell. Will only be set, if
         *      the option 'attributes' has been set to true (see above).
         *  - {Number} [cellData.count]
         *      Always set in compressed mode (see option 'compressed' above).
         *      Contains the number of consecutive cells with equal contents
         *      and formatting represented by this array element.
         *  The cells in the inner arrays will be in order of the range lists.
         *  Cells from a single range will be ordered row-by-row.
         */
        this.queryCellContents = function (rangesArray, options) {

            var // the deferred object that will be resolved with the cell contents
                def = $.Deferred(),
                // the abortable promise returned by this method
                promise = this.createAbortablePromise(def),
                // whether to include hidden cells
                hidden = Utils.getBooleanOption(options, 'hidden', false),
                // whether to insert the merged formatting attributes
                attributes = Utils.getBooleanOption(options, 'attributes', false),
                // compressed mode
                compressed = Utils.getBooleanOption(options, 'compressed', false),
                // the maximum number of cells in the result
                maxCount = Utils.getIntegerOption(options, 'maxCount', null, 1),
                // the maximum number of server requests
                maxRequests = Utils.getIntegerOption(options, 'maxRequests', 20, 1);

            // inserts the passed result entry into the array (handles compressed mode)
            function pushResult(results, entry, count) {

                var // the preceding result, for compressed mode
                    prevEntry = compressed ? _.last(results) : null;

                // increase count of previous entry, if the new entry is equal
                if (prevEntry) {
                    entry.count = prevEntry.count;
                    if ((prevEntry.result === entry.result) && (prevEntry.display === entry.display) && (!attributes || _.isEqual(prevEntry.attrs, entry.attrs))) {
                        prevEntry.count += count;
                        return;
                    }
                }

                // insert the new entry into the array (prepare count for compressed mode)
                if (compressed) {
                    entry.count = count;
                    results.push(entry);
                } else {
                    for (; count > 0; count -= 1) { results.push(entry); }
                }
            }

            // pushes result entries for blank cells without formatting attributes
            function pushBlanks(results, count) {
                if (count > 0) {
                    pushResult(results, { result: null, display: '' }, count);
                }
            }

            // collects all cells in the passed range without formatting attributes
            function collectWithoutAttributes(results, range) {

                var // all existing cells, needed for ordering
                    existingCells = [],
                    // number of columns in the range
                    colCount = range.cols(),
                    // linear index of next expected cell in the range (used to fill blank cells)
                    nextIndex = 0;

                // collect all existing cells in the range
                iterateExistingCellModels(range, function (cellModel) {
                    var index = (cellModel.address[1] - range.start[1]) * colCount + (cellModel.address[0] - range.start[0]);
                    existingCells.push({ index: index, cellModel: cellModel });
                });

                // sort by cell address (row-by-row)
                existingCells.sort(function (entry1, entry2) { return entry1.index - entry2.index; });

                // fill the result array with existing cells and preceding blank cells
                _.each(existingCells, function (entry) {
                    pushBlanks(results, entry.index - nextIndex);
                    pushResult(results, { result: entry.cellModel.result, display: entry.cellModel.display }, 1);
                    nextIndex = entry.index + 1;
                });

                // fill up following blank cells
                pushBlanks(results, range.cells() - nextIndex);
            }

            // collects all cells in the passed range with formatting attributes
            function collectWithAttributes(results, range) {
                iterateCellsInRange(range, function (colEntry, rowEntry, cellModel, address) {
                    // resolve formatting attributes (including column/row formatting)
                    var cellDesc = self.getCellEntry(address);
                    // push a single result entry
                    pushResult(results, {
                        result: cellDesc.result,
                        display: cellDesc.display,
                        attrs: cellDesc.attributes,
                        format: cellDesc.format
                    }, 1);
                });
            }

            // collects all cells in the passed ranges, and returns a flat result array
            function collectCellData(ranges) {
                var results = [];
                ranges.forEach(function (range) {
                    if (attributes) {
                        collectWithAttributes(results, range);
                    } else {
                        collectWithoutAttributes(results, range);
                    }
                });
                return results;
            }

            // reduce the ranges to their visible parts
            if (!hidden) {
                rangesArray.forEach(function (ranges, index) {
                    rangesArray[index] = sheetModel.getVisibleRanges(ranges);
                });
            }

            // shorten the ranges if specified
            if (_.isNumber(maxCount)) {
                rangesArray.forEach(function (ranges, index) {
                    rangesArray[index] = ranges.shortenTo(maxCount);
                });
            }

            // reject the entire request, if the ranges are too large (in compressed
            // mode, the size of the result array will be checked later)
            if (!compressed && rangesArray.some(function (ranges) {
                return ranges.cells() > SheetUtils.MAX_QUERY_CELL_COUNT;
            })) {
                def.reject('overflow');
                return promise;
            }

            // request missing cell data from server, resolve the ranges to the result data
            (function resolveRanges() {

                var // the missing ranges not yet covered by the cell collection
                    missingRanges = RangeArray.map(rangesArray, _.identity).difference(boundRanges),
                    // the array of all result arrays
                    resultsArray = null;

                // creates and pushes a new result array to the overall result, checks array length
                function pushResultsForRanges(ranges) {
                    var results = collectCellData(ranges);
                    if (compressed && (results.length > SheetUtils.MAX_QUERY_CELL_COUNT)) {
                        def.reject('overflow');
                        return false;
                    }
                    resultsArray.push(results);
                    return true;
                }

                // request missing ranges from server, repeat resolving ranges recursively
                if (!missingRanges.empty() && (maxRequests > 0)) {
                    maxRequests -= 1;
                    requestRangeContents(missingRanges).done(resolveRanges).fail(_.bind(def.reject, def));
                    return;
                }

                // resolve the deferred object with the resulting array of arrays
                // of cell entries, if no ranges are missing anymore
                resultsArray = [];
                if (rangesArray.every(pushResultsForRanges)) {
                    def.resolve(resultsArray);
                }
            }());

            // return the abortable promise
            return promise;
        };

        /**
         * Inserts new cells into the passed cell ranges, and moves the old
         * cells down or to the right.
         *
         * @param {Range} range
         *  The address of the cell range to be inserted.
         *
         * @param {Boolean} columns
         *  True to move the existing cells horizontally (to the right or left
         *  according to the sheet orientation), or false to move the existing
         *  cells down.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.insertCells = function (range, columns) {

            var // the addresses of all changed (inserted and moved) cells
                changedCells = insertCells(range, columns);

            // notify all change listeners
            triggerChangeCellsEvent(null, changedCells);
            return this;
        };

        /**
         * Removes the passed cell ranges from the collection, and moves the
         * remaining cells up or to the left.
         *
         * @param {Range} range
         *  The address of the cell range to be deleted.
         *
         * @param {Boolean} columns
         *  True to move the existing cells horizontally (to the left or right
         *  according to the sheet orientation), or false to move the existing
         *  cells up.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.deleteCells = function (range, columns) {

            var // the addresses of all changed (deleted and moved) cells
                changedCells = deleteCells(range, columns);

            // notify all change listeners
            triggerChangeCellsEvent(null, changedCells);
            return this;
        };

        // initialization -----------------------------------------------------

        // application notifies changed contents/results of cells
        this.listenTo(docModel.getApp(), 'docs:update:cells', updateCellsHandler);

        // update cell entries after inserting/deleting columns or rows in the sheet
        sheetModel.registerTransformationHandler(transformationHandler);

        // update cell entries after formatting entire columns in the sheet
        colCollection.on('change:entries', function (event, interval, attributes, options) {
            changeIntervalHandler(interval, true, attributes, options);
        });

        // update cell entries after formatting entire rows in the sheet
        rowCollection.on('change:entries', function (event, interval, attributes, options) {
            changeIntervalHandler(interval, false, attributes, options);
        });

        // update cell entries after merging or unmerging ranges
        mergeCollection.on({
            'insert:merged': insertMergedHandler,
            'delete:merged': deleteMergedHandler
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(cellMap, 'destroy');
            tokenArray.destroy();
            self = docModel = grammarConfig = cellStyles = numberFormatter = null;
            sheetModel = colCollection = rowCollection = mergeCollection = null;
            cellMap = boundRanges = null;
        });

    } // class CellCollection

    // constants --------------------------------------------------------------

    /**
     * Standard cell data object for empty cells.
     *
     * @constant
     */
    CellCollection.DEFAULT_CELL_DATA = { display: '', result: null, formula: null };

    /**
     * Display text for pending cells (e.g. for new formula cells waiting for
     * the server response with the formula result).
     *
     * @constant
     */
    CellCollection.PENDING_DISPLAY = Utils.ELLIPSIS_CHAR;

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the cell with the passed descriptor does not contain any
     * value.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object. If missing, the cell is considered to be
     *  blank, and this method returns true.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor is a blank cell.
     */
    CellCollection.isBlank = function (cellDesc) {
        return !cellDesc || _.isNull(cellDesc.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a numeric cell,
     * including formula cells resulting in a number, including positive and
     * negative infinity.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object. If missing, the cell is considered to be
     *  blank, and this method returns false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a numeric result.
     */
    CellCollection.isNumber = function (cellDesc) {
        return !!cellDesc && _.isNumber(cellDesc.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is a Boolean cell,
     * including formula cells resulting in a Boolean value.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a Boolean result.
     */
    CellCollection.isBoolean = function (cellDesc) {
        return !!cellDesc && _.isBoolean(cellDesc.result);
    };

    /**
     * Returns whether the cell with the passed descriptor is an error code
     * cell, including formula cells resulting in an error code.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has an error code (instance
     *  of the class ErrorCode) as result.
     */
    CellCollection.isError = function (cellDesc) {
        return !!cellDesc && (cellDesc.result instanceof ErrorCode);
    };

    /**
     * Returns whether the cell with the passed descriptor is a text cell,
     * including formula cells resulting in a text value.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell with the passed descriptor has a text result.
     */
    CellCollection.isText = function (cellDesc) {
        return !!cellDesc && _.isString(cellDesc.result);
    };

    /**
     * Returns effective text orientation settings for the passed cell data,
     * according to the cell result value and the horizontal alignment value.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank.
     *
     * @returns {Object}
     *  A descriptor containing various orientation properties:
     *  - {String} textDir
     *      The effective writing direction in the single text lines of the
     *      display text, as value supported by the HTML element attribute
     *      'dir' (either 'ltr' or 'rtl').
     *  - {String} cssTextAlign
     *      The effective horizontal CSS text alignment, as value supported by
     *      the CSS property 'text-align'. For cells with automatic alignment,
     *      the effective text alignment is dependent on the data type of the
     *      cell result. Number cells will be aligned to the right, Boolean
     *      cells and cells with error codes will be centered, and text cells
     *      will be aligned according to the writing direction of the result
     *      text (NOT the cell display text which may differ from the result,
     *      e.g. by using number formats).
     *      - 'left': The text lines are aligned at the left cell border.
     *      - 'center': The text lines are centered in the cell.
     *      - 'right': The text lines are ligned at the right cell border.
     *      - 'justify': The text lines will be expanded to the cell width. The
     *          alignment of a single text line, or of the the last text line
     *          in multi-line cells is dependent on the default writing
     *          direction of the .
     *  - {String} baseBoxAlign
     *      The resolved horizontal alignment of the single text lines inside
     *      their text box. Will be equal to the property 'cssTextAlign'
     *      except for justified alignment (property cssTextAlign contains the
     *      value 'justify') which will be resolved to 'left' or 'right',
     *      depending on the default writing direction of the text result (NOT
     *      the display string).
     */
    CellCollection.getTextOrientation = function (cellDesc) {

        // converts the passed text direction to left/right alignment
        function getAlignFromDir(dir) {
            return (dir === 'rtl') ? 'right' : 'left';
        }

        var // the typed cell result
            result = cellDesc ? cellDesc.result : null,
            // the display text of the cell
            display = cellDesc ? cellDesc.display : '',
            // the value of the alignment attribute
            alignHor = cellDesc ? cellDesc.attributes.cell.alignHor : 'auto',
            // the effective writing direction of the display text
            displayDir = _.isString(display) ? Font.getTextDirection(display) : LocaleData.DIR,
            // the effective CSS text alignment
            cssTextAlign = null,
            // base alignment of the text lines in the cell box
            baseBoxAlign = null;

        switch (alignHor) {
            case 'left':
            case 'center':
            case 'right':
            case 'justify':
                cssTextAlign = alignHor;
                break;

            default: // automatic alignment

                // strings (depends on writing direction of the text result, not display string!)
                if (_.isString(result)) {
                    cssTextAlign = getAlignFromDir((result === display) ? displayDir : Font.getTextDirection(result));

                // numbers (always right aligned, independent from display string)
                } else if (_.isNumber(result)) {
                    cssTextAlign = 'right';

                // blank cells (depending on system writing direction)
                } else if (_.isNull(result)) {
                    cssTextAlign = getAlignFromDir(Font.DEFAULT_TEXT_DIRECTION);

                // Booleans and error codes (always centered)
                } else {
                    cssTextAlign = 'center';
                }
        }

        // base box alignment in justified cells depends on writing direction of display string
        baseBoxAlign = (cssTextAlign === 'justify') ? getAlignFromDir(displayDir) : cssTextAlign;

        // return the orientation descriptor object
        return { textDir: displayDir, cssTextAlign: cssTextAlign, baseBoxAlign: baseBoxAlign };
    };

    /**
     * Returns whether the passed cell descriptor will force to wrap the text
     * contents of a cell, either by the cell attribute 'wrapText' set to true,
     * or by horizontally or vertically justified alignment.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be undefined, and this method
     *  returns false.
     */
    CellCollection.hasWrappingAttributes = function (cellDesc) {

        var // shortcut to the cell attributes
            cellAttrs = cellDesc && cellDesc.attributes.cell;

        return !!cellAttrs && (cellAttrs.wrapText || (cellAttrs.alignHor === 'justify') || (cellAttrs.alignVert === 'justify'));
    };

    /**
     * Returns whether the cell with the passed descriptor wraps its text
     * contents at the left and right cell borders. This happens for all cells
     * with text result value, which either have the cell attribute 'wrapText'
     * set to true, or contain a justifying horizontal alignment.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell wraps its text contents automatically.
     */
    CellCollection.isWrappedText = function (cellDesc) {
        return CellCollection.isText(cellDesc) && CellCollection.hasWrappingAttributes(cellDesc);
    };

    /**
     * Returns whether the cell with the passed descriptor overflows its text
     * contents over the left and right cell borders. This happens for all
     * cells with text result value, which do not have the cell attribute
     * 'wrapText' set to true, and do not contain a justifying horizontal
     * alignment.
     *
     * @param {CellDescriptor} [cellDesc]
     *  The cell descriptor object, as received from the cell collection. If
     *  missing, the cell is considered to be blank, and this method returns
     *  false.
     *
     * @returns {Boolean}
     *  Whether the cell contains text that can overflow the cell borders.
     */
    CellCollection.isOverflowText = function (cellDesc) {
        return CellCollection.isText(cellDesc) && !CellCollection.hasWrappingAttributes(cellDesc);
    };

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: CellCollection });

});
