/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */
define('io.ox/office/presentation/utils/presentationutils', [
    'io.ox/office/editframework/utils/attributeutils'
], function (AttributeUtils) {

    'use strict';

    // static class PresentationUtils =========================================

    var PresentationUtils = {};

    /**
     * The default type for place holders in presentation attributes.
     *
     * @constant
     */
    PresentationUtils.DEFAULT_PLACEHOLDER_TYPE = 'body';

    /**
     * The title types for place holders title drawings.
     *
     * @constant
     */
    PresentationUtils.TITLE_PLACEHOLDER_TYPES = {
        title: 1,
        ctrTitle: 1
    };

    /**
     * The default index for place holders in presentation attributes.
     *
     * @constant
     */
    PresentationUtils.DEFAULT_PLACEHOLDER_INDEX = 0;

    // methods ----------------------------------------------------------------

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes. This means, that the phType is set, or that the phIndex is specified. It is possible,
     * that only the phIndex is specified. In this case the phType is defaulted to 'body'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes.
     */
    PresentationUtils.isPlaceHolderAttributeSet = function (attrs) {
        return attrs && attrs.presentation && (attrs.presentation.phType || _.isNumber(attrs.presentation.phIndex));
    };

    /**
     * Check, whether the specified attribute set (explicit attributes from a drawing) contain the presentation
     * place holder attributes of a title place holder drawing. This means, that the phType is set to 'title' or
     * 'ctrTitle'.
     *
     * @param {Object} attrs
     *  The attribute object. This should be the explicit attributes from a drawing.
     *
     * @returns {Boolean}
     *  Whether the specified attribute set contains presentation place holder attributes of 'title' drawing.
     */
    PresentationUtils.isTitlePlaceHolderAttributeSet = function (attrs) {
        return attrs && attrs.presentation && attrs.presentation.phType && PresentationUtils.TITLE_PLACEHOLDER_TYPES[attrs.presentation.phType];
    };

    /**
     * Check, whether the specified drawing is a drawing that contains place holder attributes. This means, that
     * the phType is set, or that the phIndex is specified. It is possible, that only the phIndex is specified. In
     * this case the phType is defaulted to 'body'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes.
     */
    PresentationUtils.isPlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isPlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Check, whether the specified drawing is a title drawing that contains place holder attributes. This means, that
     * the phType is set to 'title' or 'ctrTitle'.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     *
     * @returns {Boolean}
     *  Whether the specified drawing contains presentation place holder attributes of type 'title'.
     */
    PresentationUtils.isTitlePlaceHolderDrawing = function (drawing) {
        return PresentationUtils.isTitlePlaceHolderAttributeSet(AttributeUtils.getExplicitAttributes(drawing));
    };

    /**
     * Getting the default place holder type.
     *
     * @returns {String}
     *  The default place holder type.
     */
    PresentationUtils.getDefaultPlaceHolderType = function () {
        return PresentationUtils.DEFAULT_PLACEHOLDER_TYPE;
    };

    /**
     * Getting the default place holder index.
     *
     * @returns {Number}
     *  The default place holder index.
     */
    PresentationUtils.getDefaultPlaceHolderIndex = function () {
        return PresentationUtils.DEFAULT_PLACEHOLDER_INDEX;
    };

    // exports ================================================================

    return PresentationUtils;

});
