/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/components/drawing/drawingresize', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position'
], function (Utils, Tracking, AttributeUtils, DrawingFrame, Operations, DOM, Position) {

    'use strict';

    var // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // static class DrawingSlideResize ========================================

    var DrawingSlideResize = {};

    // static methods ---------------------------------------------------------

    /**
     * Draws a selection box for the specified drawing node and registers
     * mouse handlers for moving and resizing.
     *
     * @param {PresentationApplication} app
     *  The application instance containing the drawing.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node to be selected, as DOM node or jQuery object.
     */
    DrawingSlideResize.drawDrawingSelection = function (app, drawingNode) {

        var // the model object
            model = app.getModel(),
            // the current root node
            rootNode = model.getCurrentRootNode(),
            // the node containing the scroll bar
            scrollNode = app.getView().getContentRootNode(),
            // object containing information about movable and resizable
            options = { movable: true, resizable: true },  // -> parameter?
            // the container element used to visualize the selection
            selectionBox = null,
            // saving the selected resize node
            resizeNode = null,
            // the container element used to visualize the movement and resizing
            moveBox = null,
            //zoom factor in floating point notation
            zoomFactor,
            // target ID of currently active root node, if existing
            target = model.getActiveTarget(),
            // the resize nodes of the selection
            resizeHandles = null,
            // an object collecting the attributes for an operation
            operationProperties = {};

        /**
         * Handling all tracking events while moving a drawing.
         */
        var drawingMoveHandler = (function () {

            var // the current position change (in px)
                shiftX = 0, shiftY = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the start scroll position
                startScrollX = 0, startScrollY = 0,
                // whether the move box was at least once painted
                moveBoxPainted = false;

            // initialing the moving of the drawing according to the passed tracking event
            function startMoveDrawing() {

                zoomFactor = app.getView().getZoomFactor() / 100;

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();

                moveBoxPainted = false;
            }

            // updating the drawing position according to the passed tracking event
            function updateMove(event) {

                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode, true);

                // reading scrollPosition again. Maybe it was not updated or not updated completely.
                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;

                shiftX = (event.pageX - event.startX + scrollX) / zoomFactor;
                shiftY = (event.pageY - event.startY + scrollY) / zoomFactor;

                // only move the moveBox, if the mouse was really moved
                if ((shiftX !== 0) || (shiftY !== 0)) {
                    if ((_.isNumber(shiftX)) && (_.isNumber(shiftY)) && (shiftX !== 0) || (shiftY !== 0)) {
                        moveBox.css({ left: shiftX, top: shiftY, width: drawingNode.width(), height: drawingNode.height() });
                        moveBoxPainted = true;
                    }
                }
            }

            // updates scroll position according to the passed tracking event
            function updateMoveScroll(event) {

                // update scrollPosition with suggestion from event
                if (event.scrollX) {
                    scrollNode.scrollLeft(scrollNode.scrollLeft() + event.scrollX);
                }

                if (event.scrollY) {
                    scrollNode.scrollTop(scrollNode.scrollTop() + event.scrollY);
                }

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // handling the drawing position, when moving is stopped according to the passed tracking event
            function stopMoveDrawing() {

                var // the operations generator
                    generator = model.createOperationsGenerator();

                if (moveBoxPainted && !($(drawingNode).parent().is('.notselectable'))) { // only move drawing, if it is selectable

                    // setting the new attributes of the drawing
                    operationProperties = {
                        start: Position.getOxoPosition(rootNode, drawingNode, 0),
                        attrs: { drawing: {
                            left: Utils.convertLengthToHmm((moveBox.offset().left - rootNode.offset().left) / zoomFactor, 'px'),
                            top: Utils.convertLengthToHmm((moveBox.offset().top - rootNode.offset().top) / zoomFactor, 'px') }
                        }
                    };

                    model.extendPropertiesWithTarget(operationProperties, target);
                    generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                    // apply the operations (undo group is created automatically)
                    model.applyOperations(generator);
                }
            }

            // finalizes the move tracking
            function finalizeMoveDrawing() {
                leaveTracking();
                drawingNode.off(TRACKING_EVENT_NAMES);

                // Resetting variables for new mouseup events without mousemove
                shiftX = shiftY = scrollX = scrollY = 0;
            }

            // return the actual drawingMoveHandler() function
            return function (event) {

                // handling for text frame drawings, if the click happened inside the internal text frame element
                if ($(event.target).closest(DrawingFrame.TEXTFRAME_NODE_SELECTOR).length > 0) {

                    Tracking.disableTracking(drawingNode)
                        .off('tracking:start', trackingStartHandler)
                        .off(TRACKING_EVENT_NAMES);

                    return;
                }

                switch (event.type) {
                    case 'tracking:start':
                        startMoveDrawing(event);
                        break;
                    case 'tracking:move':
                        updateMove(event);
                        break;
                    case 'tracking:scroll':
                        updateMoveScroll(event);
                        break;
                    case 'tracking:end':
                        stopMoveDrawing(event);
                        finalizeMoveDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeMoveDrawing(event);
                        break;
                }
            };

        }()); // end of drawingMoveHandler() local scope

        /**
         * Handles all tracking events while resizing a drawing.
         */
        var drawingResizeHandler = (function () {

            var // original size of the drawing
                oldWidth = 0, oldHeight = 0,
                // the size of the resized drawing (in px)
                finalWidth = 0, finalHeight = 0,
                // the current scroll position
                scrollX = 0, scrollY = 0,
                // the initial scroll position
                startScrollX = 0, startScrollY = 0,
                // whether resizing is available in horizontal/vertical direction
                useX = false, useY = false,
                // whether resizing is available in left or top direction
                useLeft = false, useTop = false,
                // correction factor for resizing to the left/top
                scaleX = 0, scaleY = 0;

            // initializes resizing the drawing according to the passed tracking event
            function startResizeDrawing(event) {

                var // evaluating which resize element is active
                    pos = $(event.target).attr('data-pos');

                zoomFactor = app.getView().getZoomFactor() / 100;

                // storing old height and width of drawing
                oldWidth = finalWidth = drawingNode.width();
                oldHeight = finalHeight = drawingNode.height();

                // collecting information about the handle node
                useX = /[lr]/.test(pos);
                useY = /[tb]/.test(pos);

                useLeft = /[l]/.test(pos);
                useTop = /[t]/.test(pos);

                if (/l/.test(pos)) {
                    scaleX = -1;
                    moveBox.css({ left: 'auto', right: 0 });
                } else {
                    scaleX = 1;
                    moveBox.css({ left: 0, right: 'auto' });
                }

                if (/t/.test(pos)) {
                    scaleY = -1;
                    moveBox.css({ top: 'auto', bottom: 0 });
                } else {
                    scaleY = 1;
                    moveBox.css({ top: 0, bottom: 'auto' });
                }

                startScrollX = scrollNode.scrollLeft();
                startScrollY = scrollNode.scrollTop();
            }

            // updates scroll position according to the passed tracking event
            function updateResizeScroll(event) {

                // update scrollPosition with suggestion from event
                scrollNode
                    .scrollLeft(scrollNode.scrollLeft() + event.scrollX)
                    .scrollTop(scrollNode.scrollTop() + event.scrollY);

                scrollX = scrollNode.scrollLeft() - startScrollX;
                scrollY = scrollNode.scrollTop() - startScrollY;
            }

            // updates resizing the drawing according to the passed tracking event
            function updateResize(event) {

                var // the horizontal shift
                    deltaX = useX ? ((event.pageX / zoomFactor - event.startX / zoomFactor + scrollX) * scaleX) : 0,
                    // the vertical shift
                    deltaY = useY ? ((event.pageY / zoomFactor - event.startY / zoomFactor + scrollY) * scaleY) : 0,
                    // the scaling factor for the width
                    scaleWidth = 1,
                    // the scaling factor for the height
                    scaleHeight = 1;

                // update drawing size
                finalWidth = Math.max(0, oldWidth + deltaX);
                finalHeight = Math.max(0, oldHeight + deltaY);

                // use the same scaling factor for vertical and horizontal resizing, if both are enabled
                // -> the larger number wins
                if (useX && useY) {
                    scaleWidth = finalWidth / oldWidth;
                    scaleHeight = finalHeight / oldHeight;

                    if (scaleWidth > scaleHeight) {
                        finalHeight = scaleWidth * oldHeight;
                    } else {
                        finalWidth = scaleHeight * oldWidth;
                    }
                }

                // make move box visible when tracking position has moved
                DrawingFrame.toggleTracking(drawingNode, true);
                moveBox.css({ width: finalWidth, height: finalHeight });
            }

            // stop resizing of drawing
            function stopResizeDrawing() {

                var // the operations generator
                    generator = model.createOperationsGenerator(),
                    // the logical position of the drawing
                    drawingPosition = Position.getOxoPosition(rootNode, drawingNode, 0),
                    // the affected drawings with auto resize functionality
                    allAutoResizeNodes = null;

                // setting the new attributes of the drawing
                operationProperties = {
                    start: drawingPosition,
                    attrs: { drawing: {
                        width: Utils.convertLengthToHmm(moveBox.outerWidth(true), 'px'), // 'outer' is required to avoid shrinking in unused dimension
                        height: Utils.convertLengthToHmm(moveBox.outerHeight(true), 'px') }
                    }
                };

                // evaluating useTop and useLeft to avoid sending of superfluous drawing information
                if (useTop) { operationProperties.attrs.drawing.top = Utils.convertLengthToHmm((moveBox.offset().top - rootNode.offset().top) / zoomFactor, 'px'); }
                if (useLeft) { operationProperties.attrs.drawing.left = Utils.convertLengthToHmm((moveBox.offset().left - rootNode.offset().left) / zoomFactor, 'px'); }

                // after resizing an automatic resized text frame in vertical direction by the user,
                // the automatic resize will be removed
                if (useY) {
                    if (DrawingFrame.isAutoResizeHeightDrawingFrame(drawingNode)) {
                        operationProperties.attrs.shape = { autoResizeHeight: false };
                    } else if (DrawingFrame.isGroupDrawingFrame(drawingNode)) {
                        // disable autoResizeHeight for all children inside a resized group
                        allAutoResizeNodes = drawingNode.find(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
                        if (allAutoResizeNodes.length > 0) {
                            _.each(allAutoResizeNodes, function (resizeNode) {

                                var // the logical positions of the children
                                    localDrawingPos = Position.getOxoPosition(rootNode, $(resizeNode).parent(), 0);

                                generator.generateOperation(Operations.SET_ATTRIBUTES, {
                                    start: localDrawingPos,
                                    attrs: { shape: { autoResizeHeight: false } }
                                });
                            });
                        }
                    }
                }

                model.extendPropertiesWithTarget(operationProperties, target);
                generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

                // Applying the operations (undo group is created automatically)
                model.applyOperations(generator);
            }

            // finalizes the resize tracking
            function finalizeResizeDrawing() {
                leaveTracking();
                resizeNode.off(TRACKING_EVENT_NAMES);
                app.getView().scrollToChildNode(drawingNode);
            }

            // return the actual drawingResizeHandler() function
            return function (event) {

                event.preventDefault();

                switch (event.type) {
                    case 'tracking:start':
                        startResizeDrawing(event);
                        break;
                    case 'tracking:move':
                        updateResize(event);
                        break;
                    case 'tracking:scroll':
                        updateResizeScroll(event);
                        break;
                    case 'tracking:end':
                        stopResizeDrawing(event);
                        finalizeResizeDrawing(event);
                        break;
                    case 'tracking:cancel':
                        finalizeResizeDrawing(event);
                        break;
                }

                // events must not bubble up to drawing node (move handling)
                event.stopPropagation();
            };

        }()); // end of drawingResizeHandler() local scope

        /**
         * Leaving the tracking mode.
         */
        function leaveTracking() {
            DrawingFrame.toggleTracking(drawingNode, false);
            moveBox.css({ left: '', top: '', right: '', bottom: '', width: '', height: '' });
        }

        /**
         * Handler for 'tracking:start' events for 'div.handle' and 'div.move'
         * nodes. This handler has to decide, which node got the event. For this
         * node all other tracking events have to be registered. Additionally the
         * handler is called with the 'tracking:start' event, so that the drawing
         * initialization happens.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event, that starts the moving or the resizing
         *  of a drawing.
         */
        function trackingStartHandler(event) {

            var pos = DrawingFrame.getResizerHandleType(event.target);
            if (_.isString(pos)) {
                resizeNode = $(event.target);
                resizeNode.off(TRACKING_EVENT_NAMES);
                resizeNode.on(TRACKING_EVENT_NAMES, drawingResizeHandler);
                drawingResizeHandler.call(this, event);
            } else {
                drawingNode.off(TRACKING_EVENT_NAMES); // <- should not be necessary, but it is (more and more setAttributes OPs)
                drawingNode.on(TRACKING_EVENT_NAMES, drawingMoveHandler);
                drawingMoveHandler.call(this, event);
            }
        }

        // starting code of static method drawDrawingSelection()
        drawingNode = $(drawingNode);

        if (drawingNode.length !== 1) {
            Utils.error('DrawingResize.drawDrawingSelection(): single drawing node expected');
            drawingNode = drawingNode.first();
        }

        if (drawingNode.hasClass('horizontal-line')) {
            options.movable = false;
        } else if (drawingNode.hasClass('grouped')) {
            // options.movable = options.resizable = false;
            drawingNode = drawingNode.closest(DrawingFrame.GROUPCONTENT_SELECTOR).parent();
        }

        if (!model.getEditMode()) {
            options.movable = options.resizable = false;
        }

        // the container element used to visualize the selection
        selectionBox = DrawingFrame.drawSelection(drawingNode, options);

        // set visible drawing anchor
        app.getView().setVisibleDrawingAnchor(drawingNode, AttributeUtils.getExplicitAttributes(drawingNode));

        // initialize move tracking
        moveBox = selectionBox.children('.tracker');

        if (options.movable) {
            Tracking.enableTracking(drawingNode, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250
            });
            drawingNode.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(drawingNode);
            drawingNode.off('tracking:start', trackingStartHandler);
        }

        // initialize resize tracking
        resizeHandles = selectionBox.find('>.resizers>[data-pos]');
        if (options.resizable) {
            Tracking.enableTracking(resizeHandles, {
                autoScroll: true,
                borderNode: scrollNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 10,
                maxSpeed: 250
            });
            resizeHandles.on('tracking:start', trackingStartHandler);
        } else {
            Tracking.disableTracking(resizeHandles);
            resizeHandles.off('tracking:start', trackingStartHandler);
        }
    };

    // exports ================================================================

    return DrawingSlideResize;

});
