/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/app/controller', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/app/editcontroller',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/presentation/format/paragraphstyles',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/components/hyperlink/hyperlink',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/format/tablestyles'
], function (Utils, Color, Border, MixedBorder, LineHeight, EditController, DrawingUtils, DrawingLabels, Image, ParagraphStyles, Config, Hyperlink, Position, TableStyles) {

    'use strict';

    // class TextController ===================================================

    /**
     * The controller of a OX Presentation application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function PresentationController(app, docModel, docView) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        EditController.call(this, app, docModel, docView, { updateDelay: 200 });

        // private methods ----------------------------------------------------

        // initialization -----------------------------------------------------

        if (_.browser.Android) {
            var originalUpdate = self.update;

            self.update = function () {
                if (docModel.isAndroidTyping()) {
                    self.executeDelayed(self.update, 1000, 'Text: android: update');
                } else {
                    originalUpdate();
                }
            };
        }

        // register item definitions
        this.registerDefinitions({

            // view -------------------------------------------------------

            'view/zoom': {
                parent: 'app/valid',
                get: function () { return docView.getZoomFactor() / 100; },
                set: function (zoom) { docView.setZoomType(zoom * 100); }
            },

            'view/zoom/dec': {
                parent: 'app/valid',
                enable: function () { return docView.getZoomFactor() > docView.getMinZoomFactor(); },
                set: function () { docView.decreaseZoomLevel(); }
            },

            'view/zoom/inc': {
                parent: 'app/valid',
                enable: function () { return docView.getZoomFactor() < docView.getMaxZoomFactor(); },
                set: function () { docView.increaseZoomLevel(); }
            },

            'view/zoom/type': {
                parent: 'app/valid',
                get: function () { return docView.getZoomType(); },
                set: function (zoomType) { docView.setZoomType(zoomType); }
            },

            'view/toolbars/sidebar': {
                get: function () { return docView.getSlidePane().isVisible(); },
                set: function (state) { docView.getSlidePane().toggle(state); }
            },

            'document/editable/text': {
                parent: 'document/editable',
                enable: function () { return docModel.isTextSelected(); }
            },

            'document/editable/text/notextframe': {
                parent: 'document/editable/text',
                enable: function () {
                    return !docModel.getSelection().isAdditionalTextframeSelection();
                }
            },

            'document/selectall': {
                // enabled in read-only mode
                set: function () { docModel.selectAll(); },
                // restrict keyboard shortcut to application pane
                shortcut: { keyCode: 'A', ctrlOrMeta: true, selector: '.app-pane' },
                preserveFocus: true // do not return focus to document
            },

            'document/selectDrawing': {
                set: function (options) {
                    docModel.getSelection().selectNextDrawing({ backwards: Utils.getBooleanOption(options, 'backwards', false) });
                },
                shortcut: { keyCode: 'F4', shift: true }
            },
            'document/setCursorIntoTextframe': {
                set: function () {
                    docModel.getSelection().setSelectionIntoTextframe();
                },
                shortcut: { keyCode: 'F2' }
            },

            // document contents
            'document/cut': {
                parent: 'document/editable',
                enable: function () { return docModel.hasSelectedRange(); },
                set: function () { docModel.cut(); }
            },
            'document/copy': {
                // enabled in read-only mode
                enable: function () { return docModel.hasSelectedRange(); },
                set: function () { docModel.copy(); }
            },
            'document/paste': {
                parent: 'document/editable',
                enable: function () { return docModel.hasInternalClipboard(); },
                set: function () { docModel.pasteInternalClipboard(); }
            },

            // spelling

            'document/spelling/available': {
                parent: 'document/editable',
                enable: function () { return Config.SPELLING_ENABLED; }
            },
            'document/onlinespelling': {
                parent: 'document/spelling/available',
                get: function () { return docModel.getSpellChecker().isOnlineSpelling(); },
                set: function (state) { docModel.getSpellChecker().setOnlineSpelling(state); }
            },
            'document/spelling/replace': {
                parent: 'document/editable',
                set: function (replacement) {
                    var selection = docModel.getSelection(),
                        startPosition = selection.getStartPosition(),
                        wordPos = Position.getWordSelection(selection.getEnclosingParagraph(), startPosition[startPosition.length - 1]);

                    docModel.getSpellChecker().replaceWord(startPosition, wordPos, replacement);
                }
            },
            'document/spelling/ignoreword': {
                parent: 'document/editable',
                set: function (options) {
                    return docModel.getSpellChecker().addWordToIgnoreSpellcheck(options);
                }
            },
            'document/spelling/userdictionary': {
                parent: 'document/editable',
                set: function (options) {
                    return docModel.getSpellChecker().addWordToIgnoreSpellcheck(options);
                }
            },
            // Page settings

            'document/pagesettings': {
                parent: 'document/editable/text',
                set: function () { return docView.showPageSettingsDialog(); }
            },

            // paragraphs

            'paragraph/group': {
                parent: 'document/editable/text',
                get: function () { return docModel.getAttributes('paragraph', { maxIterations: 10 }) || {}; }
            },
            'paragraph/attributes': {
                parent: 'paragraph/group',
                get: function (paragraph) { return paragraph.paragraph || {}; }
            },
            'paragraph/alignment': {
                parent: ['paragraph/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return attributes.alignment; },
                set: function (alignment) { docModel.setAttribute('paragraph', 'alignment', alignment); }
                // removed for the 7.2.0 due to clashes with browser shortcuts
                /*shortcut: [
                    { keyCode: 'L', ctrlOrMeta: true, value: 'left' },
                    { keyCode: 'R', ctrlOrMeta: true, value: 'right' },
                    { keyCode: 'E', ctrlOrMeta: true, value: 'center' },
                    { keyCode: 'J', ctrlOrMeta: true, value: 'justify' }
                ]*/
            },
            'paragraph/lineheight': {
                parent: ['paragraph/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return attributes.lineHeight; },
                set: function (lineHeight) { docModel.setAttribute('paragraph', 'lineHeight', lineHeight); }
                // removed for the 7.2.0 due to clashes with browser shortcuts
                /*shortcut: [
                    { keyCode: '1', ctrlOrMeta: true, value: LineHeight.SINGLE },
                    { keyCode: '5', ctrlOrMeta: true, value: LineHeight.ONE_HALF },
                    { keyCode: '2', ctrlOrMeta: true, value: LineHeight.DOUBLE }
                ]*/
            },

            // parent for controller items only enabled inside lists
            'paragraph/list/enabled': {
                parent: 'paragraph/attributes',
                enable: function () {
                    var indent = this.getValue().level;
                    return _.isNumber(indent) && (indent >= 0) && (indent <= 8);
                }
            },

            // toggle default bullet list, or select bullet type
            'paragraph/list/bullet': {
                parent: ['paragraph/attributes'],
                get: _.partial(docModel.getListStyleId, 'bullet'),
                set: function (value) { docModel.setListStyleId('bullet', value, this.getValue()); }
            },
            // toggle default numbered list, or select numbering type
            'paragraph/list/numbered': {
                parent: ['paragraph/attributes'],
                get: _.partial(docModel.getListStyleId, 'numbering'),
                set: function (value) { docModel.setListStyleId('numbering', value, this.getValue()); }
            },

            // change list level
            'paragraph/list/indent': {
                parent: 'paragraph/list/enabled',
                get: function (attributes) { return attributes.level; },
                set: function (indent) { docModel.setAttribute('paragraph', 'level', indent); }
            },
            // increase list level by one
            'paragraph/list/incindent': {
                parent: 'paragraph/list/indent',
                enable: function () { return docModel.isListIndentChangeable({ increase: true }); },
                set: function () { docModel.changeListIndent({ increase: true, validatedLevel: true }); }
            },
            // decrease list level by one
            'paragraph/list/decindent': {
                parent: 'paragraph/list/indent',
                enable: function () { return docModel.isListIndentChangeable({ increase: false }); },
                set: function () { docModel.changeListIndent({ increase: false, validatedLevel: true }); }
            },
            // slides

            'slide/insertslide': {
                enable: function () { return !docModel.isMasterView(); },
                set: function () { docModel.insertSlide(); }
            },
            'slide/deleteslide': {
                enable: function () { return !docModel.isMasterView() && !docModel.isOnlySlideInView(); },
                set: function () { docModel.deleteSlide(); }
            },

            // characters

            'character/group': {
                parent: 'document/editable/text',
                get: function () { return docModel.getAttributes('character', { maxIterations: 10 }) || {}; }
            },
            'character/stylesheet': {
                parent: 'character/group',
                get: function (character) { return character.styleId; },
                set: function (styleId) { docModel.setAttributes('character', { styleId: styleId }, { clear: true }); }
            },
            'character/attributes': {
                parent: 'character/group',
                get: function (character) { return character.character || {}; }
            },
            'character/fontname': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.fontName; },
                set: function (fontName) { docModel.setAttribute('character', 'fontName', fontName); }
            },
            'character/fontsize': {
                parent: ['character/attributes', 'insert/comment/supported/odf'],
                get: function (attributes) { return attributes.fontSize; },
                set: function (fontSize) { docModel.setAttribute('character', 'fontSize', fontSize); }
            },
            'character/bold': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.bold; },
                set: function (state) { docModel.setAttribute('character', 'bold', state); },
                shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
            },
            'character/italic': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.italic; },
                set: function (state) { docModel.setAttribute('character', 'italic', state); },
                shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
            },
            'character/underline': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.underline; },
                set: function (state) { docModel.setAttribute('character', 'underline', state); },
                shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
            },
            'character/strike': {
                parent: 'character/attributes',
                get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                set: function (state) { docModel.setAttribute('character', 'strike', state ? 'single' : 'none'); }
            },
            'character/vertalign': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.vertAlign; },
                set: function (align) { docModel.setAttribute('character', 'vertAlign', align); }
            },
            'character/color': {
                parent: ['character/attributes', 'insert/comment/odf/supported'],
                get: function (attributes) { return attributes.color; },
                set: function (color) { docModel.setAttribute('character', 'color', color); }
            },
            'character/language': {
                parent: 'character/attributes',
                get: function (attributes) { return attributes.language; },
                set: function (language) { docModel.setAttribute('character', 'language', language); }
            },
            'character/hyperlink': {
                parent: 'character/attributes',
                enable: function () { return docModel.hasEnclosingParagraph(); },
                get: function (attributes) { return attributes.url; }
                // no direct setter (see item 'character/hyperlink/dialog')
            },
            'character/hyperlink/dialog': {
                parent: 'character/hyperlink',
                get: function () { return Hyperlink.hasPopup(docModel); },
                set: function () { return docModel.insertHyperlinkDialog(); }
                // removed for the 7.2.0 due to clashes with browser shortcuts
                //shortcut: { keyCode: 'K', ctrlOrMeta: true }
            },
            'character/hyperlink/remove': {
                parent: 'character/hyperlink',
                set: function () { return docModel.removeHyperlink(); }
            },

            'character/reset': {
                parent: 'document/editable/text',
                set: function () { docModel.resetAttributes(); },
                shortcut: { keyCode: 'SPACE', ctrl: true }
            },

            'character/format': {
                parent: 'app/valid'
            },

            'character/insert/break': {
                parent: 'document/editable/text',
                set: function () { return docModel.insertHardBreak(); }
            },
            'character/insert/tab': {
                parent: 'document/editable/text',
                set: function () { return docModel.insertTab(); }
            },

            // reduced odf text frame functionality
            'insert/textframe/supported': {
                enable: function () { return !docModel.isReducedOdfTextframeFunctionality(); }
            },

            // reduced odf comment functionality
            'insert/comment/odf/supported': {
                enable: function () { return !docModel.isOdfCommentFunctionality(); }
            },

            'insert/comment/supported': {
                enable: function () { return !docModel.isCommentFunctionality(); }
            },

            'insert/comment/supported/odf': {
                enable: function () { return app.isODF() || !docModel.isCommentFunctionality(); }
            },

            'insert/margin/supported': {
                enable: function () { return !docModel.isHeaderFooterEditState(); }
            },

            'insert/margin/supported/odf': {
                enable: function () { return app.isODF() || !docModel.isHeaderFooterEditState(); }
            },

            // tables

            'table/group': {
                parent: 'document/editable/text',
                get: function () { return docModel.getAttributes('table') || {}; }
            },
            'table/insert/available': {
                enable: function () { return Config.MAX_TABLE_CELLS > 0; }
            },
            'table/insert': {
                parent: ['document/editable/text', 'table/insert/available', 'insert/textframe/supported', 'insert/comment/odf/supported'],
                set: function (size) { docModel.insertTextFrame({ insertTable: true, size: size }); },
                preserveFocus: true // do not return focus to document
            },

            'document/editable/table': {
                parent: 'table/group',
                enable: function () { return docModel.isPositionInTable(); }
            },
            'document/show/tabletab': {
                parent: 'document/editable/table',
                enable: function () { return docModel.isCellRangeSelected() || (!docModel.isCellRangeSelected() && !docModel.hasSelectedRange()); }
            },
            'table/delete': {
                parent: 'document/editable/table',
                set: function () { return docModel.deleteTable(); }
            },
            'table/insert/row': {
                parent: 'document/editable/table',
                set: function () { docModel.insertRow(); }
            },
            'table/insert/column': {
                parent: 'document/editable/table',
                set: function () { docModel.insertColumn(); }
            },
            'table/delete/row': {
                parent: 'document/editable/table',
                set: function () { return docModel.deleteRows(); }
            },
            'table/delete/column': {
                parent: 'document/editable/table',
                set: function () { return docModel.deleteColumns(); }
            },

            'table/stylesheet': {
                parent: 'document/editable/table',
                get: function (table) { return table.styleId; },
                set: function (styleId) { docModel.setAttributes('table', { styleId: styleId }, { clear: true }); }
            },
            'table/attributes': {
                parent: 'document/editable/table',
                get: function (table) { return table.table || {}; }
            },
            'table/cellattributes': {
                parent: 'document/editable/table',
                get: function () { return docModel.getAttributes('cell').cell || {}; }
            },
            'table/cellborder': {
                parent: 'table/attributes',
                get: function (attributes) { return MixedBorder.getBorderMode(attributes); },
                set: function (borderMode) {
                    var borderAttrs = MixedBorder.getBorderAttributes(borderMode, this.getParentValue(), ParagraphStyles.SINGLE_BORDER);
                    docModel.setAttributes('table', { table: borderAttrs }, { cellSpecificTableAttribute: true });
                }
            },
            'table/borderwidth': {
                parent: 'table/cellattributes',
                get: function (attributes) { return TableStyles.getBorderStyleFromAttributes(attributes); },
                set: function (borderWidth) {
                    docModel.setAttributes('table', {
                        table: TableStyles.getAttributesFromBorderStyle(borderWidth, docModel.getAttributes('table').table)
                    }, { onlyVisibleBorders: true });
                }
            },
            'table/fillcolor': {
                parent: 'table/cellattributes',
                get: function (attributes) { return attributes.fillColor; },
                set: function (color) { docModel.setAttribute('cell', 'fillColor', color); }
            },
            'table/split': {
                parent: 'document/editable/table',
                set: function () { docModel.splitTable(); }
            },

            // drawing

            'document/editable/drawing': {
                parent: 'document/editable',
                enable: function () { return docModel.isDrawingSelected(); }
            },
            'document/editable/anydrawing': {
                parent: 'document/editable',
                enable: function () { return docModel.getSelection().isAnyDrawingSelection(); }
            },
            'document/editable/onlysupporteddrawings': {
                parent: 'document/editable',
                enable: function () { return docModel.getSelection().areOnlySupportedDrawingsSelected(); }
            },
            'drawing/shapes/support/fill': {
                parent: 'document/editable/onlysupporteddrawings',
                enable: function () {
                    var selection = docModel.getSelection();
                    return (/^(shape|group)$/).test(selection.getSelectedDrawingType()) || (selection.getSelectionType() === 'text');
                }
            },
            'drawing/shapes/support/line': {
                parent: 'document/editable/onlysupporteddrawings',
                enable: function () {
                    var selection = docModel.getSelection();
                    return (/^(shape|group|image)$/).test(selection.getSelectedDrawingType()) || (selection.getSelectionType() === 'text');
                }
            },

            'drawing/attributes/full': {
                parent: 'document/editable/anydrawing',
                get: function () { return docModel.getAttributes('drawing') || {}; }
            },

            'drawing/attributes': {
                parent: 'drawing/attributes/full',
                get: function (attributes) { return attributes.drawing || {}; }
            },
            'drawing/group/attributes': {
                parent: 'document/editable/anydrawing',
                get: function () { return docModel.getAttributes('drawing', { groupOnly: true }).drawing || {}; }
            },

            'drawing/delete': {
                parent: 'document/editable/anydrawing',
                set: function () {
                    var selection = docModel.getSelection();
                    if (docModel.isDrawingSelected()) {
                        return docModel.deleteSelected();

                    } else if (selection.isAdditionalTextframeSelection()) {
                        var start = Position.getOxoPosition(docModel.getCurrentRootNode(), selection.getSelectedTextFrameDrawing(), 0),
                            end = Position.increaseLastIndex(start);

                        selection.setTextSelection(start, end);
                        return docModel.deleteSelected();
                    }
                }
            },

            'drawing/type/label': {
                parent: 'document/editable/drawing',
                get: function () { return DrawingLabels.getDrawingTypeLabel(docModel.getSelection().getClosestSelectedDrawingType()); }
            },

            // drawing fill formatting

            'drawing/fill/attributes': {
                parent: 'drawing/attributes/full',
                get: function (attributes) { return attributes.fill || {}; }
            },

            'drawing/fill/color': {
                parent: 'drawing/fill/attributes',
                get: function (attributes) { return attributes.color; },
                set: function (color) { docModel.setDrawingFillColor(color); }
            },

            // drawing line formatting

            'drawing/line/attributes': {
                parent: 'drawing/attributes/full',
                get: function (attributes) { return attributes.line || {}; }
            },

            'drawing/border/style': {
                parent: 'drawing/line/attributes',
                get: function (attributes) { return DrawingUtils.getPresetBorder(attributes); },
                set: function (border) { docModel.setDrawingBorder(border); }
            },

            'drawing/border/color': {
                parent: 'drawing/line/attributes',
                get: function (attributes) { return (attributes.type !== 'none') ? attributes.color : Color.AUTO; },
                set: function (color) { docModel.setDrawingBorderColor(color); }
            },

            // text frames

            'textframe/insert': {
                parent: ['document/editable/text', 'insert/textframe/supported', 'insert/comment/odf/supported'],
                set: function () { docModel.insertTextFrame(); }
            },

            // comments

            'comment/insert': {
                parent: ['document/editable/text/notextframe', 'insert/comment/supported', 'insert/margin/supported'],
                set: function () { docModel.getCommentLayer().insertComment(); }
            },

            'comment/displayModeParent': {
                enable: function () { return !docModel.getCommentLayer().isEmpty(); }
            },

            'comment/displayMode': {
                get: function () { return docModel.getCommentLayer().getDisplayMode(); },
                set: function (mode) { docModel.getCommentLayer().switchCommentDisplayMode(mode); }
            },

            'comment/authorFilter': {
                get: function () { return docModel.getCommentLayer().getListOfUnfilteredAuthors(); },
                set: function (authorList) { docModel.getCommentLayer().setAuthorFilter(authorList); }
            },

            'comment/nextComment': {
                // enabled in read-only mode
                enable: function () { return !docModel.getCommentLayer().isEmpty(); },
                set: function () { docModel.getCommentLayer().selectNextComment({ next: true }); }
            },
            'comment/prevComment': {
                // enabled in read-only mode
                enable: function () { return !docModel.getCommentLayer().isEmpty(); },
                set: function () { docModel.getCommentLayer().selectNextComment({ next: false }); }
            },

            'comment/deleteAll': {
                parent: ['document/editable'],
                enable: function () { return !docModel.getCommentLayer().isEmpty(); },
                set: function () { docModel.getCommentLayer().deleteComment(null, { all: true }); }
            },

            // images

            'image/insert/dialog': {
                parent: ['document/editable/text', 'insert/textframe/supported', 'insert/comment/odf/supported'],
                set: function (dialogType) { return docModel.showInsertImageDialog(dialogType); }
            },

            // debug
            'debug/textframeautofit': {
                enable: function () { return docModel.getSelection().isAnyTextFrameSelection({ allDrawingTypesAllowed: true }); },
                get: function () { return docModel.isAutoResizableTextFrame(); },
                set: function (state) { docModel.handleTextFrameAutoFit(state); }
            },
            'debug/recordoperations': {
                get: function () { return docModel.isRecordingOperations(); },
                set: function (state) { docModel.setRecordingOperations(state); }
            },
            'debug/replayoperations': {
                parent: 'document/editable',
                enable: function () { return docView.getDebugOperationsContent() !== null; },
                set: function () { docModel.replayOperations(docView.getDebugOperationsContent(), docView.getDebugOperationsOSN()); }
            },
            'debug/draftmode/toggle': {
                parent: 'debug/enabled',
                enable: function () { return !Utils.SMALL_DEVICE; },
                get: function () { return docModel.isDraftMode(); },
                set: function (state) { docModel.toggleDraftMode(state); }
            },
            'debug/verticalalignment': {
                enable: docModel.getSelection().isAnyTextFrameSelection({ allDrawingTypesAllowed: true }),
                get: function () { return docModel.getVerticalAlignmentMode(); },
                set: function (mode) { docModel.setVerticalAlignmentMode(mode); }
            },
            'debug/insertslide': {
                enable: function () { return !docModel.isMasterView(); },
                set: function () { docModel.insertSlide(); }
            },
            'debug/deleteslide': {
                enable: function () { return !docModel.isMasterView() && !docModel.isOnlySlideInView(); },
                set: function () { docModel.deleteSlide(); }
            },
            'debug/followmastershapes': {
                get: function () { return docModel.isFollowMasterShapeSlide(); },
                set: function () { docModel.setFollowMasterShape(!docModel.isFollowMasterShapeSlide()); }
            },
            'debug/resetbackground': {
                enable: function () { return docModel.isSlideWithBackground(); },
                set: function () { docModel.resetBackground(); }
            },
            'debug/slideattributes': {
                get: function () { return docModel.getSlideAttributes(); }
            },
            'debug/setbackgroundcolor': {
                parent: ['debug/slideattributes'],
                get: function (attrs) { return attrs && attrs.fill && attrs.fill.color; },
                set: function (color) { docModel.setBackgroundColor(color); }
            },
            'debug/setbackgroundimage': {
                parent: ['debug/slideattributes'],
                get: function (attrs) { return attrs && attrs.fill && attrs.fill.color; },
                set: function (dialogType) {
                    return docModel.showInsertImageDialogForSlideBackground(dialogType);
                }
            },
            'debug/normalslideview': {
                enable: function () { return docModel.isMasterView(); },
                set: function () { docModel.selectSlideView(); }
            },
            'debug/masterslideview': {
                enable: function () { return !docModel.isMasterView(); },
                set: function () { docModel.selectSlideView({ showMaster: true }); }
            },
            'debug/previousslide': {
                enable: function () { return !docModel.isFirstSlideActive(); },
                set: function () { docModel.changeToNeighbourSlide({ next: false }); }
            },
            'debug/nextslide': {
                enable: function () { return !docModel.isLastSlideActive(); },
                set: function () { docModel.changeToNeighbourSlide({ next: true }); }
            },

            // fields
            updateField: {
                parent: 'document/editable',
                set: function () { docModel.getFieldManager().updateHighlightedField(); }
            },
            updateAllFields: {
                parent: 'document/editable',
                set: function () { docModel.getFieldManager().updateAllFields(); }
            },
            removeField: {
                parent: 'document/editable',
                set: function () { docModel.getFieldManager().removeField(); }
            },

            'document/insertfield': {
                parent: 'document/editable/text',
                set: function (value) { docModel.getFieldManager().dispatchInsertField(value, 'default'); }
            },

            'document/formatfield': {
                parent: 'document/editable',
                get: function () { return docModel.getFieldManager().getSelectedFieldFormat(); },
                set: function (value) { docModel.getFieldManager().updateFieldFormatting(value); }
            },

            // slide Pane
            slidePaneShowFirstSlide: {
                set: function () {  docModel.setActiveSlideId(docModel.getIdOfFirstSlideOfActiveView()); }
            },

            slidePaneShowLastSlide: {
                set: function () { docModel.setActiveSlideId(docModel.getIdOfLastSlideOfActiveView()); }
            },

            slidePaneSlideUp: {
                set: function () { docModel.changeToNeighbourSlide({ next: false }); }
            },

            slidePaneSlideDown: {
                set: function () { docModel.changeToNeighbourSlide({ next: true }); }
            },

            slidePaneSetActiveSlide: {
                set: function (id) { docModel.setActiveSlideId(id); }
            },

            // android-entry for selection which cant be deleted by softkeyboard
            'document/editable/selection': {
                parent: 'document/editable',
                enable: function () {
                    var selection = docModel.getSelection();
                    return !_.isEqual(selection.getStartPosition(), selection.getEndPosition());
                }
            },
            'selection/delete': {
                parent: 'document/editable/selection',
                set: function () { return docModel.deleteSelected(); }
            }
        });

        // update GUI after changed selection
        this.listenTo(docModel, 'selection', function () { self.update(); });

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = docView = null;
        });

    } // class PresentationController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: PresentationController });

});
