/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/languagepicker', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/control/radiolist',
    'io.ox/office/editframework/utils/editconfig',
    'io.ox/office/tk/config',
    'settings!io.ox/office',
    'gettext!io.ox/office/editframework/main'
], function (Utils, LocaleData, RadioList, EditConfig, Config, Settings, gt) {

    'use strict';

    // locale codes of all supported languages
    var LOCALES = [
        'en_US',
        'en_GB',
        'de_DE',
        'fr_FR',
        'es_ES',
        'cs_CZ',
        'da_DK',
        'nl_NL',
        'fi_FI',
        'el_GR',
        'hu_HU',
        'it_IT',
        'pl_PL',
        'pt_PT',
        'ro_RO',
        'ru_RU',
        'sv_SE'
    ];

    // a promise that will resolve when the locales with dictionaries have been received
    var definitionsPromise = (function () {

        // configuration data for all supported locales
        var localeDefinitions = LOCALES.map(function (locale) {
            return {
                lc: locale, // the original locale identifier
                value: locale.replace('_', '-'), // formatting attribute value (dash instead of underscore)
                supported: false, // whether a dictionary is available, will be updated after server request)
                label: LocaleData.getLanguageName(locale) // translated name of the language
            };
        });

        return EditConfig.getLocalesWithDictionary().then(function (supportedLocales) {
            localeDefinitions.forEach(function (definition) {
                definition.supported = _.contains(supportedLocales, definition.lc);
            });
            return localeDefinitions;
        });
    }());

    // class LanguagePicker ===================================================

    /**
     * A drop-down list with all supported languages.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the RadioList base class.
     *  Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.showListIcons=true]
     *      Whether to show the check boxes in the list.
     *  @param {Boolean} [initOptions.showAllLanguages=true]
     *      If true show "Most Recently Languages", None and all supported Languages  see@ LOCALES.
     *      If false show "Most Recently Languages", None and the Language of the current
     *      selected word.
     */
    function LanguagePicker(initOptions) {

        var // self reference
            self = this,
            showListIcons = Utils.getBooleanOption(initOptions, 'showListIcons', true),
            showAllLanguages = Utils.getBooleanOption(initOptions, 'showAllLanguages', true),
            mostRecentlyUsedNumber = showAllLanguages ? 4 : 0,
            mostRecentlyBlackList = showAllLanguages ? ['none'] : [],
            mostRecentlyUsedSettingsKey = 'languagepicker/mostrecentlyused',
            localeDefinitions,
            uiLocale = Config.LOCALE.replace('_', '-');

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions(initOptions, {
            tooltip: gt('Text language'),
            sortItems: true,
            mostRecentlyUsedNumber: mostRecentlyUsedNumber,
            mostRecentlyBlackList: mostRecentlyBlackList,
            mostRecentlyUsedSettingsKey: mostRecentlyUsedSettingsKey
        }));

        // private methods ----------------------------------------------------

        /**
         * Add languages to the Language Picker if the initOptions.showAllLanguages is true;
         */
        function addLanguagePickerLanguages() {
            self.getMenu().createMostRecentlyUsedSection();
            createNoneLanguageButton();

            addLanguages();
        }

        /**
         * Add languages to the Language Picker if the initOptions.showAllLanguages is false;
         */
        function addReduceLanguagePickeLanguages() {
            var mostRecentlyLanguages = Settings.get(mostRecentlyUsedSettingsKey, []);

            function isMostRecentlyLanguage(value) {
                return _.find(mostRecentlyLanguages, function (item) {
                    return item.value === value;
                });
            }

            addLanguages(function (value) {
                return isMostRecentlyLanguage(value) || self.getValue() === value || uiLocale === value;
            });
            createNoneLanguageButton();
        }

        function createNoneLanguageButton() {
            self.createOptionButton('none', {
                label: /*#. None is a "special" text language used in the languagepicker or means "Do not check spelling". Please do not remove the square brackets. */ gt('[None]'),
                section: '[None]'
            });
        }

        /**
         * Add the languages from the local definition to the menu list.
         * @param {type} [languageFilter] filter for the languages the 1param is the Local,
         *  if the filter return true the language will be added to the list.
         */
        function addLanguages(languageFilter) {
            var filter = _.isFunction(languageFilter) ? languageFilter : _.constant(true);

            localeDefinitions.forEach(function (definition) {

                if (filter(definition.value)) {
                    var options = { label: definition.label };
                    if (showListIcons) {
                        options.icon = 'docs-dict-available';
                        if (!definition.supported) {
                            options.iconStyle = 'visibility: hidden';
                        }
                    }

                    self.createOptionButton(definition.value, options);
                }
            });

        }

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('language-picker');

        this.waitForSuccess(definitionsPromise, function (localeDef) {
            localeDefinitions = localeDef;
            if (Utils.getBooleanOption(initOptions, 'showAllLanguages', true)) {
                addLanguagePickerLanguages();
            } else {
                addReduceLanguagePickeLanguages();
            }
        });

        if (!showAllLanguages) {
            // Register the update handler to add the language of the selected word to the minimized language picker
            this.registerUpdateHandler(function (newLanguage) {
                if (localeDefinitions && _.isString(newLanguage) && self.getMenu().findItemNodes(newLanguage).length === 0) {

                    addLanguages(function (language) {
                        return newLanguage === language;
                    });
                }
            });
        }

        // destroy all members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class LanguagePicker

    // exports ================================================================

    // derive this class from class RadioList
    return RadioList.extend({ constructor: LanguagePicker });

});
