/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Malte Timmermann <malte.timmermann@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/undomanager', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, ModelObject) {

    'use strict';

    // class UndoAction =======================================================

    /**
     * An instance of UndoAction is used by the undo manager to store undo and
     * redo operations and to apply them in the document model.
     *
     * @constructor
     *
     * @param {Object|Object[]|Function} [undoOperations]
     *  One ore more operations that form a logical undo action. When the undo
     *  action is executed, the operations will be applied in the exact order
     *  as passed in this parameter; they will NOT be applied in reversed order
     *  as it would be the case if the operations have been added in multiple
     *  undo actions. The parameter may be a single operation object, an array
     *  of operation objects, or omitted.
     *
     * @param {Object|Object[]|Function} [redoOperations]
     *  One ore more operations that form a logical redo action. When the redo
     *  action is executed, the operations will be applied in the exact order
     *  as passed in this parameter. The parameter may be a single operation
     *  object, an array of operation objects, or omitted.
     */
    function UndoAction(undoOperations, redoOperations) {
        this.undoOperations = [];
        this.redoOperations = [];
        this.appendOperations(undoOperations, redoOperations);
    }

    /**
     * Appends the passed undo and redo operations/functions to the own operations
     * arrays. Undo operations/functions will be inserted at the beginning of the undo
     * operations array, and redo operations will be appended to the redo
     * operations array.
     */
    UndoAction.prototype.appendOperations = function (undoOperations, redoOperations) {

        // add the undo operations at the beginning of the array
        if (_.isArray(undoOperations)) {
            this.undoOperations = _.copy(undoOperations, true).concat(this.undoOperations);
        } else if (_.isFunction(undoOperations)) {
            this.undoOperations.unshift(undoOperations);
        } else if (_.isObject(undoOperations)) {
            this.undoOperations.unshift(_.copy(undoOperations, true));
        }

        // add the redo operations at the end of the array
        if (_.isArray(redoOperations)) {
            this.redoOperations = this.redoOperations.concat(_.copy(redoOperations, true));
        } else if (_.isFunction(redoOperations)) {
            this.redoOperations.push(redoOperations);
        } else if (_.isObject(redoOperations)) {
            this.redoOperations.push(_.copy(redoOperations, true));
        }
    };

    // class UndoManager ======================================================

    /**
     * The UndoManager mix-in class adds undo/redo functionality to the
     * document model. Also the grouping and merging of undo/redo operations is
     * a task of the undo manager.
     *
     * Triggers the following events:
     * - 'undo:before': Before undo operations are about to be applied by the
     *      method UndoManager.undo().
     * - 'undo:after': After undo operations have been applied by the method
     *      UndoManager.undo().
     * - 'redo:before': Before redo operations are about to be applied by the
     *      method UndoManager.redo().
     * - 'redo:after': After redo operations have been applied by the method
     *      UndoManager.redo().
     * - 'undogroup:open': After a new undo group has been opened by the method
     *      UndoManager.enterUndoGroup().
     * - 'undogroup:close': After an undo group has been closed by the method
     *      UndoManager.enterUndoGroup().
     * - 'change:count': After the number of available undo/redo actions has
     *      changed. Event listeners receive the current number of available
     *      undo and redo actions.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this undo manager instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {Boolean} [initOptions.remoteUndo=false]
     *      If set to true, the application does not manage its undo and redo
     *      operations, but sends 'undo' and 'redo' requests to the server
     *      which executes the actual undo or redo operations from its internal
     *      undo/redo stack, and answers with all the operations to be applied
     *      at the document model, and the new number of undo and redo actions
     *      available.
     *  @param {Function} [initOptions.mergeUndoActionHandler]
     *      A function that will be called while adding a new undo action onto
     *      the undo stack, and that can try to merge the operations of the new
     *      undo action into the operations of the last undo action already
     *      stored in the undo stack. Receives the following parameters:
     *      (1) {UndoAction} prevAction
     *          The last undo action stored in the undo stack.
     *      (2) {UndoAction} nextAction
     *          The new action about to be pushed onto the undo stack.
     *      Must return true, if the new undo action has been merged into the
     *      existing undo action, in that case the new undo action will be
     *      discarded. All other return values result in pushing the new undo
     *      action onto the undo stack.
     */
    function UndoManager(docModel, initOptions) {

        var // self reference
            self = this,

            // the application instance
            app = docModel.getApp(),

            // whether the application manages undo/redo operations on the server
            remoteUndo = Utils.getBooleanOption(initOptions, 'remoteUndo', false),

            // callback function that implements merging two actions to one action
            mergeUndoActionHandler = Utils.getFunctionOption(initOptions, 'mergeUndoActionHandler', $.noop),

            // all undo actions
            actions = [],

            // index of the next redo action in the 'actions' array
            currentAction = 0,

            // number of nested action groups
            groupLevel = 0,

            // current undo action in grouped mode
            groupAction = null,

            // whether undo or redo operations are currently processed
            processing = false,

            // number of undo actions available in the remote undo manager
            remoteUndoCount = 0,

            // number of redo actions available in the remote undo manager
            remoteRedoCount = 0;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // private methods ----------------------------------------------------

        /**
         * Triggers a 'change:count' event with the current number of available
         * undo actions, and redo actions.
         */
        function triggerChangeCountEvent() {
            self.trigger('change:count', self.getUndoCount(), self.getRedoCount());
        }

        /**
         * Opens a new undo group. If no undo group is open yet, triggers an
         * 'undogroup:open' event.
         */
        function openUndoGroup() {

            // notify listeners when top-level undo group has been opened
            if (groupLevel === 0) {
                self.trigger('undogroup:open');
            }

            groupLevel += 1;
        }

        /**
         * Closes the most recent undo group. If no undo group is open anymore
         * afterwards, triggers an 'undogroup:close' event.
         */
        function closeUndoGroup() {

            // push existing group action to action stack on last group level
            groupLevel -= 1;
            if ((groupLevel === 0) && groupAction) {
                if (!app.isOperationsBlockActive()) { // the operations were not cancelled
                    pushAction(groupAction);
                }
                groupAction = null;
            }

            // notify listeners when top-level undo group has been closed
            if (groupLevel === 0) {
                self.trigger('undogroup:close');
            }
        }

        /**
         * Pushes the passed new undo action onto the undo stack. Shortens the
         * undo stack if the current undo action is not the topmost action on
         * the stack.
         *
         * @param {UndoAction} newAction
         *  The new undo action to be pushed onto the undo stack.
         */
        function pushAction(newAction) {

            var // last action on action stack
                lastAction = null;

            // truncate main undo stack and push the new action
            if (currentAction < actions.length) {

                // remove undone actions, push new action without merging
                actions.splice(currentAction);

            } else {

                // try to merge with last action, if stack has not been truncated
                lastAction = _.last(actions);

                // try to merge an 'insertText' operation for a single character with the last action on stack
                if (_.isObject(lastAction) && (mergeUndoActionHandler.call(self, lastAction, newAction) === true)) {
                    return;
                }
            }

            actions.push(newAction);
            currentAction = actions.length;
            triggerChangeCountEvent();
        }

        /**
         * Handles update push messages sent from the application after a
         * remote undo or redo action has been applied. Updates the internal
         * state according to the passed state of the remote undo manager.
         */
        function remoteUpdateHandler(data) {

            var newUndoCount = Utils.getIntegerOption(data, 'undoCount', remoteUndoCount),
                newRedoCount = Utils.getIntegerOption(data, 'redoCount', remoteRedoCount);

            if ((remoteUndoCount !== newUndoCount) || (remoteRedoCount !== newRedoCount)) {
                remoteUndoCount = newUndoCount;
                remoteRedoCount = newRedoCount;
                triggerChangeCountEvent();
            }
        }

        /**
         * Requests undo or redo actions from the remote undo manager, and
         * applies the associated operations at the document model.
         *
         * @param {String} type
         *  Either 'undo' or 'redo'.
         *
         * @param {Number} [count=1]
         *  The number of undo/redo actions whose operations will be applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the undo/redo
         *  actions have been applied completely.
         */
        function requestAction(type/*, count*/) {

            // sends the undo/redo query to the real-time framework
            function sendRealTimeQuery() {
                // TODO: pass undo/redo count
                return app.sendRealTimeQuery(type);
            }

            // wait for all pending actions, then send the real-time query
            return app.saveChanges().then(sendRealTimeQuery).then(function (data) {
                // returns another Promise that will be resolved after all
                // undo/redo operations have been applied
                return app.applyUpdateMessageData(data);
            });
        }

        /**
         * Apply the operations and return a promise.
         * @param {[]} operations array with operations
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the undo/redo action
         *  has been applied completely.
         */
        function applyOperations(operations) {
            var promise = null,
                undoFunction = _.first(operations);

            // block keyboard events while processing undo operations
            if (_.isFunction(docModel.setBlockKeyboardEvent)) {
                docModel.setBlockKeyboardEvent(true);
            }

            if (_.isFunction(undoFunction)) {
                promise = $.when(undoFunction());
            } else {
                // apply all collected operations
                // bug 30954: always create a deep copy of the operations
                promise = docModel.applyOperations(_.copy(operations, true), { async: true });
            }

            // unblock keyboard events again
            promise.always(function () {
                if (docModel && _.isFunction(docModel.setBlockKeyboardEvent)) {
                    docModel.setBlockKeyboardEvent(false);
                }
            });

            return promise;
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether the undo stack manages its undo actions locally (the
         * constructor option 'remoteUndo' was not set).
         *
         * @returns {Boolean}
         *  Whether the undo stack manages its undo actions locally.
         */
        this.isLocalUndo = function () {
            return !remoteUndo;
        };

        /**
         * Returns whether the undo manager is enabled (importing the document
         * has been finished, and the undo manager is currently not processing
         * any undo or redo operations).
         *
         * @returns {Boolean}
         *  Whether the undo manager is enabled.
         */
        this.isUndoEnabled = function () {
            return !processing && this.isImportFinished();
        };

        /**
         * Clears all undo actions from the stack of the undo manager.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.clearUndoActions = function () {
            if (actions.length > 0) {
                actions = [];
                currentAction = 0;
                triggerChangeCountEvent();
            }
            return this;
        };

        /**
         * Clears all undo actions in the stack that are above the current
         * pointer position. This disables the redo stack.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.disableRedo = function () {

            if (currentAction < actions.length) {
                actions.splice(currentAction);
            }
            return this;
        };

        /**
         * Opens an undo action group and executes the specified callback
         * function. All undo operations added by the callback function will be
         * collected in the action group and act like a single undo action.
         * Nested calls are supported.
         *
         * @param {Function} callback
         *  The callback function that will be executed while the undo action
         *  group is open. If the function returns a promise, the undo group
         *  remains opened asynchronously until this object will be resolved or
         *  rejected by the callback.
         *
         * @param {Object} [context]
         *  The context the callback function will be bound to.
         *
         * @returns {Any}
         *  The return value of the callback function. In asynchronous mode,
         *  this will be the promise returned by the callback function.
         */
        this.enterUndoGroup = function (callback, context) {

            // open a new undo group
            openUndoGroup();

            // invoke the callback function, wait for its result
            var result = callback.call(context);
            this.waitForAny($.when(result), closeUndoGroup);

            return result;
        };

        /**
         * Creates a new undo action that will apply the passed undo and redo
         * operations in the document model.
         *
         * @constructor
         *
         * @param {Object|Object[]|Function} [undoOperations]
         *  One ore more operations that form a logical undo action. When the
         *  undo action is executed, the operations will be applied in the
         *  exact order as passed in this parameter; they will NOT be applied
         *  in reversed order (as would be the case if the operations had been
         *  added in multiple undo actions). The parameter may be a single
         *  operation object, an array of operation objects, or omitted.
         *
         * @param {Object|Object[]|Function} [redoOperations]
         *  One ore more operations that form a logical redo action. When the
         *  redo action is executed, the operations will be applied in the
         *  exact order as passed in this parameter. The parameter may be a
         *  single operation object, an array of operation objects, or omitted.
         *
         * @returns {UndoManager}
         *  A reference to this instance.
         */
        this.addUndo = function (undoOperations, redoOperations) {

            // check that undo manager is valid and either undo or redo operations have been passed
            if (!remoteUndo && this.isUndoEnabled() && (_.isObject(undoOperations) || _.isFunction(undoOperations) || _.isFunction(redoOperations) || _.isObject(redoOperations))) {

                if (groupLevel > 0) {
                    // active group action: insert operations into its operation arrays
                    if (groupAction) {
                        groupAction.appendOperations(undoOperations, redoOperations);
                    } else {
                        groupAction = new UndoAction(undoOperations, redoOperations);
                    }
                } else {
                    // create and insert a new action
                    pushAction(new UndoAction(undoOperations, redoOperations));
                }
            }

            return this;
        };

        /**
         * Returns the number of undo actions available on the undo stack.
         *
         * @returns {Number}
         *  The number of undo actions available on the stack.
         */
        this.getUndoCount = function () {
            return remoteUndo ? remoteUndoCount : currentAction;
        };

        /**
         * Applies the undo next undo action.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the undo action
         *  has been applied completely.
         */
        this.undo = function () {

            var // the resulting Promise
                promise = null,
                // collect all undo operations into a single array
                operations = [];

            this.trigger('undo:before');
            processing = true;

            if (remoteUndo) {
                // request undo from remote undo manager
                promise = requestAction('undo');
            } else {

                if (currentAction > 0) {
                    currentAction -= 1;
                    operations = actions[currentAction].undoOperations;
                    promise = applyOperations(operations);
                }

            }

            // notify listeners after all operations have been applied
            this.waitForAny(promise, function () {
                processing = false;
                self.trigger('undo:after', operations);
                triggerChangeCountEvent();
            });

            return promise;
        };

        /**
         * Returns the number of redo actions available on the redo stack.
         *
         * @returns {Number}
         *  The number of redo actions available on the stack.
         */
        this.getRedoCount = function () {
            return remoteUndo ? remoteRedoCount : (actions.length - currentAction);
        };

        /**
         * Applies the redo operations of the next undo action.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected after the redo action
         *  has been applied completely.
         */
        this.redo = function () {

            var // the resulting Promise
                promise = null,
                // collect all redo operations into a single array
                operations = [];

            this.trigger('redo:before');
            processing = true;

            if (remoteUndo) {
                // request redo from remote undo manager
                promise = requestAction('redo');
            } else {

                if (currentAction < actions.length) {
                    operations = actions[currentAction].redoOperations;

                    promise = applyOperations(operations);

                    currentAction += 1;
                }
            }

            // notify listeners after all operations have been applied
            this.waitForAny(promise, function () {
                processing = false;
                self.trigger('redo:after', operations);
                triggerChangeCountEvent();
            });

            return promise;
        };

        // initialization -----------------------------------------------------

        // initial notification about undo stack size
        this.waitForImportSuccess(triggerChangeCountEvent);

        // remote undo stack: listen to update push messages from application
        if (remoteUndo) {
            this.listenTo(app, 'docs:update', remoteUpdateHandler);
        }

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = initOptions = null;
            app = docModel = mergeUndoActionHandler = actions = null;
        });

    } // class UndoManager

    // exports ================================================================

    return ModelObject.extend({ constructor: UndoManager });

});
