/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/editcontroller', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/app/basecontroller',
    'io.ox/office/editframework/utils/editconfig'
], function (Utils, BaseController, Config) {

    'use strict';

    // class EditController ===================================================

    /**
     * The base class for controller classes of all OX Documents allowing to
     * edit a document.
     *
     * @constructor
     *
     * @extends BaseController
     *
     * @param {EditApplication} app
     *  The application that has created this controller instance.
     *
     * @param {EditModel} docModel
     *  The document model created by the passed application.
     *
     * @param {EditView} docView
     *  The document view created by the passed application.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  BaseController.
     */
    function EditController(app, docModel, docView, initOptions) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        BaseController.call(this, app, docModel, docView, initOptions);

        // initialization -----------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // application and view

            // enabled unless the application is in 'internal error' state
            'app/valid': {
                parent: 'app/imported',
                enable: function () { return !app.isInternalError(); }
            },

            // toggle visibility of the tool bars
            'view/toolbars/show': {
                // always enabled (also in read-only mode, and in application error state)
                get: function () { return docView.getToolPane().isVisible(); },
                set: function (state) { docView.getToolPane().toggle(state); }
            },

            // the identifier of the active tool bar
            'view/toolbars/tab': {
                // always enabled (also in read-only mode, and in application error state)
                get: function () { return docView.getToolBarTabs().getActiveTabId(); },
                set: function (tabId) { docView.getToolBarTabs().activateTab(tabId); },
                focusTarget: function (sourceType) {
                    // when activating via ENTER key, move focus into tool pane
                    return (sourceType === 'keyboard') ? docView.getToolPane() : null;
                }
            },

            // dummy item for the 'Save in Drive' drop-down menu
            'view/saveas/menu': {
                parent: 'app/valid',
                enable: function () { return !app.getLaunchOption('disableSaveAs'); }
            },

            'view/searchgroup': {
                enable: 'app/valid',
                get: function () { return docView.getSearchGroup().isVisible(); },
                set: function (state) { docView.getSearchGroup().toggle(state); },
                focusTarget: function () { return docView.getSearchGroup(); },
                // shortcut always enables the search pane (no toggling)
                shortcut: { keyCode: 'F', ctrlOrMeta: true, value: true }
            },

            // document

            'document/acquireedit': {
                parent: 'app/valid',
                enable: function () { return app.isAcquireEditRightsEnabled(); },
                set: function () { app.acquireEditRights(); }
            },

            'document/reload': {
                parent: 'app/imported',
                enable: function () { return (app.isInternalError() && app.isReloadEnabled()); },
                set: function () { app.reloadDocument(); }
            },

            // to be used as parent item for all items that require edit mode
            'document/editable': {
                parent: 'app/valid',
                enable: function () { return docModel.getEditMode(); }
            },

            'document/fullname': {
                get: function () { return app.getFullFileName(); }
            },

            'document/rename': {
                parent: ['document/editable', 'app/bundled'],
                enable: function () { return !Config.RENAME_DISABLED; },
                get: function () { return app.getShortFileName(); },
                set: function (fileName) { return app.rename(fileName); }
            },

            'document/rename/dialog': {
                parent: 'document/rename',
                set: function () { return docView.showRenameDialog(); }
            },

            'document/undo': {
                parent: 'document/editable',
                enable: function () { return docView.isUndoAvailable(); },
                set: function () { return docView.undo(); },
                shortcut: [
                   // bug 33077: restrict to application pane, to not interfere with text field's
                   // native undo/redo and automatic item execution after losing focus
                    { keyCode: 'Z', ctrlOrMeta: true, selector: '.app-pane' },
                    { keyCode: 'BACKSPACE', alt: true, selector: '.app-pane' }
                ]
            },

            'document/redo': {
                parent: 'document/editable',
                enable: function () { return docView.isRedoAvailable(); },
                set: function () { return docView.redo(); },
                shortcut: [
                    // bug 33077: restrict to application pane, to not interfere with text field's
                    // native undo/redo and automatic item execution after losing focus
                    { keyCode: 'Y', ctrlOrMeta: true, selector: '.app-pane' },
                    { keyCode: 'BACKSPACE', shift: true, alt: true, selector: '.app-pane' }
                ]
            },

            'document/saveas/dialog': {
                parent: 'app/valid',
                set: function (type) { return docView.showSaveAsDialog(type); }
            },

            'document/autosave': {
                parent: 'app/valid',
                get: function () { return app.isAutoSaveEnabled(); },
                set: function (state) { app.toggleAutoSave(state); }
            },

            'document/users': {
                parent: 'app/valid',
                enable: function () { return app.getActiveClients().length > 1; },
                get: function () { return docView.getCollaboratorPopup().isVisible(); },
                set: function (state) { docView.toggleCollaboratorPopup(state); }
            },

            // base item for search/replace, returns the search configuration object
            'document/search': {
                // enabled in read-only mode
                parent: 'app/valid',
                get: function () { return docView.getSearchSettings(); }
            },

            'document/search/text': {
                parent: 'document/search',
                get: function (settings) { return settings.query; },
                set: function (text) { return docView.executeSearchOperation('search:start', { query: text }); }
            },

            'document/search/regexp': {
                parent: 'document/search',
                get: function (settings) { return settings.regexp; },
                set: function (state) { return docView.executeSearchOperation('search:start', { regexp: state }); }
            },

            'document/search/start': {
                parent: 'document/search',
                set: function () { return docView.executeSearchOperation('search:start'); }
            },

            'document/search/prev': {
                parent: 'document/search',
                set: function () { return docView.executeSearchOperation('search:prev'); },
                shortcut: [{ keyCode: 'G', ctrlOrMeta: true, shift: true }, { keyCode: 'F3', shift: true }]
            },

            'document/search/next': {
                parent: 'document/search',
                set: function () { return docView.executeSearchOperation('search:next'); },
                shortcut: [{ keyCode: 'G', ctrlOrMeta: true }, { keyCode: 'F3' }]
            },

            'document/search/close': {
                parent: 'document/search',
                set: function () { return docView.getSearchGroup().hide(); }
            },

            // base item for replace, disabled in read-only documents
            'document/replace': {
                parent: ['document/search', 'document/editable']
            },

            'document/replace/text': {
                parent: 'document/replace',
                get: function (settings) { return settings.replace; },
                set: function (text) { docView.changeSearchSettings({ replace: text }); }
            },

            'document/replace/next': {
                parent: 'document/replace',
                set: function () { return docView.executeSearchOperation('replace:next'); }
            },

            'document/replace/all': {
                parent: 'document/replace',
                set: function () { return docView.executeSearchOperation('replace:all'); }
            },

            // debug mode

            'debug/enabled': {
                parent: 'app/valid',
                enable: function () { return Config.DEBUG; }
            }
        });

        // update controller after operations, or changed state of edit mode
        this.waitForImport(function () {
            this.listenTo(app, 'docs:state:error docs:users', function () { self.update(); });
            this.listenTo(docModel, 'operations:after change:editmode', function () { self.update(); });
            this.listenTo(docModel.getUndoManager(), 'change:count', function () { self.update(); });
        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            app = initOptions = self = docModel = docView = null;
        });

    } // class EditController

    // exports ================================================================

    // derive this class from class BaseController
    return BaseController.extend({ constructor: EditController });

});
