/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/baseframework/model/modelobject', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/app/appobjectmixin'
], function (Utils, TriggerObject, AppObjectMixin) {

    'use strict';

    // class ModelObject ======================================================

    /**
     * An abstract model object that allows to trigger change events to event
     * listeners. As long as the application still imports the document, this
     * instance will not trigger any events (performance optimization).
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends AppObjectMixin
     *
     * @param {BaseModel} docModel
     *  The document model containing this model object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.trigger='runtime']
     *      The behavior of this instance when the trigger() method has been
     *      called to trigger a new event. The following modes are supported:
     *      - 'runtime' (default):
     *          Events are triggered during runtime of the application, but not
     *          when importing the document (performance optimization).
     *      - 'always':
     *          Events are always triggered, also during document import.
     *      - 'never':
     *          Events will never be triggered (the method trigger() of this
     *          instance is empty). Nevertheless, this instance provides the
     *          full event API (the on(), one(), off(), and trigger() methods).
     *          Useful if a few sub classes in a class hierarchy chose to be
     *          permanently silent although the classes derive indirectly from
     *          this class.
     */
    function ModelObject(docModel, initOptions) {

        // base constructors --------------------------------------------------

        TriggerObject.call(this);
        AppObjectMixin.call(this, docModel.getApp());

        // public methods -----------------------------------------------------

        /**
         * Returns the application that owns this instance.
         *
         * @returns {BaseApplication}
         *  The application that owns this instance.
         */
        this.getApp = function () {
            return docModel.getApp();
        };

        /**
         * Returns the document model that owns this instance.
         *
         * @returns {BaseModel}
         *  The document model that owns this instance.
         */
        this.getDocModel = function () {
            return docModel;
        };

        // initialization -----------------------------------------------------

        (function (triggerMode) {

            // set dummy event methods for silent objects
            if (triggerMode === 'never') {
                this.on = this.one = this.off = this.trigger = Utils.NOOP;
                return;
            }

            // disable triggering until import is finished
            if ((triggerMode !== 'always') && !this.isImportFinished()) {

                // overwrite the trigger() method with a dummy, restore original method after import
                this.trigger = (function (trigger) {
                    this.waitForImport(function () {
                        this.trigger = trigger; // back to original method
                    }, this);
                    return Utils.NOOP;
                }.call(this, this.trigger));
            }

        }.call(this, Utils.getStringOption(initOptions, 'trigger', 'runtime')));

    } // class ModelObject

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: ModelObject });

});
