/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range'
], function (Address, Interval, Range) {

    'use strict';

    // class Range ============================================================

    describe('Spreadsheet class Range', function () {

        it('should exist', function () {
            expect(Range).to.be.a('function');
        });
        it('should be extendable', function () {
            expect(Range).itself.to.respondTo('extend');
        });

        // private helpers ----------------------------------------------------

        var ic = Interval.parseAsCols,
            ir = Interval.parseAsRows,
            a = Address.parse,
            r = Range.parse;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            var a1 = a('A2'), a2 = a('C4');
            it('should create a range address', function () {
                var r1 = new Range(a1, a2);
                expect(r1.start).to.equal(a1);
                expect(r1.end).to.equal(a2);
            });
            it('should create a range address from a single address', function () {
                var r1 = new Range(a1);
                expect(r1.start).to.equal(a1);
                expect(r1.end).to.deep.equal(a1);
                expect(r1.end).to.not.equal(a1);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('create');
            });
            it('should return the adjusted range address', function () {
                var r1 = new Range(new Address(0, 1), new Address(2, 3));
                expect(Range.create(0, 1, 2, 3)).to.deep.equal(r1);
                expect(Range.create(2, 1, 0, 3)).to.deep.equal(r1);
                expect(Range.create(0, 3, 2, 1)).to.deep.equal(r1);
                expect(Range.create(2, 3, 0, 1)).to.deep.equal(r1);
            });
        });

        describe('method "createFromIntervals"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromIntervals');
            });
            it('should create range address from intervals', function () {
                expect(Range.createFromIntervals(ic('B:C'), ir('3:4'))).to.deep.equal(r('B3:C4'));
            });
        });

        describe('method "createFromColInterval"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromColInterval');
            });
            it('should create range address from column interval and rows', function () {
                expect(Range.createFromColInterval(ic('B:C'), 2, 3)).to.deep.equal(r('B3:C4'));
                expect(Range.createFromColInterval(ic('B:C'), 2)).to.deep.equal(r('B3:C3'));
            });
        });

        describe('method "createFromRowInterval"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromRowInterval');
            });
            it('should create range address from row interval and columns', function () {
                expect(Range.createFromRowInterval(ir('3:4'), 1, 2)).to.deep.equal(r('B3:C4'));
                expect(Range.createFromRowInterval(ir('3:4'), 1)).to.deep.equal(r('B3:B4'));
            });
        });

        describe('method "createFromAddresses"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('createFromAddresses');
            });
            it('should return the adjusted range address', function () {
                expect(Range.createFromAddresses(a('B3'), a('D5'))).to.deep.equal(Range.create(1, 2, 3, 4));
                expect(Range.createFromAddresses(a('B5'), a('D3'))).to.deep.equal(Range.create(1, 2, 3, 4));
                expect(Range.createFromAddresses(a('D3'), a('B5'))).to.deep.equal(Range.create(1, 2, 3, 4));
                expect(Range.createFromAddresses(a('D5'), a('B3'))).to.deep.equal(Range.create(1, 2, 3, 4));
            });
            it('should return range address for single cell', function () {
                expect(Range.createFromAddresses(a('B3'))).to.deep.equal(Range.create(1, 2, 1, 2));
            });
            it('should clone addresses', function () {
                var a1 = a('A2'), a2 = a('C4'), r1 = Range.createFromAddresses(a1, a2);
                expect(r1.start).to.deep.equal(a1);
                expect(r1.start).to.not.equal(a1);
                expect(r1.end).to.deep.equal(a2);
                expect(r1.end).to.not.equal(a2);
            });
        });

        describe('method "parse"', function () {
            it('should exist', function () {
                expect(Range).itself.to.respondTo('parse');
            });
            var r1 = Range.create(0, 1, 2, 3);
            it('should return the correct range address', function () {
                expect(Range.parse('A2:C4')).to.deep.equal(r1);
                expect(Range.parse('A1:ZZZZ99999999')).to.deep.equal(Range.create(0, 0, 475253, 99999998));
            });
            it('should return the adjusted range address', function () {
                expect(Range.parse('C2:A4')).to.deep.equal(r1);
                expect(Range.parse('A4:C2')).to.deep.equal(r1);
                expect(Range.parse('C4:A2')).to.deep.equal(r1);
            });
            it('should accept lower-case columns and leading zero characters', function () {
                expect(Range.parse('a01:aa010')).to.deep.equal(Range.create(0, 0, 26, 9));
                expect(Range.parse('A001:aA0010')).to.deep.equal(Range.create(0, 0, 26, 9));
            });
            it('should return null for invalid range names', function () {
                expect(Range.parse('')).to.equal(null);
                expect(Range.parse('A')).to.equal(null);
                expect(Range.parse('1')).to.equal(null);
                expect(Range.parse('A1')).to.equal(null);
                expect(Range.parse('$')).to.equal(null);
                expect(Range.parse('$A$1:$A$1')).to.equal(null);
                expect(Range.parse('A0:A1')).to.equal(null);
                expect(Range.parse('AAAAA1:AAAAA1')).to.equal(null);
                expect(Range.parse('A100000000:A100000000')).to.equal(null);
            });
        });

        describe('method "compare"', function () {
            var r1 = r('C3:E5');
            it('should exist', function () {
                expect(Range).itself.to.respondTo('compare');
            });
            it('should return 0 for equal ranges', function () {
                expect(Range.compare(r1, r1)).to.equal(0);
            });
            it('should return negative value for ranges in order', function () {
                expect(Range.compare(r1, r('B4:D4'))).to.be.below(0);
                expect(Range.compare(r1, r('D3:D4'))).to.be.below(0);
                expect(Range.compare(r1, r('C3:D6'))).to.be.below(0);
                expect(Range.compare(r1, r('C3:F5'))).to.be.below(0);
            });
            it('should return positive value for ranges in reversed order', function () {
                expect(Range.compare(r1, r('D2:F6'))).to.be.above(0);
                expect(Range.compare(r1, r('B3:F6'))).to.be.above(0);
                expect(Range.compare(r1, r('C3:F4'))).to.be.above(0);
                expect(Range.compare(r1, r('C3:D5'))).to.be.above(0);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var r1 = r('A2:C4'), r2 = r1.clone();
                expect(r2).to.be.an['instanceof'](Range);
                expect(r2).to.not.equal(r1);
                expect(r2.start).to.not.equal(r1.start);
                expect(r2.end).to.not.equal(r1.end);
                expect(r2).to.deep.equal(r1);
            });
        });

        describe('method "equals"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('equals');
            });
            var range = r('A2:C4');
            it('should return true for equal ranges', function () {
                expect(r('A2:A4').equals(r('A2:A4'))).to.equal(true);
                expect(r('A2:C2').equals(r('A2:C2'))).to.equal(true);
                expect(r('A2:C4').equals(r('A2:C4'))).to.equal(true);
                expect(range.equals(range)).to.equal(true);
            });
            it('should return false for different ranges', function () {
                expect(range.equals(r('A2:C5'))).to.equal(false);
                expect(range.equals(r('A2:D4'))).to.equal(false);
                expect(range.equals(r('A3:C4'))).to.equal(false);
                expect(range.equals(r('B2:C4'))).to.equal(false);
            });
        });

        describe('method "differs"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('differs');
            });
            var range = r('A2:C4');
            it('should return false for equal ranges', function () {
                expect(r('A2:A4').differs(r('A2:A4'))).to.equal(false);
                expect(r('A2:C2').differs(r('A2:C2'))).to.equal(false);
                expect(r('A2:C4').differs(r('A2:C4'))).to.equal(false);
                expect(range.differs(range)).to.equal(false);
            });
            it('should return true for different ranges', function () {
                expect(range.differs(r('A2:C5'))).to.equal(true);
                expect(range.differs(r('A2:D4'))).to.equal(true);
                expect(range.differs(r('A3:C4'))).to.equal(true);
                expect(range.differs(r('B2:C4'))).to.equal(true);
            });
        });

        describe('method "compareTo"', function () {
            var r1 = r('C3:E5');
            it('should exist', function () {
                expect(Range).to.respondTo('compareTo');
            });
            it('should return 0 for equal ranges', function () {
                expect(r1.compareTo(r1)).to.equal(0);
            });
            it('should return negative value for ranges in order', function () {
                expect(r1.compareTo(r('B4:D4'))).to.be.below(0);
                expect(r1.compareTo(r('D3:D4'))).to.be.below(0);
                expect(r1.compareTo(r('C3:D6'))).to.be.below(0);
                expect(r1.compareTo(r('C3:F5'))).to.be.below(0);
            });
            it('should return positive value for ranges in reversed order', function () {
                expect(r1.compareTo(r('D2:F6'))).to.be.above(0);
                expect(r1.compareTo(r('B3:F6'))).to.be.above(0);
                expect(r1.compareTo(r('C3:F4'))).to.be.above(0);
                expect(r1.compareTo(r('C3:D5'))).to.be.above(0);
            });
        });

        describe('method "cols"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('cols');
            });
            it('should return the correct count', function () {
                expect(r('A2:A3').cols()).to.equal(1);
                expect(r('C2:C3').cols()).to.equal(1);
                expect(r('C2:D3').cols()).to.equal(2);
                expect(r('C2:E3').cols()).to.equal(3);
            });
        });

        describe('method "rows"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('rows');
            });
            it('should return the correct count', function () {
                expect(r('B1:C1').rows()).to.equal(1);
                expect(r('B3:C3').rows()).to.equal(1);
                expect(r('B3:C4').rows()).to.equal(2);
                expect(r('B3:C5').rows()).to.equal(3);
            });
        });

        describe('method "size"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('size');
            });
            it('should return the correct coumn count', function () {
                expect(r('B1:C1').size(true)).to.equal(2);
            });
            it('should return the correct row count', function () {
                expect(r('B1:C1').size(false)).to.equal(1);
            });
        });

        describe('method "cells"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('cells');
            });
            it('should return the correct count', function () {
                expect(r('A1:A1').cells()).to.equal(1);
                expect(r('C3:C3').cells()).to.equal(1);
                expect(r('C3:D3').cells()).to.equal(2);
                expect(r('C3:E3').cells()).to.equal(3);
                expect(r('C3:C4').cells()).to.equal(2);
                expect(r('C3:C5').cells()).to.equal(3);
                expect(r('C3:D4').cells()).to.equal(4);
                expect(r('C3:E5').cells()).to.equal(9);
            });
        });

        describe('method "single"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('single');
            });
            it('should return the correct result', function () {
                expect(r('A1:A1').single()).to.equal(true);
                expect(r('B3:B3').single()).to.equal(true);
                expect(r('B3:B4').single()).to.equal(false);
                expect(r('B3:C3').single()).to.equal(false);
                expect(r('B2:B3').single()).to.equal(false);
                expect(r('A3:B3').single()).to.equal(false);
            });
        });

        describe('method "startsAt"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('startsAt');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.startsAt(a('B3'))).to.equal(true);
                expect(r1.startsAt(a('B2'))).to.equal(false);
                expect(r1.startsAt(a('B4'))).to.equal(false);
                expect(r1.startsAt(a('A3'))).to.equal(false);
                expect(r1.startsAt(a('D3'))).to.equal(false);
                expect(r1.startsAt(a('D5'))).to.equal(false);
            });
        });

        describe('method "endsAt"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('endsAt');
            });
            it('should return the correct result', function () {
                var r1 = r('B3:D5');
                expect(r1.endsAt(a('D5'))).to.equal(true);
                expect(r1.endsAt(a('D4'))).to.equal(false);
                expect(r1.endsAt(a('D6'))).to.equal(false);
                expect(r1.endsAt(a('C5'))).to.equal(false);
                expect(r1.endsAt(a('E5'))).to.equal(false);
                expect(r1.endsAt(a('B3'))).to.equal(false);
            });
        });

        describe('method "containsCol"', function () {
            var range = r('C2:D9');
            it('should exist', function () {
                expect(Range).to.respondTo('containsCol');
            });
            it('should return false for columns left of the range', function () {
                expect(range.containsCol(0)).to.equal(false);
                expect(range.containsCol(1)).to.equal(false);
            });
            it('should return true for columns inside the range', function () {
                expect(range.containsCol(2)).to.equal(true);
                expect(range.containsCol(3)).to.equal(true);
            });
            it('should return false for columns right of the range', function () {
                expect(range.containsCol(4)).to.equal(false);
                expect(range.containsCol(9)).to.equal(false);
            });
        });

        describe('method "containsRow"', function () {
            var range = r('B3:I4');
            it('should exist', function () {
                expect(Range).to.respondTo('containsRow');
            });
            it('should return false for rows above the range', function () {
                expect(range.containsRow(0)).to.equal(false);
                expect(range.containsRow(1)).to.equal(false);
            });
            it('should return true for rows inside the range', function () {
                expect(range.containsRow(2)).to.equal(true);
                expect(range.containsRow(3)).to.equal(true);
            });
            it('should return false for rows below the range', function () {
                expect(range.containsRow(4)).to.equal(false);
                expect(range.containsRow(9)).to.equal(false);
            });
        });

        describe('method "containsIndex"', function () {
            var range = r('C2:C2');
            it('should exist', function () {
                expect(Range).to.respondTo('containsIndex');
            });
            it('should check column indexes', function () {
                expect(range.containsIndex(1, true)).to.equal(false);
                expect(range.containsIndex(2, true)).to.equal(true);
                expect(range.containsIndex(3, true)).to.equal(false);
            });
            it('should check row indexes', function () {
                expect(range.containsIndex(0, false)).to.equal(false);
                expect(range.containsIndex(1, false)).to.equal(true);
                expect(range.containsIndex(2, false)).to.equal(false);
            });
        });

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('containsAddress');
            });
            var range = r('B4:C5');
            it('should return false for cells outside the range', function () {
                'A3 B3 C3 D3 A4 D4 A5 D5 A6 B6 C6 D6'.split(' ').forEach(function (str) {
                    expect(range.containsAddress(a(str))).to.equal(false);
                });
            });
            it('should return true for cells inside the range', function () {
                'B4 C4 B5 C5'.split(' ').forEach(function (str) {
                    expect(range.containsAddress(a(str))).to.equal(true);
                });
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('contains');
            });
            var range = r('C3:F6');
            it('should return false for ranges outside the range', function () {
                'A1:B8 G1:H8 A1:H2 A7:H8'.split(' ').forEach(function (str) {
                    expect(range.contains(r(str))).to.equal(false);
                });
            });
            it('should return false for ranges partly overlapping the range', function () {
                'A1:C8 F1:H8 A1:H3 A6:H8 D1:E3 D6:E8 A4:C5 F5:H6'.split(' ').forEach(function (str) {
                    expect(range.contains(r(str))).to.equal(false);
                });
            });
            it('should return true for ranges inside the range', function () {
                'C3:C3 C3:F3 F3:F3 F3:F6 F6:F6 C6:F6 C6:C6 C3:C6 C3:F6 D4:E5'.split(' ').forEach(function (str) {
                    expect(range.contains(r(str))).to.equal(true);
                });
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('overlaps');
            });
            var range = r('B3:C4');
            it('should return false for distinct ranges', function () {
                expect(range.overlaps(r('D3:E4'))).to.equal(false);
                expect(r('D3:E4').overlaps(range)).to.equal(false);
                expect(range.overlaps(r('B5:C6'))).to.equal(false);
                expect(r('B5:C6').overlaps(range)).to.equal(false);
            });
            it('should return true for partly overlapping ranges', function () {
                expect(range.overlaps(r('C4:D5'))).to.equal(true);
                expect(range.overlaps(r('C2:D3'))).to.equal(true);
                expect(range.overlaps(r('A2:B3'))).to.equal(true);
                expect(range.overlaps(r('A4:B5'))).to.equal(true);
            });
            it('should return true for ranges containing each other', function () {
                expect(range.overlaps(r('B3:B3'))).to.equal(true);
                expect(range.overlaps(r('B3:B4'))).to.equal(true);
                expect(range.overlaps(r('B3:C4'))).to.equal(true);
                expect(r('B3:B4').overlaps(range)).to.equal(true);
                expect(r('B3:B3').overlaps(range)).to.equal(true);
                expect(range.overlaps(range)).to.equal(true);
            });
        });

        describe('method "colInterval"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('colInterval');
            });
            it('should return the column interval', function () {
                expect(r('B3:B3').colInterval()).to.deep.equal(ic('B:B'));
                expect(r('B3:C4').colInterval()).to.deep.equal(ic('B:C'));
                expect(r('B3:D5').colInterval()).to.deep.equal(ic('B:D'));
                expect(r('C4:D5').colInterval()).to.deep.equal(ic('C:D'));
                expect(r('D5:D5').colInterval()).to.deep.equal(ic('D:D'));
            });
        });

        describe('method "rowInterval"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('rowInterval');
            });
            it('should return the row interval', function () {
                expect(r('B3:B3').rowInterval()).to.deep.equal(ir('3:3'));
                expect(r('B3:C4').rowInterval()).to.deep.equal(ir('3:4'));
                expect(r('B3:D5').rowInterval()).to.deep.equal(ir('3:5'));
                expect(r('C4:D5').rowInterval()).to.deep.equal(ir('4:5'));
                expect(r('D5:D5').rowInterval()).to.deep.equal(ir('5:5'));
            });
        });

        describe('method "interval"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('interval');
            });
            it('should return the column interval', function () {
                expect(r('B3:C4').interval(true)).to.deep.equal(ic('B:C'));
                expect(r('B3:D5').interval(true)).to.deep.equal(ic('B:D'));
                expect(r('C4:D5').interval(true)).to.deep.equal(ic('C:D'));
            });
            it('should return the row interval', function () {
                expect(r('B3:C4').interval(false)).to.deep.equal(ir('3:4'));
                expect(r('B3:D5').interval(false)).to.deep.equal(ir('3:5'));
                expect(r('C4:D5').interval(false)).to.deep.equal(ir('4:5'));
            });
        });

        describe('method "header"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('header');
            });
            it('should return the header range', function () {
                expect(r('B3:D5').header()).to.deep.equal(r('B3:D3'));
            });
        });

        describe('method "footer"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('footer');
            });
            it('should return the footer range', function () {
                expect(r('B3:D5').footer()).to.deep.equal(r('B5:D5'));
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('boundary');
            });
            it('should return the bounding interval', function () {
                expect(r('B3:C4').boundary(r('C4:D5'))).to.deep.equal(r('B3:D5'));
                expect(r('C3:D4').boundary(r('B4:C5'))).to.deep.equal(r('B3:D5'));
                expect(r('B4:C5').boundary(r('C3:D4'))).to.deep.equal(r('B3:D5'));
                expect(r('C4:D5').boundary(r('B3:C4'))).to.deep.equal(r('B3:D5'));
                expect(r('C4:C4').boundary(r('B3:D5'))).to.deep.equal(r('B3:D5'));
                expect(r('B3:D5').boundary(r('C4:C4'))).to.deep.equal(r('B3:D5'));
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('intersect');
            });
            it('should return null for distinct ranges', function () {
                expect(r('B3:C4').intersect(r('D3:E4'))).to.equal(null);
                expect(r('D3:E4').intersect(r('B3:C4'))).to.equal(null);
                expect(r('B3:C4').intersect(r('B5:C6'))).to.equal(null);
                expect(r('B5:C6').intersect(r('B3:C4'))).to.equal(null);
            });
            it('should return intersection range for overlapping ranges', function () {
                var range = r('B3:E5');
                expect(range.intersect(r('A2:B3'))).to.deep.equal(r('B3:B3'));
                expect(range.intersect(r('A2:B6'))).to.deep.equal(r('B3:B5'));
                expect(range.intersect(r('A4:C6'))).to.deep.equal(r('B4:C5'));
                expect(range.intersect(r('A4:F4'))).to.deep.equal(r('B4:E4'));
                expect(range.intersect(r('C1:D8'))).to.deep.equal(r('C3:D5'));
                expect(range.intersect(r('A1:F8'))).to.deep.equal(r('B3:E5'));
                expect(range.intersect(r('B3:E5'))).to.deep.equal(r('B3:E5'));
                expect(range.intersect(r('C3:D5'))).to.deep.equal(r('C3:D5'));
                expect(range.intersect(r('C4:C4'))).to.deep.equal(r('C4:C4'));
                expect(range.intersect(range)).to.deep.equal(range);
                expect(range.intersect(range)).to.not.equal(range);
            });
        });

        describe('method "getStart"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('getStart');
            });
            it('should return the correct index', function () {
                var r1 = r('B3:D5');
                expect(r1.getStart(true)).to.equal(1);
                expect(r1.getStart(false)).to.equal(2);
            });
        });

        describe('method "getEnd"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('getEnd');
            });
            it('should return the correct index', function () {
                var r1 = r('B3:D5');
                expect(r1.getEnd(true)).to.equal(3);
                expect(r1.getEnd(false)).to.equal(4);
            });
        });

        describe('method "setStart"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('setStart');
            });
            it('should change the correct index', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.setStart(1, true);
                expect(r1).to.deep.equal(r('B1:F6'));
                r1.setStart(2, false);
                expect(r1).to.deep.equal(r('B3:F6'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.setStart(1, true)).to.equal(r1);
            });
        });

        describe('method "setEnd"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('setEnd');
            });
            it('should change the correct index', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.setEnd(3, true);
                expect(r1).to.deep.equal(r('A1:D6'));
                r1.setEnd(4, false);
                expect(r1).to.deep.equal(r('A1:D5'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.setEnd(1, true)).to.equal(r1);
            });
        });

        describe('method "setBoth"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('setBoth');
            });
            it('should change the correct index', function () {
                var r1 = r('A1:F6');
                expect(r1).to.deep.equal(r('A1:F6'));
                r1.setBoth(3, true);
                expect(r1).to.deep.equal(r('D1:D6'));
                r1.setBoth(4, false);
                expect(r1).to.deep.equal(r('D5:D5'));
            });
            it('should return itself', function () {
                var r1 = r('A1:F6');
                expect(r1.setBoth(1, true)).to.equal(r1);
            });
        });

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('key');
            });
            it('should return unique key for range address', function () {
                expect(r('A2:A2').key()).to.equal('0,1:0,1');
                expect(r('B3:D5').key()).to.equal('1,2:3,4');
                expect(r('A1:ZZ1000').key()).to.equal('0,0:701,999');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('toString');
            });
            it('should stringify the range', function () {
                expect(r('A2:C4').toString()).to.equal('A2:C4');
                expect(r('A1:ZZZZ99999999').toString()).to.equal('A1:ZZZZ99999999');
            });
            it('should stringify implicitly', function () {
                expect('<' + r('A2:C4') + '>').to.equal('<A2:C4>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Range).to.respondTo('toJSON');
            });
            var r1 = r('A3:B4'), r2 = { start: [0, 2], end: [1, 3] };
            it('should convert to JSON data', function () {
                expect(r1.toJSON()).to.deep.equal(r2);
            });
            it('should stringify implicitly', function () {
                expect(JSON.parse(JSON.stringify(r1))).to.deep.equal(r2);
            });
        });
    });

    // ========================================================================
});
