/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/matrix'
], function (ErrorCode, Matrix) {

    'use strict';

    // class Matrix ===========================================================

    describe('Spreadsheet class Matrix', function () {

        it('should exist', function () {
            expect(Matrix).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var a1 = [[1]],
            a2 = [[1, 2], ['a', 'b'], [false, true]];

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a matrix', function () {
                var m1 = new Matrix(a1);
                expect(m1).to.have.property('array', a1);
                var m2 = new Matrix(a2);
                expect(m2).to.have.property('array', a2);
            });
        });

        function ts(arg) {
            return arg.toString();
        }

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('create');
            });
            it('should create a filled matrix', function () {
                var m1 = Matrix.create(1, 1, 1);
                expect(m1).to.be.an['instanceof'](Matrix);
                expect(m1.array).to.deep.equal([[1]]);
                var m2 = Matrix.create(3, 2, 'a');
                expect(m2).to.be.an['instanceof'](Matrix);
                expect(m2.array).to.deep.equal([['a', 'a'], ['a', 'a'], ['a', 'a']]);
            });
        });

        describe('method "generate"', function () {
            it('should exist', function () {
                expect(Matrix).itself.to.respondTo('generate');
            });
            it('should create a filled matrix', function () {
                var m1 = Matrix.generate(1, 1, _.constant(1));
                expect(m1).to.be.an['instanceof'](Matrix);
                expect(m1.array).to.deep.equal([[1]]);
                function generator(row, col) { return a2[row][col]; }
                var spy = sinon.spy(generator), m2 = Matrix.generate(3, 2, spy, a1);
                expect(m2).to.be.an['instanceof'](Matrix);
                expect(m2.array).to.deep.equal(a2);
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a1);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "rows"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('rows');
            });
            it('should return the number of rows', function () {
                expect(new Matrix(a1).rows()).to.equal(1);
                expect(new Matrix(a2).rows()).to.equal(3);
            });
        });

        describe('method "cols"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('cols');
            });
            it('should return the number of rows', function () {
                expect(new Matrix(a1).cols()).to.equal(1);
                expect(new Matrix(a2).cols()).to.equal(2);
            });
        });

        describe('method "size"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('size');
            });
            it('should return the number of rows', function () {
                expect(new Matrix(a1).size()).to.equal(1);
                expect(new Matrix(a2).size()).to.equal(6);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('get');
            });
            var m1 = new Matrix(a1), m2 = new Matrix(a2), m3 = new Matrix([[2, 3]]), m4 = new Matrix([[2], [3]]);
            it('should return the specified element', function () {
                expect(m1.get(0, 0)).to.equal(a1[0][0]);
                expect(m2.get(0, 0)).to.equal(a2[0][0]);
                expect(m2.get(0, 1)).to.equal(a2[0][1]);
                expect(m2.get(1, 0)).to.equal(a2[1][0]);
                expect(m2.get(1, 1)).to.equal(a2[1][1]);
                expect(m2.get(2, 0)).to.equal(a2[2][0]);
                expect(m2.get(2, 1)).to.equal(a2[2][1]);
            });
            it('should ignore row index for single-row vector', function () {
                expect(m3.get(0, 0)).to.equal(2);
                expect(m3.get(0, 1)).to.equal(3);
                expect(m3.get(1, 0)).to.equal(2);
                expect(m3.get(1, 1)).to.equal(3);
                expect(m3.get(9, 0)).to.equal(2);
                expect(m3.get(9, 1)).to.equal(3);
            });
            it('should ignore column index for single-column vector', function () {
                expect(m4.get(0, 0)).to.equal(2);
                expect(m4.get(0, 1)).to.equal(2);
                expect(m4.get(0, 9)).to.equal(2);
                expect(m4.get(1, 0)).to.equal(3);
                expect(m4.get(1, 1)).to.equal(3);
                expect(m4.get(1, 9)).to.equal(3);
            });
            it('should throw #N/A on invalid indexes', function () {
                expect(m2.get.bind(m2, 2, 2)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(m2.get.bind(m2, 3, 1)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(m2.get.bind(m2, 3, 2)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(m2.get.bind(m3, 0, 2)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(m2.get.bind(m3, 9, 2)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(m2.get.bind(m4, 2, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
                expect(m2.get.bind(m4, 2, 9)).to['throw'](ErrorCode).that.equals(ErrorCode.NA);
            });
        });

        describe('method "set"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('set');
            });
            var m1 = Matrix.create(3, 2, 1);
            it('should change the specified element', function () {
                m1.set(0, 0, 'a');
                expect(m1.array[0][0]).to.equal('a');
                m1.set(2, 1, true);
                expect(m1.array[2][1]).to.equal(true);
            });
            it('should return itself', function () {
                expect(m1.set(0, 0, 1)).to.equal(m1);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('forEach');
            });
            it('should visit all matrix elements', function () {
                var m = new Matrix(a2), spy = sinon.spy();
                expect(m.forEach(spy, a1)).to.equal(m);
                sinon.assert.callCount(spy, 6);
                sinon.assert.alwaysCalledOn(spy, a1);
                sinon.assert.calledWithExactly(spy.getCall(0), a2[0][0], 0, 0);
                sinon.assert.calledWithExactly(spy.getCall(1), a2[0][1], 0, 1);
                sinon.assert.calledWithExactly(spy.getCall(2), a2[1][0], 1, 0);
                sinon.assert.calledWithExactly(spy.getCall(3), a2[1][1], 1, 1);
                sinon.assert.calledWithExactly(spy.getCall(4), a2[2][0], 2, 0);
                sinon.assert.calledWithExactly(spy.getCall(5), a2[2][1], 2, 1);
            });
        });

        describe('method "transform"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('transform');
            });
            it('should modify all matrix elements', function () {
                var m = new Matrix([[1, 2], [3, 4]]), spy = sinon.spy(function (e) { return e * e; });
                expect(m.transform(spy, a1)).to.equal(m);
                expect(m.array).to.deep.equal([[1, 4], [9, 16]]);
                sinon.assert.callCount(spy, 4);
                sinon.assert.alwaysCalledOn(spy, a1);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, 0, 0);
                sinon.assert.calledWithExactly(spy.getCall(1), 2, 0, 1);
                sinon.assert.calledWithExactly(spy.getCall(2), 3, 1, 0);
                sinon.assert.calledWithExactly(spy.getCall(3), 4, 1, 1);
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('toString');
            });
            it('should return the string representation', function () {
                var m = new Matrix(a2);
                expect(m.toString()).to.equal('{1;2|"a";"b"|FALSE;TRUE}');
                expect('<' + m + '>').to.equal('<{1;2|"a";"b"|FALSE;TRUE}>');
            });
        });

        describe('methods "getRow" & "getCol"', function () {
            it('should exist', function () {
                expect(Matrix).to.respondTo('getRow');
                expect(Matrix).to.respondTo('getCol');
            });
            var m = new Matrix([[1, 2, 3], [4, 5, 6], [7, 8, 9]]);
            it('should return single-row vector', function () {
                expect(ts(m.getRow(0))).to.equal(ts([1, 2, 3]));
                expect(ts(m.getRow(1))).to.equal(ts([4, 5, 6]));
                expect(ts(m.getRow(2))).to.equal(ts([7, 8, 9]));
            });
            it('should return single-column vector', function () {
                expect(ts(m.getCol(0))).to.equal(ts([1, 4, 7]));
                expect(ts(m.getCol(1))).to.equal(ts([2, 5, 8]));
                expect(ts(m.getCol(2))).to.equal(ts([3, 6, 9]));
            });
        });
    });

    // ========================================================================
});
