/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulacontext',
    'io.ox/office/spreadsheet/model/formula/impl/datetimefuncs'
], function (SheetUtils, FormulaContext, DateTimeFuncs) {

    'use strict';

    // initialize the formula context
    var context = null;
    before(function (done) {
        ox.test.spreadsheet.createApp('ooxml').done(function (app) {
            context = new FormulaContext(app.getModel());
            done();
        });
    });

    // module DateTimeFuncs ===================================================

    describe('Spreadsheet module DateTimeFuncs', function () {

        it('should exist', function () {
            expect(DateTimeFuncs).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var ErrorCode = SheetUtils.ErrorCode;

        // converts a date string 'YYYY-MM-DD' to an instance of the Date class
        function d(date) {
            var matches = /^(\d{4})-(\d{2})-(\d{2})$/.exec(date);
            return new Date(Date.UTC(parseInt(matches[1], 10), parseInt(matches[2], 10) - 1, parseInt(matches[3], 10)));
        }

        function ts(date) {
            return date.toString();
        }

        // function implementations -------------------------------------------

        // TODO: more test cases to cover all execution paths
        describe('function "YEARFRAC"', function () {
            var YEARFRAC = DateTimeFuncs.YEARFRAC;
            it('should exist', function () {
                expect(YEARFRAC).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(YEARFRAC).itself.to.respondTo('resolve');
            });
            it('should return correct results for date mode 0 (30/360 US)', function () {
                expect(YEARFRAC.resolve.call(context, d('2011-01-01'), d('2011-01-01'), 0)).to.equal(0);
                expect(YEARFRAC.resolve.call(context, d('2011-02-28'), d('2012-02-29'), 0)).to.equal(1);
            });
            it('should return correct results for date mode 1 (actual/actual)', function () {
                expect(YEARFRAC.resolve.call(context, d('2011-01-01'), d('2011-01-01'), 1)).to.equal(0);
                expect(YEARFRAC.resolve.call(context, d('2011-02-28'), d('2012-02-29'), 1)).to.equal(732 / 731);
            });
            it('should return correct results for date mode 2 (actual/360)', function () {
                expect(YEARFRAC.resolve.call(context, d('2011-01-01'), d('2011-01-01'), 2)).to.equal(0);
                expect(YEARFRAC.resolve.call(context, d('2011-02-28'), d('2012-02-29'), 2)).to.equal(61 / 60);
            });
            it('should return correct results for date mode 3 (actual/365)', function () {
                expect(YEARFRAC.resolve.call(context, d('2011-01-01'), d('2011-01-01'), 3)).to.equal(0);
                expect(YEARFRAC.resolve.call(context, d('2011-02-28'), d('2012-02-29'), 3)).to.equal(366 / 365);
            });
            it('should return correct results for date mode 4 (30E/360)', function () {
                expect(YEARFRAC.resolve.call(context, d('2011-01-01'), d('2011-01-01'), 4)).to.equal(0);
                expect(YEARFRAC.resolve.call(context, d('2011-02-28'), d('2012-02-29'), 4)).to.equal(361 / 360);
            });
            it('should default to date mode 0 (30/360 US)', function () {
                expect(YEARFRAC.resolve.call(context, d('2011-01-01'), d('2011-01-01'))).to.equal(0);
                expect(YEARFRAC.resolve.call(context, d('2011-02-28'), d('2012-02-29'))).to.equal(1);
            });
        });

        describe('function "MONTHS"', function () {
            var MONTHS = DateTimeFuncs.MONTHS;
            it('should exist', function () {
                expect(MONTHS).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(MONTHS).itself.to.respondTo('resolve');
            });

            it('should return correct results months between two assigned dates', function () {
                expect(MONTHS.resolve.call(context, d('2010-04-03'), d('2011-06-17'), 0)).to.equal(14);
                expect(MONTHS.resolve.call(context, d('2010-03-31'), d('2010-04-30'), 0)).to.equal(0);
                expect(MONTHS.resolve.call(context, d('2010-03-31'), d('2010-06-30'), 0)).to.equal(2);
                expect(MONTHS.resolve.call(context, d('2010-04-03'), d('2011-06-17'), 1)).to.equal(14);
                expect(MONTHS.resolve.call(context, d('2010-03-31'), d('2010-04-01'), 1)).to.equal(1);
            });
        });

        describe('function "NETWORKDAYS"', function () {
            var NETWORKDAYS = DateTimeFuncs.NETWORKDAYS;
            it('should exist', function () {
                expect(NETWORKDAYS).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(NETWORKDAYS).itself.to.respondTo('resolve');
            });

            it('should return correct results netto workdays between two assigned dates', function () {
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-05'), d('2015-10-06'))).to.equal(2);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-06'), d('2015-10-05'))).to.equal(-2);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-05'), d('2015-10-09'))).to.equal(5);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-05'), d('2015-10-12'))).to.equal(6);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-02'), d('2015-10-12'))).to.equal(7);

                expect(NETWORKDAYS.resolve.call(context, d('1983-06-23'), d('2015-10-02'))).to.equal(8422);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-02'), d('2015-10-05'))).to.equal(2);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-05'), d('1983-06-23'))).to.equal(-8423);
                expect(NETWORKDAYS.resolve.call(context, d('2015-10-05'), d('2915-12-07'))).to.equal(234845);
            });

        });

        describe('function "NETWORKDAYS.INTL"', function () {
            var NETWORKDAYS_INTL = DateTimeFuncs['NETWORKDAYS.INTL'];
            it('should exist', function () {
                expect(NETWORKDAYS_INTL).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(NETWORKDAYS_INTL).itself.to.respondTo('resolve');
            });

            it('should return correct results netto workdays between two assigned dates', function () {
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'))).to.equal(5);

            });
            it('should return correct results netto workdays between two assigned dates and optional weekenddays', function () {
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 1)).to.equal(5);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 2)).to.equal(4);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 3)).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 4)).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 5)).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 5)).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 7)).to.equal(4);

                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), true)).to.equal(5); //true is interpreted as 1
                expect(NETWORKDAYS_INTL.resolve.bind(context, d('2015-10-05'), d('2015-10-09'), 0)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(NETWORKDAYS_INTL.resolve.bind(context, d('2015-10-05'), d('2015-10-09'), 8)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(NETWORKDAYS_INTL.resolve.bind(context, d('2015-10-05'), d('2015-10-09'), '000')).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(NETWORKDAYS_INTL.resolve.bind(context, d('2015-10-05'), d('2015-10-09'), false)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(NETWORKDAYS_INTL.resolve.bind(context, d('2015-10-05'), d('2015-10-09'), 'hello')).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });

            it('should return correct results netto workdays between two assigned dates and optional weekenddays & holidays', function () {
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 1, d('2015-10-07'))).to.equal(4);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 2, d('2015-10-07'))).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 3, d('2015-10-07'))).to.equal(2);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 4, d('2015-10-07'))).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 5, d('2015-10-07'))).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 6, d('2015-10-07'))).to.equal(2);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-09'), 7, d('2015-10-07'))).to.equal(3);

                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-11'), '0000000', [d('2015-10-05'), d('2015-10-06')])).to.equal(5);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-11'), '0000000', [d('2015-10-09'), d('2015-10-10')])).to.equal(5);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-11'), '0000000', [d('2015-10-09'), d('2015-10-10'), d('2015-10-11')])).to.equal(4);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2015-10-11'), '0000000', [d('2015-10-11'), d('2015-10-09')])).to.equal(5);

                expect(NETWORKDAYS_INTL.resolve.call(context, d('1983-06-23'), d('2015-10-02'), 4, d('2015-10-02'))).to.equal(8421);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-02'), d('2015-10-05'), 4, d('2015-10-02'))).to.equal(3);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('1983-06-23'), 4, d('2015-10-02'))).to.equal(-8424);
                expect(NETWORKDAYS_INTL.resolve.call(context, d('2015-10-05'), d('2915-12-07'), 4, d('2015-10-02'))).to.equal(234844);
            });
        });

        describe('function "WEEKS"', function () {
            var WEEKS = DateTimeFuncs.WEEKS;
            it('should exist', function () {
                expect(WEEKS).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(WEEKS).itself.to.respondTo('resolve');
            });

            it('should return correct results weeks between two assigned dates', function () {
                expect(WEEKS.resolve.call(context, d('2010-05-03'), d('2010-05-17'), 0)).to.equal(2);
                expect(WEEKS.resolve.call(context, d('2010-05-04'), d('2010-05-27'), 0)).to.equal(3);
                expect(WEEKS.resolve.call(context, d('2010-05-06'), d('2010-05-17'), 1)).to.equal(2);
                expect(WEEKS.resolve.call(context, d('2010-05-09'), d('2010-05-10'), 1)).to.equal(1);
                expect(WEEKS.resolve.call(context, d('2010-05-03'), d('2010-05-09'), 1)).to.equal(0);
            });
        });

        describe('function "WORKDAY"', function () {
            var WORKDAY = DateTimeFuncs.WORKDAY;
            it('should exist', function () {
                expect(WORKDAY).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(WORKDAY).itself.to.respondTo('resolve');
            });

            it('should return correct results netto workdays between two assigned dates', function () {
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-05'), 2))).to.equal(ts(d('2015-10-07')));
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-06'), -2))).to.equal(ts(d('2015-10-02')));
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-05'), 5))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-05'), 6))).to.equal(ts(d('2015-10-13')));
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-02'), 7))).to.equal(ts(d('2015-10-13')));

                expect(ts(WORKDAY.resolve.call(context, d('1983-06-23'), 8422))).to.equal(ts(d('2015-10-05')));
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-02'), 2))).to.equal(ts(d('2015-10-06')));
                expect(ts(WORKDAY.resolve.call(context, d('2015-10-05'), -8423))).to.equal(ts(d('1983-06-22')));
//                expect(ts(WORKDAY.resolve.call(context, d('2015-10-05'), 234845))).to.equal(ts(d('2915-12-09')));
            });

        });

        describe('function "WORKDAY.INTL"', function () {
            var WORKDAY_INTL = DateTimeFuncs['WORKDAY.INTL'];
            it('should exist', function () {
                expect(WORKDAY_INTL).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(WORKDAY_INTL).itself.to.respondTo('resolve');
            });

            it('should return correct results next workday between assigned date and day count', function () {
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5))).to.equal(ts(d('2015-10-12')));

            });
            it('should return correct results next workday between assigned date, day count and optional weekenddays', function () {
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 1))).to.equal(ts(d('2015-10-12')));

                //first day is a weekend day
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-04'), 5, 1))).to.equal(ts(d('2015-10-09')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, '1000000'))).to.equal(ts(d('2015-10-10')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 2))).to.equal(ts(d('2015-10-10')));

                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 3))).to.equal(ts(d('2015-10-11')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 4))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 5))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 6))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, 7))).to.equal(ts(d('2015-10-12')));

                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, true))).to.equal(ts(d('2015-10-12'))); //true is interpreted as 1
                expect(WORKDAY_INTL.resolve.bind(context, d('2015-10-05'), 5, 0)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(WORKDAY_INTL.resolve.bind(context, d('2015-10-05'), 5, 8)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(WORKDAY_INTL.resolve.bind(context, d('2015-10-05'), 5, '000')).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(WORKDAY_INTL.resolve.bind(context, d('2015-10-05'), 5, false)).to['throw'](ErrorCode).that.equals(ErrorCode.NUM);
                expect(WORKDAY_INTL.resolve.bind(context, d('2015-10-05'), 5, 'hello')).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
                expect(WORKDAY_INTL.resolve.bind(context, d('2015-10-05'), 5, '1111111')).to['throw'](ErrorCode).that.equals(ErrorCode.VALUE);
            });

            it('should return correct results next workday between assigned date, day count and optional weekenddays & holidays', function () {
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 4, 1, d('2015-10-07')))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 3, 2, d('2015-10-07')))).to.equal(ts(d('2015-10-09')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 2, 3, d('2015-10-07')))).to.equal(ts(d('2015-10-09')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 3, 4, d('2015-10-07')))).to.equal(ts(d('2015-10-10')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 3, 5, d('2015-10-07')))).to.equal(ts(d('2015-10-10')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 2, 6, d('2015-10-07')))).to.equal(ts(d('2015-10-10')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 3, 7, d('2015-10-07')))).to.equal(ts(d('2015-10-11')));

                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, '0000000', [d('2015-10-05'), d('2015-10-06')]))).to.equal(ts(d('2015-10-11')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, '0000000', [d('2015-10-09'), d('2015-10-10')]))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 4, '0000000', [d('2015-10-09'), d('2015-10-10'), d('2015-10-11')]))).to.equal(ts(d('2015-10-12')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 5, '0000000', [d('2015-10-11'), d('2015-10-09')]))).to.equal(ts(d('2015-10-12')));

                expect(ts(WORKDAY_INTL.resolve.call(context, d('1983-06-23'), 8421, 4, d('2015-10-02')))).to.equal(ts(d('2015-10-03')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-02'), 3, 4, d('2015-10-02')))).to.equal(ts(d('2015-10-05')));
//                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), 234844, 4, d('2015-10-02')))).to.equal(ts(d('2915-12-08')));
            });

            it('should return correct results last workday (negative workdays) between assigned date, day count and optional weekenddays & holidays', function () {
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, 1))).to.equal(ts(d('1983-06-21')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, '0000000'))).to.equal(ts(d('1992-09-11')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, '0000000', d('2015-10-02')))).to.equal(ts(d('1992-09-10')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, 1, d('2015-10-02')))).to.equal(ts(d('1983-06-20')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, 2, d('2015-10-02')))).to.equal(ts(d('1983-06-21')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, 3, d('2015-10-02')))).to.equal(ts(d('1983-06-22')));
                expect(ts(WORKDAY_INTL.resolve.call(context, d('2015-10-05'), -8424, 4, d('2015-10-02')))).to.equal(ts(d('1983-06-20')));
            });
        });

        describe('function "YEARS"', function () {
            var YEARS = DateTimeFuncs.YEARS;
            it('should exist', function () {
                expect(YEARS).to.be.an('object');
            });
            it('should be implemented', function () {
                expect(YEARS).itself.to.respondTo('resolve');
            });
            it('should return correct results years between two assigned dates', function () {
                expect(YEARS.resolve.call(context, d('2009-04-03'), d('2011-11-17'), 0)).to.equal(2);
                expect(YEARS.resolve.call(context, d('2011-02-28'), d('2012-02-28'), 0)).to.equal(1);
                expect(YEARS.resolve.call(context, d('2012-02-29'), d('2013-02-28'), 0)).to.equal(0);
                expect(YEARS.resolve.call(context, d('2009-04-03'), d('2011-11-17'), 1)).to.equal(2);
                expect(YEARS.resolve.call(context, d('2009-12-31'), d('2010-01-01'), 1)).to.equal(1);
            });
        });
    });

    // ========================================================================
});
