/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/locale/parser', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/object/arraytemplate',
    'io.ox/office/tk/locale/localedata'
], function (Utils, DateUtils, ArrayTemplate, LocaleData) {

    'use strict';

    // all available text colors in format codes, as name to RGB map
    var TEXT_COLORS = {
        black: '000000',
        blue: '0000FF',
        cyan: '00FFFF',
        green: '00FF00',
        magenta: 'FF00FF',
        red: 'FF0000',
        white: 'FFFFFF',
        yellow: 'FFFF00'
    };

    // A regular expression matching a text color in brackets at the beginning of a string.
    // - Group 1: The name of the matched color, without the bracket characters.
    var TEXT_COLOR_TOKEN_RE = new RegExp('^\\[(' + _.keys(TEXT_COLORS).join('|') + ')\\]', 'i');

    // currency symbols that can appear outside of a currency bracket
    // (TODO: generate from locale data according to some smart filter rules?)
    var LITERAL_CURRENCY_SYMBOLS = ['$', '\xa3', '\u0e3f', '\u20a1', '\u20a9', '\u20aa', '\u20ab', '\u20ac', '\u20ae', '\uffe5'];

    // A regular expression matching a literal currency symbol at the beginning of a string.
    // - Group 1: The matched currency symbol.
    var LITERAL_CURRENCY_RE = new RegExp('^(' + LITERAL_CURRENCY_SYMBOLS.map(_.escapeRegExp).join('|') + ')');

    // characters for CJK date/time format code templates
    var CJK_DATE_TIME_CHARACTERS = {
        trad: { Y: '\u5e74', M: '\u6708', D: '\u65e5', h: '\u6642', m: '\u5206', s: '\u79d2' },
        simp: { Y: '\u5e74', M: '\u6708', D: '\u65e5', h: '\u65f6', m: '\u5206', s: '\u79d2' },
        hang: { Y: '\ub144', M: '\uc6d4', D: '\uc77c', h: '\uc2dc', m: '\ubd84', s: '\ucd08' }
    };

    // private global methods =================================================

    /**
     * Returns the regular expression used to match a floating-point number at
     * the beginning of a string. The regular expression will support decimal
     * and scientific notation, and will contain two matching groups:
     * - Group 1: The leading sign character (may be an empty string).
     * - Group 2: The floating-point number (decimal or scientific), without
     *      the sign character.
     *
     * @param {String} dec
     *  The decimal separator inserted into the regular expression.
     *
     * @returns {RegExp}
     *  The regular expression matching a floating-point number at the
     *  beginning of a string.
     */
    var getNumberRegExp = _.memoize(function (dec) {
        dec = _.escapeRegExp(dec);
        return new RegExp('^([-+]?)((?:\\d*' + dec + '\\d+|\\d+' + dec + '?)(?:E[-+]?\\d+)?)', 'i');
    });

    // class FormatToken ======================================================

    /**
     * A parsed token from a number format code.
     *
     * @constructor
     *
     * @property {String} type
     *  The type of this format token.
     *
     * @property {String} text
     *  The text represented by this format token, e.g. for literal text, or
     *  for specific delimiter tokens for numbers or dates.
     *
     * @property {Number} length
     *  The target length of this format token, e.g. for date/time tokens
     *  specifying the type or length of the date/time component.
     */
    function FormatToken(type, text, length) {

        this.type = type;
        this.text = text;
        this.length = length;

    } // class FormatToken

    // class FormatTokenArray =================================================

    /**
     * A typed array with instances of the class FormatToken.
     *
     * @constructor
     */
    var FormatTokenArray = ArrayTemplate.create(FormatToken);

    // class FormatSection ====================================================

    /**
     * Contains the parsed information of a single section in a format code.
     * Sections will be separated with semicolons in the format code and can be
     * used to define different number formats for specific intervals of
     * numbers, e.g. for positive and negative numbers.
     *
     * @constructor
     *
     * @property {FormatTokenArray} tokens
     *  An array with the parsed tokens of this section.
     *
     * @property {String} category
     *  The identifier of the category this format section is associated to.
     *  See property ParsedFormat.category for details.
     *
     * @property {String|Null} colorName
     *  The lower-case name of the text color to be used, if this format
     *  section is used to format a value. The value null does not change the
     *  original text color of the value.
     *
     * @property {String|Null} rgbColor
     *  The RGB value of the text color to be used, if this format section is
     *  used to format a value. The value null does not change the original
     *  text color of the value.
     *
     * @property {String|Null} operator
     *  The operator used to define the number interval covered by this code
     *  section, together with the property 'boundary'. Must be one of '<'
     *  (less than), '<=' (less than or equal), '>' (greater than), '>='
     *  (greater than or equal), '=' (equal), or '<>' (not equal). The value
     *  null represents a missing operator (this format section will match all
     *  numbers).
     *
     * @property {Number} boundary
     *  The boundary value used to define the number interval covered by this
     *  code section, together with the property 'operator'.
     *
     * @property {Boolean} autoMinus
     *  Whether this code section will cause to automatically add a leading
     *  minus sign when formatting negative numbers. Happens if the interval of
     *  this code section covers at least negative numbers, and non-negative
     *  numbers (zero and/or positive numbers).
     *
     * @property {Object} flags
     *  Internal state flags for the format section:
     *  @property {Number} flags.general
     *      Whether this section contains a 'General' format token.
     *  @property {Number} flags.integer
     *      Whether this section contains any format tokens for the integral
     *      part of numbers.
     *  @property {Number} flags.float
     *      Whether this section contains any format tokens for the fractional
     *      part of floating-point numbers.
     *  @property {Number} flags.percent
     *      The number of percent tokens contained in this section.
     *  @property {Boolean} flags.scientific
     *      Whether this section contains an exponent token resulting in
     *      scientific notation.
     *  @property {Boolean} flags.fraction
     *      Whether this section contains a fraction token with numerator and
     *      denominator.
     *  @property {Boolean} flags.currency
     *      Whether this section contains a currency symbol.
     *  @property {Boolean} flags.date
     *      Whether this section contains any date tokens.
     *  @property {Boolean} flags.time
     *      Whether this section contains any time tokens.
     *  @property {Boolean} flags.ampm
     *      Whether this section contains an AM/PM token that switches the time
     *      to 12-hours format.
     *  @property {Boolean} flags.text
     *      Whether this section contains a text token.
     *  @property {Boolean} flags.fill
     *      Whether this section contains a fill token (a character that fills
     *      the remaining space available for the formatted text).
     */
    function FormatSection() {

        this.tokens = new FormatTokenArray();
        this.category = 'custom';

        this.colorName = null;
        this.rgbColor = null;

        this.operator = null;
        this.boundary = 0;
        this.autoMinus = true;

        this.flags = {
            general: false,
            integer: false,
            float: false,
            percent: 0,
            scientific: false,
            fraction: false,
            fractionCode: '',
            floatCode: '',
            currency: false,
            date: false,
            time: false,
            ampm: false,
            text: false,
            fill: false
        };

    } // class FormatSection

    // public methods ---------------------------------------------------------

    /**
     * Adds a new format token without literal text, and without a specific
     * length to this code section.
     *
     * @param {String} type
     *  The type of the new format token.
     */
    FormatSection.prototype.pushToken = function (type) {
        this.tokens.push(new FormatToken(type, '', 0));
    };

    /**
     * Adds a new format token with literal text to this code section, or
     * extends the text of the last token, if it has the same type.
     *
     * @param {String} type
     *  The type of the new format token.
     *
     * @param {String} text
     *  The text represented by the new format token, e.g. for literal text, or
     *  for specific delimiter tokens for numbers or dates.
     */
    FormatSection.prototype.pushTextToken = function (type, text) {
        var token = this.tokens.last();
        if (token && (token.type === type)) {
            token.text += text;
        } else {
            this.tokens.push(new FormatToken(type, text, 0));
        }
    };

    /**
     * Adds a new literal text token to this code section.
     *
     * @param {String} text
     *  The text literal represented by the new token.
     */
    FormatSection.prototype.pushLiteralToken = function (text) {
        this.pushTextToken('lit', text);
    };

    /**
     * Adds a new date token to this code section.
     *
     * @param {String} type
     *  The exact type of the date token. Must be one of 'year', 'month', or
     *  'day'.
     *
     * @param {Number} length
     *  The length of the date token. The meaning of this value depends on the
     *  passed type:
     *  - 'year': 2 for short years, 4 for full years.
     *  - 'month': 1 for short month number, 2 for month number with leading
     *      zero, 3 for abbreviated month name, 4 for full month name, or 5 for
     *      the first character of the month name.
     *  - 'day': 1 for short day number, 2 for day number with leading zero, 3
     *      for abbreviated name of week day, or 4 for full name of week day.
     */
    FormatSection.prototype.pushDateToken = function (type, length) {
        this.tokens.push(new FormatToken(type, '', length));
        this.flags.date = true;
    };

    /**
     * Adds a new time token to this code section.
     *
     * @param {String} type
     *  The exact type of the time token. Must be one of 'hour', 'minute',
     *  'second', 'hours', 'minutes', or 'seconds' (the latter three types are
     *  for total number of hours, minutes, or seconds contained in a date).
     *
     * @param {Number} length
     *  The length of the time token. The meaning of this value depends on the
     *  passed type:
     *  - 'hour', 'minute', 'second': The length of the time token: 1 for short
     *      numbers, 2 for numbers with leading zero.
     *  - 'hours', 'minutes', 'seconds': The minimum length of the number, to
     *      be filled with leading zeros.
     */
    FormatSection.prototype.pushTimeToken = function (type, length) {
        this.tokens.push(new FormatToken(type, '', length));
        this.flags.time = true;
    };

    /**
     * Adds a new AM/PM token to this code section.
     *
     * @param {Number} length
     *  1, if the token represents the short variant ('a' or 'p'), or 2 for the
     *  long variant ('am' or 'pm').
     */
    FormatSection.prototype.pushAmPmToken = function (length) {
        this.pushTimeToken('ampm', length);
        this.flags.ampm = true;
    };

    /**
     * Sets the text color to be used to render a value formatted with this
     * format section.
     *
     * @param {String} colorName
     *  A valid predefined name of a text color.
     */
    FormatSection.prototype.setColor = function (colorName) {
        if (colorName in TEXT_COLORS) {
            this.colorName = colorName;
            this.rgbColor = TEXT_COLORS[colorName];
        } else {
            Utils.error('FormatSection.setColor(): invalid color "' + colorName + '"');
        }
    };

    /**
     * Changes the interval covered by this code section.
     *
     * @param {String} operator
     *  The operator used to define the number interval covered by this code
     *  section. See description of the class property 'operator' for details.
     *
     * @param {Number} boundary
     *  The boundary value used to define the number interval covered by this
     *  code section. See description of the class property 'boundary' for
     *  details.
     *
     * @param {Boolean} [def=false]
     *  Whether this invocation wants to initialize the default interval of
     *  this code section. If the section already contains a custom interval,
     *  it will not be changed.
     */
    FormatSection.prototype.setInterval = function (operator, boundary, def) {

        // do not change custom intervals, if default flag has been passed
        if (def && this.operator) { return; }

        // initialize 'autoMinus' flag, and check validity of the operator
        switch (operator) {
            case '<':  this.autoMinus = boundary > 0;  break;
            case '<=': this.autoMinus = boundary >= 0; break;
            case '>': // same auto-minus for both operators (e.g.: [>-1] and [>=-1] cover negative and zero, but [>0] and [>=0] do not)
            case '>=': this.autoMinus = boundary < 0;  break;
            case '=':  this.autoMinus = false;         break;
            case '<>': this.autoMinus = true;          break;
            default:
                Utils.error('FormatSection.setInterval(): invalid interval operator "' + operator + '"');
                return;
        }

        // store passed values, if operator is valid
        this.operator = operator;
        this.boundary = boundary;
    };

    /**
     * Returns whether this code section can be used to format the passed
     * number, according to the properties 'operator' and 'boundary'.
     *
     * @param {Number} number
     *  The number to be checked.
     *
     * @returns {Boolean}
     *  Whether this code section can be used to format the passed number.
     */
    FormatSection.prototype.contains = function (number) {
        switch (this.operator) {
            case '<':  return number < this.boundary;
            case '<=': return number <= this.boundary;
            case '>':  return number > this.boundary;
            case '>=': return number >= this.boundary;
            case '=':  return number === this.boundary;
            case '<>': return number !== this.boundary;
        }
        return true; // default for single code section covering all numbers
    };

    // static methods ---------------------------------------------------------

    /**
     * Parses the leading code section contained in the passed format code, and
     * creates a new instance of the class FormatSection if possible.
     *
     * @param {String} formatCode
     *  The number format code to be parsed.
     *
     * @returns {Object}
     *  The parser result, with the following properties:
     *  - {FormatSection|Null} section
     *      The new format section instance; or null, if the passed format code
     *      contains an unrecoverable syntax error.
     *  - {String|Null} remainder
     *      The remaining format code without the parsed code section; or null,
     *      if the entire format code has been parsed successfully.
     */
    FormatSection.parse = function (formatCode) {

        var // the current code section
            section = new FormatSection(),
            // the internal flags of the code section
            flags = section.flags,
            // the matches of the regular expressions
            matches = null,
            // whether the parser has encountered a syntax error
            parseError = false,
            // whether the parser has found a section separator
            hasSeparator = false,
            // whether last date/time token was of type hour
            lastWasHour = false;

        // stop if a code section, or the entire format code has been consumed
        while (!parseError && !hasSeparator && (formatCode.length > 0)) {

            // try/finally to be able to continue the while-loop at any place
            try {

                // code section separator
                if ((matches = /^;/.exec(formatCode))) {
                    hasSeparator = true;
                    continue;
                }

                // 'General' format token
                if ((matches = /^GENERAL/i.exec(formatCode))) {
                    section.pushToken('general');
                    flags.general = true;
                    continue;
                }

                // decimal fractional part
                if ((matches = /^\.[0?#]+/i.exec(formatCode))) {
                    section.pushTextToken('sign', matches[0]);
                    flags.float = true;
                    flags.floatCode = matches[0];
                    continue;
                }

                // fractional notation
                if ((matches = /^[0?#]+\/[0-9?#]+/i.exec(formatCode))) {
                    section.pushTextToken('frac', matches[0]);
                    flags.fraction = true;
                    flags.fractionCode = matches[0];
                    continue;
                }

                // integral digits
                if ((matches = /^,?[0#?]+,?/.exec(formatCode))) {
                    section.pushTextToken('nr', matches[0]);
                    flags.integer = true;
                    continue;
                }

                // scientific notation
                if ((matches = /^E[-+]([0?#]+)/i.exec(formatCode))) {
                    section.pushTextToken('scien', matches[1]);
                    flags.scientific = true;
                    continue;
                }

                // date/time: 12 hours, AM/PM marker
                if ((matches = /^AM\/PM/i.exec(formatCode))) {
                    section.pushAmPmToken(2);
                    continue;
                }

                // date/time: 12 hours, A/P marker
                if ((matches = /^A\/P/i.exec(formatCode))) {
                    section.pushAmPmToken(1);
                    continue;
                }

                // date/time category: year
                if ((matches = /^Y+/i.exec(formatCode))) {
                    section.pushDateToken('year', Math.min(4, Utils.roundUp(matches[0].length, 2))); // only YY and YYYY allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: month
                if ((matches = /^M+/.exec(formatCode))) {
                    section.pushDateToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: day/weekday
                if ((matches = /^D+/i.exec(formatCode))) {
                    section.pushDateToken('day', Math.min(4, matches[0].length)); // D to DDDD allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: total number of hours
                if ((matches = /^\[(h+)\]/i.exec(formatCode))) {
                    section.pushTimeToken('hours', matches[1].length); // [h], [hh], [hhh], ... (unlimited length)
                    lastWasHour = true;
                    continue;
                }

                // date/time category: total number of minutes
                if ((matches = /^\[(m+)\]/.exec(formatCode))) {
                    section.pushTimeToken('minutes', matches[1].length); // [m], [mm], [mmm], ... (unlimited length)
                    continue;
                }

                // date/time category: total number of seconds
                if ((matches = /^\[(s+)\]/i.exec(formatCode))) {
                    section.pushTimeToken('seconds', matches[1].length); // [s], [ss], [sss], ... (unlimited length)
                    lastWasHour = false;
                    continue;
                }

                // date/time category: hour
                if ((matches = /^h+/i.exec(formatCode))) {
                    section.pushTimeToken('hour', Math.min(2, matches[0].length)); // h or hh allowed
                    lastWasHour = true;
                    continue;
                }

                // date/time category: minute together with seconds
                if ((matches = /^(m+)(?=\W+s)/i.exec(formatCode))) {
                    section.pushTimeToken('minute', Math.min(2, matches[1].length)); // m or mm allowed
                    lastWasHour = false;
                    continue;
                }

                // date/time category: minute
                if ((matches = /^m+/.exec(formatCode))) {
                    if (lastWasHour) {
                        section.pushTimeToken('minute', Math.min(2, matches[0].length)); // m or mm allowed
                    } else {
                        section.pushDateToken('month', Math.min(5, matches[0].length)); // M to MMMMM allowed
                    }
                    lastWasHour = false;
                    continue;
                }

                // date/time category: second
                if ((matches = /^s+/i.exec(formatCode))) {
                    section.pushTimeToken('second', Math.min(2, matches[0].length)); // s or ss allowed
                    lastWasHour = false;
                    continue;
                }

                // percent sign
                if ((matches = /^%/.exec(formatCode))) {
                    section.pushLiteralToken('%');
                    flags.percent += 1;
                    continue;
                }

                // string literal, enclosed in double-quotes
                if ((matches = /^"([^"]*)"/.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    continue;
                }

                // character literal, preceded by backspace
                if ((matches = /^\\(.)/.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    continue;
                }

                // whitespace (TODO: store and handle exact character width)
                if ((matches = /^_./.exec(formatCode))) {
                    section.pushLiteralToken(' ');
                    continue;
                }

                // repeated fill character
                if ((matches = /^\*(.)/.exec(formatCode))) {
                    // subsequent fill characters will be inserted once literally
                    // (only the first fill character in the section will be expanded)
                    if (flags.fill) {
                        section.pushLiteralToken(matches[1]);
                    } else {
                        section.pushTextToken('fill', matches[1]);
                        flags.fill = true;
                    }
                    continue;
                }

                // number interval
                if ((matches = /^\[(<=?|>=?|<>|=)([^[\];]+)\]/.exec(formatCode))) {
                    var parseResult = Parser.parseLeadingNumber(matches[2], { dec: '.', sign: true, complete: true });
                    if (parseResult && (parseResult.remaining.length === 0)) {
                        section.setInterval(matches[1], parseResult.number);
                    } else {
                        parseError = true;
                    }
                    continue;
                }

                // text color
                if ((matches = TEXT_COLOR_TOKEN_RE.exec(formatCode))) {
                    section.setColor(matches[1].toLowerCase());
                    continue;
                }

                // locale data with hexadecimal LCID and calendar information
                if ((matches = /^\[\$-([0-9a-f]+)\]/i.exec(formatCode))) {
                    continue;
                }

                // currency symbol with hexadecimal LCID
                if ((matches = /^\[\$([^[\];-]+)-([0-9a-f]+)\]/i.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    flags.currency = true;
                    continue;
                }

                // currency symbol without hexadecimal LCID
                if ((matches = /^\[\$([^[\];-]+)\]/i.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    flags.currency = true;
                    continue;
                }

                // literal currency symbol not enclosed in brackets
                if ((matches = LITERAL_CURRENCY_RE.exec(formatCode))) {
                    section.pushLiteralToken(matches[1]);
                    flags.currency = true;
                    continue;
                }

                // ignore other unsupported specifiers
                if ((matches = /^(\[[^[\];]*\])/.exec(formatCode))) {
                    Utils.warn('FormatSection.parse(): unsupported section specifier "' + matches[1] + '"');
                    continue;
                }

                // placeholder for text value
                if ((matches = /^@/.exec(formatCode))) {
                    section.pushToken('text');
                    flags.text = true;
                    continue;
                }

                // unmatched double-quote, unmatched opening bracket, or incomplete escape sequence: syntax error
                if (/^["\[\\_*]/.test(formatCode)) {
                    parseError = true;
                    continue;
                }

                // any non-matching token is considered a character literal
                matches = [formatCode[0]];
                section.pushLiteralToken(matches[0]);

            } finally {
                // remove the current match from the formula string
                formatCode = formatCode.slice(matches[0].length);
            }
        }

        // resolve the category identifier
        if (!parseError) {
            if (flags.text) {
                section.category = 'text';
                // do not allow any numeric tokens (currency symbols are allowed)
                parseError = flags.general || flags.integer || flags.float || flags.percent || flags.scientific || flags.fraction || flags.date || flags.time;
            } else if (flags.general && (section.tokens.length === 1)) {
                // simple 'General' format code (optionally with color)
                section.category = 'standard';
            } else if (!flags.general && (flags.integer || flags.float) && !flags.date && !flags.time) {
                // numeric format codes
                if (!flags.percent && !flags.scientific && !flags.fraction && !flags.currency) {
                    section.category = 'number';
                } else if (flags.percent && !flags.scientific && !flags.fraction && !flags.currency) {
                    section.category = 'percent';
                } else if (!flags.percent && flags.scientific && !flags.fraction && !flags.currency) {
                    section.category = 'scientific';
                } else if (!flags.percent && !flags.scientific && flags.fraction && flags.integer && !flags.float && !flags.currency) {
                    section.category = 'fraction';
                } else if (!flags.percent && !flags.scientific && !flags.fraction && flags.currency) {
                    section.category = 'currency';
                }
            } else if (!flags.general && !flags.integer && !flags.percent && !flags.scientific && !flags.fraction) {
                // date/time format codes
                if (flags.date && !flags.time && !flags.float) {
                    section.category = 'date';
                } else if (!flags.date && flags.time) {
                    section.category = 'time';
                } else if (flags.date && flags.time) {
                    section.category = 'datetime';
                }
            }
        }

        // create and return the parser result
        return {
            section: parseError ? null : section,
            remainder: hasSeparator ? formatCode : null
        };
    };

    // class FormatSectionArray ===============================================

    /**
     * A typed array with instances of the class FormatSection.
     *
     * @constructor
     */
    var FormatSectionArray = ArrayTemplate.create(FormatSection);

    // class ParsedFormat =====================================================

    /**
     * Represents a parsed number format code.
     *
     * @constructor
     *
     * @property {String} formatCode
     *  The original format code this instance has been created from.
     *
     * @property {FormatSectionArray} numberSections
     *  An array of format code sections for different number intervals.
     *
     * @property {FormatSection} textSection
     *  A dedicated code section to format string values.
     *
     * @property {Boolean} syntaxError
     *  Whether the format code that has been used to create this instance
     *  contains a syntax error.
     *
     * @property {String} category
     *  The identifier of the category this format code is associated to. Will
     *  contain one of the following values:
     *  - 'standard': Reserved for the format code 'General'.
     *  - 'number': All numeric format codes without special appearance.
     *  - 'scientific': All numeric format codes with scientific notation.
     *  - 'percent': All numeric format codes shown as percentage.
     *  - 'fraction': All numeric format codes shown as fractions.
     *  - 'currency': All numeric format codes with a currency symbol.
     *  - 'date': All format codes shown as a date (without time).
     *  - 'time': All format codes shown as a time (without date).
     *  - 'datetime': All format codes shown as combined date and time.
     *  - 'text': All format codes containing a text section only.
     *  - 'custom': All format codes not fitting into any other category.
     */
    function ParsedFormat(formatCode) {

        this.formatCode = formatCode;
        this.numberSections = new FormatSectionArray();
        this.textSection = null;
        this.syntaxError = false;
        this.category = 'custom';

    } // class ParsedFormat

    // public methods ---------------------------------------------------------

    /**
     * Parses the leading code section contained in the passed format code, and
     * inserts a new instance of the class FormatSection into this object if
     * possible. If the passed format code contains a syntax error, the public
     * flag 'syntaxError' will be set to true.
     *
     * @param {String} formatCode
     *  The number format code to be parsed.
     *
     * @returns {String|Null}
     *  The remaining format code without the parsed code section; or null, if
     *  the entire format code has been parsed successfully.
     */
    ParsedFormat.prototype.parseSection = function (formatCode) {

        // result of the section parser (new code section, and remaining format code)
        var parseResult = FormatSection.parse(formatCode);

        // missing code section indicates syntax error
        if (!parseResult.section) {
            this.syntaxError = true;
            return null;
        }

        // there can only be a single text sections at the end of the format code
        if (this.textSection) {
            this.syntaxError = true;
            return null;
        }

        // store the new code section in this instance
        if (parseResult.section.category === 'text') {
            this.textSection = parseResult.section;
        } else {
            this.numberSections.push(parseResult.section);
        }

        // return the remaining format code
        return parseResult.remainder;
    };

    /**
     * Returns an existing code section, according to the passed value.
     *
     * @param {Any} value
     *  The value used to pick a code section. For finite numbers, one of the
     *  numeric sections will be returned. For strings, the text section will
     *  be returned. For all other values, null will be returned.
     *
     * @returns {FormatSection|Null}
     *  The format code section for the passed value, if available; otherwise
     *  null.
     */
    ParsedFormat.prototype.getSection = function (value) {

        // do not return sections, if the format code contains a syntax error
        if (this.syntaxError) { return null; }

        // return text section for strings
        if (_.isString(value)) { return this.textSection; }

        // reject other values than finite numbers
        if (!_.isNumber(value) || !isFinite(value)) { return null; }

        // pick first matching section for numbers
        return this.numberSections.find(function (s) { return s.contains(value); });
    };

    /**
     * Returns whether this parsed format is the 'General' format code in the
     * category 'standard'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is the 'General' format code.
     */
    ParsedFormat.prototype.isStandard = function () {
        return this.category === 'standard';
    };

    /**
     * Returns whether this parsed format is in one of the categories 'date',
     * or 'datetime'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is in a date category.
     */
    ParsedFormat.prototype.isAnyDate = function () {
        return (/^date(time)?$/).test(this.category);
    };

    /**
     * Returns whether this parsed format is in one of the categories 'time',
     * or 'datetime'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is in a time category.
     */
    ParsedFormat.prototype.isAnyTime = function () {
        return (/^(date)?time$/).test(this.category);
    };

    /**
     * Returns whether this parsed format is in one of the categories 'date',
     * 'time', or 'datetime'.
     *
     * @returns {Boolean}
     *  Whether this parsed format is in one of the date/time categories.
     */
    ParsedFormat.prototype.isAnyDateTime = function () {
        return (/^(date|time|datetime)$/).test(this.category);
    };

    // static class Parser ====================================================

    /**
     * A static class providing methods to parse numbers, dates, format codes,
     * and other locale-dependent source data.
     */
    var Parser = {};

    // static methods ---------------------------------------------------------

    /**
     * Converts the passed number to a string, using the decimal separator of
     * the current UI locale. This is a simple method that uses JavaScript's
     * conversion from numbers to strings.
     *
     * @param {Number} number
     *  The number to be converted.
     *
     * @returns {String}
     *  The passed number, converted to string, with the decimal separator of
     *  the current UI locale.
     */
    Parser.numberToString = function (number) {
        return String(number).replace(/\./g, LocaleData.DEC);
    };

    /**
     * Converts the passed string to a number, using the decimal separator of
     * the current UI locale. This is a simple method that uses JavaScript's
     * parseFloat() conversion function.
     *
     * @param {String} text
     *  The text to be converted to a number.
     *
     * @returns {Number}
     *  The passed number, converted to string. If the passed text does not
     *  represent a floating-point number, NaN will be returned. If the text
     *  consists of a number with following garbage (e.g. '123abc'), the
     *  leading number will be returned (same behavior as JavaScript's native
     *  parseFloat() function).
     */
    Parser.stringToNumber = function (text) {
        return parseFloat(text.replace(LocaleData.DEC, '.'));
    };

    /**
     * Tries to parse a floating-point number from the leading part of the
     * passed string.
     *
     * @param {String} text
     *  The text containing a floating-point number.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dec=LocaleData.DEC]
     *      If specified, a custom decimal separator character. By default, the
     *      decimal separator of the current UI language will be used.
     *  @param {Boolean} [options.sign=false]
     *      If set to true, the passed string may contain a leading sign
     *      character ('-' or '+').
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      floating-point number. No remaining 'garbage' text will be
     *      accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid floating-
     *  point number (according to the passed options); otherwise null. The
     *  result descriptor contains the following properties:
     *  - {Number} number
     *      The parsed floating-point number.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      floating-point number.
     *  - {String} sign
     *      The sign character. Will be the empty string, if no sign exists in
     *      the passed text, or if the option 'sign' has not been set
     *      (see above).
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed floating-point number. Will be the empty string, if the
     *      option 'complete' has been set (see above).
     */
    Parser.parseLeadingNumber = function (text, options) {

        // the decimal separator used to parse the number
        var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC);
        // whether to accept a leading sign
        var sign = Utils.getBooleanOption(options, 'sign', false);
        // whether to reject trailing garbage
        var complete = Utils.getBooleanOption(options, 'complete', false);
        // try to parse the passed string for a leading floating-point number
        var matches = getNumberRegExp(dec).exec(text);
        // the parsed floating-point number
        var number = null;

        // reject leading sign unless specified in the passed options
        if (!matches || (!sign && matches[1].length > 0)) { return null; }

        // reject trailing garbage if specified in the passed options
        if (complete && (matches[0].length < text.length)) { return null; }

        // parse the floating-point number
        number = parseFloat(matches[2].replace(dec, '.'));
        if (!_.isFinite(number)) { return null; }
        if (matches[1] === '-') { number = -number; }

        // build the result descriptor
        return { number: number, text: matches[0], sign: matches[1], remaining: text.substr(matches[0].length) };
    };

    /**
     * Tries to parse a date from the leading part of the passed string.
     *
     * @param {String} text
     *  The text potentially starting with a date.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dec=LocaleData.DEC]
     *      A custom decimal separator character. By default, the decimal
     *      separator of the current UI language will be used. Needed to decide
     *      whether the period character is allowed as date separator (if the
     *      decimal separator is a period, it cannot be part of a date).
     *  @param {Boolean} [options.leadingMonth=LocaleData.LEADING_MONTH]
     *      If set to true, the parser will prefer months before days if in
     *      doubt. If omitted, the preference of the current UI locale will be
     *      used. Example: The text '1/2' may be converted to the date Jan-02
     *      instead of Feb-01.
     *  @param {Boolean} [options.leadingYear=LocaleData.LEADING_YEAR]
     *      If set to true, the parser will prefer a leading year if in doubt.
     *      If omitted, the preference of the current UI locale will be used.
     *      Example: The text '1/2/3' may be converted to the date 2001-Feb-03
     *      instead of 2003-Feb-01.
     *  @param {Number} [options.centuryThreshold=100]
     *      An integer in the range 0 to 100 that specifies how to convert a
     *      number below 100 to a 4-digit year. If omitted, all short years
     *      will be moved into the current century. See DateUtils.expandYear()
     *      for more details.
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      date. No remaining 'garbage' text will be accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid date
     *  (according to the passed options); otherwise null. The result
     *  descriptor contains the following properties:
     *  - {Number|Null} D
     *      The parsed day (one-based); or null, if the text does not contain a
     *      day (i.e. month and year only).
     *  - {Number} M
     *      The parsed month (zero-based!).
     *  - {Number|Null} Y
     *      The parsed year; or null, if the text does not contain a year (i.e.
     *      day and month only).
     *  - {Date} date
     *      A UTC date object representing the parsed date. Defaults to the
     *      current year, and to the first day in the month, if the respective
     *      date component is missing in the passed text.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      date.
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed date. Will be the empty string, if the option 'complete' has
     *      been set (see above).
     */
    Parser.parseLeadingDate = (function () {

        // returns a regular expression used to match a specific date at the beginning of a string
        var getDateRegExp = _.memoize(function (comps, dec) {

            // character class for valid date separators
            var separators = (dec === '.') ? '[-/]' : '[-./]';

            var pattern = '';
            _.each(comps, function (comp) {

                // add separator character between date components
                if (pattern) { pattern += '\\s*' + separators + '\\s*'; }

                // add a capturing group for the current component
                switch (comp) {
                    case 'D':
                        pattern += '([12]\\d|3[01]|0?[1-9])'; // shortest alternative must be at the end
                        break;
                    case 'M':
                        pattern += '(1[0-2]|0?[1-9])'; // shortest alternative must be at the end
                        break;
                    case 'Y':
                        pattern += '(\\d{4}|\\d\\d?)'; // shortest alternative must be at the end
                        break;
                }
            });

            return new RegExp('^' + pattern);
        }, function (comps, dec) { return comps + dec; });

        // tries to parse a date with the specified components
        function parseDate(text, comps, dec) {

            // try to match a date with the specified components
            var matches = getDateRegExp(comps, dec).exec(text);
            if (!matches) { return null; }

            // the result object with the matched text
            var result = {
                D: null,
                M: null,
                Y: null,
                text: matches[0],
                remaining: text.substr(matches[0].length)
            };

            // convert the matched date components to numbers
            _.each(comps, function (comp, index) {
                result[comp] = parseInt(matches[index + 1], 10);
            });

            // zero-based month (by convention, as used everywhere else, e.g. in class Date)
            if (_.isNumber(result.M)) { result.M -= 1; }

            return result;
        }

        // the actual parseLeadingDate() method
        function parseLeadingDate(text, options) {

            // the decimal separator needed to select the valid date separators
            var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC);
            // whether to prefer leading months and years in the text to be parsed
            var leadingMonth = Utils.getBooleanOption(options, 'leadingMonth', LocaleData.LEADING_MONTH);
            var leadingYear = Utils.getBooleanOption(options, 'leadingYear', LocaleData.LEADING_YEAR);

            // the day/month components for getDateRegExp() according to the native order of the current locale
            var DAY_MONTH_COMPS = leadingMonth ? 'MD' : 'DM';

            // all date patterns to be matched, in the correct order
            var ALL_COMPONENTS = [
                // start with complete dates (leading and trailing year, prefer native date order)
                leadingYear ? ('Y' + DAY_MONTH_COMPS) : (DAY_MONTH_COMPS + 'Y'),
                leadingYear ? (DAY_MONTH_COMPS + 'Y') : ('Y' + DAY_MONTH_COMPS),
                // day/month without year
                DAY_MONTH_COMPS,
                // month/year without day (leading and trailing year, prefer native date order)
                leadingYear ? 'YM' : 'MY',
                leadingYear ? 'MY' : 'YM'
            ];

            // find the first matching date pattern (method some() will exit on first match)
            var result = null;
            ALL_COMPONENTS.some(function (comps) { return (result = parseDate(text, comps, dec)); });
            if (!result) { return null; }

            // reject trailing garbage if specified in the passed options
            var complete = Utils.getBooleanOption(options, 'complete', false);
            if (complete && (result.remaining.length > 0)) { return null; }

            // validate the day according to month and year
            var threshold = Utils.getIntegerOption(options, 'centuryThreshold', 100, 0, 100);
            var y = _.isNumber(result.Y) ? DateUtils.expandYear(result.Y, threshold) : new Date().getFullYear();
            var m = _.isNumber(result.M) ? result.M : 0;
            var d = _.isNumber(result.D) ? result.D : 1;
            if (d > DateUtils.getDaysInMonth(y, m)) { return null; }

            // create the date property in the result
            result.date = DateUtils.makeUTCDate({ Y: y, M: m, D: d });
            return isFinite(result.date.getTime()) ? result : null;
        }

        return parseLeadingDate;
    }());

    /**
     * Tries to parse a time from the leading part of the passed string.
     *
     * @param {String} text
     *  The text potentially starting with a time.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {String} [options.dec=LocaleData.DEC]
     *      If specified, a custom decimal separator character. By default, the
     *      decimal separator of the current UI language will be used. Needed
     *      to parse the fractional part of seconds (see option 'milliseconds'
     *      for details).
     *  @param {Boolean} [options.multipleDays=false]
     *      If set to true, the hours may be greater than the value 23,
     *      resulting in a time greater than or equal to an entire day.
     *  @param {Boolean} [options.milliseconds=false]
     *      If set to true, the seconds may contain a fractional part, e.g. in
     *      the parse text '12:23:34.567'.
     *  @param {Boolean} [options.complete=false]
     *      If set to true, the entire passed string must represent a valid
     *      time. No remaining 'garbage' text will be accepted.
     *
     * @returns {Object|Null}
     *  A result descriptor, if the passed string starts with a valid time
     *  (according to the passed options); otherwise null. The result
     *  descriptor contains the following properties:
     *  - {Number|Null} h
     *      The parsed hour; or null, if the text does not contain an hour
     *      (i.e. minutes and seconds only).
     *  - {Number} m
     *      The parsed minute.
     *  - {Number|Null} s
     *      The parsed second; or null, if the text does not contain a second
     *      (i.e. hours and minutes only).
     *  - {Number|Null} ms
     *      The parsed milliseconds (e.g. 500 for the text '12:34.5'); or null,
     *      if the text does not contain a fractional seconds part.
     *  - {Date} time
     *      A UTC date object representing the parsed time.
     *  - {String} text
     *      The leading text from the passed string that has been parsed to the
     *      time.
     *  - {String} remaining
     *      The remaining characters from the passed string following the
     *      parsed time. Will be the empty string, if the option 'complete' has
     *      been set (see above).
     */
    Parser.parseLeadingTime = (function () {

        // returns a regular expression used to match a specific time at the beginning of a string
        var getTimeRegExp = _.memoize(function (comps, dec) {
            var pattern = '';
            _.each(comps, function (comp) {

                // add separator character between time components
                if (pattern) { pattern += '\\s*' + ((comp === '0') ? _.escapeRegExp(LocaleData.DEC) : ':') + '\\s*'; }

                // add a capturing group for the current component
                switch (comp) {
                    case 'h':
                        pattern += '(\\d{1,4})'; // accept up to 4 digits for hours (multiple days)
                        break;
                    case 'm':
                    case 's':
                        pattern += '([0-5]?\\d)';
                        break;
                    case '0':
                        pattern += _.escapeRegExp(dec) + '(\\d+)';
                }
            });

            return new RegExp('^' + pattern);
        }, function (comps, dec) { return comps + dec; });

        // tries to parse a time with the specified components
        function parseTime(text, comps, dec) {

            // try to match a time with the specified components
            var matches = getTimeRegExp(comps, dec).exec(text);
            if (!matches) { return null; }

            // the result object with the matched text
            var result = {
                h: null,
                m: null,
                s: null,
                ms: null,
                text: matches[0],
                remaining: text.substr(matches[0].length)
            };

            // convert the matched time components to numbers
            _.each(comps, function (comp, index) {
                var match = matches[index + 1];
                // special handling for the fractional seconds part
                if (comp === '0') {
                    result.ms = parseFloat('0.' + match) * 1000;
                } else {
                    result[comp] = parseInt(match, 10);
                }
            });

            return result;
        }

        // the actual parseLeadingTime() method
        function parseLeadingTime(text, options) {

            // the decimal separator used to parse the milliseconds
            var dec = Utils.getStringOption(options, 'dec', LocaleData.DEC);
            // whether to accept milliseconds
            var milliseconds = Utils.getBooleanOption(options, 'milliseconds', false);

            // all time patterns to be matched, in the correct order (TODO: AM/PM)
            var ALL_COMPONENTS = milliseconds ? ['hms0', 'hm', 'ms0'] : ['hms', 'hm'];

            // find the first matching time pattern (method some() will exit on first match)
            var result = null;
            ALL_COMPONENTS.some(function (comps) { return (result = parseTime(text, comps, dec)); });
            if (!result) { return null; }

            // reject trailing garbage if specified in the passed options
            var complete = Utils.getBooleanOption(options, 'complete', false);
            if (complete && (result.remaining.length > 0)) { return null; }

            // reject hours above 23 unless specified
            var multipleDays = Utils.getBooleanOption(options, 'multipleDays', false);
            if (!multipleDays && _.isNumber(result.h) && (result.h > 23)) { return null; }

            // create the time property in the result
            result.time = new Date(0);
            if (_.isNumber(result.h)) { result.time.setUTCHours(result.h); }
            if (_.isNumber(result.m)) { result.time.setUTCMinutes(result.m); }
            if (_.isNumber(result.s)) { result.time.setUTCSeconds(result.s); }
            if (_.isNumber(result.ms)) { result.time.setUTCMilliseconds(result.ms); }
            return result;
        }

        return parseLeadingTime;
    }());

    /**
     * Parses a format code, and returns an instance of the class ParsedFormat
     * containing all properties of the format code needed to format numbers or
     * strings.
     *
     * @param {String} formatCode
     *  The format code to be parsed.
     *
     * @returns {ParsedFormat}
     *  An instance of the class ParsedFormat containing all properties of the
     *  format code needed to format numbers or strings.
     */
    Parser.parseFormatCode = _.memoize(function (formatCode) {

        // the resulting parsed format code
        var parsedFormat = new ParsedFormat(formatCode);

        // empty format code string is considered invalid
        parsedFormat.syntaxError = formatCode.length === 0;

        // stop if the entire format code has been consumed, or a syntax error was found
        while (!parsedFormat.syntaxError && _.isString(formatCode)) {
            formatCode = parsedFormat.parseSection(formatCode);
        }

        // initialize category of the format code (before adding default sections)
        var numSections = parsedFormat.numberSections;
        if (numSections.length === 0) {
            // 'text' category, if there are no numeric sections at all
            parsedFormat.category = 'text';
        } else if ((numSections.length === 1) && (numSections[0].category === 'standard')) {
            // 'standard' category only for simple 'General' format code (optionally with color)
            parsedFormat.category = 'standard';
        } else {
            // use the category of the numeric sections, if they are all equal
            var categories = _.unique(_.pluck(numSections, 'category'));
            parsedFormat.category = (categories.length === 1) ? categories[0] : 'custom';
        }

        // initialize default intervals of numeric code sections
        switch (numSections.length) {
            case 0:
                // text format only: format numbers as 'General' number format
                numSections.push(new FormatSection());
                numSections[0].pushToken('general');
                break;
            case 1:
                // one number section for all numbers
                break;
            case 2:
                // two number sections: positive and zero, negative
                numSections[0].setInterval('>=', 0, true);
                numSections[1].setInterval('<', 0, true);
                break;
            case 3:
                // three number sections: positive, negative, zero
                numSections[0].setInterval('>', 0, true);
                numSections[1].setInterval('<', 0, true);
                numSections[2].setInterval('=', 0, true);
                break;
            default:
                // invalid number of number sections
                parsedFormat.syntaxError = true;
        }

        // initialize missing text section
        if (!parsedFormat.textSection) {
            parsedFormat.textSection = new FormatSection();
            parsedFormat.textSection.pushToken('text');
            // use color of first number section, but only if it is the 'General' format code
            if (numSections[0].colorName && (parsedFormat.category === 'standard')) {
                parsedFormat.textSection.setColor(numSections[0].colorName);
            }
        } else if (parsedFormat.textSection.operator) {
            // text section with number interval is invalid
            parsedFormat.syntaxError = true;
        }

        return parsedFormat;
    });

    /**
     * Generates a date/time format code for CJK locales based on the passed
     * format code template.
     *
     * @param {String} formatCodeTemplate
     *  The format code template used to generate the date/time format code.
     *  May contain the meta tokens {YEAR}, {MONTH}, {DAY}, {HOUR}, {MINUTE},
     *  and {SECOND}, which will be replaced by the respective CJK ideographs
     *  according to the passed CJK script type (language); and the meta token
     *  {H} that will be replaced by an hour token according to the passed
     *  options.
     *
     * @param {String} script
     *  The CJK script type used to replace the meta tokens in the passed
     *  format code template. MUST be one of:
     *  - 'trad': CJK Unified Ideographs (Traditional Chinese, Japanese).
     *  - 'simp': CJK Unified Ideographs (Simplified Chinese).
     *  - 'hang': Hangul alphabet (Korean).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.ampm=false]
     *      If set to true, the token 'AM/PM' will be added in front of the
     *      hour token inserted for the meta token {H}.
     *  @param {Boolean} [options.long=false]
     *      If set to true, the hour token inserted for the meta token {H} will
     *      be 'hh', otherwise 'h'.
     *
     * @returns {String}
     *  The format code generated from the passed format code template.
     */
    Parser.generateCJKDateTimeFormat = function (formatCodeTemplate, script, options) {
        var chars = CJK_DATE_TIME_CHARACTERS[script];
        return formatCodeTemplate
            .replace(/{H}/g, ((options && options.ampm) ? 'AM/PM' : '') + ((options && options.long) ? 'hh' : 'h'))
            .replace(/{YEAR}/g, chars.Y)
            .replace(/{MONTH}/g, chars.M)
            .replace(/{DAY}/g, chars.D)
            .replace(/{HOUR}/g, chars.h)
            .replace(/{MINUTE}/g, chars.m)
            .replace(/{SECOND}/g, chars.s);
    };

    // exports ================================================================

    return Parser;

});
