/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/tk/locale/localedata', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/io',
    'gettext!io.ox/office/tk/main'
], function (Config, Utils, IO, gt) {

    'use strict';

    // the locale data cache for all locales, mapped by locale code
    var localeDataByCode = null;

    // the locale data cache for all locales, mapped by MS locale identifier
    var localeDataByLcid = null;

    // locale data of current UI locale
    var localeData = null;

    // static cache for JSON resources already loaded, mapped by resource module name
    var resourceCache = {};

    // the translated names of some supported locales, mapped by locale identifier
    var LANGUAGE_NAMES = {
        en_US: gt('English (US)'),
        en_GB: gt('English (UK)'),
        de_DE: gt('German'),
        fr_FR: gt('French'),
        es_ES: gt('Spanish'),
        cs_CZ: gt('Czech'),
        da_DK: gt('Danish'),
        nl_NL: gt('Dutch (Netherlands)'),
        fi_FI: gt('Finnish'),
        el_GR: gt('Greek'),
        hu_HU: gt('Hungarian'),
        it_IT: gt('Italian (Italy)'),
        ja_JP: gt('Japanese'),
        pl_PL: gt('Polish'),
        pt_PT: gt('Portuguese (Portugal)'),
        ro_RO: gt('Romanian'),
        ru_RU: gt('Russian'),
        sv_SE: gt('Swedish (Sweden)')
    };

    // private global functions ===============================================

    /**
     * Returns the settings for the specified UI locale.
     *
     * @param {String|Number} [locale]
     *  The locale code with underscore character (e.g. 'en_GB'), or a numeric
     *  MS locale identifier (as contained in the property 'lcid' of the locale
     *  data entries). If omitted, the locale data for the current UI locale
     *  will be returned.
     *
     * @returns {LocaleDataEntry|Null}
     *  The locale data entry for the specified locale. If no locale data is
     *  available for the passed locale code or MS locale identifier, null will
     *  be returned instead.
     */
    function getLocaleData(locale) {
        switch (typeof locale) {
            case 'undefined':
                return localeData;
            case 'string':
                return localeDataByCode[locale] || null;
            case 'number':
                return localeDataByLcid[locale] || null;
        }
        return null;
    }

    // class LocaleDataEntry ==================================================

    /**
     * Represents the locale data of a single locale.
     *
     * @constructor
     *
     * @property {String} lc
     *  The locale code, e.g. 'en_GB'.
     *
     * @property {String} language
     *  The lower-case language identifier of the locale code, e.g. 'en'.
     *
     * @property {String} country
     *  The upper-case region identifier of the locale code, e.g. 'GB'.
     *
     * @property {Number} lcid
     *  The MS locale identifier (positive integer).
     *
     * @property {String|Null} name
     *  The name of the language, translated to the current UI language; or
     *  null, if no translated language name is available.
     *
     * @property {String} dec
     *  The decimal separator used in numbers (single character).
     *
     * @property {String} group
     *  The group separator (a.k.a. thousands separator in most locales) used
     *  in numbers (single character).
     *
     * @property {String} dir
     *  The default writing direction (either 'ltr' for left-to-right, or 'rtl'
     *  for right-to-left).
     *
     * @property {Boolean} cjk
     *  Whether the locale represents one of the languages Chinese, Japanese,
     *  or Korean.
     *
     * @property {String} unit
     *  The default measurement unit (one of 'cm', 'mm', or 'in').
     *
     * @property {String} currency
     *  The default currency symbol (non-empty string).
     *
     * @property {String} shortDate
     *  The number format code for a short date (month as number).
     *
     * @property {String} longDate
     *  The number format code for a long date (with month name).
     *
     * @property {String} shortTime
     *  The number format code for a short time (without seconds).
     *
     * @property {String} longTime
     *  The number format code for a time (with seconds).
     *
     * @property {Boolean} leadingMonth
     *  Whether short dates will be formatted with the month in front of the
     *  day (regardless of the year's position).
     *
     * @property {Boolean} leadingYear
     *  Whether short dates will be formatted with the year in front of the
     *  day and month.
     *
     * @property {Boolean} hours24
     *  Whether the 24-hours time format is preferred over the 12-hours AM/PM
     *  time format.
     */
    function LocaleDataEntry(lc, jsonData) {

        // add all properties of the raw JSON data
        _.extend(this, jsonData);

        // add defaults for optional properties
        this.cjk = !!this.cjk;

        // add locale identifier, and split language/region codes
        var parts = lc.split('_');
        this.lc = lc;
        this.language = parts[0];
        this.country = parts[1];
        this.name = LANGUAGE_NAMES[lc] || null;

        // add other properties
        this.leadingMonth = /M.*D/.test(this.shortDate);
        this.leadingYear = /Y.*M/.test(this.shortDate);
        this.hours24 = !/AM\/PM/.test(this.longTime);

    } // class LocaleDataEntry

    // static class LocaleData ================================================

    /**
     * Provides various settings for the current UI locale, or for any other
     * supported locale.
     *
     * @property {String} LOCALE
     *  The locale code of the current UI locale, e.g. 'en_GB', provided as a
     *  convenience shortcut for 'LocaleData.get().lc'.
     *
     * @property {String} LANGUAGE
     *  The lower-case language identifier of the current UI locale, e.g. 'en',
     *  provided as a convenience shortcut for 'LocaleData.get().language'.
     *
     * @property {String} COUNTRY
     *  The upper-case country identifier of the current locale, e.g. 'GB',
     *  provided as a convenience shortcut for 'LocaleData.get().country'.
     *
     * @property {Number} LCID
     *  The MS locale identifier of the current UI locale (positive integer),
     *  provided as a convenience shortcut for 'LocaleData.get().lcid'.
     *
     * @property {String|Null} NAME
     *  The translated name of the current UI language; or null, if the
     *  translated language name is not available. Provided as a convenience
     *  shortcut for 'LocaleData.get().name'.
     *
     * @property {String} DEC
     *  The decimal separator of the current UI locale (single character),
     *  provided as a convenience shortcut for 'LocaleData.get().dec'.
     *
     * @property {String} GROUP
     *  The group separator (a.k.a. thousands separator in most locales) of the
     *  current UI locale (single character), provided as a convenience
     *  shortcut for 'LocaleData.get().group'.
     *
     * @property {String} DIR
     *  The default writing direction of the current UI language (either 'ltr'
     *  for left-to-right, or 'rtl' for right-to-left), provided as a
     *  convenience shortcut for 'LocaleData.get().dir'.
     *
     * @property {Boolean} CJK
     *  Whether the locale represents one of the languages Chinese, Japanese,
     *  or Korean, provided as a convenience shortcut for
     *  'LocaleData.get().cjk'.
     *
     * @property {String} UNIT
     *  The default measurement unit to be used in the current UI language
     *  (one of 'cm', 'mm', or 'in'), provided as a convenience shortcut for
     *  'LocaleData.get().unit'.
     *
     * @property {String} CURRENCY
     *  The default currency symbol of the current UI locale, provided as a
     *  convenience shortcut for 'LocaleData.get().currency'.
     *
     * @property {String} SHORT_DATE
     *  The number format code for a short date (month as number) of the
     *  current UI locale, provided as a convenience shortcut for
     *  'LocaleData.get().shortDate'.
     *
     * @property {String} LONG_DATE
     *  The number format code for a long date (with month name) of the current
     *  UI locale, provided as a convenience shortcut for
     *  'LocaleData.get().longDate'.
     *
     * @property {String} SHORT_TIME
     *  The number format code for a short time (without seconds) of the
     *  current UI locale, provided as a convenience shortcut for
     *  'LocaleData.get().shortTime'.
     *
     * @property {String} LONG_TIME
     *  The number format code for a time (with seconds) of the current UI
     *  locale, provided as a convenience shortcut for
     *  'LocaleData.get().longTime'.
     *
     * @property {Boolean} LEADING_MONTH
     *  Whether short dates will be formatted with the month in front of the
     *  day (regardless of the year's position) in the current UI locale.
     *  Provided as a convenience shortcut for 'LocaleData.get().leadingMonth'.
     *
     * @property {Boolean} LEADING_YEAR
     *  Whether short dates will be formatted with the year in front of the
     *  day and month in the current UI locale. Provided as a convenience
     *  shortcut for 'LocaleData.get().leadingYear'.
     *
     * @property {Boolean} HOURS24
     *  Whether the 24-hours time format is preferred over the 12-hours AM/PM
     *  time format in the current UI locale. Provided as a convenience
     *  shortcut for 'LocaleData.get().hours24'.
     */
    var LocaleData = {};

    // static methods ---------------------------------------------------------

    /**
     * Returns the map with the locale data entries of all supported locales.
     *
     * @returns {Object}
     *  The map with the locale data entries of all supported locales, mapped
     *  by locale identifier.
     */
    LocaleData.getAll = function () {
        return localeDataByCode;
    };

    /**
     * Returns the settings for the current UI locale, or for any locale
     * specified in the parameter.
     *
     * @param {String|Number} [locale]
     *  The locale code with underscore character (e.g. 'en_GB'), or a numeric
     *  MS locale identifier (as contained in the property 'lcid' of the locale
     *  data entries). If omitted, the locale data for the current UI locale
     *  will be returned.
     *
     * @returns {LocaleDataEntry}
     *  The locale data entry for the specified locale. If no locale data is
     *  available for the passed locale code or MS locale identifier, the
     *  locale data for 'en_US' will be returned instead.
     */
    LocaleData.get = function (locale) {
        var data = getLocaleData(locale);
        return data ? data : localeDataByCode.en_US;
    };

    /**
     * Returns the translated language name for the passed locale.
     *
     * @param {String|Number} [locale]
     *  The locale code with underscore character (e.g. 'en_GB'), or a numeric
     *  MS locale identifier (as contained in the property 'lcid' of the locale
     *  data entries). If omitted, the translated language name for the current
     *  UI locale will be returned.
     *
     * @returns {String|Null}
     *  The language name for the specified locale; or null if no translated
     *  name exists for the passed locale.
     */
    LocaleData.getLanguageName = function (locale) {
        var data = getLocaleData(locale);
        return data ? data.name : null;
    };

    /**
     * Loads the specified JSON resource module for the current UI locale. If
     * there is no resource module available for the current language, tries to
     * fall back to the language code 'en'. All JSON resources loaded once will
     * be cached internally, and will be returned immediately when calling this
     * method again.
     *
     * @param {String} modulePath
     *  The module path to the directory containing the JSON resource modules.
     *  The resource modules must be stored as JSON source file containing the
     *  plain language code as file name: '[modulePath]/[LANGUAGE].json'. The
     *  file must contain a JSON object with locale names as property keys, and
     *  the resource data as property values, e.g.:
     *      {
     *          "en": {...},
     *          "en_US": {...},
     *          "en_GB": {...}
     *      }
     *  The exact locale identifier (e.g. "en_GB") will be preferred over data
     *  mapped by a generic language name (e.g. "en"). The latter will be used
     *  as fall-back in case no data exists for the exact locale identifier
     *  (e.g., the data mapped by "en" will be loaded for the locale "en_CA" in
     *  the example above).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.merge=false]
     *      If set to true, the data mapped by the generic language name (e.g.
     *      "en"), and the data mapped by the exact locale identifier (e.g.
     *      "en_GB") will be merged.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the JSON object loaded from the
     *  specified resource file, or that will be rejected on any error (e.g.,
     *  resource file not found, or invalid JSON data).
     */
    LocaleData.loadResource = function (modulePath, options) {

        // tries to load the resource file for the passed locale
        function requireResource(language, country) {

            // load the JSON source file for the passed language
            return IO.loadJSON(modulePath + '/' + language).then(function (jsonData) {

                // if the file consists of a simple string, use it as fall-back locale
                var matches = null;
                if (_.isString(jsonData) && (matches = /^([a-z]+)(?:_([A-Z]+))?$/.exec(jsonData))) {
                    return requireResource(matches[1], matches[2]);
                }

                // IO.loadJSON() accepts any JSON data type
                if (!_.isObject(jsonData)) {
                    Utils.error('LocaleData.loadResource(): invalid JSON data');
                    return $.Deferred().reject(jsonData);
                }

                // validate resource data (must be a map with language and/or local identifiers
                var keyPattern = new RegExp(_.escapeRegExp(language) + '(_[A-Z]{2,})?');
                _.each(jsonData, function (map, key) {
                    if (!keyPattern.test(key)) {
                        Utils.error('LocaleData.loadResource(): invalid map key "' + key + '"');
                    }
                });

                // no country code available: return data for language code only
                if (!country) { return jsonData[language]; }

                // merge country specific resources (e.g. 'en_GB') over generic language resources (e.g. 'en')
                return Utils.getBooleanOption(options, 'merge', false) ?
                    Utils.extendOptions(jsonData[language], jsonData[language + '_' + country]) :
                    (jsonData[language + '_' + country] || jsonData[language]);
            },
            function (response) {
                // on error, fall back to en_US (unless English is already the requested language)
                return (language === 'en') ? response : requireResource('en', 'US');
            });
        }

        // the (pending, resolved, or rejected) promise is stored in the cache
        if (!(modulePath in resourceCache)) {
            resourceCache[modulePath] = requireResource(LocaleData.LANGUAGE, LocaleData.COUNTRY);
        }

        // return the promise containing the localized resource object
        return resourceCache[modulePath];
    };

    // exports ================================================================

    return IO.loadJSON('io.ox/office/tk/locale/localedata').then(function (jsonDataMap) {

        // store the map by locale code, add locale/language/country codes to the entries
        localeDataByCode = _.mapObject(jsonDataMap, function (jsonData, lc) {
            return new LocaleDataEntry(lc, jsonData);
        });

        // create the map by LCID
        localeDataByLcid = {};
        _.each(localeDataByCode, function (entry) {
            localeDataByLcid[entry.lcid] = entry;
        });

        // store data of current locale
        if (!(Config.LOCALE in localeDataByCode)) {
            Utils.error('LocaleData: unknown UI language: ' + Config.LOCALE);
        }
        localeData = LocaleData.get(Config.LOCALE);

        // convenience shortcuts for current locale
        LocaleData.LOCALE = localeData.lc;
        LocaleData.LANGUAGE = localeData.language;
        LocaleData.COUNTRY = localeData.country;
        LocaleData.LCID = localeData.lcid;
        LocaleData.NAME = localeData.name;
        LocaleData.DEC = localeData.dec;
        LocaleData.GROUP = localeData.group;
        LocaleData.DIR = localeData.dir;
        LocaleData.CJK = localeData.cjk;
        LocaleData.UNIT = localeData.unit;
        LocaleData.CURRENCY = localeData.currency;
        LocaleData.SHORT_DATE = localeData.shortDate;
        LocaleData.LONG_DATE = localeData.longDate;
        LocaleData.SHORT_TIME = localeData.shortTime;
        LocaleData.LONG_TIME = localeData.longTime;
        LocaleData.LEADING_MONTH = localeData.leadingMonth;
        LocaleData.LEADING_YEAR = localeData.leadingYear;
        LocaleData.HOURS24 = localeData.hours24;

        return LocaleData;
    });

});
