/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/field/simplefield', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/components/field/basefield'
], function (DateUtils, LocaleData, Utils, Operations, DOM, Position, DrawingFrame, BaseField) {

    'use strict';

    // class SimpleField =====================================================

    /**
     * An instance of this class represents the model for all simple fields in the
     * edited document.
     *
     * @constructor
     *
     * @extends Field
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function SimpleField(app) {

        var // self reference
            self = this,
            // collector for simple fields
            allSimpleFields = {},
            // counter for simple field Id
            simpleFieldID = 0,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // page layout object
            pageLayout = null,
            // change track object
            changeTrack = null,
            // number formatter object
            numberFormatter = null;

        // base constructors --------------------------------------------------

        BaseField.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one simple field in the collection (the model).
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} [target]
         *  The target, where the simple field is located.
         */
        function addSimpleFieldToCollection(fieldNode, target) {
            var $fieldNode = $(fieldNode),
                id = $fieldNode.data('sFieldId') ? $fieldNode.data('sFieldId') : 'sf' + simpleFieldID,
                isMarginalId = target && pageLayout.isIdOfMarginalNode(target),
                fieldAttrs = $fieldNode.data('attributes'),
                attrsField = null;

            // when fastload, pageNum and dateFormat formats needs to be set to field data
            if (fieldAttrs && fieldAttrs.field && fieldAttrs.field.pageNumFormat && !$fieldNode.data('pageNumFormat')) {
                $fieldNode.data('pageNumFormat', fieldAttrs.field.pageNumFormat);
            }
            if (fieldAttrs && fieldAttrs.field) {
                attrsField = fieldAttrs.field;
                if (attrsField.pageNumFormat && !$fieldNode.data('pageNumFormat')) {
                    $fieldNode.data('pageNumFormat', attrsField.pageNumFormat);
                }
                if (attrsField.dateFormat && !$fieldNode.data('dateFormat')) {
                    $fieldNode.data('dateFormat', attrsField.dateFormat);
                }

            }

            $fieldNode.data('sFieldId', id);
            allSimpleFields[id] = isMarginalId ? target : $fieldNode;

            simpleFieldID += 1;
        }

        /**
         * Removing one simple field with specified id from the collection (the model).
         *
         * @param {String} id
         *  The unique id of the simple field node.
         *
         */
        function removeSimpleFieldFromCollection(id) {
            delete allSimpleFields[id];
        }

        /**
         * Finding in a header or footer specified by the target the simple field
         * with the specified id.
         *
         * @param {String} id
         *  The id string.
         *
         * @param {String} target
         *  The target string to identify the header or footer node
         *
         * @returns {jQuery|null}
         *  The simple field node with the specified id and type, or null, if no such
         *  field exists.
         *
         */
        function getMarginalSimpleField(id, target) {
            var // the specified header or footer node
                marginalNode = null,
                // the searched complex field with the specified id
                field = null;

            marginalNode = model.getRootNode(target);

            if (marginalNode) {
                field = _.find(marginalNode.find(DOM.FIELD_NODE_SELECTOR), function (field) { return DOM.getSimpleFieldId(field) === id; });
            }

            return field ? $(field) : null;
        }

        /**
         * Update of content for simple field: Date
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of date
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.time=false]
         *      If it's time format only.
         */
        function updateDateTimeField(node, format, options) {
            var date = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow()),
                time = Utils.getBooleanOption(options, 'time', false);
            format = (format && format !== 'default') ? format : (time ? LocaleData.SHORT_TIME : LocaleData.SHORT_DATE);
            var formattedDate = numberFormatter.formatValue(date, format),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                dateFormat = DOM.getFieldDateTimeFormat(node),
                isMarginal = DOM.isMarginalNode(node),
                rootNode = isMarginal ? DOM.getClosestMarginalTargetNode(node) : selection.getRootNode(),
                target = isMarginal ? DOM.getTargetContainerId(rootNode) : '',
                startPos = Position.getOxoPosition(rootNode, node),
                operation;

            if (startPos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    if (isMarginal) {
                        pageLayout.enterHeaderFooterEditMode(rootNode);
                        selection.setNewRootNode(rootNode);
                    }
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: formattedDate, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    if (dateFormat) {
                        operation = _.extend(operation, { attrs: { field: { dateFormat: dateFormat } } });
                    }

                    model.applyOperations(operation);

                    selection.setTextSelection(Position.increaseLastIndex(startPos));
                });
            } else {
                Utils.error('simpleField.updateDateTimeField(): Wrong start postition: ', startPos);
            }
        }

        /**
         * Update the content of simple field: FILENAME
         *
         * @param {jQuery|Node} node
         *  Simple field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateFileNameField(node, format) {
            var fileName = app.getFullFileName(),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (format) {
                if ((/Lower/).test(format)) {
                    fileName = fileName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    fileName = fileName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    fileName = Utils.capitalize(fileName.toLowerCase());
                } else if ((/Caps/).test(format)) {
                    fileName = Utils.capitalizeWords(fileName);
                }
            }

            if (startPos && fileName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);

                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: fileName, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);

                    selection.setTextSelection(Position.increaseLastIndex(startPos));
                });
            } else {
                Utils.error('simpleField.updateFileNameField(): Wrong start postition or text: ', startPos, fileName);
            }
        }

        /**
         * Update the content of simple field: AUTHOR
         *
         * @param {jQuery|Node} node
         *  Simple field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateAuthorField(node, format) {
            var authorName = app.getClientOperationName(),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (format) {
                if ((/Lower/).test(format)) {
                    authorName = authorName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    authorName = authorName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    authorName = Utils.capitalize(authorName.toLowerCase());
                } else if ((/Caps/).test(format)) {
                    authorName = Utils.capitalizeWords(authorName);
                }
            }

            if (startPos && authorName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);

                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: authorName, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);

                    selection.setTextSelection(Position.increaseLastIndex(startPos));
                });
            } else {
                Utils.error('simpleField.updateAuthorField(): Wrong start postition or text: ', startPos, authorName);
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Public method to add simple field to model.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         * @param {String} [target]
         *  The target, where the simple field is located.
         *
         */
        this.addSimpleFieldToCollection = function (fieldNode, target) {
            addSimpleFieldToCollection(fieldNode, target);
        };

        /**
         * Public method to remove simple field from model.
         *
         * @param {HTMLElement|jQuery} field
         *  One simple field node.
         *
         */
        this.removeSimpleFieldFromCollection = function (node) {
            removeSimpleFieldFromCollection(DOM.getSimpleFieldId(node));
        };

        /**
         * Whether the document contains complex fields in the document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one complex field.
         */
        this.isEmpty = function () {
            return _.isEmpty(allSimpleFields);
        };

        /**
         * Public method to get collection of all simple fields from model.
         *
         * @returns {Object}
         */
        this.getAllFields = function () {
            return allSimpleFields;
        };

        /**
         * Gets simple field from collection by passed id.
         *
         * @param {String} id
         *  Id of queried field.
         *
         * @returns {jQuery|null}
         *  The simple field with the specified id, or null, if no such start marker exists.
         *
         */
        this.getSimpleField = function (id) {
            var // the value saved under the specified id
                field = allSimpleFields[id] || null;

            return _.isString(field) ? getMarginalSimpleField(id, field) : field;
        };

        /**
         * After deleting a complete paragraph, it is necessary, that all simple field nodes
         * in the deleted paragraph are also removed from the model collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.removeSimpleFieldsInNode = function (paragraph) {
            var // all simple field nodes inside the paragraph
                simpleFieldNodes = $(paragraph).find(DOM.FIELD_NODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(simpleFieldNodes, function (field) {

                var // the id of the marker node
                    fieldId = DOM.getSimpleFieldId(field);

                if (_.isString(fieldId)) {
                    removeSimpleFieldFromCollection(fieldId);
                }
            });
        };

        /**
         * Update simple field depending of type, by given instruction.
         *
         * @param {jQuery|Node} node
         *  Simple field node
         *
         * @param {String} instruction
         *  Simple field instruction, containing type,
         *  formating and optionally style, separated by \ .
         *  (In case of ODF, instruction is empty, and we fetch possible
         *  date/time format from data property of node).
         */
        this.updateByInstruction = function (node, instruction) {
            var parsedFormatInstruction,
                dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type;
            instruction = dividedData.instruction;

            if (type) {
                if (self.isCurrentDate(type)) {
                    parsedFormatInstruction = DOM.getFieldDateTimeFormat(node) || instruction;
                    updateDateTimeField(node, parsedFormatInstruction);
                } else if (self.isCurrentTime(type)) {
                    parsedFormatInstruction = DOM.getFieldDateTimeFormat(node) || instruction;
                    updateDateTimeField(node, parsedFormatInstruction, { time: true });
                } else if (self.isFileName(type)) {
                    updateFileNameField(node, instruction);
                } else if (self.isAuthor(type)) {
                    updateAuthorField(node, instruction);
                } else if (self.isNumPages(type)) {
                    self.updateNumPagesField(node, instruction);
                } else if (self.isPageNumber(type)) {
                    self.updatePageNumberField(node, instruction);
                }
            }
        };

        /**
         * Update of content for simple field: Page number
         *
         * @param {jQuery|Node} node
         *  Simple field node.
         *
         * @param {String} format
         *  Format of the number.
         */
        this.updatePageNumberField = function (node, format) {
            var number = pageLayout.getPageNumber(node),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (model.isHeaderFooterEditState()) { // no explicit update of pageNumber field in header/footer!
                return;
            }

            number = self.formatPageFieldInstruction(number, format);

            if (startPos && number) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: number, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);
                });
            } else {
                Utils.error('complexField.updatePageNumberField(): Wrong start postition or text: ', startPos, number);
            }
        };

        /**
         * Update of content for simple field: Number of pages
         *
         * @param {jQuery|Node} node
         *   Simple field node
         *
         * @param {String} format
         *  Format of the number.
         */
        this.updateNumPagesField = function (node, format) {
            var numPages = pageLayout.getNumberOfDocumentPages(),
                rootNode = selection.getRootNode(),
                startPos = Position.getOxoPosition(rootNode, node),
                operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                target = DOM.isMarginalNode(node) ? DOM.getTargetContainerId(rootNode) : '',
                operation;

            if (model.isHeaderFooterEditState()) { // no explicit update of numPages field in header/footer!
                return;
            }

            numPages = self.formatPageFieldInstruction(numPages, format, true);

            if (startPos && numPages) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: numPages, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    model.applyOperations(operation);
                });
            } else {
                Utils.error('complexField.updateNumPagesField(): Wrong start postition or text: ', startPos, numPages);
            }
        };

        /**
         * Update date and time fields on document load, or before running download or print action.
         *
         * @param {jQuery|Node} field
         *  Field to be updated.
         *
         * @param {String} instruction
         *  Instruction for updating.
         */
        this.updateDateTimeField = function (field, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type;
            instruction = dividedData.instruction;

            if (type) {
                if ((self.isCurrentDate(type) || self.isCurrentTime(type))) {
                    instruction = app.isODF() ? DOM.getFieldDateTimeFormat(field) : instruction;
                    updateDateTimeField(field, instruction, { time: self.isCurrentTime(type) });
                }
            }
        };

        /**
         * After load from local storage or fast load the collectors for the simple fields need to be filled.
         */
        this.refreshSimpleFields = function () {
            var // the page content node of the document
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // all simple field nodes in the document
                allSimpleFieldNodes = pageContentNode.find(DOM.FIELD_NODE_SELECTOR),
                // a collector for all marginal template nodes
                allMargins = pageLayout.getHeaderFooterPlaceHolder().children(),
                // the target string node
                target = '';

            // reset model
            allSimpleFields = {};

            _.each(allSimpleFieldNodes, function (fieldNode) {
                if (!DOM.isMarginalNode($(fieldNode).parent())) {
                    addSimpleFieldToCollection(fieldNode);
                }
            });

            _.each(allMargins, function (margin) {
                var // a collector for all range marker nodes inside one margin
                    allMarginSimpleFieldNodes = $(margin).find(DOM.FIELD_NODE_SELECTOR);

                if (allMarginSimpleFieldNodes.length > 0) {
                    target = DOM.getTargetContainerId(margin);
                    _.each(allMarginSimpleFieldNodes, function (oneMarginalField) {
                        addSimpleFieldToCollection(oneMarginalField, target);
                    });
                    pageLayout.replaceAllTypesOfHeaderFooters();
                }
            });

        };

        /**
         * After splitting a paragraph, it is necessary, that all simple field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateSimpleFieldCollector = function (para) {

            var // whether each single node needs to be checked
                checkMarginal = false,
                // all simple fields inside the paragraph
                allFields = null;

            // not necessary for paragraphs in header or footer -> only the target are stored, not the nodes
            if (DOM.isMarginalNode(para)) { return; }

            // if this is not a paragraph, each single node need to be checked
            checkMarginal = !DOM.isParagraphNode(para);

            allFields = $(para).find(DOM.FIELD_NODE_SELECTOR);

            // update  the complex field nodes in the collection objects
            _.each(allFields, function (fieldNode) {
                if (!checkMarginal || !DOM.isMarginalNode(fieldNode)) {
                    // simply overwriting the old simple fields with the new values
                    addSimpleFieldToCollection(fieldNode);
                }
            });
        };

        /**
         * On document load, this method loops all fields in collection
         * to find date/time fields and update them.
         */
        this.updateDateTimeFieldOnLoad = function () {
            _.each(allSimpleFields, function (id, fieldEntry) {
                var field = self.getSimpleField(fieldEntry);
                if (!DOM.isFixedSimpleField(field, app.isODF()) && (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field))) {
                    // if previously updated field is marginal, and this is not, leave header edit state
                    if (!DOM.isMarginalNode(field) && model.isHeaderFooterEditState()) {
                        pageLayout.leaveHeaderFooterEditMode();
                        selection.setNewRootNode(model.getNode());
                    }
                    self.updateDateTimeField(field, DOM.getFieldInstruction(field));
                    if (pageLayout.isIdOfMarginalNode(id) && model.isHeaderFooterEditState() && DOM.isHeaderOrFooter(selection.getRootNode())) {
                        pageLayout.getHeaderFooterPlaceHolder().children('.' + id).empty().append(selection.getRootNode().children().clone(true));
                    }
                }
            });
            if (model.isHeaderFooterEditState()) {
                pageLayout.leaveHeaderFooterEditMode();
                selection.setNewRootNode(model.getNode());
            }
            selection.setTextSelection(selection.getFirstDocumentPosition());
        };

        /**
         * Public method for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNode = function ($node, isHeader, pageCount) {
            if (isHeader) {
                if (DOM.isHeaderWrapper($node.parent())) {
                    $node.find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(1);
                } else {
                    _.each($node.find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                        $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + 1);
                    });
                }
            } else {
                if (DOM.isFooterWrapper($node.parent())) {
                    $node.find(DOM.PAGENUMBER_FIELD_SELECTOR).children().empty().html(pageCount);
                } else {
                    _.each($node.find(DOM.PAGENUMBER_FIELD_SELECTOR), function (field) {
                        $(field).children().empty().html($(field).parentsUntil('.pagecontent', '.page-break').data('page-num'));
                    });
                }
            }
        };

        /**
         * Public method for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNode = function ($node, pageCount) {
            $node.find(DOM.PAGECOUNT_FIELD_SELECTOR).children().empty().html(pageCount);
        };

        /**
         * Insert simple field (usually ODF format, or some legacy docx).
         *
         * @param {String} fieldType
         *
         * @param {String} fieldFormat
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.isFixed='true']
         *      If it's fixed date field.
         */
        this.insertField = function (fieldType, fieldFormat, options) {

            return model.getUndoManager().enterUndoGroup(function () {
                function doInsertField() {
                    var start = selection.getStartPosition(),
                        generator = model.createOperationsGenerator(),
                        operation = {},
                        // target for operation - if exists, it's for ex. header or footer
                        target = model.getActiveTarget(),
                        representation = null,
                        isFixed = Utils.getBooleanOption(options, 'isFixed', true);

                    model.doCheckImplicitParagraph(start);

                    if (fieldType) {
                        operation = { start: start, type: fieldType };

                        if (self.isCurrentDate(fieldType) || self.isCurrentTime(fieldType)) {
                            if (!fieldFormat || fieldFormat === 'default') {
                                fieldFormat = LocaleData.SHORT_DATE;
                            }
                            representation = self.getDateTimeRepresentation(fieldFormat);
                            if (app.isODF()) {
                                operation.attrs = operation.attrs || {};
                                operation.attrs.field = { dateFormat: fieldFormat, fixed: JSON.stringify(isFixed) };
                            } else {
                                operation.type = fieldType + ' \\@ ' + fieldFormat;
                            }
                        } else if (self.isFileName(fieldType) || self.isAuthor(fieldType)) {
                            representation = self.isFileName(fieldType) ? app.getFullFileName() : app.getClientOperationName();
                            if ((/Lower/i).test(fieldFormat)) {
                                representation = representation.toLowerCase();
                            } else if ((/Upper/i).test(fieldFormat)) {
                                representation = representation.toUpperCase();
                            } else if ((/FirstCap/i).test(fieldFormat)) {
                                representation = Utils.capitalize(representation.toLowerCase());
                            } else if ((/Caps/i).test(fieldFormat)) {
                                representation = Utils.capitalizeWords(representation);
                            }
                            if (app.isODF()) {
                                if (self.isFileName(fieldType)) {
                                    operation.type = 'file-name';
                                } else {
                                    operation.type = 'creator';
                                }
                            }
                            // TODO format for this types of fields
                        } else if (self.isNumPages(fieldType)) {
                            representation = self.formatPageFieldInstruction(pageLayout.getNumberOfDocumentPages(), fieldFormat, true);
                            if (app.isODF()) {
                                operation.attrs = operation.attrs || {};
                                operation.attrs.field = { pageNumFormat: fieldFormat };
                                operation.type = 'page-count';
                            } else {
                                operation.type = fieldType + ' \\* ' + fieldFormat;
                            }
                        } else if (self.isPageNumber(fieldType)) {
                            representation = self.formatPageFieldInstruction(pageLayout.getPageNumber(), fieldFormat);
                            if (app.isODF()) {
                                operation.attrs = operation.attrs || {};
                                operation.attrs.field = { pageNumFormat: fieldFormat };
                                operation.type = 'page-number';
                            } else {
                                operation.type = fieldType + ' \\* ' + fieldFormat;
                            }
                        }

                        operation.representation = representation;
                        model.extendPropertiesWithTarget(operation, target);

                        // modifying the attributes, if changeTracking is activated
                        if (changeTrack.isActiveChangeTracking()) {
                            operation.attrs = operation.attrs || {};
                            operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                        }

                        generator.generateOperation(Operations.FIELD_INSERT, operation);
                        model.applyOperations(generator.getOperations());

                        selection.setTextSelection(Position.increaseLastIndex(start));
                    }
                }

                if (selection.hasRange()) {
                    return model.deleteSelected()
                    .done(function () {
                        doInsertField();
                    });
                }

                doInsertField();
                return $.when();
            });
        };

        /**
        * Update date field with value from popup, datepicker or input field.
        * @param {jQuery} node
        *   Field node that is going to be updated
        * @param {String} fieldValue
        *   Value with which will field be updated.
        * @param {String|null} standardizedDate
        *   Odf needs this standard date value for fixed fields.
        */
        this.updateDateFromPopupValue = function (node, fieldValue, standardizedDate) {
            var operationType = $(node).data('type'),
                operationAttrs = $(node).data('attributes'),
                dateFormat = DOM.getFieldDateTimeFormat(node),
                isMarginal = DOM.isMarginalNode(node),
                rootNode = isMarginal ? DOM.getClosestMarginalTargetNode(node) : selection.getRootNode(),
                target = isMarginal ? DOM.getTargetContainerId(rootNode) : '',
                startPos = Position.getOxoPosition(rootNode, node),
                operation;

            if (startPos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    model.deleteRange(startPos);
                    operation = { name: Operations.FIELD_INSERT, start: startPos, representation: fieldValue, type: operationType };
                    if (target) {
                        operation = _.extend(operation, { target: target });
                    }
                    if (operationAttrs) {
                        operation = _.extend(operation, { attrs: operationAttrs });
                    }
                    if (dateFormat) {
                        operation = _.extend(operation, { attrs: { field: { dateFormat: dateFormat, fixed: 'true', dateValue: standardizedDate } } });
                    }

                    model.applyOperations(operation);
                    selection.setTextSelection(Position.increaseLastIndex(startPos));
                });
            } else {
                Utils.error('simpleField.updateDateFromPopupValue(): Wrong start postition: ', startPos);
            }
        };

        /**
         * Handler for updateField operation.
         *
         * @param {Number[]} start
         *  The logical start position for the new simple field.
         *
         * @param {String} type
         *  Type of the simple field.
         *
         * @param {String} representation
         *  Content of the field that's updated.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the simple field has been inserted successfully.
         */
        this.updateSimpleFieldHandler = function (start, type, representation, attrs, target) {
            var undoManager = model.getUndoManager(),
                undoOperation,
                redoOperation,
                fieldNode = Position.getDOMPosition(model.getRootNode(target), start, true),
                odfFixedField;

            fieldNode = fieldNode.node || null;

            if (attrs && fieldNode) {
                if (attrs.character && !_.isUndefined(attrs.character.autoDateField)) {
                    model.getCharacterStyles().setElementAttributes(fieldNode, attrs); // This is needed to remove property set in SetAttributes operation, in data of the element!
                    $(fieldNode).attr('data-auto-date', attrs.character.autoDateField);

                    // create the undo/redo actions
                    if (undoManager.isUndoEnabled()) {
                        undoOperation = { name: Operations.FIELD_UPDATE, start: start, type: type, attrs: { character: { autoDateField: !attrs.character.autoDateField } }, representation: representation };
                        redoOperation = { name: Operations.FIELD_UPDATE, start: start, type: type, attrs: { character: { autoDateField: attrs.character.autoDateField } }, representation: representation };
                        if (target) {
                            undoOperation.target = target;
                            redoOperation.target = target;
                        }
                        undoManager.addUndo(undoOperation, redoOperation);
                    }
                } else if (attrs.field && !_.isUndefined(attrs.field.fixed)) { // odf
                    odfFixedField = attrs.field.fixed;
                    $(fieldNode).data('fixed', odfFixedField);
                    var tempAttrs = { field: { fixed: odfFixedField === 'true' ? 'false' : 'true' } };
                    var redoTempAttrs = { field: { fixed: odfFixedField } };
                    if (_.isObject($(fieldNode).data().attributes) && _.isObject($(fieldNode).data().attributes.field)) {
                        $(fieldNode).data().attributes.field.fixed = odfFixedField;
                    } else {
                        $(fieldNode).data('attributes', tempAttrs);
                    }

                    // create the undo/redo actions
                    if (undoManager.isUndoEnabled()) {
                        undoOperation = { name: Operations.FIELD_UPDATE, start: start, type: type, attrs: tempAttrs, representation: representation };
                        redoOperation = { name: Operations.FIELD_UPDATE, start: start, type: type, attrs: redoTempAttrs, representation: representation };
                        if (target) {
                            undoOperation.target = target;
                            redoOperation.target = target;
                        }
                        undoManager.addUndo(undoOperation, redoOperation);
                    }
                }
            }
            return true;
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            selection = model.getSelection();
            pageLayout = model.getPageLayout();
            changeTrack = model.getChangeTrack();
            numberFormatter = model.getNumberFormatter();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = selection = pageLayout = changeTrack = numberFormatter = null;
        });

    } // class SimpleField

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class BaseField
    return BaseField.extend({ constructor: SimpleField });

});
