/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/rangearray', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/arraytemplate',
    'io.ox/office/spreadsheet/utils/address',
    'io.ox/office/spreadsheet/utils/interval',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/intervalarray'
], function (Utils, ArrayTemplate, Address, Interval, Range, IntervalArray) {

    'use strict';

    // private static functions ===============================================

    /**
     * Divides the passed cell range addresses into a sorted array of row band
     * objects. Each row band contains the row interval, and a merged array of
     * column intervals representing the parts of the original ranges covered
     * by the respective row band (see method IntervalArray.merge() for details
     * about merged index intervals).
     *
     * @param {RangeArray} ranges
     *  An array of cell range addresses.
     *
     * @returns {IntervalArray}
     *  An array of row intervals representing all passed cell range addresses.
     *  Each row interval object contains the additional property 'intervals'
     *  representing the parts of the original ranges covered by the row band.
     */
    function getRowBandsFromRanges(ranges) {

        var // the cell ranges mapped by row interval (used for the resulting row bands)
            rowBands = new IntervalArray(),
            // the start array index in bandRanges used to search for matching row intervals
            startIndex = 0;

        // creates a new row band instance
        function createRowBand(row1, row2, ranges) {
            var rowBand = new Interval(row1, row2);
            rowBand.ranges = ranges;
            return rowBand;
        }

        // work on a clone sorted by start row index
        ranges = ranges.clone().sort(function (range1, range2) { return range1.start[1] - range2.start[1]; });

        // build the array of row bands
        ranges.forEach(function (range) {

            var // the start and end row index of the range
                row1 = range.start[1],
                row2 = range.end[1];

            // update the row bands containing the range
            for (var bandIndex = startIndex; bandIndex < rowBands.length; bandIndex += 1) {

                // the current row band
                var rowBand = rowBands[bandIndex];

                // row band is above the range, ignore it in the next iterations
                if (rowBand.last < row1) {
                    startIndex = bandIndex + 1;
                    continue;
                }

                // row band needs to be split (range starts inside the row band)
                if (rowBand.first < row1) {
                    rowBands.splice(bandIndex, 0, createRowBand(rowBand.first, row1 - 1, rowBand.ranges.clone()));
                    rowBand.first = row1;
                    startIndex = bandIndex + 1;
                    // next iteration of the for loop will process the current row band again
                    continue;
                }

                // row band needs to be split (range ends inside the row band)
                if (row2 < rowBand.last) {
                    rowBands.splice(bandIndex + 1, 0, createRowBand(row2 + 1, rowBand.last, rowBand.ranges.clone()));
                    rowBand.last = row2;
                }

                // now, current row band contains the range completely
                rowBand.ranges.push(range);
                // continue with next range, if end of range found
                if (row2 === rowBand.last) { return; }
                // adjust start row index in case a new row band needs to be appended
                row1 = rowBand.last + 1;
            }

            // no row band found, create and append a new row band
            rowBands.push(createRowBand(row1, row2, new RangeArray(range)));
        });

        // create the column intervals for all row bands
        rowBands.forEach(function (rowBand) {
            rowBand.intervals = rowBand.ranges.colIntervals().merge();
            delete rowBand.ranges;
        });
        return rowBands;
    }

    /**
     * Converts the passed array of row bands to an array cell range addresses.
     *
     * @param {IntervalArray} rowBands
     *  An array of row bands, as returned for example by the static function
     *  getRowBandsFromRanges(). Each array element must contain an additional
     *  property 'intervals' containing the column intervals of the row band.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses resulting from the passed row bands.
     */
    function getRangesFromRowBands(rowBands) {

        var // resulting array of cell range addresses
            resultRanges = new RangeArray(),
            // current ranges whose last row ends in the current row band
            bandRanges = [];

        // appends the passed ranges to the result array
        function appendResultRanges(ranges) {
            resultRanges = resultRanges.concat(ranges);
        }

        // returns whether the passed range precedes the specified column interval,
        // or whether it starts at the interval, but ends at another column
        function isPrecedingUnextendableRange(range, interval) {
            return (range.start[0] < interval.first) || ((range.start[0] === interval.first) && (range.end[0] !== interval.last)) || ((interval.first < range.start[0]) && (range.start[0] <= interval.last));
        }

        // build ranges from the row bands and their column intervals
        rowBands.forEach(function (rowBand, bandIndex) {

            var // the preceding row band
                prevRowBand = rowBands[bandIndex - 1],
                // array index into the bandRanges array
                bandRangeIndex = 0;

            // try to extend ranges from the previous row band with ranges from the current row band
            if (prevRowBand && (prevRowBand.last + 1 === rowBand.first)) {

                // process all intervals of the current row band, extend ranges or move them to the result array
                rowBand.intervals.forEach(function (colInterval) {

                    // move all ranges from bandRanges to resultRanges that cannot be merged with the current interval anymore
                    // (taking advantage of the fact that all ranges in bandRanges are distinct and ordered by column)
                    while ((bandRangeIndex < bandRanges.length) && isPrecedingUnextendableRange(bandRanges[bandRangeIndex], colInterval)) {
                        appendResultRanges(bandRanges.splice(bandRangeIndex, 1));
                    }

                    // extend the current band range to this row band, if column interval matches
                    var bandRange = bandRanges[bandRangeIndex];
                    if (bandRange && (bandRange.start[0] === colInterval.first) && (bandRange.end[0] === colInterval.last)) {
                        bandRange.end[1] = rowBand.last;
                    } else {
                        bandRanges.splice(bandRangeIndex, 0, Range.createFromIntervals(colInterval, rowBand));
                    }
                    bandRangeIndex += 1;
                });

                // move all remaining unextended ranges from bandRanges to resultRanges
                appendResultRanges(bandRanges.splice(bandRangeIndex));

            } else {

                // store all old band ranges in the result array, create new ranges for this row band
                appendResultRanges(bandRanges);
                bandRanges = RangeArray.createFromColIntervals(rowBand.intervals, rowBand.first, rowBand.last);
            }
        });

        // append remaining band ranges to the result
        appendResultRanges(bandRanges);
        return resultRanges;
    }

    // class RangeArray =======================================================

    /**
     * Represents an array of cell range addresses. The array elements are
     * instances of the class Range.
     *
     * @constructor
     *
     * @extends Array
     */
    var RangeArray = ArrayTemplate.create(Range);

    // static methods ---------------------------------------------------------

    /**
     * Creates an array of cell range addresses from the passed column and row
     * intervals, by combining each column interval with each row interval.
     *
     * @param {IntervalArray|Interval} colIntervals
     *  The column intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @param {IntervalArray|Interval} rowIntervals
     *  The row intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses created from the passed intervals.
     */
    RangeArray.createFromIntervals = function (colIntervals, rowIntervals) {
        var result = new RangeArray();
        IntervalArray.forEach(rowIntervals, function (rowInterval) {
            IntervalArray.forEach(colIntervals, function (colInterval) {
                result.push(Range.createFromIntervals(colInterval, rowInterval));
            });
        });
        return result;
    };

    /**
     * Creates an array of cell range addresses from the passed column
     * intervals and row indexes.
     *
     * @param {IntervalArray|Interval} colIntervals
     *  The column intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter.
     *
     * @param {Number} firstRow
     *  The index of the first row in all new cell range addresses.
     *
     * @param {Number} [lastRow]
     *  The index of the last row in all new cell range addresses. If omitted,
     *  all ranges will cover a single row only.
     *
     * @returns {RangeArray}
     *  The array of cell range addresses created from the passed column
     *  intervals and row indexes.
     */
    RangeArray.createFromColIntervals = function (colIntervals, firstRow, lastRow) {
        return RangeArray.createFromIntervals(colIntervals, new Interval(firstRow, lastRow));
    };

    /**
     * Creates an array of cell range addresses from the passed row intervals
     * and column indexes.
     *
     * @param {IntervalArray|Interval} rowIntervals
     *  The row intervals of the new cell range addresses. This method also
     *  accepts a single index interval as parameter. Note that this method is
     *  one of the rare cases where row parameters precede column parameters,
     *  in order to provide an optional column parameter.
     *
     * @param {Number} firstCol
     *  The index of the first column in all new cell range addresses.
     *
     * @param {Number} [lastCol]
     *  The index of the last row in all new cell range addresses. If omitted,
     *  all ranges will cover a single column only.
     *
     * @returns {Range}
     *  The array of cell range addresses created from the passed row intervals
     *  and column indexes.
     */
    RangeArray.createFromRowIntervals = function (rowIntervals, firstCol, lastCol) {
        return RangeArray.createFromIntervals(new Interval(firstCol, lastCol), rowIntervals);
    };

    /**
     * Creates an array of cell range addresses from the passed array of single
     * cell addresses. The cell ranges in the resulting array will be merged,
     * i.e. no cell range address overlaps with or adjoins exactly to another
     * cell range address.
     *
     * @param {AddressArray} addresses
     *  An unordered array of cell addresses to be merged to cell range
     *  addresses.
     *
     * @returns {RangeArray}
     *  A merged array of cell range addresses from the passed cell addresses.
     */
    RangeArray.mergeAddresses = function (addresses) {

        // simplifications for small arrays
        switch (addresses.length) {
            case 0:
                return new RangeArray();
            case 1:
                return new RangeArray(new Range(addresses[0]));
        }

        // the row bands containing column intervals for all existing rows
        var rowBands = new IntervalArray();

        // sort and unify the cell addresses
        addresses = addresses.unify().sort();

        // create row bands with all rows that contains any cells
        addresses.forEach(function (address) {

            var // the last row band in the array of all row bands
                lastRowBand = _.last(rowBands),
                // the last interval in the current row band
                lastInterval = null;

            // create a new row band on demand
            if (!lastRowBand || (lastRowBand.first !== address[1])) {
                rowBands.push(lastRowBand = new Interval(address[1]));
                lastRowBand.intervals = new IntervalArray();
            }

            // get the last interval, extend it or create a new interval
            lastInterval =  _.last(lastRowBand.intervals);
            if (lastInterval && (lastInterval.last + 1 === address[0])) {
                lastInterval.last += 1;
            } else {
                lastRowBand.intervals.push(new Interval(address[0]));
            }
        });

        // convert the row bands to an array of cell ranges
        return getRangesFromRowBands(rowBands);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the number of cells covered by all cell range address.
     *
     * @returns {Number}
     *  The number of cells covered by all cell range address.
     */
    RangeArray.prototype.cells = function () {
        return this.reduce(function (count, range) { return count + range.cells(); }, 0);
    };

    /**
     * Returns whether a cell range address in this array contains the passed
     * column index.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed column
     *  index.
     */
    RangeArray.prototype.containsCol = function (col) {
        return this.some(function (range) { return range.containsCol(col); });
    };

    /**
     * Returns whether a cell range address in this array contains the passed
     * row index.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed row
     *  index.
     */
    RangeArray.prototype.containsRow = function (row) {
        return this.some(function (range) { return range.containsRow(row); });
    };

    /**
     * Returns whether a range address in this array contains the passed column
     * or row index.
     *
     * @param {Number} index
     *  The column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to check for a column index (true), or a row index (false).
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed column
     *  or row index.
     */
    RangeArray.prototype.containsIndex = function (index, columns) {
        return columns ? this.containsCol(index) : this.containsRow(index);
    };

    /**
     * Returns whether a cell range address in this array contains the passed
     * cell address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether a cell range address in this array contains the passed cell
     *  address.
     */
    RangeArray.prototype.containsAddress = function (address) {
        return this.some(function (range) { return range.containsAddress(address); });
    };

    /**
     * Returns whether all cell range addresses in the passed array are
     * contained completely in any of the cell range addresses in this array.
     *
     * @param {RangeArray|Range} ranges
     *  The other cell range addresses to be checked. This method also accepts
     *  a single cell range address as parameter.
     *
     * @returns {Boolean}
     *  Whether all cell range addresses in the passed array are contained
     *  completely in any of the cell range addresses in this array.
     */
    RangeArray.prototype.contains = function (ranges) {
        return RangeArray.every(ranges, function (range2) {
            return this.some(function (range1) {
                return range1.contains(range2);
            });
        }, this);
    };

    /**
     * Returns whether any cell range address in this array overlaps with any
     * cell range address in the passed array.
     *
     * @param {RangeArray|Range} ranges
     *  The other cell range addresses to be checked. This method also accepts
     *  a single cell range address as parameter.
     *
     * @returns {Boolean}
     *  Whether any cell range address in this array overlaps with any cell
     *  range address in the passed array.
     */
    RangeArray.prototype.overlaps = function (ranges) {
        return this.some(function (range1) {
            return RangeArray.some(ranges, function (range2) {
                return range1.overlaps(range2);
            });
        });
    };

    /**
     * Returns whether any pair of cell range addresses in this array overlap
     * each other.
     *
     * @returns {Boolean}
     *  Whether any pair of cell range addresses in this array overlap each
     *  other.
     */
    RangeArray.prototype.overlapsSelf = function () {

        // simplifications for small arrays
        switch (this.length) {
            case 0:
            case 1:
                return false;
            case 2:
                return this[0].overlaps(this[1]);
        }

        // work on sorted clones for performance (check adjacent array elements)
        var ranges = this.clone().sort(function (range1, range2) { return range1.start[1] - range2.start[1]; });
        return ranges.some(function (range1, index) {
            var result = false;
            Utils.iterateArray(ranges, function (range2) {
                // a range with distinct row interval and all following ranges cannot overlap anymore
                if (range1.end[1] < range2.start[1]) { return Utils.BREAK; }
                if ((result = range1.overlaps(range2))) { return Utils.BREAK; }
            }, { begin: index + 1 });
            return result;
        });
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed column index.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed
     *  column index; or null, if no range has been found.
     */
    RangeArray.prototype.findByCol = function (col) {
        return _.find(this, function (range) { return range.containsCol(col); }) || null;
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed row index.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed row
     *  index; or null, if no range has been found.
     */
    RangeArray.prototype.findByRow = function (row) {
        return _.find(this, function (range) { return range.containsRow(row); }) || null;
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed column or row index.
     *
     * @param {Number} index
     *  The zero-based column or row index to be checked.
     *
     * @param {Boolean} columns
     *  Whether to search for a column index (true), or a row index (false).
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed
     *  column or row index; or null, if no range has been found.
     */
    RangeArray.prototype.findByIndex = function (index, columns) {
        return columns ? this.findByCol(index) : this.findByRow(index);
    };

    /**
     * Returns the first cell range address in this array that contains the
     * passed cell address.
     *
     * @param {Address} address
     *  The cell address to be checked.
     *
     * @returns {Range|Null}
     *  The first cell range address in this array that contains the passed
     *  cell address; or null, if no range has been found.
     */
    RangeArray.prototype.findByAddress = function (address) {
        return _.find(this, function (range) { return range.containsAddress(address); }) || null;
    };

    /**
     * Returns the column intervals of all cell range addresses in this array.
     *
     * @returns {IntervalArray}
     *  The column intervals of all cell range addresses in this array.
     */
    RangeArray.prototype.colIntervals = function () {
        return IntervalArray.invoke(this, 'colInterval');
    };

    /**
     * Returns the row intervals of all cell range addresses in this array.
     *
     * @returns {IntervalArray}
     *  The row intervals of all cell range addresses in this array.
     */
    RangeArray.prototype.rowIntervals = function () {
        return IntervalArray.invoke(this, 'rowInterval');
    };

    /**
     * Returns the column or row intervals of all cell range addresses in this
     * array.
     *
     * @param {Boolean} columns
     *  Whether to return the column intervals (true), or the row intervals
     *  (false) of the cell range address.
     *
     * @returns {IntervalArray}
     *  The column or row intervals of all cell range addresses in this array.
     */
    RangeArray.prototype.intervals = function (columns) {
        return columns ? this.colIntervals() : this.rowIntervals();
    };

    /**
     * Returns the address of the bounding cell range of all range addresses in
     * this array (the smallest range that contains all ranges in the array).
     *
     * @returns {Range|Null}
     *  The address of the bounding cell range containing all range addresses;
     *  or null, if this range array is empty.
     */
    RangeArray.prototype.boundary = function () {

        switch (this.length) {
            case 0:
                return null;
            case 1:
                // this method must not return original array elements
                return this[0].clone();
        }

        var result = this[0].clone();
        this.forEach(function (range) {
            result.start[0] = Math.min(result.start[0], range.start[0]);
            result.start[1] = Math.min(result.start[1], range.start[1]);
            result.end[0] = Math.max(result.end[0], range.end[0]);
            result.end[1] = Math.max(result.end[1], range.end[1]);
        });
        return result;
    };

    /**
     * Returns a shallow copy of this cell range address array that does not
     * contain any duplicate cell range addresses.
     *
     * @returns {RangeArray}
     *  A copy of this cell range address array without any duplicates.
     */
    RangeArray.prototype.unify = function () {
        var result = new RangeArray(), map = {};
        this.forEach(function (range) {
            var key = range.key();
            if (!map[key]) {
                result.push(range);
                map[key] = true;
            }
        });
        return result;
    };

    /**
     * Merges the cell range addresses in this array so that no cell range
     * address in the result overlaps with or adjoins exactly to another cell
     * range address.
     *
     * @returns {RangeArray}
     *  An array of merged cell range addresses exactly covering the cell range
     *  addresses in this array.
     */
    RangeArray.prototype.merge = function () {
        // at most one range: return directly instead of running the complex row band code
        return (this.length <= 1) ? this.clone(true) : getRangesFromRowBands(getRowBandsFromRanges(this));
    };

    /**
     * Returns the cell range addresses covered by this array, and the passed
     * array of cell range addresses. More precisely, returns the existing
     * intersection ranges from all pairs of the cross product of the arrays of
     * cell range addresses.
     *
     * @param {RangeArray|Range} ranges
     *  The other cell range addresses to be intersected with the cell range
     *  addresses in this array. This method also accepts a single cell range
     *  address as parameter.
     *
     * @returns {RangeArray}
     *  The cell range addresses covered by this array and the passed array.
     */
    RangeArray.prototype.intersect = function (ranges) {
        var result = new RangeArray();
        this.forEach(function (range1) {
            RangeArray.forEach(ranges, function (range2) {
                var range = range1.intersect(range2);
                if (range) { result.push(range); }
            });
        });
        return result;
    };

    /**
     * Returns the difference of the cell range addresses in this array, and
     * the passed cell range addresses (all cell range addresses contained in
     * this array, that are not contained in the passed array).
     *
     * @param {RangeArray|Range} ranges
     *  The cell range addresses to be removed from this array. This method
     *  also accepts a single cell range address as parameter.
     *
     * @returns {RangeArray}
     *  All cell range addresses that are contained in this array, but not in
     *  the passed array. May be an empty array, if the passed cell range
     *  addresses completely cover all cell range addresses in this array.
     */
    RangeArray.prototype.difference = function (ranges) {

        // the resulting ranges
        var result = this;

        // reduce ranges in 'result' with each range from the passed array
        RangeArray.some(ranges, function (range2) {

            // initialize the arrays for this iteration
            var source = result;
            result = new RangeArray();

            // process each range in the source range list
            source.forEach(function (range1) {

                // check if the ranges cover each other at all
                if (!range1.overlaps(range2)) {
                    result.push(range1);
                    return;
                }

                // do nothing if range2 covers range1 completely (delete range1 from the result)
                if (range2.contains(range1)) {
                    return;
                }

                // if range1 starts above range2, extract the upper part of range1
                if (range1.start[1] < range2.start[1]) {
                    result.push(Range.create(range1.start[0], range1.start[1], range1.end[0], range2.start[1] - 1));
                }

                // if range1 starts left of range2, extract the left part of range1
                if (range1.start[0] < range2.start[0]) {
                    result.push(Range.create(range1.start[0], Math.max(range1.start[1], range2.start[1]), range2.start[0] - 1, Math.min(range1.end[1], range2.end[1])));
                }

                // if range1 ends right of range2, extract the right part of range1
                if (range1.end[0] > range2.end[0]) {
                    result.push(Range.create(range2.end[0] + 1, Math.max(range1.start[1], range2.start[1]), range1.end[0], Math.min(range1.end[1], range2.end[1])));
                }

                // if range1 ends below range2, extract the lower part of range1
                if (range1.end[1] > range2.end[1]) {
                    result.push(Range.create(range1.start[0], range2.end[1] + 1, range1.end[0], range1.end[1]));
                }
            });

            // early exit the loop, if all source ranges have been deleted already
            return result.length === 0;
        });

        // ensure to return a clone
        return (result === this) ? this.clone() : result;
    };

    /**
     * Returns a shortened array of cell range addresses containing at most as
     * many cells as specified. If a cell range address in this array does not
     * fit completely due to the remaining number of allowed cells, it will be
     * shortened to the appropriate number of upper rows of the range. If the
     * range cannot be shortened exactly to entire rows, it will be split into
     * two ranges (the latter range being the leading part of an inner row).
     * Overlapping cell range addresses will NOT be merged.
     *
     * Example: Shortening the ranges A1:C3,D1:F3 to 14 cells results in the
     * ranges A1:C3,D1:F1,D2:E2.
     *
     * @param {Number} maxCells
     *  The maximum number of cells allowed in the resulting array.
     *
     * @returns {RangeArray}
     *  The shortened array of cell range addresses.
     */
    RangeArray.prototype.shortenTo = function (maxCells) {

        // the resulting cell ranges
        var result = new RangeArray();

        Utils.iterateArray(this, function (range) {

            // no more cells left (e.g., previous range has occupied the last cell)
            if (maxCells <= 0) { return Utils.BREAK; }

            var // number of cells in the current range
                cells = range.cells();

            // push entirely fitting range, continue with next range
            if (cells <= maxCells) {
                result.push(range);
                maxCells -= cells;
                return;
            }

            // shorten the range to entire rows
            var rows = Math.floor(maxCells / range.cols());
            if (rows > 0) {
                range = range.clone();
                range.end[1] = range.start[1] + rows - 1;
                result.push(range);
                maxCells -= range.cols() * rows;
            }

            // adjust the range to refer to the split row only
            range = range.clone();
            range.start[1] = range.end[1] = range.start[1] + rows;

            // add remaining part of the last row
            if (maxCells > 0) {
                range.end[0] = range.start[0] + maxCells - 1;
                result.push(range);
            }

            // last range has been pushed partly, stop iteration
            return Utils.BREAK;
        });

        return result;
    };

    // exports ================================================================

    return RangeArray;

});
