/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/tokenarray', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/sheetref',
    'io.ox/office/spreadsheet/model/formula/cellref',
    'io.ox/office/spreadsheet/model/formula/tokens'
], function (Utils, ModelObject, SheetUtils, FormulaUtils, SheetRef, CellRef, Tokens) {

    'use strict';

    var // convenience shortcuts
        Address = SheetUtils.Address,
        Range3DArray = SheetUtils.Range3DArray;

    // class TokenArray =======================================================

    /**
     * Represents a formula expression that has been parsed and split into
     * single formula tokens (instances of the class BaseToken). Provides
     * support for parsing formula expressions and handling of the resulting
     * formula token arrays.
     *
     * Instances of this class trigger the following events:
     * - 'change:tokens'
     *      After the token array has been changed (manually, or by parsing a
     *      formula), or cleared.
     * - 'change:token'
     *      After a single token in the token array has been changed. The event
     *      handlers receive a reference to the changed token, and the array
     *      index of the token.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetModel|SheetModel} parentModel
     *  The spreadsheet document model, or a specific sheet model containing
     *  this instance. The parent model determines the default reference sheet
     *  index used by various methods of this class.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options supported by the base class
     *  ModelObject. Additionally, the following options are supported:
     *  @param {String} [initOptions.type='generic']
     *      The type of the token array. May cause special behavior in specific
     *      situations. The following types are supported:
     *      - 'generic': Default type. No special behavior.
     *      - 'name': The token array represents the definition of a defined
     *          name. Reference tokens that will be located outside the sheet
     *          when relocating them will be wrapped at the sheet borders.
     *  @param {Boolean} [initOptions.temp=false]
     *      If set to true, the token array will not register itself as
     *      document listener for sheet operations, and will not trigger any
     *      change events (the value 'never' will be passed automatically for
     *      the option 'trigger' to the base class ModelObject). Can be used as
     *      performance optimization in class implementations having to deal
     *      with formula expressions.
     */
    function TokenArray(parentModel, initOptions) {

        var // self reference
            self = this,

            // the document model
            docModel = parentModel.getDocModel(),

            // the sheet model of sheet-local names
            sheetModel = (parentModel === docModel) ? null : parentModel,

            // the type of the token array, needed for some special behavior
            objectType = Utils.getStringOption(initOptions, 'type', 'generic'),

            // whether to wrap relocated range addresses at the sheet borders
            wrapReferences = objectType === 'name',

            // whether the token array is only used temporarily (no event handling)
            temp = Utils.getBooleanOption(initOptions, 'temp', false),

            // the array of all formula tokens
            tokens = [],

            // cached tokens compiled to prefix notation
            compiled = null,

            // whether this token array is currently interpreted (to prevent recursion)
            interpreting = false;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel, temp ? Utils.extendOptions(initOptions, { trigger: 'never' }) : initOptions);

        // private methods ----------------------------------------------------

        /**
         * Invokes the specified method at all tokens that implement it, and
         * triggers a 'change:tokens' event, if any of the methods has returned
         * true.
         */
        function invokeForAllTokensAndTrigger(methodName) {

            var // whether any token has returned true
                changed = false,
                // the arguments to be passed to the token method
                args = _.toArray(arguments).slice(1);

            // invoke method for all tokens, and collect changed state
            tokens.forEach(function (token) {
                if (_.isFunction(token[methodName]) && token[methodName].apply(token, args)) { changed = true; }
            });

            // notify listeners
            if (changed) { self.trigger('change:tokens'); }
            return changed;
        }

        /**
         * Returns the index of the sheet that contains this token array. If
         * this token array is owned by the document itself, the value of the
         * option 'refSheet' will be returned instead.
         *
         * @returns {Number|Null}
         *  The index of the sheet that contains this token array, if such a
         *  sheet has been passed as parent model to the constructor; or the
         *  value of the option 'refSheet', if this is a global token array
         *  with the document model as parent.
         */
        function resolveRefSheet(options) {
            return sheetModel ? sheetModel.getIndex() : Utils.getIntegerOption(options, 'refSheet', null);
        }

        /**
         * Extends the passed options received by various public methods of
         * this instance, according to the type of the token array. Adds the
         * boolean option 'wrapReferences', and inserts the own reference sheet
         * index as option 'refSheet' if available.
         */
        function extendWithOwnOptions(options) {
            return _.extend({}, options, { refSheet: resolveRefSheet(options), wrapReferences: wrapReferences });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether this token array is empty.
         *
         * @returns {Boolean}
         *  Whether this token array is empty.
         */
        this.empty = function () {
            return tokens.length === 0;
        };

        /**
         * Removes all formula tokens from this token array, and triggers a
         * 'change:tokens' event.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.clear = function () {
            if (tokens.length > 0) {
                tokens = [];
                compiled = null;
                this.trigger('change:tokens');
            }
            return this;
        };

        /**
         * Inserts the passed formula tokens into this token array, and
         * triggers a 'change:tokens' event.
         *
         * @param {Array<BaseToken>} newTokens
         *  The new formula tokens to be inserted into this token array.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.setTokens = function (newTokens) {
            tokens = newTokens.slice();
            compiled = null;
            this.trigger('change:tokens');
            return this;
        };

        /**
         * Appends the passed formula token to the token array, and triggers a
         * 'change:token' event for this token.
         *
         * @param {Token} token
         *  The new formula token. This token array will take ownership.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendToken = function (token) {
            tokens.push(token);
            compiled = null;
            this.trigger('change:token', token, tokens.length - 1);
            return this;
        };

        /**
         * Appends a new reference token to this token array, that will be
         * built from the passed cell range address.
         *
         * @param {Range} range
         *  The address of the cell range to be added as reference token to
         *  this token array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.abs=false]
         *      If set to true, all column and row references will be marked as
         *      absolute (leading dollar signs).
         *  @param {Number} [options.sheet]
         *      The zero-based index of the sheet the reference token will
         *      point to. If omitted, no sheet reference will be inserted into
         *      the reference token (sheet-local reference).
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendRange = function (range, options) {

            var // whether the range spans entire columns or rows
                colRange = docModel.isColRange(range),
                rowRange = docModel.isRowRange(range),
                // the cell references
                abs = Utils.getBooleanOption(options, 'abs', false),
                cell1Ref = CellRef.create(range.start, abs || rowRange, abs || (colRange && !rowRange)),
                cell2Ref = range.single() ? null : CellRef.create(range.end, abs || rowRange, abs || (colRange && !rowRange)),
                // the sheet reference
                sheet = Utils.getIntegerOption(options, 'sheet'),
                sheetRef = (typeof sheet === 'number') ? new SheetRef(sheet, true) : null;

            return this.appendToken(new Tokens.ReferenceToken(docModel, cell1Ref, cell2Ref, sheetRef));
        };

        /**
         * Appends new reference tokens to this token array, that will be built
         * from the passed cell range addresses. If the passed range array
         * contains more than one range address, the reference tokens will be
         * separated by list operator tokens.
         *
         * @param {RangeArray} ranges
         *  The addresses of the cell ranges to be added as reference tokens to
         *  this token array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.abs=false]
         *      If set to true, all column and row references will be marked as
         *      absolute (leading dollar signs).
         *  @param {Number} [options.sheet]
         *      The zero-based index of the sheet the reference tokens will
         *      point to. If omitted, no sheet references will be inserted into
         *      the reference tokens (sheet-local references).
         *  @param {Boolean} [options.parentheses=false]
         *      If set to true, and the passed array contains more than one
         *      range, the entire range list will be enclosed into parentheses.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.appendRangeList = function (ranges, options) {

            var // whether to enclose the range list into parentheses
                parentheses = (ranges.length > 1) && Utils.getBooleanOption(options, 'parentheses', false);

            if (parentheses) {
                this.appendToken(new Tokens.ParenthesisToken(true));
            }

            ranges.forEach(function (range, index) {
                if (index > 0) { this.appendToken(new Tokens.SeparatorToken()); }
                this.appendRange(range, options);
            }, this);

            if (parentheses) {
                this.appendToken(new Tokens.ParenthesisToken(false));
            }

            return this;
        };

        /**
         * Parses the specified formula expression, and creates an array of
         * formula tokens held by this instance.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to parse the
         *  passed formula expression. See class GrammarConfig for details.
         *
         * @param {String} formula
         *  The formula expression to be parsed, without the leading equality
         *  sign.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  Tokenizer.parseFormula().
         *
         * @returns {Array<TokenDescriptor>}
         *  An array of descriptors for all parsed formula tokens, as received
         *  from the method Tokenizer.parseFormula().
         */
        this.parseFormula = function (grammar, formula, options) {

            // parse the formula string, extract all formula tokens, trigger change event
            var tokenDescs = docModel.getFormulaTokenizer().parseFormula(grammar, formula, options);
            this.setTokens(_.pluck(tokenDescs, 'token'));

            // return the complete token descriptors
            return tokenDescs;
        };

        /**
         * Transforms the cell ranges in this token array according to the
         * specified column or row operation.
         *
         * @param {Number} sheet
         *  The zero-based index of the changed sheet.
         *
         * @param {Interval} interval
         *  The column/row interval of the operation, with the index properties
         *  'first' and 'last'.
         *
         * @param {Boolean} insert
         *  Whether the specified column/row interval has been inserted into
         *  the sheet (true), or deleted from the sheet (false).
         *
         * @param {Boolean} columns
         *  Whether the specified interval is a column interval (true), or a
         *  row interval (false).
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.transformRanges = function (sheet, interval, insert, columns) {
            return invokeForAllTokensAndTrigger('transformRange', sheet, interval, insert, columns, resolveRefSheet());
        };

        /**
         * Relocates the cell ranges in this token array to a new reference
         * address by adjusting the relative column and row components in all
         * reference tokens.
         *
         * @param {Address} refAddress
         *  The address of the original reference cell the formula is related
         *  to.
         *
         * @param {Address} targetAddress
         *  The address of the new reference cell the formula will be related
         *  to.
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.relocateRanges = function (refAddress, targetAddress) {
            return refAddress.differs(targetAddress) && invokeForAllTokensAndTrigger('relocateRange', refAddress, targetAddress, wrapReferences);
        };

        /**
         * Changes all tokens with sheet references containing the specified
         * old sheet index to the new sheet index. Used for example when
         * copying existing sheets, and adjusting all references and defined
         * names that point to the original sheet.
         *
         * @param {Number} oldSheet
         *  The zero-based index of the original sheet.
         *
         * @param {Number} newSheet
         *  The zero-based index of the target sheet.
         *
         * @returns {Boolean}
         *  Whether any token in this token array has been changed.
         */
        this.relocateSheet = function (oldSheet, newSheet) {
            return (oldSheet !== newSheet) && invokeForAllTokensAndTrigger('relocateSheet', oldSheet, newSheet);
        };

        /**
         * Returns the string representation of the formula represented by this
         * token array.
         *
         * @param {String} grammar
         *  The identifier of the formula grammar to be used to generate the
         *  formula expression. See class GrammarConfig for details.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Address} [options.refAddress]
         *      The address of the original reference cell the formula is
         *      related to. If omitted, cell A1 will be used instead. If this
         *      address is different to the address resulting from the option
         *      'targetAddress', the generated formula expression will be
         *      modified, as if the token array has been relocated according to
         *      these two addresses (without actually modifying the token
         *      array).
         *  @param {Address} [options.targetAddress]
         *      The address of the target cell for relocation. If omitted, cell
         *      A1 will be used instead. If this address is different to the
         *      address resulting from the option 'refAddress', the generated
         *      formula expression will be modified, as if the token array has
         *      been relocated according to these two addresses (without
         *      actually modifying the token array).
         *
         * @returns {String}
         *  The string representation of the formula.
         */
        this.getFormula = function (grammar, options) {

            var // the grammar configuration expected by the formula tokens
                config = docModel.getGrammarConfig(grammar),
                // prepare the options for reference relocation
                newOptions = extendWithOwnOptions(options);

            // concatenate the string representations of all formula tokens
            return tokens.reduce(function (formula, token) { return formula + token.getText(config, newOptions); }, '');
        };

        /**
         * Invokes a callback function used to modify the specified token in
         * this token array. Afterwards, a 'change:token' event will be
         * triggered.
         *
         * @param {Number} index
         *  The array index of the token to be modified.
         *
         * @param {String} type
         *  A string with a single type identifier that must match the type of
         *  the token. If the token at the passed index does not match the
         *  type, the callback function will not be invoked, and no event will
         *  be triggered.
         *
         * @param {Function} callback
         *  The callback function invoked for the specified token. Receives the
         *  following parameters:
         *  (1) {Token} token
         *      The token instance.
         *  (2) {Number} index
         *      The token index, as passed to the method modifyToken().
         *  Must return a Boolean value specifying whether the token has been
         *  modified. If the callback returns true, this token array will
         *  trigger a 'change:token' event.
         *
         * @returns {TokenArray}
         *  A reference to this instance.
         */
        this.modifyToken = function (index, type, callback, context) {

            // the token to be modified
            var token = tokens[index];

            // invoke the callback, trigger change event if callback returns true
            if (token && token.isType(type) && callback.call(context, token, index)) {
                this.trigger('change:token', token, index);
            }

            return this;
        };

        /**
         * Invokes the passed callback function for all tokens contained in
         * this token array.
         *
         * @param {Function} callback
         *  The callback function invoked for all tokens. Receives the
         *  following parameters:
         *  (1) {Token} token
         *      The current token.
         *  (2) {Number} index
         *      The zero-based token index.
         *  If the callback returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Object} [options.context]
         *      The calling context for the callback function.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the tokens will be visited in reversed order.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the callback has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateTokens = function (callback, options) {
            return Utils.iterateArray(tokens, function (token, index) {
                return callback.call(this, token, index);
            }, options);
        };

        /**
         * Resolves the token array to a list of constant strings. The formula
         * represented by this token array must consist of one or more literal
         * string tokens, separated by list operators. Whitespace tokens will
         * be ignored.
         *
         * @returns {Array<String>|Null}
         *  The string literals contained in this token array. If the formula
         *  cannot be resolved successfully to a list of strings, this method
         *  returns null.
         */
        this.resolveStringList = function () {

            var // the resulting strings
                result = [],
                // a string with a single character reflecting the token types in the token array
                tokenTypes = '';

            // process all tokens as long as they are considered valid, collect token types
            tokens.every(function (token) {
                switch (token.getType()) {
                    case 'lit':
                        tokenTypes += 'L';
                        result.push(token.getValue());
                        return _.isString(_.last(result));
                    case 'sep':
                        tokenTypes += ',';
                        return true;
                    case 'ws':
                        return true;
                    default:
                        tokenTypes += 'I'; // invalid token
                        return false;
                }
            });

            // on success, check the formula structure
            return (/^L(,L)*$/).test(tokenTypes) ? result : null;
        };

        /**
         * Resolves the token array to a list of range addresses. The formula
         * represented by this token array must be completely resolvable to a
         * range list (single reference, a list of references separated by the
         * list operator, optionally enclosed in parentheses, defined names
         * that resolve to a range list if specified with the option
         * 'resolveNames', also recursively, or a mixed list of defined names
         * and references). To extract the range addresses contained in an
         * arbitrary formula, the method TokenArray.extractRanges() can be
         * used.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the original reference sheet that will be used if
         *      this token array is related to the document (does not have its
         *      own reference sheet).
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, will return the range addresses referred by all
         *      defined names contained in this token array.
         *
         * @returns {Range3DArray}
         *  The addresses of the resulting cell ranges, including their sheet
         *  indexes. If the formula cannot be resolved successfully to a range
         *  list, this method returns an empty array.
         */
        this.resolveRangeList = function (options) {

            // the resulting range addresses
            var resultRanges = new Range3DArray();

            // prevent recursive processing of the token array, e.g. in defined name
            if (interpreting) { return resultRanges; }

            // whether to resolve defined name tokens
            var resolveNames = Utils.getBooleanOption(options, 'resolveNames', false);

            // add more options (refSheet and wrapReferences)
            options = extendWithOwnOptions(options);

            // extracts the cell range address from the passed reference token
            function resolveReferenceRange(refToken) {

                var // the cell range address, will be null for invalid ranges (#REF! errors)
                    range = refToken.getRange3D(options),
                    // whether the token contains a valid range (formula must not contain multi-sheet references)
                    valid = _.isObject(range) && range.singleSheet();

                if (valid) { resultRanges.push(range); }
                return valid;
            }

            // extracts all cell range addresses from the passed name token
            function resolveNamedRanges(nameToken) {

                // the model of the defined name referred by the passed token
                var nameModel = resolveNames ? nameToken.resolveNameModel(options.refSheet) : null;
                if (!nameModel) { return false; }

                // defined names are always defined relative to cell A1
                var newOptions = _.clone(options);
                newOptions.refAddress = Address.A1;

                // prevent recursive invocation of this token array via other defined names
                interpreting = true;
                var ranges = nameModel.getTokenArray().resolveRangeList(newOptions);
                interpreting = false;

                // append the new descriptors to the result
                resultRanges.append(ranges);
                return !ranges.empty();
            }

            // process all tokens as long as they are considered valid, build a signature
            // string with a single character for each token to check the formula syntax
            var tokenTypes = '';
            tokens.every(function (token) {
                switch (token.getType()) {
                    case 'ref':
                        tokenTypes += 'R';
                        return resolveReferenceRange(token);
                    case 'name':
                        tokenTypes += 'R';
                        return resolveNamedRanges(token);
                    case 'open':
                        tokenTypes += '<';
                        return true;
                    case 'close':
                        tokenTypes += '>';
                        return true;
                    case 'sep':
                        tokenTypes += ',';
                        return true;
                    case 'ws':
                        return true;
                    default:
                        tokenTypes += 'I'; // invalid token
                        return false;
                }
            });

            // reset resulting ranges if formula structure is invalid
            return (/^(R(,R)*|<R(,R)*>)$/).test(tokenTypes) ? resultRanges : resultRanges.clear();
        };

        /**
         * Returns the addresses of all cell ranges this token array refers to.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the original reference sheet that will be used if
         *      this token array is related to the document (does not have its
         *      own reference sheet).
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *  @param {Boolean} [options.resolveNames=false]
         *      If set to true, will return the range addresses referred by all
         *      defined names contained in this token array.
         *
         * @returns {Array<Object>}
         *  The addresses of all cell ranges contained in this token array with
         *  additional information, in an array of objects with the following
         *  properties:
         *  - {Range3D} range
         *      The cell range address represented by a formula token, with
         *      sheet indexes.
         *  - {Number} index
         *      The internal array index of the token containing the range
         *      (may be equal for different objects in the array, for example
         *      for a single name token referring to multiple cell ranges).
         *  - {String} type
         *      The type of the token containing the range. Will be either
         *      'ref' for a regular reference token, or 'name' for a defined
         *      name that evaluates to one or more ranges in the specified
         *      sheet (only if option 'resolveNames' is set, see above).
         */
        this.extractRanges = function (options) {

            // prevent recursive processing of the token array, e.g. in defined name
            if (interpreting) { return []; }

            var // descriptors for the resulting ranges
                rangeInfos = [],
                // whether to resolve defined name tokens
                resolveNames = Utils.getBooleanOption(options, 'resolveNames', false);

            // add more options (refSheet and wrapReferences)
            options = extendWithOwnOptions(options);

            // extracts the cell range address from the passed reference token
            function extractReferenceRange(refToken, tokenIndex) {
                var range = refToken.getRange3D(options); // returns null for invalid ranges (#REF! errors)
                if (range) { rangeInfos.push({ range: range, index: tokenIndex, type: 'ref' }); }
            }

            // extracts all cell range addresses from the passed name token
            function extractNamedRanges(nameToken, tokenIndex) {

                // the model of the defined name referred by the passed token
                var nameModel = resolveNames ? nameToken.resolveNameModel(options.refSheet) : null;
                if (!nameModel) { return false; }

                // defined names are always defined relative to cell A1
                var newOptions = _.clone(options);
                newOptions.refAddress = Address.A1;

                // prevent recursive invocation of this token array via other defined names
                interpreting = true;
                var nameRangeInfos = nameModel.getTokenArray().extractRanges(newOptions);
                interpreting = false;

                // insert the range descriptors with adjusted properties
                nameRangeInfos.forEach(function (rangeInfo) {
                    rangeInfos.push({ range: rangeInfo.range, index: tokenIndex, type: 'name' });
                });
            }

            tokens.forEach(function (token, index) {
                switch (token.getType()) {
                    case 'ref':
                        extractReferenceRange(token, index);
                        break;
                    case 'name':
                        extractNamedRanges(token, index);
                        break;
                }
            });

            return rangeInfos;
        };

        /**
         * Compiles the formula tokens contained in this token array to prefix
         * notation. See class Compiler for details.
         *
         * @returns {Object}
         *  The result descriptor of the compilation process. See method
         *  Compiler.compileTokens() for details.
         */
        this.compileFormula = function () {
            // compile the infix tokens to prefix notation, and cache the result
            return compiled || (compiled = docModel.getFormulaCompiler().compileTokens(tokens));
        };

        /**
         * Calculates the result of the formula represented by this token
         * array.
         *
         * @param {String} contextType
         *  The formula context type influencing the type of the formula
         *  result. See method Interpreter.interpretTokens() for details.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.refSheet]
         *      The index of the original reference sheet that will be used if
         *      this token array is related to the document (does not have its
         *      own reference sheet). Used to resolve reference tokens and
         *      defined names without explicit sheet reference.
         *  @param {Address} [options.refAddress]
         *      The source reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as source reference cell.
         *  @param {Address} [options.targetAddress]
         *      The target reference address used to relocate reference tokens
         *      with relative column/row components. If omitted, cell A1 will
         *      be used as target reference cell.
         *
         * @returns {Object}
         *  The result descriptor for the formula. See description of the
         *  method Interpreter.interpretTokens() for details.
         */
        this.interpretFormula = function (contextType, options) {

            // compile the infix tokens to prefix notation
            var compileInfo = this.compileFormula();

            // check compiler result
            if (compileInfo.error) {
                return { type: 'error', value: compiled.error };
            }

            // prevent recursive interpretation
            if (interpreting) {
                return { type: 'warn', value: 'circular' };
            }

            // interpret the formula with extended options
            try {
                interpreting = true;
                return docModel.getFormulaInterpreter().interpretTokens(contextType, compileInfo.tokens, extendWithOwnOptions(options));
            } finally {
                interpreting = false;
            }
        };

        // initialization -----------------------------------------------------

        // reparse formula after the document has been imported and all sheets exist
        if (!this.isImportFinished()) {
            (function () {

                // rescue the original parseFormula() method
                var parseFormula = this.parseFormula;

                // do not parse if document is still importing (referenced sheets may be missing)
                // store formula text in a bad token which will be reparsed after import is done
                this.parseFormula = function (grammar, formula) {
                    FormulaUtils.log('TokenArray.parseFormula(): caching formula: ' + formula);
                    tokens = [new Tokens.FixedToken('cache', grammar + ':' + formula)];
                    return this;
                };

                // parse formula after import, and restore the original method (use waitForSuccess() -
                // the token array may be destroyed before import finishes, e.g. operations that
                // delete the object containing this token array)
                this.waitForImportSuccess(function () {
                    this.parseFormula = parseFormula;
                    if ((tokens.length === 1) && tokens[0].isType('cache')) {
                        var text = tokens[0].getValue(), sepPos = text.indexOf(':');
                        this.parseFormula(text.substr(0, sepPos), text.substr(sepPos + 1));
                    }
                }, this);
            }.call(this));
        }

        // refresh all sheet references after the collection of sheets has been changed
        if (!temp) {
            this.listenTo(docModel, 'insert:sheet:after', function (event, sheet) { invokeForAllTokensAndTrigger('transformSheet', sheet, null); });
            this.listenTo(docModel, 'delete:sheet:after', function (event, sheet) { invokeForAllTokensAndTrigger('transformSheet', null, sheet); });
            this.listenTo(docModel, 'move:sheet:after', function (event, to, from) { invokeForAllTokensAndTrigger('transformSheet', to, from); });
            this.listenTo(docModel, 'rename:sheet', function (event, sheet) { invokeForAllTokensAndTrigger('renameSheet', sheet); });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = parentModel = docModel = tokens = compiled = null;
        });

    } // class TokenArray

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: TokenArray });

});
