/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/spreadsheet/model/formula/resource', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/desccollection'
], function (Utils, BaseObject, LocaleData, ErrorCode, FormulaUtils, DescriptorCollection) {

    'use strict';

    // internal implementations of all operators
    var OPERATOR_IMPL_MAP = null;

    // internal implementations of all functions
    var FUNCTION_IMPL_MAP = null;

    // raw formula resource data loaded from JSON
    var RAW_RESOURCE_DATA = null;

    // the native prefix characters for R1C1 reference notation
    var NATIVE_R1C1_PREFIX_CHARS = 'RC';

    // global cache of formula resource promises, mapped by file format
    var FORMULA_RESOURCE_CACHE = {};

    // maps the keys of error codes to their native names used in file format
    var NATIVE_ERROR_NAMES = {
        NULL: '#NULL!',
        DIV0: '#DIV/0!',
        VALUE: '#VALUE!',
        REF: '#REF!',
        NAME: '#NAME?',
        NUM: '#NUM!',
        NA: '#N/A',
        DATA: '#GETTING_DATA'
    };

    // maps resource keys of operators to their local names
    var LOCAL_OPERATOR_NAMES = {
        add: '+',
        sub: '-',
        mul: '*',
        div: '/',
        pow: '^',
        con: '&',
        pct: '%',
        lt: '<',
        le: '<=',
        gt: '>',
        ge: '>=',
        eq: '=',
        ne: '<>',
        list: (LocaleData.DEC === ',') ? ';' : ',',
        isect: ' ',
        range: ':'
    };

    // maps native operator names of the CalcEngine to local names (native names as keys)
    var CE_LOCAL_OPERATOR_NAMES = {
        '+': '+',
        '-': '-',
        '*': '*',
        '/': '/',
        '^': '^',
        '&': '&',
        '%': '%',
        '<': '<',
        '<=': '<=',
        '>': '>',
        '>=': '>=',
        '=': '=',
        '<>': '<>',
        ',': null, // list operator needs to be added later according to CalcEngine data received from server
        ' ': ' ',
        ':': ':'
    };

    // imports ================================================================

    // load localized resource data
    var resourcePromise = LocaleData.loadResource('io.ox/office/spreadsheet/resource/formula', { merge: true }).done(function (data) {
        RAW_RESOURCE_DATA = data;
    });

    // import all operator implementations
    var operatorPromise = require([
        'io.ox/office/spreadsheet/model/formula/impl/operators'
    ], function (OPERATORS) {
        OPERATOR_IMPL_MAP = OPERATORS;
    });

    // import all function implementations
    var functionPromise = require([
        'io.ox/office/spreadsheet/model/formula/impl/complexfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/conversionfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/databasefuncs',
        'io.ox/office/spreadsheet/model/formula/impl/datetimefuncs',
        'io.ox/office/spreadsheet/model/formula/impl/engineeringfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/financialfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/informationfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/logicalfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/mathfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/matrixfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/referencefuncs',
        'io.ox/office/spreadsheet/model/formula/impl/statisticalfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/textfuncs',
        'io.ox/office/spreadsheet/model/formula/impl/webfuncs'
    ], function () {
        // merge all imported function descriptors into a single map
        FUNCTION_IMPL_MAP = _.extend.apply(_, [{}].concat(_.toArray(arguments)));
    });

    // wait for all promises
    var modulePromise = $.when(resourcePromise, operatorPromise, functionPromise);

    // private global functions ===============================================

    /**
     * Creates a descriptor collection containing the operator or function
     * implementations supported by the specified file format, resolved from
     * the passed internal implementation map.
     *
     * @param {Object} implementationMap
     *  The internal implementations of the operators or functions to be
     *  converted.
     *
     * @param {String} fileFormat
     *  The identifier of the file format.
     *
     * @param {Object} localNames
     *  The localized names of the operators or functions, mapped by native
     *  name as reported by the CalcEngine (in CalcEngine mode, see parameter
     *  'calcEngine'); otherwise by unique resource key.
     *
     * @param {Boolean} calcEngine
     *  Whether to use the formula syntax of the CalcEngine as native syntax
     *  (true); or the raw syntax of the respective file format (false).
     *
     * @returns {DescriptorCollection}
     *  The resulting descriptor collection.
     */
    function createCollection(implementationMap, fileFormat, localNames, calcEngine) {

        // create a clone for deleting entries (needed to detect missing descriptors)
        localNames = _.clone(localNames);

        // process all raw descriptor objects
        var collection = DescriptorCollection.create(implementationMap, function (implementation, key) {

            // resolve file format dependent properties of the implementation
            var properties = Utils.mapProperties(implementation, function (value) {
                if (!_.isObject(value) || _.isArray(value) || _.isFunction(value)) { return value; }
                // if an object, pick a property by file format identifier
                if (_.isObject(value) && (fileFormat in value)) { return value[fileFormat]; }
            });

            // get the native name(s) of the function or operator
            var nativeNames = properties[calcEngine ? 'ceName' : 'name'];
            if (_.isUndefined(nativeNames)) {
                nativeNames = [key]; // no name property: fall-back to resource key
            } else if (_.isString(nativeNames)) {
                nativeNames = [nativeNames];
            }

            // skip descriptors not supported in the specified file format
            if (!_.isArray(nativeNames) || (nativeNames.length === 0)) { return null; }

            // find the translated name
            var localName = null;
            if (calcEngine) {
                // CalcEngine sends a map from its own native names (English function names used in OOo)
                // to its translations, pick first available translation
                nativeNames.forEach(function (name) {
                    var lName = localNames[name];
                    delete localNames[name];
                    if (!lName) { Utils.error('\xa0 missing translation: key=' + name); }
                    if (!localName && lName) { localName = lName; }
                });
            } else {
                localName = localNames[key];
                delete localNames[key];
                // value may be an object with different translations per file format
                if (_.isObject(localName)) { localName = localName[fileFormat]; }
                if (!localName) { Utils.error('\xa0 missing translation: key=' + key); }
            }

            // skip descriptors without translation
            if (!localName) { return null; }

            // create the resource descriptor with preferred native name (first array element)
            var descriptor = DescriptorCollection.newDescriptor(key, nativeNames[0], localName);

            // store the alternative native names contained in the array
            descriptor.altNativeNames = nativeNames.slice(1);

            // convert category list to a set
            descriptor.category = Utils.makeSet(Utils.getTokenListOption(properties, 'category', []));

            // add the hidden and volatile flag
            descriptor.hidden = Utils.getBooleanOption(properties, 'hidden', false);
            descriptor.volatile = Utils.getBooleanOption(properties, 'volatile', false);

            // copy parameter counts
            descriptor.minParams = properties.minParams;
            if (_.isNumber(properties.maxParams)) { descriptor.maxParams = properties.maxParams; }
            if (_.isNumber(properties.repeatParams)) { descriptor.repeatParams = properties.repeatParams; }

            // copy result type, convert signature strings to arrays
            descriptor.type = properties.type;
            descriptor.signature = Utils.getTokenListOption(properties, 'signature', []);

            // copy the resolver (callback function, or resource key, the latter will be resolved later)
            descriptor.resolve = properties.resolve;

            return descriptor;
        });

        // replace 'resolve' string properties with the implementation of the referred descriptor
        collection.forEach(function (descriptor) {
            var srcDescriptor = null;
            while (_.isString(descriptor.resolve) && (srcDescriptor = collection.get(descriptor.resolve))) {
                descriptor.resolve = srcDescriptor.resolve;
            }
        });

        // print error messages for unknown translations
        Utils.withLogging(function () {
            _.each(localNames, function (name, key) {
                Utils.warn('\xa0 unexpected translation: key=' + key + ', name=' + name);
            });
        });

        return collection;
    }

    // class ParameterHelpDescriptor ==========================================

    /**
     * A descriptor with translated help texts for a single parameter of a
     * spreadsheet function.
     *
     * @constructor
     *
     * @property {String} name
     *  The translated name of the parameter.
     *
     * @property {String} description
     *  The description text of the parameter.
     *
     * @property {Boolean} optional
     *  Whether the parameter is required or optional.
     */
    function ParameterHelpDescriptor(name, description, optional) {

        this.name = _.isString(name) ? Utils.trimString(name) : '';
        this.description = _.isString(description) ? Utils.trimString(description) : '';
        this.optional = !!optional;

    } // class ParameterHelpDescriptor

    // class FunctionHelpDescriptor ===========================================

    /**
     * A descriptor with translated help texts for a spreadsheet function and
     * its parameters.
     *
     * @constructor
     *
     * @property {String} name
     *  The translated upper-case name of the function.
     *
     * @property {String} description
     *  A short description for the function itself.
     *
     * @property {Array<ParameterHelpDescriptor>} params
     *  An array with the names and descriptions of all parameters supported by
     *  the function.
     *
     * @property {Number} repeatStart
     *  The zero-based index of the first trailing parameter that can be used
     *  repeatedly in the function; or -1, if the function does not support
     *  parameter repetition.
     *
     * @property {Number} cycleLength
     *  The number of trailing repeated parameters; or 0, if the function does
     *  not support parameter repetition.
     *
     * @property {Number} cycleCount
     *  The maximum number of repetition cycles supported by the function,
     *  according to the index of first repeated parameter, and the size of the
     *  repetition cycles. Will be 0, if the function does not support
     *  parameter repetition.
     */
    function FunctionHelpDescriptor(name, description) {

        this.name = name;
        this.description = _.isString(description) ? Utils.trimString(description) : '';
        this.params = null;
        this.repeatStart = -1;
        this.cycleLength = 0;
        this.cycleCount = 0;

    } // class FunctionHelpDescriptor

    // public methods ---------------------------------------------------------

    /**
     * Returns a shallow clone of this function help descriptor with another
     * function name.
     *
     * @param {String} name
     *  The new function name.
     *
     * @returns {FunctionHelpDescriptor}
     *  A shallow clone of this function help descriptor.
     */
    FunctionHelpDescriptor.prototype.clone = function (name) {
        var helpDesc = new FunctionHelpDescriptor(name, this.description);
        helpDesc.params = this.params;
        helpDesc.repeatStart = this.repeatStart;
        helpDesc.cycleLength = this.cycleLength;
        helpDesc.cycleCount = this.cycleCount;
        return helpDesc;
    };

    // class FormulaResource ==================================================

    /**
     * Contains and provides access to all native and localized resources used
     * in spreadsheet formulas, for example operator and function descriptors
     * (syntax descriptions, and actual implementations), function names, names
     * of error codes and boolean literals, and the help texts for functions
     * and function parameters. An instance of this class is not bound to a
     * single application instance but will be created once for all documents
     * loaded from files with a specific file format.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {String} fileFormat
     *  The identifier of the file format related to the formula resource data
     *  represented by this instance.
     *
     * @param {Object} [rawServerData]
     *  The raw formula resource data received from the server (CalcEngine). If
     *  omitted, the localized formula resources loaded from JSON will be used
     *  instead.
     */
    var FormulaResource = BaseObject.extend({ constructor: function (fileFormat, rawServerData) {

        var // native and localized names of boolean values
            booleanCollection = null,

            // descriptors of all supported error codes
            errorCollection = null,

            // collection of all supported operators
            operatorCollection = null,

            // descriptors of all supported functions
            functionCollection = null,

            // the native function parameter separator
            nativeSeparator = null,

            // the localized function parameter separator
            localSeparator = null,

            // localized R1C1 prefix characters
            localPrefixChars = null,

            // version of the server-side Calc Engine
            engineVersion = '';

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // public methods -----------------------------------------------------

        /**
         * Returns the identifier of the file format represented by this
         * instance.
         *
         * @returns {String}
         *   The identifier of the file format represented by this instance.
         */
        this.getFileFormat = function () {
            return fileFormat;
        };

        /**
         * Returns whether this formula resource is based on remote server data
         * from the CalcEngine.
         *
         * @returns {Boolean}
         *  Whether this formula resource is based on remote server data from
         *  the CalcEngine (true), or on resource data loaded from local JSON
         *  files (false).
         */
        this.isCalcEngine = function () {
            return !!rawServerData;
        };

        /**
         * Returns the list separator for the current UI language. The list
         * separator is used to separate parameters of functions, and for the
         * range list operator.
         *
         * @returns {String}
         *  The list separator for the current UI language.
         */
        this.getSeparator = function (localized) {
            return localized ? localSeparator : nativeSeparator;
        };

        /**
         * Returns both prefix characters used in R1C1 references (for column
         * and row).
         *
         * @param {Boolean} localized
         *  Whether to return the localized prefix characters according to the
         *  UI language (true), or the native (English) prefix characters.
         *
         * @returns {String}
         *  The prefix characters used in R1C1 references (a string with two
         *  characters, first the row prefix, then the column prefix).
         */
        this.getR1C1PrefixChars = function (localized) {
            return localized ? localPrefixChars : NATIVE_R1C1_PREFIX_CHARS;
        };

        /**
         * Returns the collection with the descriptors of all boolean names.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all boolean names. The
         *  descriptor for the boolean value false is mapped by the resource
         *  key 'f', and the descriptor for the boolean value true is mapped by
         *  the resource key 't'.
         */
        this.getBooleanCollection = function () {
            return booleanCollection;
        };

        /**
         * Returns the collection with the descriptors of all error codes
         * supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all error codes supported by
         *  the current file format.
         */
        this.getErrorCollection = function () {
            return errorCollection;
        };

        /**
         * Returns the collection with the descriptors of all unary and binary
         * operators supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all unary and binary
         *  operators supported by the current file format.
         */
        this.getOperatorCollection = function () {
            return operatorCollection;
        };

        /**
         * Returns the collection with the descriptors of all functions
         * supported by the current file format.
         *
         * @returns {DescriptorCollection}
         *  The collection with the descriptors of all functions supported by
         *  the current file format.
         */
        this.getFunctionCollection = function () {
            return functionCollection;
        };

        /**
         * Returns a help text descriptor for the specified function.
         *
         * @param {String} key
         *  The unique resource key of the function.
         *
         * @returns {FunctionHelpDescriptor|Null}
         *  A help descriptor for the specified function; or null for invalid
         *  keys, or functions without help texts.
         */
        this.getFunctionHelp = function (key) {
            return functionCollection.getProperty(key, 'help', null);
        };

        /**
         * Returns the version of the server-side Calc Engine.
         *
         * @returns {String}
         *  The version of the Calc Engine.
         */
        this.getEngineVersion = function () {
            return engineVersion;
        };

        // initialization -----------------------------------------------------

        // names of unsupported functions contained in the resource data
        var unknownFunctions = [];

        // create descriptors for the names of boolean values
        booleanCollection = DescriptorCollection.create({ f: 'FALSE', t: 'TRUE' }, function (nativeName, key) {
            var localName = !rawServerData ? RAW_RESOURCE_DATA[nativeName] : (key === 'f') ? rawServerData.falseLiteral : rawServerData.trueLiteral;
            if (!localName) {
                Utils.error('\xa0 missing boolean translation: key=' + key);
                localName = nativeName;
            }
            return DescriptorCollection.newDescriptor(key, nativeName, localName);
        });

        // create error code descriptors
        errorCollection = DescriptorCollection.create(ErrorCode.NATIVE_CODES, function (error, key) {
            var nativeName = NATIVE_ERROR_NAMES[key];
            var localName = rawServerData ? rawServerData.errorCodes[nativeName] : RAW_RESOURCE_DATA.errors[key];
            if (!localName) {
                Utils.error('\xa0 missing error code translation: key=' + key);
                localName = nativeName;
            }
            return DescriptorCollection.newDescriptor(key, nativeName, localName);
        });

        // resolve operator descriptors, and the maps with native and translated operator names
        var operatorNames = rawServerData ? _.extend({}, CE_LOCAL_OPERATOR_NAMES, { ',': rawServerData.listSeparator }) : LOCAL_OPERATOR_NAMES;
        operatorCollection = createCollection(OPERATOR_IMPL_MAP, fileFormat, operatorNames, !!rawServerData);

        // resolve function descriptors, and the maps with native and translated function names
        var functionNames = rawServerData ? rawServerData.functions : RAW_RESOURCE_DATA.functions;
        functionCollection = createCollection(FUNCTION_IMPL_MAP, fileFormat, functionNames, !!rawServerData);

        // function parameter separator
        nativeSeparator = ',';
        localSeparator = rawServerData ? rawServerData.listSeparator : LOCAL_OPERATOR_NAMES.list;

        // the R1C1 prefix characters (ODF does not translate these characters in the UI)
        localPrefixChars = (fileFormat === 'odf') ? NATIVE_R1C1_PREFIX_CHARS : Utils.getStringOption(RAW_RESOURCE_DATA, 'RC', 'RC').toUpperCase();

        // get the version of the CalcEngine
        engineVersion = Utils.getStringOption(rawServerData, 'engineVersion', '');

        // print error messages to browser console
        Utils.withLogging(function () {
            // check validity of decimal separator
            if (LocaleData.DEC !== rawServerData.decSeparator) {
                Utils.error('\xa0 decimal separator mismatch: LocaleData.DEC="' + LocaleData.DEC + '" rawServerData.decSeparator="' + rawServerData.decSeparator + '"');
            }
            // check validity of group separator
            if (LocaleData.GROUP !== rawServerData.groupSeparator) {
                Utils.error('\xa0 group separator mismatch: LocaleData.GROUP="' + LocaleData.GROUP + '" rawServerData.groupSeparator="' + rawServerData.groupSeparator + '"');
            }
            // show differences between passed function names and implemented function names
            if (unknownFunctions.length > 0) {
                Utils.error('\xa0 unknown functions in resource data: ' + unknownFunctions.sort().join(', '));
            }
        });

        // process each entry in the function help map (build system ensures valid data)
        _.each(RAW_RESOURCE_DATA.help, function (helpEntry, funcKey) {

            var // the internal raw function descriptor (for all file formats)
                functionImpl = FUNCTION_IMPL_MAP[funcKey],
                // the function descriptor resolved for the current file format
                functionDesc = functionCollection.get(funcKey);

            // writes a warning or error message for the current function
            function log(type, msg) {
                Utils[type]('\xa0 ' + msg + ' function ' + funcKey);
            }

            // returns the maximum of a file-format-dependent numeric function descriptor property
            function resolveMaxNumber(value) {
                return _.isNumber(value) ? value : _.isObject(value) ? _.reduce(value, function (a, b) { return Math.max(a, b); }, 0) : value;
            }

            // function name must exist in locale data
            if (!functionImpl) {
                return log('warn', 'help entry found for unknown');
            }

            // silently skip function help for functions not supported by the current file format
            if (!functionDesc) {
                return;
            }

            // function data entry must be an object
            if (!_.isObject(helpEntry)) {
                return log('error', 'invalid map entry for');
            }

            // the resulting function help descriptor
            var funcHelp = functionDesc.help = new FunctionHelpDescriptor(functionDesc.localName, helpEntry.d);
            if (!helpEntry.d) {
                log('error', 'missing description for');
            }

            // check parameter help availability of required parameters
            var params = _.clone(helpEntry.p);
            var rawMinParams = resolveMaxNumber(functionImpl.minParams);
            if (params.length < rawMinParams) {
                log('error', 'missing help for required parameters for');
                while (params.length < rawMinParams) { params.push({ n: '', d: '' }); }
            }

            // check parameter help availability of fixed number of optional parameters
            var rawMaxParams = resolveMaxNumber(functionImpl.maxParams);
            if (_.isNumber(rawMaxParams)) {
                if (params.length < rawMaxParams) {
                    log('error', 'missing help for optional parameters for');
                } else if (rawMaxParams < params.length) {
                    log('warn', 'superfluous optional parameters for');
                }
            }

            // validate number of array elements of function parameter help
            if (_.isNumber(functionDesc.maxParams)) {
                while (params.length < functionDesc.maxParams) { params.push({ n: '', d: '' }); }
                params.splice(functionDesc.maxParams);
            }

            // check parameter help availability of variable number of optional parameters
            var repeatParams = _.isNumber(functionDesc.maxParams) ? null : (functionDesc.repeatParams || 1);
            if (_.isNumber(repeatParams)) {
                var optParams = params.length - functionDesc.minParams;
                // remove help texts of superfluous optional parameters
                if ((optParams !== 0) && (optParams !== repeatParams)) {
                    log('warn', 'superfluous variable parameters for');
                    params.splice(functionDesc.minParams + ((optParams > repeatParams) ? repeatParams : 0));
                }
            }

            // create a single parameter array with name, description, and additional information
            funcHelp.params = _.map(params, function (param, index) {
                return new ParameterHelpDescriptor(param.n, param.d, functionDesc.minParams <= index);
            });

            // add properties for repeating parameters
            if (_.isNumber(repeatParams)) {
                // absolute index of first repeated parameter
                funcHelp.repeatStart = funcHelp.params.length - repeatParams;
                funcHelp.cycleLength = repeatParams;
                funcHelp.cycleCount = Math.floor((FormulaUtils.MAX_PARAM_COUNT - funcHelp.repeatStart) / repeatParams);
            }
        });

        // print all function names without help texts
        Utils.withLogging(function () {
            var missingHelpDescs = _.reject(functionCollection._map, 'help');
            if (missingHelpDescs.length > 0) {
                Utils.error('\xa0 missing help for functions: ' + _.pluck(missingHelpDescs, 'key').sort().join(', '));
            }
        });

    } }); // class FormulaResource

    // static methods ---------------------------------------------------------

    /**
     * Downloads localized formula resource data (translated function names,
     * help texts for functions and their parameters, etc.) from the server,
     * and initializes an instance of the class FormulaResource. The resource
     * instances created by this method will be cached internally and reused
     * per file format identifier.
     *
     * @param {SpreadsheetApplication} app
     *  The spreadsheet application instance.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.remote=false]
     *      If set to true, the localized formula resources will be queried
     *      from the server-side CalcEngine. Otherwise, the local JSON resource
     *      data will be used.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved when the formula resource data has been
     *  initialized correctly.
     */
    FormulaResource.create = function (app, options) {

        var // the file format of the edited document
            fileFormat = app.getFileFormat(),
            // whether to load remote resource data from CalcEngine
            remoteData = Utils.getBooleanOption(options, 'remote', false),
            // the unique cache key for the resource instance
            cacheKey = fileFormat + ':' + (remoteData ? 'remote' : 'local'),
            // existing promise with formula resource data
            promise = FORMULA_RESOURCE_CACHE[cacheKey];

        // formula resource may already exist from importing another document, but it may
        // be still pending (race condition when loading two documents at the same time)
        if (!promise) {

            // request resource data from server
            if (remoteData) {

                // bug 40851: validate the server response
                var serverRequest = app.sendQueryRequest('localeData', { locale: LocaleData.LOCALE });
                serverRequest = serverRequest.then(function (rawServerData) {

                    // returns whether the raw resource data object contains a valid string property
                    function checkStringProperty(propName, maxLen) {
                        var value = rawServerData[propName];
                        var valid = _.isString(value) && (value.length > 0) && (!maxLen || (value.length <= maxLen));
                        if (!valid) { Utils.error('FormulaResource.create(): missing or invalid property "' + propName  + '" in resource data'); }
                        return valid;
                    }

                    // returns whether the raw resource data object contains a valid object property
                    function checkObjectProperty(propName) {
                        var value = rawServerData[propName];
                        var valid = _.isObject(value) && !_.isEmpty(value);
                        if (!valid) { Utils.error('FormulaResource.create(): missing or invalid property "' + propName  + '" in resource data'); }
                        return valid;
                    }

                    // check for required string and object properties
                    var valid = checkStringProperty('decSeparator', 1) && checkStringProperty('groupSeparator', 1) && checkStringProperty('listSeparator', 1) &&
                        checkStringProperty('trueLiteral') && checkStringProperty('falseLiteral') &&
                        checkObjectProperty('functions') && checkObjectProperty('errorCodes');

                    // reject the resulting server request, if response data is invalid
                    return valid ? rawServerData : $.Deferred().reject();
                });

                // create the formula resource object for the current file format
                promise = serverRequest.then(function (rawServerData) {
                    return new FormulaResource(fileFormat, rawServerData);
                });
            } else {
                promise = $.when(new FormulaResource(fileFormat));
            }

            // immediately store the (pending) promise in the global cache,
            // in case other documents will be loaded simultaneously
            FORMULA_RESOURCE_CACHE[cacheKey] = promise;

            // if the server request fails, remove the cached promise in order to place another request with next document
            promise.fail(function () {
                delete FORMULA_RESOURCE_CACHE[cacheKey];
            });
        }

        return promise;
    };

    // exports ================================================================

    return modulePromise.then(_.constant(FormulaResource));

});
