/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/operand', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/matrix'
], function (SheetUtils, FormulaUtils, Matrix) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode,
        Address = SheetUtils.Address,
        Range3D = SheetUtils.Range3D,
        Range3DArray = SheetUtils.Range3DArray;

    // class Operand ==========================================================

    /**
     * Represents a single operand stored in the operand stack of a formula
     * token interpreter.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Any} value
     *  The value to be stored as operand. Can be a scalar value (numbers,
     *  strings, boolean values, null), a Date object (will be converted to a
     *  floating-point number immediately), an error code (instance of the
     *  class ErrorCode), a matrix (instance of the class Matrix), a single
     *  cell range address (instance of the class Range3D), or an array of cell
     *  range addresses (instance of the class Range3DArray).
     */
    function Operand(context, value) {

        this._context = context;

        // initialization -----------------------------------------------------

        // cell range addresses passed: validate size
        if ((value instanceof Range3D) || (value instanceof Range3DArray)) {
            value = Range3DArray.get(value);
            if (value.empty()) {
                value = ErrorCode.NULL;
                // continue below, operand becomes an error code (a scalar value)
            } else if (value.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                value = FormulaUtils.UNSUPPORTED_ERROR;
                // continue below, operand becomes an error code (a scalar value)
            } else {
                //ReferenceOperandMixin.call(this, context, value);
                this._type = 'ref';
                this._value = value;
                return;
            }
        }

        // matrix passed: validate matrix size, and all values in the matrix
        if (value instanceof Matrix) {
            FormulaUtils.ensureMatrixSize(value.rows(), value.cols());
            value.transform(context.validateValue, context);
            this._type = 'mat';
            this._value = value;
            return;
        }

        // otherwise: scalar value (as last check, preceding code may have changed the value)
        this._type = 'val';
        this._value = context.validateValue(value);

    } // class Operand

    // public methods ---------------------------------------------------------

    /**
     * Returns the effective data type of this operand.
     *
     * @returns {String}
     *  The effective data type of this operand. Will be one of the following
     *  strings:
     *  - 'val': This operand contains a scalar value (number, string, etc.).
     *  - 'mat': This operand contains a matrix.
     *  - 'ref': This operand contains an array of cell range addresses.
     */
    Operand.prototype.getType = function () {
        return this._type;
    };

    /**
     * Returns whether this operand contains a scalar value (type 'val').
     *
     * @returns {Boolean}
     *  Whether this operand contains a scalar value (type 'val').
     */
    Operand.prototype.isValue = function () {
        return this._type === 'val';
    };

    /**
     * Returns whether this operand contains a matrix (type 'mat').
     *
     * @returns {Boolean}
     *  Whether this operand contains a matrix (type 'mat').
     */
    Operand.prototype.isMatrix = function () {
        return this._type === 'mat';
    };

    /**
     * Returns whether this operand contains a cell reference (type 'ref').
     *
     * @returns {Boolean}
     *  Whether this operand contains a cell reference (type 'ref').
     */
    Operand.prototype.isReference = function () {
        return this._type === 'ref';
    };

    /**
     * Returns whether this operand represents the empty value (e.g. the second
     * operand in the formula =SUM(1,,2) is empty).
     *
     * @returns {Boolean}
     *  Whether this operand represents the empty value.
     */
    Operand.prototype.isEmpty = function () {
        return this._value === null;
    };

    /**
     * Converts this operand to a scalar value.
     *
     * @returns {Any}
     *  The scalar value represented by this operand. Returns an existing
     *  scalar value, or the specified element of a matrix, or the contents of
     *  the specified cell from a cell reference (only if it does not cover the
     *  reference cell of the formula context passed to the constructor).
     *
     * @throws {ErrorCode}
     *  - CIRCULAR_ERROR, if the operand results in a circular reference.
     *  - The #VALUE! error code, if the reference contains more than one
     *      cell range, or if the cell range cannot be resolved to a single
     *      value according to the reference cell.
     */
    Operand.prototype.getValue = function (row, col) {

        // return scalar value, and an element of a matrix
        switch (this._type) {
            case 'val': return this._value;
            case 'mat': return this._value.get(row || 0, col || 0);
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error)
        var range = this._context.convertToValueRange(this._value);

        // pick value from a single cell
        if (range.single()) {
            return this._context.getCellValue(range.sheet1, range.start);
        }

        // the reference cell of the formula context
        var refAddress = this._context.getRefAddress();

        // pick matching cell from left or right (row interval must cover the reference cell)
        if ((range.end[0] < refAddress[0]) || (refAddress[0] < range.start[0])) {
            if (range.start[0] !== range.end[0]) { throw ErrorCode.VALUE; }
            if (!range.containsRow(refAddress[1])) { throw ErrorCode.VALUE; }
            return this._context.getCellValue(range.sheet1, new Address(range.start[0], refAddress[1]));
        }

        // pick matching cell from above or below (column interval must cover the reference cell)
        if ((range.end[1] < refAddress[1]) || (refAddress[1] < range.start[1])) {
            if (range.start[1] !== range.end[1]) { throw ErrorCode.VALUE; }
            if (!range.containsCol(refAddress[0])) { throw ErrorCode.VALUE; }
            return this._context.getCellValue(range.sheet1, new Address(refAddress[0], range.start[1]));
        }

        // range is not left of, right of, above, or below the reference cell
        throw ErrorCode.VALUE;
    };

    /**
     * Converts this operand to a matrix.
     *
     * @returns {Matrix}
     *  The matrix represented by this operand. Returns an existing matrix, or
     *  a scalar value packed into a 1x1 matrix, or the contents of the
     *  referenced cells packed into a matrix. The reference must consist of a
     *  single cell range address that does not cover the reference cell, and
     *  the range must not exceed specific limits in order to prevent
     *  performance problems.
     *
     * @throws {ErrorCode}
     *  - UNSUPPORTED_ERROR, if the resulting matrix would be too large.
     *  - CIRCULAR_ERROR, if the operand results in a circular reference.
     *  - The #VALUE! error code, if the operand contains a reference with more
     *      than one cell range address.
     */
    Operand.prototype.getMatrix = function () {

        // pack scalar value into a 1x1 matrix, or return an existing matrix
        switch (this._type) {
            case 'val': return new Matrix([[this._value]]);
            case 'mat': return this._value;
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error)
        var range = this._context.convertToValueRange(this._value);

        // the sheet model containing the cell with safety check (#REF! error if sheet index is invalid)
        var sheetModel = this._context.getDocModel().getSheetModel(range.sheet1);
        if (!sheetModel) { throw ErrorCode.REF; }

        // size of the resulting matrix
        var rows = range.rows();
        var cols = range.cols();
        FormulaUtils.ensureMatrixSize(rows, cols);

        // build an empty matrix
        var matrix = Matrix.create(rows, cols, null);

        // fill existing cell values
        sheetModel.getCellCollection().iterateCellsInRanges(range.toRange(), function (cellData) {
            matrix.set(cellData.address[1] - range.start[1], cellData.address[0] - range.start[0], cellData.result);
        }, { type: 'content', hidden: 'all' });

        return matrix;
    };

    /**
     * Converts this operand to an unresolved array of cell range addresses.
     *
     * @returns {Range3DArray}
     *  The unresolved cell range addresses contained in this operand.
     *
     * @throws {ErrorCode}
     *  The error code #VALUE!, if this operand is not reference type.
     */
    Operand.prototype.getRanges = function () {

        // throw #VALUE! error for scalar values and matrixes
        switch (this._type) {
            case 'val':
            case 'mat': throw ErrorCode.VALUE;
            // otherwise, this operand is a reference
        }

        // return the value (always an instance of Range3DArray)
        return this._value;
    };

    /**
     * Returns the string representation of this operand for debug logging.
     */
    Operand.prototype.toString = function () {
        return (this._type === 'val') ? FormulaUtils.valueToString(this._value) : this._value.toString();
    };

    // exports ================================================================

    return Operand;

});
