/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/impl/conversionfuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/errorcode'
], function (Utils, ErrorCode) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with numeric
     * conversion, e.g. number systems such as binary, octal, hexadecimal, or
     * Roman numbers; and conversion between different measurement units.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // string with zero characters for expansion of converted numbers
    var ZEROS = Utils.repeatString('0', 255);

    // private global functions ===============================================

    /**
     * Expands the passed string (binary, octal, or hexadecimal number) with
     * leading zero characters, according to the passed length.
     *
     * @param {String} number
     *  A number as string (binary, octal, or hexadecimal).
     *
     * @param {Number|Undefined} length
     *  If specified, the target length of the passed number. If omitted, the
     *  passed number will be returned unmodified.
     *
     * @param {Number} maxLength
     *  The maximum allowed length. Larger values will result in throwing a
     *  #NUM! error code.
     *
     * @param {Boolean} [ensureMin=false]
     *  If set to true, and the passed number is longer than the value passed
     *  in the 'length' parameter, a #NUM! error code will be thrown. By
     *  default, larger numbers will be returned as passed in this case.
     *
     * @returns {String}
     *  The resulting number with leading zero characters.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed length is invalid.
     */
    function expandWithZeros(number, length, maxLength, ensureMin) {

        // always throw, if the passed number is larger than the specified maximum
        if (number.length > maxLength) { throw ErrorCode.NUM; }

        // check and expand the number with leading zeros
        if (_.isNumber(length)) {
            // length parameter must be valid
            if ((ensureMin && (length < number.length)) || (length > maxLength)) {
                throw ErrorCode.NUM;
            }
            // expand, but do not truncate
            if (number.length < length) {
                number = (ZEROS + number).slice(-length);
            }
        }

        return number;
    }

    /**
     * Converts the string representing of a binary number to a decimal number.
     *
     * @param {String} value
     *  A binary number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid binary number, or if it is too long
     *  (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertBinToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[01]{0,10}$/.test(value)) {
            throw ErrorCode.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 2) : 0;

        // negative number, if 10th digit is 1
        if ((value.length === 10) && /^1/.test(value)) {
            result -= 0x400;
        }

        return result;
    }

    /**
     * Converts the string representing of an octal number to a decimal number.
     *
     * @param {String} value
     *  An octal number string to be converted to a number if it is not longer
     *  than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid octal number, or if it is too long
     *  (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertOctToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[0-7]{0,10}$/.test(value)) {
            throw ErrorCode.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 8) : 0;

        // negative number, if 10th digit is greater than 3
        if ((value.length === 10) && /^[4-7]/.test(value)) {
            result -= 0x40000000;
        }

        return result;
    }

    /**
     * Converts the string representing of a hexadecimal number to a decimal
     * number.
     *
     * @param {String} value
     *  A hexadecimal number string to be converted to a number if it is not
     *  longer than 10 characters.
     *
     * @returns {Number}
     *  The resulting decimal number.
     *
     * @throws {ErrorCode}
     *  If the passed value is not a valid hexadecimal number, or if it is too
     *  long (more than 10 digits), a #NUM! error will be thrown.
     */
    function convertHexToDec(value) {

        // check input value, maximum length of the input string is 10 digits
        if (!/^[0-9a-f]{0,10}$/i.test(value)) {
            throw ErrorCode.NUM;
        }

        var // the parsed result (empty string is interpreted as zero)
            result = (value.length > 0) ? parseInt(value, 16) : 0;

        // negative number, if 10th digit is greater than 7
        if ((value.length === 10) && /^[89a-f]/i.test(value)) {
            result -= 0x10000000000;
        }

        return result;
    }

    /**
     * Converts the passed decimal number to its binary representation. Inserts
     * leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its binary representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting binary number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  binary digits, or if the passed length is invalid.
     */
    function convertDecToBin(number, length) {
        // restrict passed number to signed 10-digit binary
        if ((-number > 0x200) || (number > 0x1FF)) { throw ErrorCode.NUM; }
        // convert negative numbers to positive resulting in a 10-digit binary number
        if (number < 0) { number += 0x400; }
        // convert to hexadecimal number, expand to specified length
        return expandWithZeros(number.toString(2), length, 10, true);
    }

    /**
     * Converts the passed decimal number to its octal representation. Inserts
     * leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its octal representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting octal number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  octal digits, or if the passed length is invalid.
     */
    function convertDecToOct(number, length) {
        // restrict passed number to signed 10-digit octal
        if ((-number > 0x20000000) || (number > 0x1FFFFFFF)) { throw ErrorCode.NUM; }
        // convert negative numbers to positive resulting in a 10-digit octal number
        if (number < 0) { number += 0x40000000; }
        // convert to octal number, expand to specified length
        return expandWithZeros(number.toString(8), length, 10, true);
    }

    /**
     * Converts the passed decimal number to its hexadecimal representation.
     * Inserts leading zero characters if specified.
     *
     * @param {Number} number
     *  The number to be converted to its hexadecimal representation.
     *
     * @param {Number} [length]
     *  If specified, the target length of the passed number. See function
     *  expandWithZeros() for details.
     *
     * @returns {String}
     *  The resulting hexadecimal number.
     *
     * @throws {ErrorCode}
     *  The #NUM! error code, if the passed number results in more than 10
     *  hexadecimal digits, or if the passed length is invalid.
     */
    function convertDecToHex(number, length) {
        // restrict passed number to signed 10-digit hexadecimal
        if ((-number > 0x8000000000) || (number > 0x7FFFFFFFFF)) { throw ErrorCode.NUM; }
        // convert negative numbers to positive resulting in a 10-digit hex number
        if (number < 0) { number += 0x10000000000; }
        // convert to hexadecimal number, expand to specified length
        return expandWithZeros(number.toString(16).toUpperCase(), length, 10, true);
    }

    // exports ================================================================

    return {

        ARABIC: {
            category: 'conversion',
            name: { ooxml: '_xlfn.ARABIC' },
            minParams: 1,
            maxParams: 1,
            type: 'val'
        },

        BASE: {
            category: 'conversion',
            name: { ooxml: '_xlfn.BASE' },
            minParams: 2,
            maxParams: 3,
            type: 'val',
            signature: 'val:int val:int val:int',
            resolve: function (number, radix, length) {
                if ((number < 0) || (number > 0x1FFFFFFFFFFFFF)) { throw ErrorCode.NUM; }
                if ((radix < 2) || (radix > 36)) { throw ErrorCode.NUM; }
                if (_.isNumber(length) && ((length < 0) || (length > 255))) { throw ErrorCode.NUM; }
                return expandWithZeros(number.toString(radix).toUpperCase(), length, 255);
            }
        },

        BIN2DEC: {
            category: 'conversion',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: convertBinToDec
        },

        BIN2HEX: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToHex(convertBinToDec(number), length);
            }
        },

        BIN2OCT: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToOct(convertBinToDec(number), length);
            }
        },

        COLOR: {
            category: 'conversion',
            name: { ooxml: null, odf: 'ORG.LIBREOFFICE.COLOR' },
            ceName: null,
            minParams: 3,
            maxParams: 4,
            type: 'val',
            signature: 'val:int val:int val:int val:int'
        },

        CONVERT: {
            category: 'conversion',
            ceName: { odf: 'CONVERT_ADD' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        'CONVERT.ODF': {
            category: 'conversion',
            name: { ooxml: null, odf: 'ORG.OPENOFFICE.CONVERT' },
            ceName: { ooxml: null, odf: 'CONVERT' },
            minParams: 3,
            maxParams: 3,
            type: 'val'
        },

        DEC2BIN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: convertDecToBin
        },

        DEC2HEX: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: convertDecToHex
        },

        DEC2OCT: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: convertDecToOct
        },

        DECIMAL: {
            category: 'conversion',
            name: { ooxml: '_xlfn.DECIMAL' },
            minParams: 2,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: (function () {
                var getTestRE = _.memoize(function (radix) {

                    // for radices 2-10: decimal digits only; otherwise all digits and additional letters from A to radix
                    var charClass = (radix <= 10) ? ('0-' + (radix - 1)) : ('0-9A-' + String.fromCharCode(54 + radix));

                    // create the regular expression
                    return new RegExp('^[' + charClass + ']+$', 'i');
                });

                return function (input, radix) {

                    if ((radix < 2) || (radix > 36)) { throw ErrorCode.NUM; }

                    // test the input value
                    if (!getTestRE(radix).test(input)) { throw ErrorCode.NUM; }

                    var decimal = parseInt(input, radix);

                    if (decimal < 0) { throw ErrorCode.NUM; }
                    if (!isFinite(decimal)) { throw ErrorCode.NUM; }

                    return decimal;
                };
            }())

        },

        HEX2BIN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToBin(convertHexToDec(number), length);
            }
        },

        HEX2DEC: {
            category: 'conversion',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: convertHexToDec
        },

        HEX2OCT: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToOct(convertHexToDec(number), length);
            }
        },

        OCT2BIN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToBin(convertOctToDec(number), length);
            }
        },

        OCT2DEC: {
            category: 'conversion',
            minParams: 1,
            maxParams: 1,
            type: 'val',
            signature: 'val:str',
            resolve: convertOctToDec
        },

        OCT2HEX: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:str val:int',
            resolve: function (number, length) {
                return convertDecToHex(convertOctToDec(number), length);
            }
        },

        ROMAN: {
            category: 'conversion',
            minParams: 1,
            maxParams: 2,
            type: 'val',
            signature: 'val:int val:int',
            resolve: (function () {

                //copied from open office (calc) source code

                var pChars = ['M', 'D', 'C', 'L', 'X', 'V', 'I'];
                var pValues = [1000, 500, 100, 50, 10, 5, 1];
                var nMaxIndex = pValues.length - 1;

                return function (fVal, fMode) {
                    if (fMode < 0 || fMode >= 5 || fVal < 0 || fVal >= 4000) {
                        throw ErrorCode.VALUE;
                    }
                    if (_.isUndefined(fMode)) {
                        fMode = 0;
                    }

                    var aRoman = '';
                    var nVal = fVal;
                    var nMode = fMode;

                    for (var i = 0; i <= nMaxIndex / 2; i++) {
                        var nIndex = 2 * i;
                        var nDigit = Math.floor(nVal / pValues[nIndex]);

                        if ((nDigit % 5) === 4) {
                            var nIndex2 = (nDigit === 4) ? nIndex - 1 : nIndex - 2;
                            var nSteps = 0;

                            while ((nSteps < nMode) && (nIndex < nMaxIndex)) {
                                nSteps++;
                                if (pValues[nIndex2] - pValues[nIndex + 1] <= nVal) {
                                    nIndex++;
                                } else {
                                    nSteps = nMode;
                                }
                            }

                            aRoman += pChars[nIndex];
                            aRoman += pChars[nIndex2];
                            nVal += pValues[nIndex];
                            nVal -= pValues[nIndex2];
                        } else {
                            if (nDigit > 4) {
                                aRoman += pChars[nIndex - 1];
                            }
                            var nPad = nDigit % 5;
                            if (nPad) {
                                aRoman += Utils.repeatString(pChars[nIndex], nPad);
                            }
                            nVal %= pValues[nIndex];
                        }
                    }
                    return aRoman;

                };
            }())
        }
    };

});
