/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/dialog/hyperlinkdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialogs',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/editframework/view/editlabels',
    'gettext!io.ox/office/editframework/main'
], function (Utils, Dialogs, HyperlinkUtils, Labels, gt) {

    'use strict';

    var // a unique DOM identifier for the label of the display text input field
        DISPLAY_LABEL_ID = _.uniqueId('display'),

        // a unique DOM identifier for the display text input field
        DISPLAY_INPUT_ID = _.uniqueId('display'),

        // a unique DOM identifier for the label of the URL input field
        URL_LABEL_ID = _.uniqueId('url'),

        // a unique DOM identifier for the URL input field
        URL_INPUT_ID = _.uniqueId('url');

    // class HyperlinkDialog ==================================================

    /**
     * A modal dialog with two text fields allowing to edit the URL and the
     * text representation of a hyperlink.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {EditView} docView
     *  The document view instance that has created this dialog.
     *
     * @param {String} [url]
     *  The initial URL to be shown in the dialog. If omitted, the text field
     *  for the URL will appear empty.
     *
     * @param {String} [display]
     *  The initial display string for the URL to be shown in the dialog. If
     *  omitted, the text field for the display string will appear empty.
     */
    var HyperlinkDialog = Dialogs.ModalDialog.extend({ constructor: function (docView, url, display) {

        var // self reference
            self = this,

            // whether to keep URL and display text in sync
            syncDisplay = !display || (url === display),

            // the input field for the URL
            urlInputNode = $('<input>', {
                value: url || '',
                tabindex: 0,
                id: URL_INPUT_ID,
                role: 'textbox',
                'aria-labelledby': URL_LABEL_ID,
                'aria-required': true
            }).addClass('form-control'),

            // the input field for the display text of the URL
            displayInputNode = $('<input>', {
                value: display || url,
                tabindex: 0,
                id: DISPLAY_INPUT_ID,
                role: 'textbox',
                'aria-labelledby': DISPLAY_LABEL_ID,
                'aria-required': true
            }).addClass('form-control');

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, {
            width: 400,
            title: url ? Labels.EDIT_HYPERLINK_LABEL : Labels.INSERT_HYPERLINK_LABEL,
            okLabel: gt('Insert')
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the trimmed URL from the input field.
         *
         * @returns {String}
         *  The trimmed URL from the input field.
         */
        function getUrl() {
            return Utils.trimString(urlInputNode.val());
        }

        function updateUrlField() {
            var currUrl = getUrl(), hasUrl = currUrl.length > 0;
            self.enableOkButton(hasUrl);
            urlInputNode.attr('aria-invalid', !hasUrl);
            if (syncDisplay) { displayInputNode.val(currUrl); }
        }

        function updateDisplayField() {
            syncDisplay = false;
        }

        // public methods -----------------------------------------------------

        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(function (action) {
                switch (action) {

                    case 'ok':
                        var currUrl = getUrl();
                        var display = displayInputNode.val();

                        // add http: as default if protocol is missing
                        var newUrl = (currUrl.indexOf(':') < 0) ? ('http://' + currUrl) : currUrl;

                        // show warning and reopen the dialog, if the URL is invalid
                        if (!HyperlinkUtils.hasSupportedProtocol(newUrl) || !HyperlinkUtils.isValidURL(newUrl)) {
                            docView.yell({ type: 'warning', message: Labels.INVALID_HYPERLINK_LABEL });
                            return new HyperlinkDialog(docView, currUrl, display).show();
                        }

                        // return result object for a valid URL
                        return { url: newUrl, text: display };

                    case 'remove':
                        return { url: null, text: null };
                }

                return $.Deferred().reject();
            });
        });

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // Bug 43868: ios auto correction should be of for input fields
        // TODO: use Forms.createInputMarkup() to create inputs here, it alrady has this fix
        if (Utils.IOS) {
            urlInputNode.attr({ autocorrect : 'off', autocapitalize : 'none' });
            displayInputNode.attr({ autocorrect : 'off', autocapitalize : 'none' });
        }

        // add the input controls to the dialog body
        this.append(
            $('<div class="form-group">').append(
                $('<label id="' +  URL_LABEL_ID + '" for="' + URL_INPUT_ID + '">')
                    .text(/*#. "Insert URL" dialog: The URL itself */ gt('URL')),
                urlInputNode
            ),
            $('<div class="form-group">').append(
                $('<label id="' + DISPLAY_LABEL_ID + '" for="' + DISPLAY_INPUT_ID + '">')
                    .text(/*#. "Insert URL" dialog: The display text of the URL */ gt('Text')),
                displayInputNode
            )
        );

        // add a Remove button, if a URL has been passed
        if (url) {
            this.addAlternativeButton('remove', gt('Remove'));
            this.getButton('remove').addClass('btn-warning');
        }

        // register a change handlers to toggle the insert button
        urlInputNode.on('input', updateUrlField);
        displayInputNode.on('input', updateDisplayField);

        // Bugfix 40844: when the user focus the displayInputNode via tab or mouse click,
        // the text in the displayInputNode should be selected
        displayInputNode.on('focus', function () {
            // a workaround is needed for the browsers below
            // otherwise on mouseup the selection is removed again
            if (_.browser.Chrome || _.browser.Safari || _.browser.Edge) {
                // the workaround: it's also mentioned here https://code.google.com/p/chromium/issues/detail?id=4505
                window.setTimeout(function () { displayInputNode.select(); }, 0);
            // this is how it's supposed to work normally
            } else {
                displayInputNode.select();
            }
        });

        // set initial focus
        this.on('show', function () {
            updateUrlField();

            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus shoud not be in an input element
            if (Utils.SMALL_DEVICE) {
                self.getOkButton().focus();
            } else {
                urlInputNode.focus();
            }
        });

    } }); // class HyperlinkDialog

    // exports ================================================================

    return HyperlinkDialog;

});
