/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
  * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/modelattributesmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/fontcollection',
    'io.ox/office/editframework/model/themecollection'
], function (Utils, TriggerObject, Color, Border, AttributeUtils, FontCollection, ThemeCollection) {

    'use strict';

    // definitions for global document attributes -----------------------------

    var DOCUMENT_ATTRIBUTE_DEFINITIONS = {

        /**
         * The actual file format of the edited document. The following values
         * are supported:
         * - 'ooxml': Microsoft's Office Open XML file format.
         * - 'odf': The OASIS OpenDocument file format.
         */
        fileFormat: { def: 'ooxml' }
    };

    // mix-in class ModelAttributesMixin ======================================

    /**
     * A mix-in class for the document model class EditModel that provides the
     * global document attributes, the default values for all formatting
     * attributes of all registered attribute families, the style collections
     * for all attribute families used in a document, a font collection, and a
     * collection of themes.
     *
     * @constructor
     */
    function ModelAttributesMixin() {

        var // self reference (the document model)
            self = this,

            // the collection of document fonts
            fontCollection = new FontCollection(this),

            // the collection of document themes
            themeCollection = new ThemeCollection(this),

            // style collections mapped by attribute family
            styleCollections = {},

            // attribute definitions registry (mapped by attribute family)
            definitionsRegistry = {},

            // default values for all supported attribute families
            defaultAttributeSet = {};

        // public methods -----------------------------------------------------

        /**
         * Registers formatting attributes for a specific attribute family.
         *
         * @param {String} family
         *  The name of the attribute family the new formatting attributes are
         *  associated with. The attribute family 'document' is reserved for
         *  the definitions of global document attributes.
         *
         * @param {Object} definitions
         *  The attribute definitions map. Contains attribute definition
         *  objects for all attributes supported by the specified attribute
         *  family, mapped by the names of the attributes. This method may be
         *  called repeatedly for the same attribute family to add new
         *  attributes, or to overwrite existing attribute definitions. Each
         *  definition object contains the following properties:
         *  - {Number|String|Boolean|RegExp|Object} definition.def
         *      Specifies the default value of the attribute which will be used
         *      if neither a style sheet nor an explicit value exists for the
         *      attribute.
         *  - {String} [definition.scope='any']
         *      Specifies where the attribute can be used. If set to 'element',
         *      the attribute can be set at explicitly only (not in style
         *      sheets). If set to 'style', the attribute can be contained in
         *      style sheets only. By default, the attribute can be set
         *      explicitly as well as in style sheets.
         *  - {Function} [definition.merge]
         *      A function that will be called while merging attribute sets,
         *      for example while collecting attributes from style sheets and
         *      explicit attribute sets. Will be called to merge two existing
         *      values of this attribute, where the second value has to
         *      overwrite the first value in some way. Will be called in the
         *      context of this instance. The function receives the 'old'
         *      attribute value in the first parameter, and the 'new' attribute
         *      value in the second parameter. By default, the new value wins
         *      and the first value will be overwritten completely.
         *  The attribute definitions may contain any other user-defined
         *  properties.
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.registerAttributeDefinitions = function (family, definitions) {

            var // get existing or create a new entry in the global registry
                definitionsMap = definitionsRegistry[family] || (definitionsRegistry[family] = {}),
                // the container for the default attribute values
                defaultAttributes = defaultAttributeSet[family] || (defaultAttributeSet[family] = {});

            // store definitions, and the default attribute values
            _.each(definitions, function (definition, name) {
                if (_.isNull(definition.def) || _.isUndefined(definition.def) || _.isFunction(definition.def)) {
                    Utils.error('ModelAttributesMixin.registerAttributeDefinitions(): invalid attribute default value, family=' + family + ', name=' + name + ', value=', definition.def);
                } else {
                    definitionsMap[name] = _.clone(definition);
                    defaultAttributes[name] = definition.def;
                }
            });

            // reset all old settings for child attributes
            _.each(definitionsMap, function (definition) { delete definition.childAttrs; });

            // collect settings for child attributes
            _.each(definitionsMap, function (definition, name) {
                var parentName = Utils.getStringOption(definition.parent, 'name');
                if (_.isString(parentName) && (parentName in definitionsMap)) {
                    var parentDef = definitionsMap[parentName],
                        childAttrs = parentDef.childAttrs || (parentDef.childAttrs = []);
                    childAttrs.push({ name: name, value: definition.parent.value });
                }
            });

            return this;
        };

        /**
         * Returns the static definitions map of all attributes registered for
         * the specified attribute family.
         *
         * @param {String} family
         *  The attribute family whose attribute definitions will be returned.
         *
         * @returns {Object|Null}
         *  A reference to the static attribute definitions map, or null if the
         *  passed attribute family is invalid.
         */
        this.getAttributeDefinitions = function (family) {
            return (family in definitionsRegistry) ? definitionsRegistry[family] : null;
        };

        /**
         * Returns the global attributes of the document.
         *
         * @param {Object}
         *  A deep copy of the global document attributes.
         */
        this.getDocumentAttributes = function () {
            return _.copy(defaultAttributeSet.document, true);
        };

        /**
         * Returns the default attribute values of the specified attribute
         * families.
         *
         * @param {String|Array<String>} families
         *  A single attribute family, or an array of attribute families.
         *
         * @returns {Object}
         *  The attribute set containing all default values of the specified
         *  attribute families.
         */
        this.getDefaultAttributes = function (families) {

            var // the attribute set containing all specified families
                attributeSet = {};

            _.getArray(families).forEach(function (family) {
                if ((family in defaultAttributeSet) && (family !== 'document')) {
                    attributeSet[family] = _.copy(defaultAttributeSet[family], true);
                }
            });

            return attributeSet;
        };

        /**
         * Changes the global document attributes, and/or the default values
         * for various other formatting attributes. These values override the
         * defaults of the attribute definitions passed in the method
         * ModelAttributesMixin.registerAttributeDefinitions(), and will be
         * used before the values of any style sheet attributes and explicit
         * element attributes will be resolved.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set with new attribute default values.
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.setDocumentAttributes = function (attributeSet) {

            var // whether any document attribute has been changed
                docChanged = false,
                // whether any other default value has been changed
                defChanged = false,
                // copy of the old attribute defaults for change listeners
                oldAttributeSet = _.copy(defaultAttributeSet, true),
                // all changed attributes
                changedAttributeSet = {};

            // update defaults of all attribute families
            _.each(attributeSet, function (attributes, family) {
                if (_.isObject(attributes) && (family in defaultAttributeSet)) {

                    var defAttributes = defaultAttributeSet[family],
                        changedAttributes = changedAttributeSet[family] = {};

                    _.each(attributes, function (value, name) {
                        if ((name in defAttributes) && !_.isNull(value) && !_.isUndefined(value) && !_.isEqual(defAttributes[name], value)) {
                            defAttributes[name] = _.copy(value, true);
                            changedAttributes[name] = _.copy(value, true);
                            if (family === 'document') { docChanged = true; } else { defChanged = true; }
                        }
                    });
                    if (_.isEmpty(changedAttributeSet[family])) {
                        delete changedAttributeSet[family];
                    }
                }
            });

            // notify all change listeners for document attributes
            if (docChanged) {
                this.trigger('change:attributes', this.getDocumentAttributes(), oldAttributeSet.document, changedAttributeSet.document);
                delete oldAttributeSet.document;
                delete changedAttributeSet.document;
            }

            // notify all change listeners for default attribute values
            if (defChanged) {
                var newAttributeSet = _.copy(defaultAttributeSet, true);
                delete newAttributeSet.document;
                this.trigger('change:defaults', newAttributeSet, oldAttributeSet, changedAttributeSet);
            }

            return this;
        };

        /**
         * Builds an attribute set containing all formatting attributes of the
         * specified attribute families, set to the null value.
         *
         * @param {String|Array<String>} styleFamilies
         *  The name of a single attribute family, or an array with names of
         *  attribute families.
         *
         * @returns {Object}
         *  An attribute set with null values for all attributes registered for
         *  the specified attribute families.
         */
        this.buildNullAttributes = function (families) {

            var // the resulting attribute set
                attributeSet = {};

            _.getArray(families).forEach(function (family) {

                var // the attribute definitions of the current family
                    definitions = self.getAttributeDefinitions(family),
                    // attribute values of the current family
                    attributes = attributeSet[family] = {};

                // add null values for all registered attributes
                _.each(definitions, function (definition, name) {
                    if (!definition.special) {
                        attributes[name] = null;
                    }
                });
            });

            return attributeSet;
        };

        /**
         * Extends the passed attribute set with the existing values of the
         * second attribute set. If the definitions of a specific attribute
         * contain a merger function, and both attribute sets contain an
         * attribute value, uses that merger function to merge the values from
         * both attribute sets, otherwise the value of the second attribute set
         * will be copied to the first attribute set.
         *
         * @param {Object} attributeSet1
         *  (in/out) The first attribute set that will be extended in-place, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         *
         * @param {Object} attributeSet2
         *  The second attribute set, as map of attribute value maps
         *  (name/value pairs), keyed by attribute family, whose attribute
         *  values will be inserted into the first attribute set.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clone=false]
         *      If set to true, the first attribute set will be cloned deeply,
         *      instead of being extended in-place. This method still returns
         *      the new attribute set, but the attribute set passed in the
         *      parameter 'attributeSet1' will not be modified.
         *  @param {Boolean} [options.special=false]
         *      If set to true, includes special attributes (attributes that
         *      are marked with the 'special' flag in the attribute definitions
         *      passed to the constructor) to the result set.
         *  @param {Boolean} [options.autoClear=false]
         *      If set to true, formatting attributes with the value null in
         *      the second attribute set will be removed from the first
         *      attribute set. By default, attributes with the value null will
         *      be inserted into the first attribute set.
         *
         * @returns {Object}
         *  A reference to the first passed and extended attribute set.
         */
        this.extendAttributes = function (attributeSet1, attributeSet2, options) {

            // whether to delete null attributes from first attribute set
            var autoClear = Utils.getBooleanOption(options, 'autoClear', false);

            // clone the first attribute set if specified
            if (Utils.getBooleanOption(options, 'clone', false)) {
                attributeSet1 = _.copy(attributeSet1, true);
            }

            // add attributes existing in attributeSet2 to attributeSet1
            _.each(attributeSet2, function (attributes2, family) {

                // the attribute definitions of the current family
                var definitions = this.getAttributeDefinitions(family);

                // copy style identifier directly (may be string or null)
                if (family === 'styleId') {
                    attributeSet1.styleId = attributes2;
                    return;
                }

                // restrict to valid attribute families
                if (!definitions || !_.isObject(attributes2)) { return; }

                // insert attribute values of attributes2
                var attributes1 = attributeSet1[family] || (attributeSet1[family] = {});
                _.each(attributes2, function (value, name) {
                    if (AttributeUtils.isRegisteredAttribute(definitions, name, options)) {

                        // try to find merger function from attribute definition
                        var merger = definitions[name].merge;

                        // either set return value from merger, or copy the attribute directly
                        if (autoClear && (value === null)) {
                            delete attributes1[name];
                        } else if (merger && (name in attributes1) && (value !== null) && (attributes1[name] !== null)) {
                            attributes1[name] = merger.call(this, attributes1[name], value);
                        } else {
                            attributes1[name] = value;
                        }
                    }
                });

                // clean up -> deleting an empty attribute set
                if (_.isEmpty(attributes1)) { delete attributeSet1[family]; }

            }, this);

            return attributeSet1;
        };

        /**
         * Adds a style collection to the document.
         *
         * @param {StyleCollection} styleCollection
         *  The new style collection to be inserted into the document.
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.addStyleCollection = function (styleCollection) {
            styleCollections[styleCollection.getStyleFamily()] = styleCollection;
            return this;
        };

        /**
         * Returns the style collection for the specified attribute family.
         *
         * @param {String} family
         *  The name of the attribute family.
         *
         * @returns {StyleCollection|Null}
         *  The specified style collection if existing, otherwise null.
         */
        this.getStyleCollection = function (family) {
            return (family in styleCollections) ? styleCollections[family] : null;
        };

        /**
         * Returns the collection with all registered fonts.
         *
         * @returns {FontCollection}
         *  The collection with all registered fonts.
         */
        this.getFontCollection = function () {
            return fontCollection;
        };

        /**
         * Returns the collection with all registered themes.
         *
         * @returns {ThemeCollection}
         *  The collection with all registered themes.
         */
        this.getThemeCollection = function () {
            return themeCollection;
        };

        /**
         * Returns the current theme from the document's theme collection.
         *
         * @returns {Theme}
         *  The current theme from the document's theme collection.
         */
        this.getTheme = function () {
            return themeCollection.getTheme();
        };

        /**
         * Returns whether the current theme contains scheme color definitions.
         *
         * @returns {Boolean}
         *  Whether the current theme contains scheme color definitions.
         */
        this.hasSchemeColors = function () {
            return themeCollection.getTheme().hasSchemeColors();
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {
            return fontCollection.getCssFontFamily(fontName);
        };

        /**
         * Resolves the passed color. Scheme colors will be resolved using the
         * current document theme.
         *
         * @param {Color} color
         *  The color to be resolved.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolveColor = function (color, auto) {
            return color.resolve(auto, themeCollection.getTheme());
        };

        /**
         * Returns a color descriptor with details about the passed color,
         * assuming it to be a text color on a specific background style.
         * Scheme colors will be resolved using the current document theme. See
         * method Color.resolveText() for more details.
         *
         * @param {Color} textColor
         *  The color to be resolved.
         *
         * @param {Array<Color>} fillColors
         *  The source fill colors, from outermost to innermost level.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolveTextColor = function (textColor, fillColors) {
            return textColor.resolveText(fillColors, themeCollection.getTheme());
        };

        /**
         * Converts the passed color attribute value to a CSS color value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} jsonColor
         *  The JSON representation of the source color, as used in document
         *  operations.
         *
         * @param {String|Object} jsonAuto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @returns {String}
         *  The CSS color value converted from the passed color object.
         */
        this.getCssColor = function (jsonColor, jsonAuto) {
            var auto = _.isString(jsonAuto) ? jsonAuto : Color.parseJSON(jsonAuto);
            return this.resolveColor(Color.parseJSON(jsonColor), auto).css;
        };

        /**
         * Resolves the passed text color to an explicit color if it is set to
         * 'auto', according to the specified fill colors.
         *
         * @param {Object} jsonTextColor
         *  The JSON representation of the source text color, as used in
         *  document operations.
         *
         * @param {Array} jsonFillColors
         *  The JSON representation of the source fill colors, from outermost
         *  to innermost level, as used in document operations.
         *
         * @returns {String}
         *  The CSS text color value converted from the passed color objects.
         */
        this.getCssTextColor = function (jsonTextColor, jsonFillColors) {

            var // the text color, as instance of the class Color
                textColor = _.isObject(jsonTextColor) ? Color.parseJSON(jsonTextColor) : new Color('auto'),
                // all fill colors, as array of Color instances
                fillColors = jsonFillColors.map(Color.parseJSON);

            // convert to CSS color string
            return this.resolveTextColor(textColor, fillColors).css;
        };

        /**
         * Extends a JSON color object with the passed optional transformation,
         * and the effective RGB value for ODF files.
         *
         * @param {Object} jsonColor
         *  The original JSON color descriptor. MUST NOT contain a color
         *  transformation.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @param {Object} [transform]
         *  An optional color transformation to be inserted into the resulting
         *  JSON color.
         *
         * @returns {Object}
         *  The resulting JSON color with the color transformation, and for ODF
         *  files with an additional property 'fallbackValue' containing the
         *  resulting RGB color value.
         */
        this.createJSONColor = function (jsonColor, auto, transform) {

            // create a Color instance from the passed JSON color, and color transformation
            var color = new Color(jsonColor.type, jsonColor.value, transform ? [transform] : null);
            jsonColor = color.toJSON();

            // add the fall-back RGB value for ODF
            if (this.getApp().isODF()) { jsonColor.fallbackValue = this.resolveColor(color, auto).hex; }
            return jsonColor;
        };

        /**
         * Returns the effective CSS attributes for the passed border value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *  @param {Object} [options.autoColor]
         *      If specified, a replacement color to be used in case the border
         *      object contains the automatic color.
         *
         * @returns {Object}
         *  A map with CSS border attributes. Contains the property 'style'
         *  with the effective CSS border style as string; the property 'width'
         *  with the effective border width in pixels (as number, rounded to
         *  entire pixels); and the property 'color' with the effective CSS
         *  color (as string with leading hash sign).
         */
        this.getCssBorderAttributes = function (border, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorderAttributes(border, themeCollection.getTheme(), options);
        };

        /**
         * Converts the passed border attribute object to a CSS border value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.clearNone=false]
         *      If set to true, the return value for invisible borders will be
         *      the empty string instead of the keyword 'none'.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *  @param {Object} [options.autoColor]
         *      If specified, a replacement color to be used in case the border
         *      object contains the automatic color.
         *
         * @returns {String}
         *  The CSS border value converted from the passed border object.
         */
        this.getCssBorder = function (border, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorder(border, themeCollection.getTheme(), options);
        };

        /**
         * Returns the value of the CSS property 'text-decoration' for the
         * passed character attributes.
         *
         * @param {Object} charAttributes
         *  A map of character attributes.
         *
         * @returns {String}
         *  The resulting value of the CSS property 'text-decoration'.
         */
        this.getCssTextDecoration = function (charAttributes) {

            var result = '';

            function addToken(token) {
                if (result) { result += ' '; }
                result += token;
            }

            if (_.isBoolean(charAttributes.underline) && charAttributes.underline) { addToken('underline'); }
            if (_.isString(charAttributes.strike) && (charAttributes.strike !== 'none')) { addToken('line-through'); }

            return result || 'none';
        };

        // initialization -----------------------------------------------------

        this.registerAttributeDefinitions('document', DOCUMENT_ATTRIBUTE_DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.invoke(styleCollections, 'destroy');
            themeCollection.destroy();
            fontCollection.destroy();
            fontCollection = themeCollection = styleCollections = null;
        });

    } // class ModelAttributesMixin

    // exports ================================================================

    return ModelAttributesMixin;

});
