/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.io.InputStream;
import java.util.HashMap;

//------------
//- IManager -
//------------

/**
 * {@link IManager}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface IManager {

    /**
     * @param backendDescriptors the array of engine descriptors, containing the properties of each single backend engine
     * @param cacheDescriptor contains properties to setup a cache
     * @return the JobProcessor to execute single jobs or null in case of failure
     */
    public IJobProcessor init(ReaderengineDescriptor readerengineDescriptors[], CacheDescriptor cacheDescriptor);

    /**
     * kills all engine instances etc. and leaves the service with a clean status
     */
    public void end();

    /**
     * @param feature the feature that is queried to be supported
     * @return true if the given feature is supported by this IManager instance
     */
    boolean hasFeature(Feature feature, Object... params);

    /**
     * @return the successfully initialized JobProcessor or null in case of failure
     */
    IJobProcessor getJobProcessor();

    /**
     * @return The used converter cache or null if none is used
     */
    ICache getCache();

    /**
     * @return the installation path of the underlying DocumentReaderEngine or null if the engine wasn't properly initialized
     */
    public String getEnginePath();

    /**
     * @param jobType the export format for the conversion
     * @param jobProperties contains additional properties to customize the job
     * @return the new Job or null in case of failure
     */
    public IJob createServerJob(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @param jobType The job type fiter to be executed.
     * @param jobProperties Input parameter, containing properties to customize the job (e.g. the InputFile)
     * @param resultProperties Output parameter,  containing additional conversion result properties like e.g. the PageCount of the conversion result.
     * @return The InputStream with the conversion result in case of success or null in case of a failure
     */
    public InputStream convert(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @param jobType The job type fiter to be executed.
     * @param documentInputStream The InputStream, containing the source document to be converted.
     * @param jobProperties Input parameter, containing properties to customize the job (e.g. the InputFile)
     * @param resultProperties Output parameter,  containing additional conversion result properties like e.g. the error code of the conversion result.
     * @return The job id string to be used in further getConversionPage/endPageConversion jobs or null in case of a failure
     */
    public String beginPageConversion(String jobType, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @param jobIdStr The job id gotten from a previous call to beginPageConversion
     * @param pageNumber The number of the page to be returned as input stream.
     * @param jobProperties Input parameter, containing properties to further customize the job, can be empty in this case
     * @param resultProperties Output parameter,  containing additional conversion result properties like e.g. the PageCount of the conversion result.
     * @return The stream the converted page can be read from or null in case of a failure
     */
    public InputStream getConversionPage(String jobIdStr, int pageNumber, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @param jobIdStr The job id gotten from a previous call to beginPageConversion
     * @param jobProperties Input parameter, containing properties to further customize the job, can be empty in this case
     * @param resultProperties Output parameter, containing additional conversion result properties like e.g. the error code of the conversion result.
     */
    public void endPageConversion(String jobIdStr, HashMap<String, Object> jobProperties, HashMap<String, Object> resultProperties);

    /**
     * @return the created temporary File
     * @param sourceFileName if source document file name is set then the temp file will be created with this file name within a new folder
     */
    public File createTempFile(String prefix, String sourceFileName);

    /**
     * @return
     */
    public Config getConverterConfig();
}
