/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.documentconverter;

import java.io.File;
import java.io.Serializable;
import java.util.Date;
import java.util.HashMap;

/**
 * {@link ICacheEntry}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public interface ICacheEntry extends Serializable {

    /**
     * @return True if this is a valid cache entry.
     */
    public boolean isValid();

    /**
     * Frees resources associated with this entry. The entry is not valid after this call anymore.
     * Depending on the value of <code>clearPersist</code> and the persistence status of the entry,
     * the persistent part of the entry is freed/deleted as well
     *
     * @param clearPersist Set to true, if this entry should be deleted finally, even if the persist flag is set.
     */
    public void clear(boolean clearPersist);

    /**
     * @return The hash value of this entry
     */
    public String getHash();

    /**
     * @return The result properties of this entry;
     */
    public HashMap<String, Object> getResultProperties();

    /**
     * @return The path to the file containing the result data of this entry
     */
    public File getResultFile();

    /**
     * @return The local filesystem directory, where the entries persistent data is stored
     */
    public File getPersistentEntryDirectory();

    /**
     * @return The size of the entries persistent directory on its local file system
     */
    public long getPersistentSize();

    /**
     * Locks the entry, so that it won't be removed from the cache.
     */
    public void acquireLock(boolean synchronize);

    /**
     * Releases the entry lock, so that it can be removed from the cache.
     */
    public void releaseLock(boolean synchronize);

    /**
     * @return True if this object is currently locked
     */
    public boolean isLocked();

    /**
     * Updates the entries last access time.
     */
    public void touch();

    /**
     * @return The point of time, when the entry was accessed the last time
     */
    public Date getAccessTime();

    /**
     * Makes the data of this entry persistent, so that it can be completely recreated
     * at a later point of time; if the entry has been made
     * persistent before and doesn't need an update, the method returns true, also.
     *
     * @param persistRootDirectory The root directory (e.g. cache directory) where to
     *  make the entry persistent
     * @return true, if the file has been made persistent (even in a previous call)
     */
    boolean makePersistent(File persistRootDirectory);

    /**
     * @return true, if the entry has been made persistent before, not changing its content anymore
     */
    boolean isPersistent();
}
