#!/bin/bash

function help {
    cat << EOF
${0} [OPTIONS] --release=<RELEASE VERSION> <DIR 1> <Dir 2> ...

    OPTIONS:
        --release=<RELEASE VERSION>                   (mandatory)
        --branch=<BRANCH>                             Base branch to create a new release branch
                                                      (default develop)
        --OX6-frontend-release=<OX6_FRONTEND_RELEASE> (optional)
        --mobile-release=<MOBILE_RELEASE>             (optional)
        --auto-push=yes/no                            (optional)

    For updating the snapshot dependencies to after an release to 'backend-master,frontend-master'
    you can use the following options:

        --branch=<branch>
        --master-snapshot-dependency
        --auto-push=yes/no (optional)

    Examples:
        The following command creates the release-7.4.0 branch and mst common changes
        automatically. Please take a close look to the auto commited stuff afterwards.

        ${0} --release=7.4.0 --OX6-frontend-release=6.22.4 --auto-push=YES oxcs oxio
        ${0} --master-snapshot-dependency --branch=develop --auto-push=YES oxcs oxio
        ${0} --branch=release.7.6.2 --release=7.6.3 --OX6-frontend-release=6.22.11 -auto-push=yes oxcs

EOF
}

NEEDED_TOOLs="\
    ant
    git
"

# Check for needed tools
set +e
for tool in ${NEEDED_TOOLs}; do
    which ${tool} &> /dev/null
    if [ ! '0' -eq "${?}" ]; then
        echo "${0}: Error, needed tool not found: ${tool}" >&2
        exit 1
    fi
done
set -e

for argument in $@; do
    case ${argument} in
        --release=*)
            RELEASE=${argument##--release=}
        ;;
        --OX6-frontend-release=*)
            OX6_FRONTEND_RELEASE=${argument##--OX6-frontend-release=}
        ;;
        --mobile-release=*)
            MOBILE_RELEASE=${argument##--mobile-release=}
        ;;
        --master-snapshot-dependency*)
            MASTER_SNAPSHOT_DEPENDENCY=YES
        ;;
        --branch=*)
            BRANCH=${argument##--branch=}
        ;;
        --auto-push=*)
            AUTO_PUSH=$(echo ${argument##--auto-push=} | tr [A-Z] [a-z])
            if [ 'yes' == ${AUTO_PUSH} -o 'true' == "${AUTO_PUSH}" ]; then
                AUTO_PUSH=yes
            elif [ 'no' == ${AUTO_PUSH} -o 'false' == "${AUTO_PUSH}" ]; then
                AUTO_PUSH=no
            else
                echo "${0}: Error, unexpected value in --auto-push option: ${argument##--auto-push=}" >&2
                help
                exit 1
            fi
        ;;
        --*|-*)
            help
            exit 0
        ;;
        *)
            if [ -d "${argument}" ]; then
                CUSTOMER_LIST="${CUSTOMER_LIST} ${argument}"
            else
                echo "Directory not found: ${argument}"
                help
                exit 1
            fi
    esac
done

SNAPSHOT_IDENTIFIER=dependingProjectNames
if [ -n "${MASTER_SNAPSHOT_DEPENDENCY}" ]; then
    SNAPSHOT_DEPENDENCIES="${SNAPSHOT_IDENTIFIER}=backend-master,frontend-master"
else
    SNAPSHOT_DEPENDENCIES="${SNAPSHOT_IDENTIFIER}=backend-release-${RELEASE},frontend-release-${RELEASE}"
fi

if [ -n "${MASTER_SNAPSHOT_DEPENDENCY}" -a -n "${BRANCH}" -a -n "${CUSTOMER_LIST}" ]; then
    for customer in ${CUSTOMER_LIST}; do
        echo "Entering directory: ${customer}"
        cd ${customer}
        git remote update
        # Drop uncommited changes
        git checkout .
        git checkout -b TMP_LOCAL remotes/origin/${BRANCH}

        # Update snaptshot repositories
        sed -i "/${SNAPSHOT_IDENTIFIER}/d" ${customer}-packages/buildservice.properties
        echo ${SNAPSHOT_DEPENDENCIES} >> ${customer}-packages/buildservice.properties
        git add ${customer}-packages/buildservice.properties
        git commit -m "Change snapshot build dependency to: ${SNAPSHOT_DEPENDENCIES}"
            if [ 'yes' == "${AUTO_PUSH}" ]; then
                git push origin TMP_LOCAL:${BRANCH}
                git checkout develop
                git branch -D TMP_LOCAL
            fi
        cd ..
    done
    exit 0
fi

if [ -z "${RELEASE}" ]; then
    echo "Error, parameter '--release' is not defined"
    help
    exit 1
fi

###############################################################################
if [ -z "${BRANCH}" ]; then
    BRANCH=develop
fi

for customer in ${CUSTOMER_LIST}; do
    cd ${customer}
    BACKEND_APPSUITE_PSF_FILE=${customer}-packages/${customer}-packages-appsuite-and-backend.psf
    OX6_PSF_FILE=${customer}-packages/${customer}-packages-OX6-frontend.psf
    MOBILE_PSF_FILE=${customer}-packages/${customer}-packages-mobile-web-interface.list

    if [ ! -f "${BACKEND_APPSUITE_PSF_FILE}" -a -n "${OX6_FRONTEND_RELEASE}" ]; then
        NEW_BRANCH=release-${OX6_FRONTEND_RELEASE}
    else
        NEW_BRANCH=release-${RELEASE}
    fi

    echo "Prepare customer ${customer} repo for ${BRANCH}"

    echo "Update local copy of repository"
    git remote update

    # Check if release branch already exsist and skip
    if [ -n "$(git branch -a | grep remotes/origin/${NEW_BRANCH})" ]; then
        echo "Existing ${NEW_BRANCH} branch on remote repsository detected"
        echo "Skipping next steps for customer ${customer}"
    else
        # Drop uncommited changes
        git checkout .
        git checkout "${BRANCH}"
        git reset --hard "remotes/origin/${BRANCH}"
        git checkout -b ${NEW_BRANCH}

        # Update build tools
        echo "Update build tools"
        git submodule init
        git submodule update
        cd build
        git checkout master
        git pull
        cd ..

        # Commit build tools update
        if [ -n "$(git status | grep build)" ]; then
            git add build
            git commit -m "Update build tools to latest version"
        else
            echo "Build tools are already up-to-date"
        fi

        # Update snaptshot repositories
        sed -i "/${SNAPSHOT_IDENTIFIER}/d" ${customer}-packages/buildservice.properties
        echo ${SNAPSHOT_DEPENDENCIES} >> ${customer}-packages/buildservice.properties

        # Commit updated snapshot build settings
        git add ${customer}-packages/buildservice.properties
        git commit -m "Update snapshot build settings to ${RELEASE} build dependencies"

        # Prepare backend and appsuite custom plugins for new release
        if [ -f "${BACKEND_APPSUITE_PSF_FILE}" ]; then
            echo "Updating backend and appsuite custom plugins for new release"
            cd build
            ant -lib lib \
                      -f releaseNumbers.xml \
                      -DversionNumber=${RELEASE} \
                      "-DreleaseNotes=Prepare backend and appsuite custom plugins for ${RELEASE} release" \
                      -DprojectSets=${customer}-packages \
                      '-DpsfPattern=*${customer}-packages-appsuite-and-backend' \
                      loadProjectSets \
                      newVersion \
                      ;
            # Commit updated backend and appsuite custom plugins plugin changes
            cd ..
            git add .
            git commit -m "Prepare backend and appsuite custom plugins for ${RELEASE} release"
        else
            echo "Skipping, no backend and appsuite custom plugins found"
        fi

        # Prepare for OX6 frontend custom plugins for new release
        if [ -f "${OX6_PSF_FILE}" -a -n "${OX6_FRONTEND_RELEASE}" ]; then
            echo "Updating OX6 frontend custom plugins for new release"
            cd build
            ant -lib lib \
                      -f releaseNumbers.xml \
                      -DversionNumber=${OX6_FRONTEND_RELEASE} \
                      "-DreleaseNotes=Prepare OX6 frontend custom plugins for ${OX6_FRONTEND_RELEASE} release" \
                      -DprojectSets=${customer}-packages \
                      '-DpsfPattern=*${customer}-packages-OX6-frontend' \
                      loadProjectSets \
                      newVersion \
                      ;
            # Commit updated OX6 frontend custom plugins changes
            cd ..
            git add .
            git commit -m "Prepare OX6 frontend custom plugins for ${OX6_FRONTEND_RELEASE} release"
        else
            echo "Skipping, no OX6 frontend custom plugins found"
        fi

        # Prepare for mobile UI custom plugins for new release
        if [ -f "${MOBILE_PSF_FILE}" -a -n "${MOBILE_RELEASE}" ]; then
            echo "Updating mobile UI custom plugins for new release"
            cd build
            ant -lib lib \
                      -f releaseNumbers.xml \
                      -DversionNumber=${MOBILE_RELEASE} \
                      "-DreleaseNotes=Prepare mobile UI custom plugins for ${MOBILE_RELEASE} release" \
                      -DprojectSets=${customer}-packages \
                      '-DpsfPattern=*${customer}-packages-mobile-web-interface' \
                      loadProjectSets \
                      newVersion \
                      ;
            # Commit updated mobile UI custom plugin changes
            cd ..
            git add .
            git commit -m "Prepare mobile UI custom plugins for ${MOBILE_RELEASE} release"
        fi
        if [ 'yes' == "${AUTO_PUSH}" ]; then
            echo "Pushing ${customer} changes to remote ${BRANCH} branch"
            git push origin ${NEW_BRANCH}:${NEW_BRANCH}
            # Remove and recreate to point for pull on the correct branch
            git checkout develop
            git branch -D ${NEW_BRANCH}
            git checkout -b ${NEW_BRANCH} remotes/origin/${NEW_BRANCH}
        fi
    fi
    cd ..
done
