/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.sync;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.ConflictingChange;
import com.openexchange.usm.api.exceptions.ConflictingChangeException;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.api.session.exceptions.SynchronizationConflictException;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.DataObjectUtil;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.USMSessionManagementErrorCodes;

public class SlowContentSyncer extends BaseContentSyncer {

    public SlowContentSyncer(SessionManagerImpl sessionManager, int retryCount) {
        super(sessionManager, retryCount);
    }

    public SyncResult syncWithServer(ContentSyncerStorage storage, ConflictResolution conflictResolution, int resultLimit, DataObjectFilter filter, Comparator<? super DataObject> sorter, DataObject... clientData) throws USMException {
        if (_journal.isDebugEnabled())
            _journal.debug(storage + " Starting slow sync, resolution=" + conflictResolution);

        setParentFolderOwnerIDField(clientData);
        for (int count = 0; count <= _retryCount; count++) {
            if (filter != null)
                filter.initialize();
            DataObjectSet newestCachedData = null;
            // First, make sure that provided elements are in state unmodified for correct comparison with cache / server data
            for (DataObject f : clientData)
                f.commitChanges();
            // 0) clear cache for folder or folder hierarchy
            // 1) retrieve server information on DataObjects
            DataObjectSet serverDataObjects = getCurrentOrCachedServerData(storage, clientData.length == 0);
            for (DataObject f : serverDataObjects) {
                newestCachedData = _sessionManager.insertStoredUUID(f, null, newestCachedData);
                f.commitChanges();
            }
            DataObjectSet onlyOnServer = new DataObjectSet(serverDataObjects);
            // 2) compare server with client information
            // - build list of changes to send to server
            // - build list of changes to report to client
            // - determine if conflicts exist
            List<DataObject> onlyOnClient = new ArrayList<DataObject>();
            List<ConflictingChange> uuidErrors = new ArrayList<ConflictingChange>();
            List<ConflictingChange> conflictingChanges = new ArrayList<ConflictingChange>();
            DataObjectSet clientDataObjects = new DataObjectSet(clientData);
            List<DataObject> changesForServer = new ArrayList<DataObject>();
            List<DataObject> changesForClient = new ArrayList<DataObject>();
            // First, try to match by IDs
            for (Iterator<DataObject> i = onlyOnServer.iterator(); i.hasNext();) {
                DataObject sf = i.next();
                DataObject cf = clientDataObjects.get(sf.getID());
                if (cf == null) {
                    cf = clientDataObjects.get(sf.getUUID());
                    if (cf != null) {
                        String clientID = cf.getID();
                        if (clientID != null && clientID.length() > 0) {
                            if (!sf.getID().equals(clientID))
                                uuidErrors.add(new ConflictingChange(cf, sf));
                        } else {
                            cf.setID(sf.getID());
                        }
                    }
                }
                if (cf != null) {
                    checkPossibleConflicts(uuidErrors, conflictingChanges, sf, cf);
                    // remove identical objects
                    i.remove();
                    clientDataObjects.remove(cf);
                }
            }
            if (!uuidErrors.isEmpty()) {
                throw new SynchronizationConflictException(
                    USMSessionManagementErrorCodes.CONFLICTS_SLOW_SYNC_UUID_ERRORS,
                    "UUID conflicts between server and client",
                    uuidErrors.toArray(new ConflictingChange[uuidErrors.size()]));
            }
            // Next, try to match by complete equality without IDs for remaining unresolved elements
            for (Iterator<DataObject> i = onlyOnServer.iterator(); i.hasNext();) {
                DataObject sf = i.next();
                matchEqualityWithoutId(clientDataObjects, changesForClient, i, sf);
            }
            // Finally, try to find possible matches by consulting the ContentType handlers
            clientDataObjects = findPossibleChanges(_journal, onlyOnServer, conflictingChanges, clientDataObjects);
            onlyOnClient.addAll(clientDataObjects);
            // 3) resolve conflicts (move to list of changes to server or to client or generate error)
            if (!conflictingChanges.isEmpty())
                resolveConflicts(conflictResolution, conflictingChanges, changesForServer, changesForClient);

            Map<DataObject, USMException> errorMap = new HashMap<DataObject, USMException>();
            // 4) send all updates to server
            // - if sync error occurs, retry from step 0
            List<DataObject> serverChanges = new ArrayList<DataObject>();
            try {
                ContentSyncerSupport.updateOnServer(
                    _sessionManager,
                    serverDataObjects,
                    onlyOnClient,
                    changesForServer,
                    null,
                    errorMap,
                    serverChanges);
            } catch (ConflictingChangeException sce) {
                _journal.warn(storage + " Conflicting change while performing slow sync", sce);
                continue;
            }

            boolean incomplete = false;
            Set<Object> objectGroups = new HashSet<Object>();

            // Add server modifications - only as much as the limit allows
            for (DataObject modification : sortElements(serverChanges, sorter)) {
                if (shouldBeAddedToResult(resultLimit, changesForClient.size(), filter, ChangeState.MODIFIED, modification, objectGroups)) {
                    changesForClient.add(modification);
                } else {
                    incomplete = true;
                    // the server changes should not be removed from the data object list because they are still not in the list
                }
            }

            // Only add as many server creations to the result as the limit dictates, undo the remaining server creations in sync state
            for (DataObject sf : sortElements(onlyOnServer, sorter)) {
                if (shouldBeAddedToResult(resultLimit, changesForClient.size(), filter, ChangeState.CREATED, sf, objectGroups)) {
                    sf = sf.createCopy(true);
                    sf.setChangeState(ChangeState.CREATED);
                    changesForClient.add(sf);
                } else {
                    incomplete = true;
                    serverDataObjects.remove(sf.getID());
                }
            }
            if (_oxDataCache != null)
                _oxDataCache.setSyncComplete(!incomplete);
            // 5) write DataObjects in cache
            // 6) report changes to client
            SyncResult result = ContentSyncerSupport.storeDataObjectsAndBuildClientResult(
                incomplete,
                storage,
                serverDataObjects,
                changesForClient,
                errorMap);
            if (_journal.isDebugEnabled())
                _journal.debug(storage + " Slow sync finished: " + result.getDescription());
            return result;
        }
        throw new SynchronizationConflictException(
            USMSessionManagementErrorCodes.TOO_MANY_CONFICTS_SLOW_SYNC,
            "Too many repeated conflicts on slow sync");
    }

    private void checkPossibleConflicts(List<ConflictingChange> uuidErrors, List<ConflictingChange> conflictingChanges, DataObject sf, DataObject cf) {
        UUID clientUUID = cf.getUUID();
        UUID serverUUID = sf.getUUID();
        if (clientUUID == null) {
            cf.setUUID((serverUUID == null) ? UUID.randomUUID() : serverUUID);
        } else {
            if (serverUUID != null && !serverUUID.equals(clientUUID)) {
                uuidErrors.add(new ConflictingChange(cf, sf));
                return;
            }
        }
        // if conflict, store it
        if (!sf.equals(cf))
            conflictingChanges.add(new ConflictingChange(cf, sf));
        sf.linkUUIDTo(cf);
    }

    private void matchEqualityWithoutId(DataObjectSet clientDataObjects, List<DataObject> changesForClient, Iterator<DataObject> i, DataObject sf) throws StorageAccessException, USMStorageException {
        for (Iterator<DataObject> j = clientDataObjects.iterator(); j.hasNext();) {
            DataObject cf = j.next();
            if (cf.getUUID() == null && sf.equalsWithoutID(cf)) {
                updateProtocolInformation(cf, sf);
                UUID serverUUID = insertMissingUUID(sf);
                DataObject clientChange = cf.createCopy(false);
                clientChange.setUUID(serverUUID);
                clientChange.setID(sf.getID());
                changesForClient.add(clientChange);
                i.remove();
                j.remove();
                return;
            }
        }
        insertMissingUUID(sf);
    }

    private UUID insertMissingUUID(DataObject sf) {
        UUID serverUUID = sf.getUUID();
        if (serverUUID == null) { // Object is either mail or mail folder -> create random session specific UUID
            serverUUID = UUID.randomUUID();
            sf.setUUID(serverUUID);
        }
        return serverUUID;
    }

    private DataObjectSet findPossibleChanges(Log journal, DataObjectSet onlyOnServer, List<ConflictingChange> conflictingChanges, DataObjectSet clientDataObjects) {
        DataObjectSet result = new DataObjectSet();
        for (DataObject cf : clientDataObjects) {
            ConflictingChange bestMatch = getBestMatch(cf, onlyOnServer);
            if (bestMatch != null) {
                if (journal.isDebugEnabled())
                    journal.debug(bestMatch.getServerChange().getSession() + " Found match: " + bestMatch);
                DataObject serverChange = bestMatch.getServerChange();
                DataObject clientChange = bestMatch.getClientChange();
                onlyOnServer.remove(serverChange);
                conflictingChanges.add(bestMatch);
                updateProtocolInformation(clientChange, serverChange);
            } else {
                result.add(cf);
            }
        }
        return result;
    }

    private void resolveConflicts(ConflictResolution conflictResolution, List<ConflictingChange> conflictingChanges, List<DataObject> changesForServer, List<DataObject> changesForClient) throws SynchronizationConflictException, InternalUSMException {
        switch (conflictResolution) {
        case ERROR:
        case ERROR_DELETE_OVER_CHANGE:
            throw new SynchronizationConflictException(
                USMSessionManagementErrorCodes.CONFLICTS_SLOW_SYNC,
                "Conflicting changes between server and client",
                conflictingChanges.toArray(new ConflictingChange[conflictingChanges.size()]));
        case USE_CLIENT:
        case USE_CLIENT_DELETE_OVER_CHANGE:
            for (ConflictingChange a : conflictingChanges) {
                DataObject cf = a.getClientChange();
                DataObject sf = a.getServerChange();
                // Even if we use client information, we make sure to update the id to the server id,
                // the timestamp to the current server timestamp and if no parent folder information
                // is set for the client data, we use that of the server
                DataObject clientChange = cf.createCopy(true);
                clientChange.setTimestamp(sf.getTimestamp());
                clientChange.setID(sf.getID());
                if (clientChange.getParentFolderID() == null)
                    clientChange.setParentFolderID(sf.getParentFolderID());
                DataObject serverChange = DataObjectUtil.copyAndModify(sf, clientChange, true);
                if (serverChange.isModified())
                    changesForServer.add(serverChange);
                if (clientChange.isModified())
                    changesForClient.add(clientChange);
            }
            return;
        case USE_SERVER:
        case USE_SERVER_DELETE_OVER_CHANGE:
            for (ConflictingChange a : conflictingChanges) {
                DataObject cf = a.getClientChange();
                DataObject sf = a.getServerChange();
                changesForClient.add(DataObjectUtil.copyAndModify(cf, sf, true));
            }
            return;
        }
        throw new InternalUSMException(
            USMSessionManagementErrorCodes.ILLEGAL_CONFLICT_RESOLUTION_NUMBER2,
            "Illegal ConflictResolution " + conflictResolution);
    }

    private ConflictingChange getBestMatch(DataObject clientObject, DataObjectSet serverDataObjects) {
        int rating = -1;
        ConflictingChange bestMatch = null;
        for (DataObject sf : serverDataObjects) {
            int currentRating = clientObject.getContentType().getMatchRating(clientObject, sf);
            if (currentRating > rating) {
                rating = currentRating;
                bestMatch = new ConflictingChange(clientObject, sf);
            }
        }
        return bestMatch;
    }

    private void updateProtocolInformation(DataObject clientChange, DataObject serverChange) {
        if (clientChange.getProtocolInformation() != null) {
            if (serverChange.getProtocolInformation() == null)
                serverChange.setProtocolInformation(clientChange.getProtocolInformation());
        } else {
            if (serverChange.getProtocolInformation() != null) // Normally not possible (the protocol doesn't know about the new server
                                                               // object yet), just for completeness
                clientChange.setProtocolInformation(serverChange.getProtocolInformation());
        }
    }
}
