/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.jmx;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.management.MBeanFeatureInfo;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanParameterInfo;
import javax.management.NotCompliantMBeanException;
import javax.management.StandardMBean;
import com.openexchange.usm.session.impl.PersistentSessionData;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;

public class USMSessionInformation extends StandardMBean implements USMSessionInformationMBean {

    private final static Map<String, String> DESCRIPTIONS = new HashMap<String, String>();

    private final static Map<String, String[]> PARAMETER_NAMES = new HashMap<String, String[]>();

    private final static Map<String, String[]> PARAMETER_DESCRIPTIONS = new HashMap<String, String[]>();

    static {
        DESCRIPTIONS.put(
            "WaitForChangesEmailPullDelay",
            "Delay in ms. (>= 0) or percent (< 0) after which an internal pull is performed if a client waits on email folder changes");
        DESCRIPTIONS.put(
            "WaitForChangesEmailMinPullDelay",
            "Minimum time in ms. before an internal pull is performed if a client waits on email folder changes. A value < 0 disables internal email pulling completely");
        DESCRIPTIONS.put("MaxSyncStatesInDB", "Maximum number of SyncStates kept in database for any folder of a session");
        DESCRIPTIONS.put("ActiveSessionCount", "Number of currently active (i.e. stored in local memory) USM sessions");
        DESCRIPTIONS.put("listActiveSessions", "Lists all currently active USM sessions");
        DESCRIPTIONS.put(
            "getPropertiesOfActiveSessions",
            "Lists persistent session properties for active sessions (i.e. in memory), filtered by the given parameters. 0, null, \"\" or \"String\" can be used to ignore a specific filter");
        DESCRIPTIONS.put(
            "updatePropertiesOfActiveSessions",
            "Updates a persistent session property for all active sessions (i.e. in memory) indicated by the filter parameters. To clear a property, use a value of null or \"\"");
        DESCRIPTIONS.put(
            "getPropertiesOfAllSessions",
            "Lists persistent session properties for all sessions (i.e. in database), filtered by the given parameters. 0, null, \"\" or \"String\" can be used to ignore a specific filter");
        DESCRIPTIONS.put(
            "updatePropertiesOfAllSessions",
            "Updates a persistent session property for all sessions (i.e. in database) indicated by the filter parameters. To clear a property, use a value of null or \"\"");

        DESCRIPTIONS.put(
            "USMSesssionCacheInactivityTimeout",
            "Currently set timeout (in seconds), after which an USM session will be explicitly removed from the memory cache if it hasn't been accessed");
        DESCRIPTIONS.put(
            "SyncStateCacheInactivityTimeout",
            "Currently set timeout (in seconds), after which a SyncState will be removed from the memory cache if it hasn't been accessed");
        DESCRIPTIONS.put("TotalUSMSessionsInMemory", "Number of USM sessions currently stored in memory");
        DESCRIPTIONS.put("TotalSyncStatesInMemory", "Number of USM SyncStates (for all sessions) currently stored in memory");
        DESCRIPTIONS.put("clearCacheCounters", "Resets all statistics counters for USM memory caches");
        DESCRIPTIONS.put(
            "NewUSMSessionsInMemory",
            "Number of USM sessions that have been created or loaded from DB (since last counter reset)");
        DESCRIPTIONS.put(
            "USMSessionsRemovedDueToInactivity",
            "Number of USM sessions that have been removed from the memory cache due to long inactivity (since the last counter reset)");
        DESCRIPTIONS.put(
            "USMSessionsRemovedByJVM",
            "Number of USM sessions that have been removed from the memory cache by the Java VM because the memory was needed for other data (since the last counter reset)");
        DESCRIPTIONS.put(
            "SyncStatesSavedToDatabase",
            "Number of new SyncStates that have been saved to the database (since the last counter reset)");
        DESCRIPTIONS.put(
            "SyncStatesLoadedFromDatabase",
            "Number of SyncStates that have been restored from the database (since the last counter reset)");
        DESCRIPTIONS.put(
            "SyncStatesRemovedDueToInactivity",
            "Number of SyncStates that have been removed from the memory cache due to inactivity (since the last counter reset)");
        DESCRIPTIONS.put(
            "MaxTotalConnections",
            "The overall maximum number of (concurrent) connections the USM may use for requests to the OX server.");
        DESCRIPTIONS.put("ConnectionsInUse", "Overall number of http connections in use (USM -> OX server)");

        PARAMETER_NAMES.put("getPropertiesOfActiveSessions", new String[] { "cid", "id", "user", "protocol", "device", "field" });
        PARAMETER_DESCRIPTIONS.put("getPropertiesOfActiveSessions", new String[] {
            "Context ID, set to 0 to not filter for a specific context", "User ID, set to 0 to not filter for a specific user id",
            "User Login, set to empty or \"String\" to not filter for a specific user login",
            "USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
            "Device-ID, set to empty or \"String\" to not filter for a specific device id",
            "Field, set to empty or \"String\" to show all fields with non-empty values" });

        PARAMETER_NAMES.put(
            "updatePropertiesOfActiveSessions",
            new String[] { "cid", "id", "user", "protocol", "device", "field", "value" });
        PARAMETER_DESCRIPTIONS.put("updatePropertiesOfActiveSessions", new String[] {
            "Context ID, set to 0 to not filter for a specific context", "User ID, set to 0 to not filter for a specific user id",
            "User Login, set to empty or \"String\" to not filter for a specific user login",
            "USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
            "Device-ID, set to empty or \"String\" to not filter for a specific device id",
            "Field, must be present, no more that 18 characters long", "Value, no more than 32 characters long" });

        PARAMETER_NAMES.put("getPropertiesOfAllSessions", new String[] { "cid", "id", "protocol", "device", "field" });
        PARAMETER_DESCRIPTIONS.put("getPropertiesOfAllSessions", new String[] {
            "Context ID, set to 0 to not filter for a specific context", "User ID, set to 0 to not filter for a specific user id",
            "USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
            "Device-ID, set to empty or \"String\" to not filter for a specific device id",
            "Field, set to empty or \"String\" to show all fields with non-empty values" });

        PARAMETER_NAMES.put("updatePropertiesOfAllSessions", new String[] { "cid", "id", "protocol", "device", "field", "value" });
        PARAMETER_DESCRIPTIONS.put("updatePropertiesOfAllSessions", new String[] {
            "Context ID, set to 0 to not filter for a specific context", "User ID, set to 0 to not filter for a specific user id",
            "USM Protocol, set to empty or \"String\" to not filter for a specific protocol",
            "Device-ID, set to empty or \"String\" to not filter for a specific device id",
            "Field, must be present, no more that 18 characters long", "Value, no more than 32 characters long" });
    }

    private final SessionManagerImpl _sessionManager;

    private final USMCacheInformation _usmCacheInformation;

    public USMSessionInformation(SessionManagerImpl sessionManager, USMCacheInformation usmCacheInformation) throws NotCompliantMBeanException {
        super(USMSessionInformationMBean.class);
        _sessionManager = sessionManager;
        _usmCacheInformation = usmCacheInformation;
    }

    @Override
    public int getActiveSessionCount() {
        return _sessionManager.getSessionStorage().getSessionCount();
    }

    @Override
    public int getMaxSyncStatesInDB() {
        return _sessionManager.getMaxSyncStatesInDB();
    }

    @Override
    public void setMaxSyncStatesInDB(int states) {
        _sessionManager.setMaxSyncStatesInDB(states);
    }

    @Override
    public int getWaitForChangesEmailMinPullDelay() {
        return _sessionManager.getWaitForChangesEmailMinPullDelay();
    }

    @Override
    public void setWaitForChangesEmailMinPullDelay(int value) {
        _sessionManager.setWaitForChangesEmailMinPullDelay(value);
    }

    @Override
    public int getWaitForChangesEmailPullDelay() {
        return _sessionManager.getWaitForChangesEmailPullDelay();
    }

    @Override
    public void setWaitForChangesEmailPullDelay(int value) {
        _sessionManager.setWaitForChangesEmailPullDelay(value);
    }

    @Override
    public String[] listActiveSessions() {
        List<SessionImpl> list = _sessionManager.getSessionStorage().getSessionList();
        if (list.isEmpty())
            return new String[] { ">> No active sessions <<" };
        String[] result = new String[list.size()];
        for (int i = 0; i < result.length; i++)
            result[i] = list.get(i).getDescription();
        return result;
    }

    @Override
    public String[] getPropertiesOfActiveSessions(int cid, int id, String user, String protocol, String device, String field) {
        List<String> result = new ArrayList<String>();
        for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
            if (accept(session, cid, id, user, protocol, device)) {
                if (field != null && field.length() > 0 && !field.equals("String")) {
                    result.add(session.getDescription() + ": " + field + " = " + session.getPersistentField(field));
                } else {
                    result.add(session.getDescription());
                    for (Map.Entry<String, String> entry : session.getPersistentFields().entrySet())
                        result.add("    " + entry.getKey() + " = " + entry.getValue());
                }
            }
        }
        if (result.isEmpty())
            return new String[] { ">> No matching active sessions or fields <<" };
        return result.toArray(new String[result.size()]);
    }

    @Override
    public String updatePropertiesOfActiveSessions(int cid, int id, String user, String protocol, String device, String field, String value) {
        if (field == null || field.length() == 0 || field.equals("String"))
            throw new IllegalArgumentException("No field specified");
        if (field.length() > 18)
            throw new IllegalArgumentException("Field name too long (18 characters maximum)");
        if (value != null && value.length() > 32)
            throw new IllegalArgumentException("Field value too long (32 characters maximum)");
        int count = 0;
        for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
            if (accept(session, cid, id, user, protocol, device)) {
                try {
                    session.setPersistentField(field, value);
                } catch (Exception e) {
                    throw new IllegalStateException("Error while updating " + session.getDescription() + ": " + e, e);
                }
                count++;
            }
        }
        return ">> " + count + " fields updated <<";
    }

    private static boolean accept(SessionImpl session, int cid, int id, String user, String protocol, String device) {
        if (cid != 0 && session.getContextId() != cid)
            return false;
        if (id != 0 && session.getUserIdentifier() != id)
            return false;
        if (user != null && user.length() > 0 && !user.equals("String") && !session.getUser().equals(user))
            return false;
        if (protocol != null && protocol.length() > 0 && !protocol.equals("String") && !session.getProtocol().equals(protocol))
            return false;
        if (device != null && device.length() > 0 && !device.equals("String") && !session.getDevice().equals(device))
            return false;
        return true;
    }

    @Override
    protected String getDescription(MBeanInfo info) {
        return "Management Interface for USM Sessions";
    }

    @Override
    protected String getDescription(MBeanFeatureInfo info) {
        String v = DESCRIPTIONS.get(info.getName());
        return (v != null) ? v : super.getDescription(info);
    }

    @Override
    protected String getDescription(MBeanOperationInfo op, MBeanParameterInfo param, int sequence) {
        String[] v = PARAMETER_DESCRIPTIONS.get(op.getName());
        return (v != null) ? v[sequence] : super.getDescription(op, param, sequence);
    }

    @Override
    protected String getParameterName(MBeanOperationInfo op, MBeanParameterInfo param, int sequence) {
        String[] v = PARAMETER_NAMES.get(op.getName());
        return (v != null) ? v[sequence] : super.getDescription(op, param, sequence);
    }

    @Override
    public String[] getPropertiesOfAllSessions(int cid, int id, String protocol, String device, String field) {
        List<String> result = PersistentSessionData.getFieldsFromDB(_sessionManager, cid, id, protocol, device, field);
        return result.toArray(new String[result.size()]);
    }

    @Override
    public String[] updatePropertiesOfAllSessions(int cid, int id, String protocol, String device, String field, String value) {
        List<String> result = PersistentSessionData.updateFieldsInDB(_sessionManager, cid, id, protocol, device, field, value);
        return result.toArray(new String[result.size()]);
    }

    @Override
    public int getUSMSesssionCacheInactivityTimeout() {
        return _usmCacheInformation.getUSMSesssionCacheInactivityTimeout();
    }

    @Override
    public void setUSMSesssionCacheInactivityTimeout(int timeout) {
        _usmCacheInformation.setUSMSesssionCacheInactivityTimeout(timeout);
    }

    @Override
    public int getSyncStateCacheInactivityTimeout() {
        return _usmCacheInformation.getSyncStateCacheInactivityTimeout();
    }

    @Override
    public void setSyncStateCacheInactivityTimeout(int timeout) {
        _usmCacheInformation.setSyncStateCacheInactivityTimeout(timeout);
    }

    @Override
    public int getTotalUSMSessionsInMemory() {
        return _usmCacheInformation.getTotalUSMSessionsInMemory();
    }

    @Override
    public int getTotalSyncStatesInMemory() {
        return _usmCacheInformation.getTotalSyncStatesInMemory();
    }

    @Override
    public void clearCacheCounters() {
        _usmCacheInformation.clearCacheCounters();
    }

    @Override
    public int getNewUSMSessionsInMemory() {
        return _usmCacheInformation.getNewUSMSessionsInMemory();
    }

    @Override
    public int getUSMSessionsRemovedDueToInactivity() {
        return _usmCacheInformation.getUSMSessionsRemovedDueToInactivity();
    }

    @Override
    public int getUSMSessionsRemovedByJVM() {
        return _usmCacheInformation.getUSMSessionsRemovedByJVM();
    }

    @Override
    public int getSyncStatesSavedToDatabase() {
        return _usmCacheInformation.getSyncStatesSavedToDatabase();
    }

    @Override
    public int getSyncStatesLoadedFromDatabase() {
        return _usmCacheInformation.getSyncStatesLoadedFromDatabase();
    }

    @Override
    public int getSyncStatesRemovedDueToInactivity() {
        return _usmCacheInformation.getSyncStatesRemovedDueToInactivity();
    }

    @Override
    @SuppressWarnings("deprecation")
    public int getMaxTotalConnections() {
        return _sessionManager.getHttpConnectionManager().getMaxTotalConnections();
    }

    @Override
    public int getConnectionsInUse() {
        return _sessionManager.getHttpConnectionManager().getConnectionsInPool();
    }
}
