/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.io.Serializable;
import java.util.Arrays;

/**
 * @author afe
 */
public class CachedData {

    private final long[] _timestamps;

    private final long[] _lastAccess;

    private final Serializable[][][] _data;

    private int _insertionIndex = 0;

    public CachedData(int cacheSize) {
        _timestamps = new long[cacheSize];
        _lastAccess = new long[cacheSize];
        _data = new Serializable[cacheSize][][];
        for (int i = 0; i < cacheSize; i++)
            _timestamps[i] = Long.MAX_VALUE;
    }

    public long loadFromDB(long timestamp, Serializable[][] data) {
        return internalStore(timestamp, data, 0L);
    }

    public long store(long timestamp, Serializable[][] data, long timestampToKeep) {
        if (_insertionIndex > 0) {
            // We have old entries, make sure the new timestamp is newer than the newest one already stored
            long lastTimestamp = _timestamps[_insertionIndex - 1];
            if (timestamp <= lastTimestamp)
                timestamp = lastTimestamp + 1;
        }
        return internalStore(timestamp, data, timestampToKeep);
    }

    private synchronized long internalStore(long timestamp, Serializable[][] data, long timestampToKeep) {
        // Search for first(i.e. oldest) index new enough to keep
        int keepIndex = (timestampToKeep == 0L) ? 0 : Arrays.binarySearch(_timestamps, timestampToKeep);
        if (keepIndex < 0) // This should never be possible, just to be sure
            keepIndex = 1 + keepIndex;
        int size = _timestamps.length;
        if (keepIndex > 0) {
            // We need to delete at least 1 entry per "normal" algorithm, therefore we always can store the new entry
            for (int i = keepIndex; i < size; i++) {
                int j = i - keepIndex;
                _timestamps[j] = _timestamps[i];
                _lastAccess[j] = _lastAccess[i];
                _data[j] = _data[i];
            }
            for (int i = size - keepIndex; i < size; i++) {
                _timestamps[i] = Long.MAX_VALUE;
                _lastAccess[i] = 0L;
                _data[i] = null;
            }
            _insertionIndex -= keepIndex;
            _timestamps[_insertionIndex] = timestamp;
            _lastAccess[_insertionIndex] = System.currentTimeMillis();
            _data[_insertionIndex++] = data;
        } else if (_insertionIndex >= size) {
            // We have to delete an old index because we reached the cache size limit
            // keepIndex must be 0, based on keepSpecifiedTimestamp we have to delete either the first or the second entry
            int start = (timestampToKeep > 0L) ? 2 : 1;
            for (int i = start; i < size; i++) {
                _timestamps[i - 1] = _timestamps[i];
                _lastAccess[i - 1] = _lastAccess[i];
                _data[i - 1] = _data[i];
            }
            _timestamps[size - 1] = timestamp;
            _lastAccess[size - 1] = System.currentTimeMillis();
            _data[size - 1] = data;
        } else {
            // We can simply add the new entry at the end of the cache list, there is nothing to delete and the cache limit has not been
            // reached
            _timestamps[_insertionIndex] = timestamp;
            _lastAccess[_insertionIndex] = System.currentTimeMillis();
            _data[_insertionIndex++] = data;
        }
        return timestamp;
    }

    public synchronized long getTimestamp(int index) {
        return _timestamps[index];
    }

    public synchronized long getNewestTimestamp() {
        return _insertionIndex > 0 ? _timestamps[_insertionIndex - 1] : 0L;
    }

    public long touchNewestTimestamp() {
        if (_insertionIndex <= 0)
            return 0L;
        _lastAccess[_insertionIndex - 1] = System.currentTimeMillis();
        return _timestamps[_insertionIndex - 1];
    }

    public synchronized Serializable[][] get(long timestamp) {
        int index = Arrays.binarySearch(_timestamps, timestamp);
        if (index < 0)
            return null;
        _lastAccess[index] = System.currentTimeMillis();
        return _data[index];
    }

    public synchronized int getSize() {
        for (int i = 0; i < _timestamps.length; i++) {
            if (_timestamps[i] == Long.MAX_VALUE)
                return i;
        }
        return _timestamps.length;
    }

    /**
     * Remove all entries for which the lastAccess timestamp is earlier than the specified limit.
     * 
     * @param limit
     * @return Number of entries that have been removed
     */
    public synchronized int removeOldData(long limit) {
        int i = 0;
        int j = 0;
        while (i < _timestamps.length && _timestamps[i] != Long.MAX_VALUE) {
            if (_lastAccess[i] >= limit) {
                if (i != j) {
                    _timestamps[j] = _timestamps[i];
                    _lastAccess[j] = _lastAccess[i];
                    _data[j] = _data[i];
                }
                j++;
            }
            i++;
        }
        for (int k = j; k < i; k++) {
            _timestamps[k] = Long.MAX_VALUE;
            _lastAccess[k] = 0L;
            _data[k] = null;
        }
        int removed = i - j;
        _insertionIndex -= removed;
        return removed;
    }

    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder(100);
        sb.append("CachedData(").append(_timestamps.length);
        char c = ':';
        for (int i = 0; i < _insertionIndex; i++) {
            sb.append(c).append(_timestamps[i]);
            c = ',';
        }
        return sb.append(')').toString();
    }
}
