/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.mimemail;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Properties;
import javax.activation.DataHandler;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimePart;
import javax.mail.util.ByteArrayDataSource;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.connector.commands.CommandConstants;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.TempFile;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.Toolkit;

/**
 * This class converts the JSON structure with information about the mail to the mime mail. The JAVA Mail API is used for this aim.
 * 
 * @author ibr
 */
public class StreamingMimeMailBuilder extends AbstractMimeMailBuilder {

    private static void checkCorrectSMIMEFields(JSONObject mail, String... prohibitedFields) throws USMJSONAPIException {
        for (String field : prohibitedFields) {
            if (mail.has(field))
                throw new USMJSONAPIException(
                    ConnectorBundleErrorCodes.MIMEMAIL_MULTIPLE_SMIME_BODY_FIELDS,
                    ResponseStatusCode.BAD_REQUEST,
                    "Message contains more than 1 field of (body, smime_body_data, tempid)");
        }
    }

    private static byte[] getContentBytes(String data, String encoding, String contentType) throws USMJSONAPIException {
        if (CommandConstants.BASE64.equals(encoding))
            return Toolkit.decodeBase64(data);
        try {
            return data.getBytes(extractEncodingCharset(contentType));
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_OF_PART,
                ResponseStatusCode.INTERNAL_ERROR,
                "unsupported encoding of content: " + contentType);
        }

    }

    private final Session _session;
    private Collection<String> _tempIDs;

    public StreamingMimeMailBuilder(Session session) {
        super();
        _session = session;
        _tempIDs = new HashSet<String>();
    }

    public InputStreamProvider convertJSONObjectToMimeMail(JSONObject data) throws USMJSONAPIException {
        if (data.has(SMIME_BODY_DATA))
            return buildSMIMEMailFromSMIMEBodyData(data);
        if (data.has(CommandConstants.TEMPID))
            return buildSmimeMailFromSMIMETempId(data);
        buildMimeMailFromJSONObject(data);
        return new MailInputStreamProvider(_message);
    }

    private InputStreamProvider buildSmimeMailFromSMIMETempId(JSONObject data) throws USMJSONAPIException {
        checkCorrectSMIMEFields(data, BODY, SMIME_BODY_DATA);
        String tempid = data.optString(CommandConstants.TEMPID);
        try {
            final TempFile file = TempFileStorage.getTempFileForRead(_session, tempid);
            // keep tracking of tempid for cleanup
            getTempIDs().add(tempid);
            final byte[] headers = buildSMIMEHeaders(data);
            return new InputStreamProvider() {

                @Override
                public InputStream getInputStream() throws IOException {
                    InputStream bodyInput;
                    bodyInput = file.getInputStream();
                    ByteArrayInputStream headerInput = new ByteArrayInputStream(headers);
                    return new SequenceInputStream(headerInput, bodyInput);
                }
            };
        } catch (IOException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_TEMP_FILE_NOT_FOUND_FOR_MAIL_BODY,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Temporary data for mail body not found: " + e.getMessage(),
                e);
        }
    }

    private InputStreamProvider buildSMIMEMailFromSMIMEBodyData(JSONObject data) throws USMJSONAPIException {
        checkCorrectSMIMEFields(data, BODY, CommandConstants.TEMPID);
        byte[] headers = buildSMIMEHeaders(data);
        byte[] body = Toolkit.decodeBase64(data.optString(SMIME_BODY_DATA));
        final byte[] mail = new byte[headers.length + body.length];
        System.arraycopy(headers, 0, mail, 0, headers.length);
        System.arraycopy(body, 0, mail, headers.length, body.length);
        return new InputStreamProvider() {

            @Override
            public InputStream getInputStream() {
                return new ByteArrayInputStream(mail);
            }
        };
    }

    private byte[] buildSMIMEHeaders(JSONObject data) throws USMJSONAPIException {
        JSONObject headers = data.optJSONObject(HEADERS);
        StringBuilder sb = new StringBuilder(1000);
        try {
            Object contentType;
            contentType = (headers != null && headers.has(CONTENT_TYPE)) ? headers.get(CONTENT_TYPE) : null;
            String contentTypeOfPart = buildContentTypeAsString(contentType);
            // add headers to message
            MimePart part = new MimeBodyPart();
            appendHeadersToPart(part, headers, contentTypeOfPart);
            for (Enumeration<?> e = part.getAllHeaderLines(); e.hasMoreElements();) {
                sb.append((String) e.nextElement());
                sb.append("\r\n");
            }
            return sb.toString().getBytes(CommandConstants.UTF_8);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SMIMEMAIL_JSONEXCEPTION_ERROR,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "JSONException: " + e.getMessage());
        } catch (MessagingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SMIMEMAIL_JAVAPI_MESSAGING_ERROR,
                ResponseStatusCode.INTERNAL_ERROR,
                "error by building of body part: " + e.getMessage());
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.SMIMEMAIL_UNSUPPORTED_ENCODING_ERROR,
                ResponseStatusCode.INTERNAL_ERROR,
                "error by building of body part: " + e.getMessage());
        }
    }

    @Override
    protected void buildPart(MimePart part, JSONObject json) throws USMJSONAPIException {
        try {
            JSONObject headers = json.optJSONObject(HEADERS);
            Object contentType = (headers != null && headers.has(CONTENT_TYPE)) ? headers.get(CONTENT_TYPE) : _DEFAULT_CONTENT_TYPE;
            String contentTypeOfPart = buildContentTypeAsString(contentType);
            // add headers to message
            appendHeadersToPart(part, headers, contentTypeOfPart);
            if (contentTypeOfPart.startsWith(MULTIPART)) {
                // set multi part content
                String subtype = contentTypeOfPart.replaceFirst(MULTIPART, "");
                part.setContent(buildMimePartForMultipart(json, subtype), contentTypeOfPart);
            } else {
                JSONObject body = json.has(BODY) ? json.getJSONObject(BODY) : null;
                if (contentTypeOfPart.startsWith(CONTENTTYPE_RFC822)) {
                    USMMimeMessage nestedPart = new USMMimeMessage(javax.mail.Session.getInstance(new Properties()));
                    buildPart(nestedPart, body);
                    part.setContent(nestedPart, CONTENTTYPE_RFC822);
                } else if (contentTypeOfPart.startsWith(CONTENTTYPE_TEXT_PREFIX)) {
                        //add headers
                        String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, "quoted-printable");
                        setHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
                        part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
                        setBodyOfPart(part, contentTypeOfPart, body, headers);
                } else {
                    String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null);
                    setHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
                    part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
                    setBodyOfPart(part, contentTypeOfPart, body, headers);
                }
            }
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_JSONEXCEPTION_ERROR,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "JSONException: " + e.getMessage());
        } catch (MessagingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_JAVAPI_MESSAGING_ERROR,
                ResponseStatusCode.INTERNAL_ERROR,
                "error by building of body part: " + e.getMessage());
        }
    }

    private void setBodyOfPart(MimePart part, String contentType, JSONObject body, JSONObject headers) throws MessagingException, JSONException, USMJSONAPIException {
        if (body == null || headers == null)
            return;
        String simpleContentType = extractPlainContentType(contentType);
        if (body.has(CommandConstants.TEMPID)) {
            String tempid = body.getString(CommandConstants.TEMPID);
            // TempidDataSource calls TempFileStorage.getTempFileForRead, keep tracking of tempid for cleanup
            getTempIDs().add(tempid);
            setHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, CommandConstants.BASE64);
            part.setDataHandler(new DataHandler(new TempidDataSource(_session, tempid, simpleContentType)));
        } else if (body.has(CommandConstants.DATA_KEY)) {
            String data = body.getString(CommandConstants.DATA_KEY);
            byte[] content = getContentBytes(data, part.getEncoding(), contentType);
            part.setDataHandler(new DataHandler(new ByteArrayDataSource(content, simpleContentType)));
        } else {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_PART_HAS_NO_DATA_OR_TEMPID,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "Body of MimePart has neither 'data' nor 'tempid'");
        }
    }

    public Collection<String> getTempIDs() {
        return _tempIDs;
    }
}
