/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.DESTINATIONFOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.DESTINATIONSYNCID;
import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.SOURCEFOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.SOURCESYNCID;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_DETAILS;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_MESSAGE;
import static com.openexchange.usm.json.response.ResponseObject.ERROR_STATUS;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.OperationDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.util.OXJSONUtil;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * Handler for the MoveItems USM-JSON-Command.
 * @author ldo
 *
 */
public class MoveItemsHandler extends NormalCommandHandler {

	private static final String MOVE_ITEMS = "moveItems";
    private static final String ERRORS = "errors";
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID, SOURCEFOLDERID,  DESTINATIONFOLDERID, OBJECTIDS };
	private static final String[] OPTIONAL_PARAMETERS = { SOURCESYNCID, DESTINATIONSYNCID};

	private ErrorStatusCode _errorStatus = ErrorStatusCode.OTHER;
	private Map<String, ErrorObject> _errorsMap = new HashMap<String, MoveItemsHandler.ErrorObject>();

    public MoveItemsHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	private class ErrorObject {

	    private JSONObject _errorDetails;
	    private ErrorStatusCode _errorStatusCode;
	    private DataObject _dataObject;

        public ErrorObject(JSONObject _errorDetails, ErrorStatusCode _errorStatusCode, DataObject _dataObject) {
            this._errorDetails = _errorDetails;
            this._errorStatusCode = _errorStatusCode;
            this._dataObject = _dataObject;
        }


        public JSONObject getErrorDetails() {
            return _errorDetails;
        }


        public ErrorStatusCode getErrorStatusCode() {
            return _errorStatusCode;
        }

        public DataObject getDataObject() {
            return _dataObject;
        }

	}

	@Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        String sourceFolderUUID = getStringParameter(SOURCEFOLDERID);
        String destinationFolderUUID = getStringParameter(DESTINATIONFOLDERID);

        if (sourceFolderUUID.equals(destinationFolderUUID))
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MOVE_ITEMS_SAME_SOURCE_DEST,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                generateErrorMessage(
                    "Source and Destination are the same.",
                    sourceFolderUUID,
                    destinationFolderUUID));

        String acquirer = "moveItems:" + sourceFolderUUID + " -> " + destinationFolderUUID;
        String previousAcquirer = _session.tryLock(sourceFolderUUID, acquirer);
        if (previousAcquirer != null)
            return createCannotLockFolderResponse(MOVE_ITEMS, "source ", sourceFolderUUID, previousAcquirer);
        try {
            previousAcquirer = _session.tryLock(destinationFolderUUID, acquirer);
            if (previousAcquirer != null)
                return createCannotLockFolderResponse(MOVE_ITEMS, "destination ", destinationFolderUUID, previousAcquirer);
            try {
                Folder sourceFolder = getFolderByUUID(sourceFolderUUID);
                Folder destinationFolder = getFolderByUUID(destinationFolderUUID);

                JSONArray objectsToMove = getJSONArray(_parameters, OBJECTIDS);
                // move items
                JSONObject response = moveItems(sourceFolder, destinationFolder, objectsToMove);

                if (DefaultContentTypes.MAIL_CODE == sourceFolder.getElementsContentType().getCode())  {
                    // for mail folders a move of items results in a change of the folder hierarchy
                    _session.invalidateCachedData(null);
                }

                long sourceFolderSyncID = getLongParameter(SOURCESYNCID, 0L);
                long destinationSyncID = getLongParameter(DESTINATIONSYNCID, 0L);
                if(sourceFolderSyncID > 0 || destinationSyncID > 0) {
                    response.put(SOURCESYNCID, sourceFolderSyncID);
                    if(!response.has(DESTINATIONSYNCID)) response.put(DESTINATIONSYNCID, destinationSyncID);
                }
                return new ResponseObject(ResponseStatusCode.SUCCESS, response);
            } catch (JSONException e) {
                throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.MOVE_ITEMS_JSON_ERROR_2, e);
            } finally {
                _session.unlock(destinationFolderUUID);
            }
        } finally {
            _session.unlock(sourceFolderUUID);
        }
    }

    private boolean checkExisting(Folder folder) {
		BitSet fields = new BitSet();
		try {
			folder.getContentType().getTransferHandler().readFolder(folder, fields);
		} catch (USMException e) {
			return false;
		}
		return true;
	}

	private String generateErrorMessage(String message, Folder source, Folder dest) {
		return message + '(' + source.getUUID() + '/' + source.getID()  + "->" + dest.getUUID()	+ '/' + dest.getID()  + ')';
	}

	private String generateErrorMessage(String message, String sourceUUID, String destUUID) {
        return message + '(' + sourceUUID  + "->" + destUUID + ')';
    }

	private JSONObject moveItems(Folder sourceFolder, Folder destinationFolder, JSONArray objectsToMove) throws USMJSONAPIException {
		JSONObject responseData = new JSONObject();
		try {
			if (!sourceFolder.getElementsContentTypeID().equals(destinationFolder.getElementsContentTypeID()))
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_NOT_MATCHING_FOLERS,
						ResponseStatusCode.WRONG_MISSING_PARAMETERS, generateErrorMessage(
								"Source and Destination are not of same type.", sourceFolder, destinationFolder));
			DataObject[] cachedSourceElements = _session.getCachedFolderElements(sourceFolder.getID(),
					sourceFolder.getElementsContentType());
            if (cachedSourceElements == null)
                throw new USMJSONAPIException(ConnectorBundleErrorCodes.MOVE_ITEMS_INVALID_SYNC_ID,
                        ResponseStatusCode.UNKNOWN_SYNCID, generateErrorMessage("No sync state known for source folder. Please resync the source folder ", sourceFolder, destinationFolder));
            DataObjectSet sourceElements = new DataObjectSet(cachedSourceElements);

			List<DataObject> changesList = new ArrayList<DataObject>();
			int length = objectsToMove.length();
            for (int i = 0; i < length; i++) {
                String uuidString = getString(objectsToMove, i);
                try {
                    DataObject object = getDataObjectByUUID(sourceElements, UUIDToolkit.extractUUIDFromString(uuidString));
                    object.getContentType().getTransferHandler().readDataObject(object, _session.getFieldFilter(object.getContentType()));
                    object.commitChanges();
                    object.setParentFolder(destinationFolder);
                    changesList.add(object);
                } catch (DataObjectNotFoundException e) {
                    addErrorToMap(uuidString, null, e, ErrorStatusCode.UNKNOWN_UUID);
                } catch (MultipleOperationsOnDataObjectException e) {
                    addErrorToMap(uuidString, e.getDataObject(), e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
                    removeDuplicatesAndAddToError(changesList, uuidString, e);
                } catch (USMException e) {
                    if(isObjectMissingInSourceFolder(e.getOxErrorForJSONResponse()))
                        addErrorToMap(uuidString, null , e, ErrorStatusCode.OBJECT_DOES_NOT_EXIST_IN_SOURCE);
                    else
                        addErrorToMap(uuidString, null, e, ErrorStatusCode.OTHER);
                }
            }
            Map<DataObject, USMException> updateErrorsMap = new HashMap<DataObject, USMException>();
			for (DataObject dataObject : changesList) {
			    try {
                    dataObject.getContentType().getTransferHandler().writeUpdatedDataObject(dataObject, dataObject.getTimestamp());
                } catch (USMException e) {
                    updateErrorsMap.put(dataObject, e);
                }
            }

            // for the destination folder save and modify the new sync state - generates new desinationsyncid
                DataObject[] newDestinationState = getNewDestinationStateToSave(destinationFolder, changesList, updateErrorsMap);
                long newestTimestamp = _session.getNewestTimestamp(destinationFolder.getID());
                long newDestinationSyncID = _session.storeSyncState(
                    newestTimestamp,
                    newestTimestamp + 1,
                    destinationFolder.getID(),
                    newDestinationState);
                responseData.put(DESTINATIONSYNCID, newDestinationSyncID);


			initializeErrorMap(updateErrorsMap);

			if (!_errorsMap.isEmpty()) {
				JSONObject notmoved = new JSONObject();
				//reset to other
				_errorStatus = checkFolderExistance(sourceFolder, destinationFolder);
				int destinationRigths = getOwnRights(destinationFolder);
				int sourceRights = getOwnRights(sourceFolder);
				addErrorsToErrorResponse(notmoved, destinationRigths, sourceRights);
				responseData.put(ERRORS, notmoved);
			}
		} catch (USMStorageException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR, e);
		} catch (StorageAccessException e) {
			throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_2, e);
		} catch (USMJSONAPIException e) {
			throw e;
		} catch (JSONException e) {
			throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.MOVE_ITEMS_JSON_ERROR, e);
		} catch (USMException e) {
		    throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_3, e);
        }
		return responseData;
	}

	private boolean isObjectMissingInSourceFolder(JSONObject oxError) {
	    return OXJSONUtil.isMissingObjectErrorCode(oxError);
	}


	private DataObject[] getNewDestinationStateToSave(Folder folder, List<DataObject> requestedChanges, Map<DataObject, USMException> updateErrorsMap)
        throws USMException {
    DataObject[] oldState = _session.getCachedFolderElements(folder.getID(), folder.getElementsContentType());
    List<DataObject> resultList = new ArrayList<DataObject>();
    if (oldState != null)  resultList.addAll(Arrays.asList(oldState));
        for (DataObject o : requestedChanges) {
            if (!updateErrorsMap.containsKey(o)) {
                if (o.getParentFolderID().equals(folder.getID())) {
                    o.commitChanges();
                    resultList.add(o);
                } else {
                    updateErrorsMap.put(o, new USMException(
                        ConnectorBundleErrorCodes.MOVE_ITEMS_INTERNAL_ERROR_4,
                        "Trying to save data objects from different folders at the destination "));
                }
            }
        }
    return resultList.toArray(new DataObject[resultList.size()]);
}

	private void initializeErrorMap(Map<DataObject, USMException> updateResultMap) throws JSONException {
        for (Map.Entry<DataObject, USMException> entry : updateResultMap.entrySet()) {
            DataObject object = entry.getKey();
            USMException error = entry.getValue();
            if ((error instanceof OperationDeniedException) && object.getChangeState() == ChangeState.DELETED) {
                addErrorToMap(object.getUUID().toString(), object, error, ErrorStatusCode.NO_DELETE_PERMISSION);
            } else {
                addErrorToMap(object.getUUID().toString(), object, error, ErrorStatusCode.OTHER);
            }
        }
    }

	private void removeDuplicatesAndAddToError(List<DataObject> changesList, String uuidString, MultipleOperationsOnDataObjectException e) throws JSONException {
	    for(Iterator<DataObject> i = changesList.iterator(); i.hasNext();) {
	        DataObject o = i.next();
	        if(o.getUUID().toString().equals(uuidString)) {
	            i.remove();
                addErrorToMap(uuidString, o, e, ErrorStatusCode.MULTIPLE_OPERATIONS_ON_SAME_UUID);
	        }
	    }
    }

    private void addErrorsToErrorResponse(JSONObject notmoved, int destinationRigths, int sourceRights) throws JSONException {
        for (Map.Entry<String, ErrorObject> entry : _errorsMap.entrySet()) {
            String uuid = entry.getKey();
            ErrorObject data = entry.getValue();
            addErrorToNotMovedArray(sourceRights, destinationRigths, notmoved, uuid, data);
        }
    }

	private void addErrorToNotMovedArray(int sourceRights, int destinationRights, JSONObject notmoved, String uuid,
			ErrorObject error) throws JSONException {
		JSONObject errorObject = new JSONObject();

		errorObject.put(ERROR_DETAILS, error.getErrorDetails());
		errorObject.put(ERROR_STATUS, getErrorStatusCode(destinationRights, sourceRights, error.getDataObject(), error.getErrorStatusCode())
				.getStatusCode());
		notmoved.put(uuid, errorObject);
	}

    private void addErrorToMap(String uuid, DataObject dataObject, USMException exception, ErrorStatusCode errorStatus) throws JSONException {
        JSONObject details = exception.getErrorDetailsForJSONResponse();
        if (details == null)
            details = exception.getOxErrorForJSONResponse();
        if (details == null) {
            details = new JSONObject();
            details.put(ERROR_MESSAGE, exception.getMessage());
        }
        ErrorObject errorObject = new ErrorObject(details, errorStatus, dataObject);
        _errorsMap.put(uuid, errorObject);
    }


	private ErrorStatusCode getErrorStatusCode(int destinationRigths, int sourceRights, DataObject errorDataObject, ErrorStatusCode errorStatus) {
	    if(_errorStatus != ErrorStatusCode.OTHER)
            return _errorStatus;
	    if(errorStatus != ErrorStatusCode.OTHER)
	        return errorStatus;
		if (errorDataObject != null) {
			if (!checkCreatePermissionsAllObjects(destinationRigths))
				return ErrorStatusCode.DESTINATION_NO_PERMISSION;
			if (isOwnObject(errorDataObject) && !checkDeletePermissionsOwnObjects(sourceRights))
				return ErrorStatusCode.SOURCE_NO_PERMISSION_OWN_OBJECTS;
			if (!checkDeletePermissionsAllObjects(sourceRights))
				return ErrorStatusCode.SOURCE_NO_PERMISSION_ALL_OBJECTS;
		}
		return ErrorStatusCode.OTHER;
	}

	private ErrorStatusCode checkFolderExistance(Folder sourceFolder, Folder destinationFolder) {
		//check special error cases
		if (!checkExisting(destinationFolder))
			return ErrorStatusCode.DESTINATION_NOT_EXIST;
		if (!checkExisting(sourceFolder))
			return ErrorStatusCode.SOURCE_NOT_EXIST;
		return ErrorStatusCode.OTHER;
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}

}
