/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.clt.eas12;

import java.util.Arrays;

/**
 * {@link EASBase64Query} copied from com.openexchange.usm.eas.servlet
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class EASBase64Query {

    public static enum Parameter {
        AttachmentName("AttachmentName"), CollectionId("CollectionId"), CollectionName("CollectionName"), ItemId("ItemId"), LongId("LongId"), ParentId("ParentId"), Occurrence("Occurrence"), Options("Options"), RoundTripId("RoundtripId");
        
        
        private final String _name;
        
        private Parameter(String name) {
            _name = name;
        }
        
        @Override
        public String toString() {
            return _name;
        }
        
        public static Parameter parse(String name) {
            for (Parameter v : values()) {
                if (v.toString().equals(name))
                    return v;
            }
            return null;
        }
    }

    public static final String[] COMMAND_LIST = {
        "Sync", "SendMail", "SmartForward", "SmartReply", "GetAttachment", "GetHierarchy", "CreateCollection", "DeleteCollection",
        "MoveCollection", "FolderSync", "FolderCreate", "FolderDelete", "FolderUpdate", "MoveItems", "GetItemEstimate", "MeetingResponse",
        "Search", "Settings", "Ping", "ItemOperations", "Provision", "ResolveRecipients", "ValidateCert" };

    private final static String _HEX_CHARS = "0123456789abcdef";

    private static String convertToString(int[] data) {
        if (data == null)
            return null;
        boolean hasInvalidCharacters = false;
        for (int i = 0; i < data.length; i++) {
            if (data[i] < 0x20 || data[i] > 0x7f) {
                hasInvalidCharacters = true;
                break;
            }
        }
        StringBuilder sb = new StringBuilder(hasInvalidCharacters ? data.length * 2 + 2 : data.length);
        if (hasInvalidCharacters) {
            sb.append('0').append('x');
            for (int c : data)
                sb.append(_HEX_CHARS.charAt(c >> 4)).append(_HEX_CHARS.charAt(c & 0xF));
        } else {
            for (int c : data)
                sb.append((char) c);
        }
        return sb.toString();
    }

    private final EASProtocolVersion _protocolVersion;

    private final String _command;

    private final int _locale;

    private final String _deviceID;

    private final Long _policyKey;

    private final String _deviceType;

    private final int[][] _parameters = new int[Parameter.values().length][];

    public EASBase64Query(EASBase64RequestParser requestParser) {
        _protocolVersion = EASProtocolVersion.fromCode(requestParser.readByte());
        _command = extractCommand(requestParser.readByte());
        _locale = requestParser.readShort();
        _deviceID = requestParser.readString();
        int policyKeyLength = requestParser.readByte();
        _policyKey = policyKeyLength == 0 ? null : requestParser.readUnsignedInt();
        _deviceType = requestParser.readString();
        while (requestParser.hasMoreData()) {
            int parameterTag = requestParser.readByte();
            if (parameterTag < 0 || parameterTag >= _parameters.length)
                throw new IllegalArgumentException("Unknown parameter tag " + parameterTag);
            _parameters[parameterTag] = requestParser.readByteArray();
        }
    }

    private static String extractCommand(int commandCode) {
        if (commandCode < 0 || commandCode >= COMMAND_LIST.length)
            throw new IllegalArgumentException("Invalid encoded query String");
        return COMMAND_LIST[commandCode];
    }

    public String getDeviceID() {
        return _deviceID;
    }

    public String getDeviceType() {
        return _deviceType;
    }

    public String getCommand() {
        return _command;
    }

    public EASProtocolVersion getProtocolVersion() {
        return _protocolVersion;
    }

    public int getLocale() {
        return _locale;
    }

    /**
     * @return PolicyKey if set in query, null if not set
     */
    public Long getPolicyKey() {
        return _policyKey;
    }

    public int[] getParameter(Parameter parameter) {
        return _parameters[parameter.ordinal()];
    }

    public String getStringParameter(Parameter parameter) {
        return convertToString(_parameters[parameter.ordinal()]);
    }

    @Override
    public String toString() {
        return "Base64Query [protocol=" + _protocolVersion + ", command=" + _command + ", locale=" + _locale + ", deviceID=" + _deviceID + ", policyKey=" + _policyKey + ", deviceType=" + _deviceType + ", parameters=" + Arrays.toString(_parameters) + "]";
    }
}
