/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.clt;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServerConnection;
import javax.management.ReflectionException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import com.openexchange.usm.clt.eas12.EASBase64Query;
import com.openexchange.usm.clt.eas12.EASBase64RequestParser;

/**
 * {@link USMSessionTool} Tool that displays information about USM sessions and allows enabling/disabling of the EAS debug logging.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMSessionTool {

    private static final int _WIDTH = 76;

    private static final String _USM_PROTOCOL_JSON = "JSON";

    private static final String _USM_PROTOCOL_EAS = "EAS";

    private static final String _USM_FIELD_EAS_DEBUG_LOG = "eas.DebugLog";

    private static final Object _EAS_USER_PARAMETER = "User";

    private static final Object _EAS_DEVICE_ID_PARAMETER = "DeviceId";

    private static final Options toolOptions;

    private static final Options normalOptions;

    private static final Options helpOptions;

    static {
        final Options opts = new Options();
        final Options helpOpts = new Options();
        final Options normalOpts = new Options();
        OptionGroup easLogging = new OptionGroup();
        Option o = new Option(
            "e",
            "enableEASDebugLog",
            true,
            "Enable EAS debug logging with the given keyword in the log messages (only for EAS sessions)");
        o.setArgName("keyword");
        easLogging.addOption(o);
        easLogging.addOption(new Option(
            "r",
            "reportEASDebugLog",
            false,
            "Report if (and under which keyword) debug logging is enabled (only for EAS sessions)"));
        easLogging.addOption(new Option("d", "disableEASDebugLog", false, "Disable debug logging (only for EAS sessions)"));
        easLogging.setRequired(false);
        opts.addOptionGroup(easLogging);
        normalOpts.addOptionGroup(easLogging);
        OptionGroup verbosity = new OptionGroup();
        verbosity.addOption(new Option(
            "q",
            "quiet",
            false,
            "Quiet mode: Only print errors/warnings. If -r is specified, print only EAS debug logging keyword (or empty line if EAS debug logging is disabled)"));
        verbosity.addOption(new Option("v", "verbose", false, "Verbose output: Report all persistent fields for matching USM sessions"));
        verbosity.setRequired(false);
        opts.addOptionGroup(verbosity);
        normalOpts.addOptionGroup(verbosity);
        OptionGroup help = new OptionGroup();
        help.addOption(new Option("h", "help", false, "Prints this help text"));
        help.addOption(new Option("X", "examples", false, "Prints this help text with usage examples"));
        opts.addOptionGroup(help);
        helpOpts.addOptionGroup(help);
        o = new Option("l", "login", true, "The optional JMX login (if JMX has authentication enabled)");
        o.setArgName("login");
        opts.addOption(o);
        normalOpts.addOption(o);
        o = new Option("p", "port", true, "The optional JMX port (default:9999)");
        o.setArgName("port");
        opts.addOption(o);
        normalOpts.addOption(o);
        o = new Option("s", "password", true, "The optional JMX password (if JMX has authentication enabled)");
        o.setArgName("password");
        opts.addOption(o);
        normalOpts.addOption(o);
        toolOptions = opts;
        normalOptions = normalOpts;
        helpOptions = helpOpts;
    }

    private static void printHelpAndExit(boolean addExamples) {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.setOptionComparator(new Comparator<Option>() {

            @Override
            public int compare(Option o1, Option o2) {
                int prioDiff = getOptionPriority(o1.getId()) - getOptionPriority(o2.getId());
                return (prioDiff == 0) ? o1.getId() - o2.getId() : prioDiff;
            }

            private int getOptionPriority(int id) {
                if (id == 'e')
                    return -100;
                if (id == 'r')
                    return -90;
                if (id == 'd')
                    return -80;
                if (id == 'q')
                    return -70;
                if (id == 'v')
                    return -60;
                if (id == 'h')
                    return -50;
                if (id == 'X')
                    return -40;
                if (id == 'l')
                    return -30;
                if (id == 'p')
                    return -20;
                if (id == 's')
                    return -10;
                return 0;
            }
        });
        System.out.println(getUsageString(helpFormatter, helpOptions, false));
        System.out.println(getUsageString(helpFormatter, normalOptions, true) + " (<Session-Filter>|<Request-Parameters>)+");
        System.out.println();
        System.out.println("Session-filter:      [<ContextID>]:[<UserID>]:[<Session-ID>]:[<Protocol>]:[<DeviceID>]");
        System.out.println("Protocol:            (EAS|JSON)");
        System.out.println("Request-Parameters:  <URL> or <URL-Parameters>");
        System.out.println();
        PrintWriter pw = new PrintWriter(System.out);
        helpFormatter.printOptions(pw, _WIDTH, toolOptions, 1, 3);
        pw.println();
        helpFormatter.printWrapped(pw, _WIDTH, "Options -d,-e,-r are mutually exclusive, as are options -q,-v.");
        pw.println();
        helpFormatter.printWrapped(pw, _WIDTH, "Arguments may be given as Session-filter or Request-URL-parameters.");
        pw.println();
        helpFormatter.printWrapped(pw, _WIDTH, "Currently available protocols are EAS (Active Sync) and JSON (OX Connector for Outlook).");
        helpFormatter.printWrapped(
            pw,
            _WIDTH,
            "PLEASE NOTE: Enabling the EAS debug log for specific sessions is only possible for EAS sessions. If enabled, it logs communication data at level INFO. It does not enable logging of debug log messages at a higher level.");
        if (addExamples) {
            pw.println();
            helpFormatter.printWrapped(pw, _WIDTH, "Examples:");
            pw.println();
            helpFormatter.printWrapped(pw, _WIDTH, "Show all active EAS sessions:");
            pw.println("  # usmsessions :::EAS:");
            pw.println();
            helpFormatter.printWrapped(
                pw,
                _WIDTH,
                "For a given Apache log entry, show the session information, including context-ID and user-ID of the accounts for which the requests were made. This may be useful in situations where a single client misbehaves and it is necessary to either disable that account or contact the end user for more information.");
            helpFormatter.printWrapped(pw, _WIDTH, "EAS:");
            pw.println("  # usmsessions \"Microsoft-Server-ActiveSync?Cmd=<SomeCommand>&User=<user>&DeviceId=<SomeDeviceId>&DeviceType=<SomeType>\"");
            helpFormatter.printWrapped(pw, _WIDTH, "EAS12:");
            pw.println("  # usmsessions \"Microsoft-Server-ActiveSync?<SomeBase64EncodedData>\"");
            helpFormatter.printWrapped(pw, _WIDTH, "OLOX2:");
            pw.println("  # usmsessions http://example.com/usm-json/syncUpdate?id=<SomeId>");
            pw.println();
            helpFormatter.printWrapped(
                pw,
                _WIDTH,
                "Enable EAS debug logging for the session(s) of user 4 in context 1, with keyword \"mykeyword\":");
            pw.println("  # usmsessions -e mykeyword 1:4:::");
            pw.println();
            helpFormatter.printWrapped(pw, _WIDTH, "Show all EAS sessions, stating for which ones EAS debug logging is enabled:");
            pw.println("  # usmsessions -r :::EAS:");
            pw.println();
            helpFormatter.printWrapped(pw, _WIDTH, "Disable EAS debug logging for all sessions:");
            pw.println("  # usmsessions -d ::::");
            pw.println();
        }
        pw.flush();
        System.exit(0);
    }

    private static String getUsageString(HelpFormatter formatter, Options options, boolean removeUsagePrefix) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        formatter.printUsage(pw, 76, "usmsessions", options);
        pw.flush();
        String s = sw.toString().trim();
        return removeUsagePrefix ? ("      " + s.substring(6)) : s;
    }

    public static void main(String[] args) {
        new USMSessionTool(args).execute();
    }

    private String _jmxLogin;

    private String _jmxPassword;

    private int _jmxPort = 9999;

    private boolean _verbose;

    private String _enableEASDebugLogging;

    private boolean _disableLogging;

    private boolean _quiet;

    private boolean _reportEasDebugLog;

    private List<String> _data = new ArrayList<String>();

    private MBeanServerConnection _connection;

    private USMSessionTool(String[] args) {
        CommandLineParser parser = new PosixParser();
        CommandLine cmd;
        try {
            cmd = parser.parse(toolOptions, args);
            if (cmd.hasOption('X')) {
                printHelpAndExit(true);
            }
            if (cmd.hasOption('h')) {
                printHelpAndExit(false);
            }
            if (cmd.hasOption('l'))
                _jmxLogin = CLToolsUtility.getOptionStringValue(cmd, 'l');
            if (cmd.hasOption('p'))
                _jmxPort = CLToolsUtility.getOptionIntValue(cmd, 'p', 1, 65535);
            if (cmd.hasOption('s'))
                _jmxPassword = cmd.getOptionValue('s');
            if (cmd.hasOption('e'))
                _enableEASDebugLogging = cmd.getOptionValue('e');
            else if (cmd.hasOption('d'))
                _disableLogging = true;
            else if (cmd.hasOption('r'))
                _reportEasDebugLog = true;
            if (cmd.hasOption('v'))
                _verbose = true;
            else if (cmd.hasOption('q'))
                _quiet = true;
            String[] data = cmd.getArgs();
            if (data == null || data.length == 0) {
                System.err.println("No arguments specified");
                printHelpAndExit(false);
            } else {
                for (String entry : data)
                    _data.add(entry);
            }
        } catch (ParseException e) {
            System.err.println("Unable to parse command line: " + e.getMessage());
            printHelpAndExit(false);
        }
    }

    private void execute() {
        try {
            final JMXServiceURL url = new JMXServiceURL(
                new StringBuilder("service:jmx:rmi:///jndi/rmi://localhost:").append(_jmxPort).append("/server").toString());
            final JMXConnector jmxConnector = JMXConnectorFactory.connect(url, CLToolsUtility.createJMXEnvironment(_jmxLogin, _jmxPassword));
            try {
                _connection = jmxConnector.getMBeanServerConnection();
                for (String data : _data) {
                    if (handleCustomData(data))
                        continue;
                    Map<String, String> urlParameters = CLToolsUtility.extractURLParameters(data);
                    if (handleBase64ActiveSyncData(data))
                        continue;
                    if (handleActiveSyncData(urlParameters))
                        continue;
                    if (handleOlox2Data(urlParameters))
                        continue;
                    // Option: Add more parsers here
                    System.err.println("Ignoring unrecognized data: " + data);
                }
            } finally {
                jmxConnector.close();
            }
        } catch (Exception e) {
            System.err.println("An error occurred while executing your request:");
            e.printStackTrace();
            printHelpAndExit(false);
        }
    }

    private boolean handleCustomData(String data) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        String[] parts = data.split(":", -1);
        if (parts.length == 5) {
            try {
                int cid = CLToolsUtility.parseIntegerValue(parts[0]);
                int id = CLToolsUtility.parseIntegerValue(parts[1]);
                int usmSessionId = CLToolsUtility.parseIntegerValue(parts[2]);
                workOnUSMSession(cid, id, usmSessionId, parts[3], parts[4]);
                return true;
            } catch (NumberFormatException nfe) {
                System.err.println("Warning: Found possible custom parameter with correct structure but invalid data: " + nfe.getMessage());
            }
        }
        return false;
    }

    private boolean handleActiveSyncData(Map<String, String> urlParameters) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        // String user = urlParameters.get(_EAS_USER_PARAMETER);
        String deviceId = urlParameters.get(_EAS_DEVICE_ID_PARAMETER);
        return handleCommonActiveSyncData(deviceId);
    }

    private boolean handleBase64ActiveSyncData(String data) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        int split = data.lastIndexOf('?');
        if (split >= 0)
            data = data.substring(split + 1);
        EASBase64Query query;
        try {
            query = new EASBase64Query(new EASBase64RequestParser(data));
        } catch (Exception e) {
            return false;
        }
        // String user = query.getStringParameter(Parameter.RoundTripId);
        String deviceId = query.getDeviceID();
        return handleCommonActiveSyncData(deviceId);
    }

    protected boolean handleCommonActiveSyncData(String deviceId) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        if (!CLToolsUtility.provided(deviceId))
            return false;
        workOnUSMSession(0, 0, 0, _USM_PROTOCOL_EAS, deviceId);
        return true;
    }

    private boolean handleOlox2Data(Map<String, String> urlParameters) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        for (String paramValue : urlParameters.values()) {
            String[] parts = paramValue.split("_");
            if (parts.length == 3) {
                int cid, usmSessionId;
                try {
                    UUID.fromString(parts[0]); // To check for correct syntax
                    cid = Integer.parseInt(parts[1]);
                    usmSessionId = Integer.parseInt(parts[2]);
                } catch (Exception e) {
                    System.err.println("Warning: Found possible OLOX2-parameter with correct structure but invalid data: " + e.getMessage());
                    continue;
                }
                workOnUSMSession(cid, 0, usmSessionId, _USM_PROTOCOL_JSON, null);
                return true;
            }
        }
        return false;
    }

    private void workOnUSMSession(int cid, int userId, int usmSessionId, String protocol, String deviceId) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Object[] params = new Object[] { cid, userId, protocol, deviceId, null };
        String[] signature = new String[] {
            int.class.getName(), int.class.getName(), String.class.getName(), String.class.getName(), String.class.getName() };
        String[] result = (String[]) _connection.invoke(
            JMXObjectNames.USM_SESSION_INFO_MBEAN,
            "getPropertiesOfAllSessions",
            params,
            signature);
        List<JMXUSMSessionInfo> sessions = JMXUSMSessionInfo.fromJMXResult(result, _quiet);
        for (JMXUSMSessionInfo session : sessions) {
            if (usmSessionId == 0 || session.getUsmSessionId() == usmSessionId) {
                if (_verbose) {
                    System.out.println("Session: " + session);
                    reportEASDebugLogging(session);
                    for (Entry<String, String> entry : session.getFields().entrySet()) {
                        System.out.println("   " + entry.getKey() + " = " + entry.getValue());
                    }
                } else if (_quiet) {
                    reportEASDebugLogging(session);
                } else {
                    System.out.println(session);
                    reportEASDebugLogging(session);
                }
                if (CLToolsUtility.provided(_enableEASDebugLogging)) {
                    if (session.getProtocol().equals(_USM_PROTOCOL_EAS)) {
                        setSessionProperty(session, _USM_FIELD_EAS_DEBUG_LOG, _enableEASDebugLogging);
                        if (_verbose)
                            System.out.println("--> EAS-Debug-Logging activated with keyword \"" + _enableEASDebugLogging + "\"");
                    } else
                        System.err.println("Warning: Ignoring option -e for non-EAS-session " + session);
                } else if (_disableLogging) {
                    if (session.getProtocol().equals(_USM_PROTOCOL_EAS)) {
                        setSessionProperty(session, _USM_FIELD_EAS_DEBUG_LOG, "");
                        if (_verbose)
                            System.out.println("--> EAS-Debug-Logging deactivated");
                    } else
                        System.err.println("Warning: Ignoring option -d for non-EAS-session " + session);
                }
                if (_verbose)
                    System.out.println();
            }
        }
    }

    private void reportEASDebugLogging(JMXUSMSessionInfo session) {
        if (!_reportEasDebugLog)
            return;
        if (session.getProtocol().equals(_USM_PROTOCOL_EAS)) {
            String easLogging = session.getFields().get(_USM_FIELD_EAS_DEBUG_LOG);
            if (_quiet)
                System.out.println(CLToolsUtility.provided(easLogging) ? easLogging : "");
            else if (CLToolsUtility.provided(easLogging))
                System.out.println("   EAS Debug Logging is activated with keyword \"" + easLogging + "\"");
            else
                System.out.println("   EAS Debug Logging is not activated");
        } else
            System.err.println("Warning: Ignoring option -r for non-EAS-session " + session);
    }

    private void setSessionProperty(JMXUSMSessionInfo session, String field, String value) throws InstanceNotFoundException, MBeanException, ReflectionException, IOException {
        Object[] params = new Object[] { session.getCID(), session.getID(), session.getProtocol(), session.getDevice(), field, value };
        String[] signature = new String[] {
            int.class.getName(), int.class.getName(), String.class.getName(), String.class.getName(), String.class.getName(),
            String.class.getName() };
        String[] result = (String[]) _connection.invoke(
            JMXObjectNames.USM_SESSION_INFO_MBEAN,
            "updatePropertiesOfAllSessions",
            params,
            signature);
        if (result.length == 1) {
            String expectedResult = session.getCID() + ": >> 1 sessions updated <<";
            if (result[0].equals(expectedResult))
                return;
        }
        if (result.length == 0) {
            System.err.println("Warning: Empty result received");
        } else {
            System.err.println("Warning: Received unexpected result from update:");
            for (String line : result)
                System.err.println(line);
        }
    }

}
