/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.session.assets;

import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;

/**
 * An SyncResult contains all information that is reported as a result of a slow or incremental sync. The provided information is the
 * timestamp which should be used for future incremental synchronizations and all operations (as DataObjects) that need to be performed by
 * the client to reach the synchronized state. Additionally, the SyncResult also provides a copy of the current state on the server (i.e.
 * the synchronized state after the pending changes have been performed on the client).
 * 
 * @author afe
 */
public class SyncResult {

    public static final DataObject[] EMPTY_DATA_OBJECT_ARRAY = new DataObject[0];

    public static final SyncResult EMPTY_RESULT = new SyncResult();

    private final long _timestamp;

    private final DataObject[] _newState;

    private final DataObject[] _changes;

    private final Map<DataObject, USMException> _errors;

    private final boolean _incomplete;

    @SuppressWarnings("unchecked")
    public SyncResult(boolean incomplete, long timestamp, DataObject[] newState, DataObject... changes) {
        this(incomplete, timestamp, newState, Collections.EMPTY_MAP, changes);
    }

    public SyncResult(boolean incomplete, long timestamp, DataObject[] newState, Map<DataObject, USMException> errors, DataObject... changes) {
        _timestamp = timestamp;
        _newState = newState;
        _changes = changes;
        _errors = errors;
        _incomplete = incomplete;
    }

    private SyncResult() {
        _timestamp = 0L;
        _newState = EMPTY_DATA_OBJECT_ARRAY;
        _changes = EMPTY_DATA_OBJECT_ARRAY;
        _errors = Collections.emptyMap();
        _incomplete = false;
    }

    /**
     * @return Resulting timestamp that should be used for future incremental synchronizations
     */
    public long getTimestamp() {
        return _timestamp;
    }

    /**
     * @return List of actions that need to be performed by the client to reach synchronized state
     */
    public DataObject[] getChanges() {
        return _changes;
    }

    /**
     * @return New synchronized state (will be synchronized after client executed the requested changes)
     */
    public DataObject[] getNewState() {
        return _newState;
    }

    /**
     * This flag will be set if for a synchronization a limit was specified and not all changes could be reported due to that limit.
     * 
     * @return true if the synchronization was cut short due to limit (and more changes are available)
     */
    public boolean isIncomplete() {
        return _incomplete;
    }

    /**
     * Returns a map that contains for all client updates that failed due to an unexpected error during the update (e.g. the object is not
     * accessible, a field would be set to an invalid value) the Exception that caused the failure.
     * 
     * @return Map that contains for all failed server updates the exception that caused the problem
     */
    public Map<DataObject, USMException> getErrors() {
        return _errors;
    }

    @Override
    public String toString() {
        return "SyncResult(" + _timestamp + ',' + Arrays.toString(_changes) + ')';
    }

    public String getDescription() {
        StringBuilder sb = new StringBuilder();
        sb.append(_newState == null ? 0 : _newState.length).append(" objects");
        if (_changes != null && _changes.length > 0)
            sb.append(", ").append(_changes.length).append(" changes");
        if (_errors != null && _errors.size() > 0)
            sb.append(", ").append(_errors.size()).append(" errors");
        if (_incomplete)
            sb.append(", incomplete");
        return sb.toString();
    }
}
