/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.exceptions;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Base class for all Exceptions that are thrown by components of USM.
 * 
 * @author afe
 *
 */
public class USMException extends Exception implements USMJSONException {

	private static final boolean SEND_STACK_TRACES = false;

	public static JSONObject appendStacktrace(Throwable t, JSONObject errorDetailsObject) {
		if (t != null) {
			StackTraceElement[] stackTraces = t.getStackTrace();
			JSONArray stackTrace = new JSONArray();
			for (StackTraceElement stackTraceElement : stackTraces) {
				stackTrace.put(stackTraceElement.toString());
			}
			try {
				if (SEND_STACK_TRACES)
					errorDetailsObject.put(STACKTRACE, stackTrace);
				errorDetailsObject.put(EXCEPTION, t.getClass().getCanonicalName());
				String s = t.getMessage();
				if (s != null)
					errorDetailsObject.put(EXCEPTION_MESSAGE, s);
			} catch (JSONException ignored) {
				//do nothing.. empty stacktrace will be returned
			}
		}
		return errorDetailsObject;
	}

	private static final long serialVersionUID = 1L;

	private static final String EXCEPTION = "exception";
	private static final String EXCEPTION_MESSAGE = "exception_message";
	private static final String STACKTRACE = "stacktrace";
	private static final String CAUSE = "cause";
	private static final String ERROR_MESSAGE = "errorMessage";
	private static final String ERROR_DETAILS = "errorDetails";
	private static final String OX_ERROR = "oxError";

	/**
	 * Unique error code for this Exception as defined in Bundle-IDs.txt and bundle specific sub-definitions.
	 * 
	 */
	private final int _errorCode;

	public USMException(int errorCode) {
		super();
		_errorCode = errorCode;
	}

	public USMException(int errorCode, String message) {
		super(message);
		_errorCode = errorCode;
	}

	public USMException(int errorCode, String message, Throwable cause) {
		super(message, cause);
		_errorCode = errorCode;
	}

	public USMException(int errorCode, Throwable cause) {
		super(cause);
		_errorCode = errorCode;
	}

	@Override
    public int getErrorCode() {
		return _errorCode;
	}

	@Override
    public JSONObject getErrorDetailsForJSONResponse() {
		JSONObject errorDetailsObject = new JSONObject();
		addStacktrace(this, errorDetailsObject);
		addCauseData(errorDetailsObject, getCause());
		return (errorDetailsObject.length() == 0) ? null : errorDetailsObject;
	}

	protected void addStacktrace(Throwable t, JSONObject errorDetailsObject) {
		appendStacktrace(t, errorDetailsObject);
	}

	protected void addCauseData(JSONObject o, Throwable cause) {
		if (cause != null) {
			try {
				JSONObject causeData = new JSONObject();
				causeData.put(EXCEPTION, cause.getClass().getCanonicalName());
				if (cause instanceof USMJSONException) {
					USMJSONException e = (USMJSONException) cause;
					JSONObject d = e.getErrorDetailsForJSONResponse();
					if (d != null)
						causeData.put(ERROR_DETAILS, d);
					d = e.getOxErrorForJSONResponse();
					if (d != null)
						causeData.put(OX_ERROR, d);
					String s = e.getErrorMessage();
					if (s != null)
						causeData.put(ERROR_MESSAGE, s);
				} else {
					causeData.put(ERROR_MESSAGE, cause.getMessage());
					addStacktrace(cause, causeData);
					addCauseData(causeData, cause.getCause());
				}
				o.put(CAUSE, causeData);
			} catch (JSONException ignored) {
				//do nothing.. empty or partial details will be returned
			}
		}
	}

	@Override
    public JSONObject getOxErrorForJSONResponse() {
		return null;
	}

	@Override
    public String getErrorMessage() {
		return getMessage();
	}

	@Override
	public String toString() {
		return Integer.toHexString(_errorCode) + ": " + super.toString();
	}

}
