/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.ant.tasks;

import java.io.File;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.Task;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import com.openexchange.jgit.OXReleaseDescribeCommand;

/**
 * {@link GitDescription} - Uses the <code>repoName</code> directory to
 * generate git describe --long infos based on the next tag matching
 * <code>tagPattern</code> and stores the description in
 * <code>propertyName</code>.
 *
 * <p><cite>
 * "The command finds the most recent tag that is reachable from a commit. If the
 * tag points to the commit, then only the tag is shown. Otherwise, it suffixes
 * the tag name with the number of additional commits on top of the tagged
 * object and the abbreviated object name of the most recent commit."
 * </cite></p>
 * ...
 * <p> <cite>
 * "Always output the long format (the tag, the number of commits and the
 * abbreviated commit name) even when it matches a tag. This is useful when you
 * want to see parts of the commit object name in "describe" output, even when
 * the commit in question happens to be a tagged version. Instead of just
 * emitting the tag name, it will describe such a commit as v1.2-0-gdeadbee (0th
 * commit since tag v1.2 that points at object deadbee....)"
 * </cite></p>
 *
 *<p>
 * but instead of using simply the next tag that might indicate something like
 * "when was this published" we only use tags that match our release pattern of
 * generation.major.minor-rev.
 * </p>
 *
 * <p>
 * Property will not be set if building the description fails e.g. when there
 * are no tags. TagPattern is actually optional and will default to the pattern
 * used below.
 * </p>
 *
 * <ul>
 *   <li> Default propertyName: gitDescription </li>
 *   <li> Default tagPattern: \d+\.\d+\.\d+-\d+ </li>
 * </ul>
 *
 * <h4>Example></h4>
 * <pre>
 *   &ltgitDescription propertyName="gitDescription" repositoryName="${checkoutDir}/.git" tagPattern="\d+\.\d+\.\d+-\d+"/&gt
 * </pre>
 *
 * @author <a href="mailto:marc.arens@open-xchange.com">Firstname Lastname</a>
 * @since v7.8.1
 */
public class GitDescription extends Task {

    private final static String DEFAULT_PROPERTY_NAME = "gitDescription";

    private String propertyName;

    private String repositoryName;

    private String tagPattern;

    public void setPropertyName(String propertyName) {
        this.propertyName = propertyName;
    }

    public void setRepositoryName(String repositoryName) {
        this.repositoryName = repositoryName;
    }

    public void setTagPattern(String tagPattern) {
        this.tagPattern = tagPattern;
    }

    @Override
    public void execute() throws BuildException {
        if (propertyName == null) {
            log("Missing parameter propertyName, going to use default: " + DEFAULT_PROPERTY_NAME);
            propertyName = DEFAULT_PROPERTY_NAME;
        }
        if(repositoryName == null || repositoryName.isEmpty()) {
            throw new BuildException("Missing parameter: repositoryName");
        }

        try {
            FileRepositoryBuilder frb = new FileRepositoryBuilder();
            frb.setGitDir(new File(repositoryName))
                .readEnvironment()
                .findGitDir()
                .build();
            Repository repository;
            repository = frb.build();
            OXReleaseDescribeCommand ordc = new OXReleaseDescribeCommand(
                repository);
            if(tagPattern != null && !tagPattern.isEmpty()) {
                ordc.settagPattern(tagPattern);
            }
            ordc.setLong(true);
            String releaseDescription = ordc.call();
            if (releaseDescription != null) {
                getProject().setInheritedProperty(propertyName,
                    releaseDescription);
                log("Setting property '" + propertyName + "' for repository '" + repositoryName + "' with tagPattern '" + tagPattern + "' to value: " + releaseDescription);
            } else {
                log("Unable to determine git description for repo: "
                    + repositoryName, Project.MSG_WARN);
                getProject().setInheritedProperty(propertyName,
                        "not available");
            }
        } catch (Exception e) {
            log("Unable to determine git description for repo: "
                + repositoryName + ". Reason: " + e.getLocalizedMessage(),
                Project.MSG_WARN);
            getProject().setInheritedProperty(propertyName,
                    "not available");
        }

    }

}
