/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.sync;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.impl.SessionManagerImpl;

public abstract class BaseContentSyncer {

    protected final SessionManagerImpl _sessionManager;

    protected final int _retryCount;

    protected CachedOXData _oxDataCache;

    protected final Log _journal;

    public BaseContentSyncer(SessionManagerImpl sessionManager, int retryCount) {
        _sessionManager = sessionManager;
        _retryCount = retryCount;
        _journal = _sessionManager.getJournal();
    }

    protected boolean shouldBeAddedToResult(int resultLimit, int clientsResultSize, DataObjectFilter filter, ChangeState action, DataObject object, Set<Object> objectGroups) {
        Object objectGroupOwner = object.getContentType().getObjectGroupOwner(object);
        if (objectGroups.contains(objectGroupOwner))
            return true;
        if (resultLimit != Session.NO_LIMIT && clientsResultSize >= resultLimit)
            return false;
        if (filter != null && !filter.addOperationToResult(action, object))
            return false;
        objectGroups.add(objectGroupOwner);
        return true;
    }

    protected Collection<? extends DataObject> sortElements(Collection<? extends DataObject> data, Comparator<? super DataObject> sorter) {
        if (sorter == null || data.size() <= 1)
            return data;
        List<? extends DataObject> list = (data instanceof List) ? (List<? extends DataObject>) data : (data instanceof DataObjectSet) ? ((DataObjectSet) data).toList() : new ArrayList<DataObject>(
            data);
        Collections.sort(list, sorter);
        return list;
    }

    protected DataObjectSet getCurrentOrCachedServerData(ContentSyncerStorage storage, boolean allowCache) throws USMException {
        if (allowCache) {
            _oxDataCache = _sessionManager.getOXDataCache().getCachedData(storage.getCacheID());
            if (_oxDataCache != null) {
                if (_journal.isDebugEnabled())
                    _journal.debug(storage.getCacheID() + " Using cached OX server data for sync");
                return new DataObjectSet(_oxDataCache.getData());
            }
        }
        DataObjectSet serverDataObjects = new DataObjectSet();
        boolean isComplete = storage.getCurrentServerData(serverDataObjects);
        if (allowCache && isComplete) {
            _oxDataCache = _sessionManager.getOXDataCache().storeCachedData(storage.getCacheID(), serverDataObjects.toArray());
        } else {
            _oxDataCache = null;
            _sessionManager.getOXDataCache().removeCachedData(storage.getCacheID());
        }
        return serverDataObjects;
    }

    protected void setParentFolderOwnerIDField(DataObject[] dataObjects) throws StorageAccessException, USMStorageException {
        Folder parentFolder = null;
        boolean searchedCachedFolders = false;
        for (DataObject dataObject : dataObjects) {
            Folder folder = dataObject.getParentFolder();
            if (folder != null) {
                parentFolder = folder;
                dataObject.setParentFolderOwnerID(folder.getOwnerID());
            } else {
                if (!searchedCachedFolders && parentFolder == null)
                    parentFolder = findParentFolder(dataObject);
                if (parentFolder != null) {
                    dataObject.setParentFolderOwnerID(parentFolder.getOwnerID());
                    dataObject.setParentFolder(parentFolder);
                }
            }
        }
    }

    private Folder findParentFolder(DataObject object) throws StorageAccessException, USMStorageException {
        String parentFolderID = object.getOriginalParentFolderID();
        if (parentFolderID == null)
            parentFolderID = object.getParentFolderID();
        if (parentFolderID != null) {
            Folder[] cachedFolders = object.getSession().getCachedFolders();
            if (cachedFolders != null) {
                for (Folder folder2 : cachedFolders) {
                    if (folder2.getID().equals(parentFolderID))
                        return folder2;
                }
            }
        }
        return null;
    }
}
