/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.json;

import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.TemporaryDownOrBusyException;
import com.openexchange.usm.api.exceptions.USMAccessDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.json.response.ResponseStatusCode;

public class USMJSONAPIException extends USMException {
	private static final long serialVersionUID = 1L;

	public static USMJSONAPIException createInternalError(int errorCode, USMException cause) {
		if (cause instanceof USMJSONAPIException)
			return (USMJSONAPIException) cause;
		if(cause instanceof InvalidUUIDException)
		    return createInvalidUUIDException((InvalidUUIDException) cause);
        if(cause instanceof DataObjectNotFoundException)
            return createObjectNotFoundException((DataObjectNotFoundException) cause);
		ResponseStatusCode code = ResponseStatusCode.INTERNAL_ERROR;
		if (cause instanceof USMAccessDeniedException || cause instanceof AuthenticationFailedException)
			code = ResponseStatusCode.ACCESS_DENIED;
		else if (cause instanceof TemporaryDownOrBusyException)
			code = ResponseStatusCode.TEMPORARY_NOT_AVAILABLE;
		else if (cause instanceof OXCommunicationException)
			code = ResponseStatusCode.OX_SERVER_ERROR;
		else if (cause instanceof StorageAccessException || cause instanceof USMStorageException)
			code = ResponseStatusCode.DATABASE_ERROR;
		else if (cause instanceof FolderNotFoundException)
			code = ResponseStatusCode.FOLDER_NOT_FOUND;
		return new USMJSONAPIException(errorCode, code, cause.getErrorMessage(), cause.getOxErrorForJSONResponse(),
				cause.getErrorDetailsForJSONResponse(), cause);
	}

    public static USMJSONAPIException createInvalidUUIDException(InvalidUUIDException e) {
        return new USMJSONAPIException(e.getErrorCode(), ResponseStatusCode.WRONG_MISSING_PARAMETERS, e.getMessage());
    }

    public static USMJSONAPIException createObjectNotFoundException(DataObjectNotFoundException e) {
        return new USMJSONAPIException(e.getErrorCode(), ResponseStatusCode.WRONG_MISSING_PARAMETERS, e.getMessage());
    }

    public static USMJSONAPIException createMultipleOperationsOnDataObjectException(MultipleOperationsOnDataObjectException e) {
        return new USMJSONAPIException(e.getErrorCode(), ResponseStatusCode.WRONG_MISSING_PARAMETERS, e.getMessage());
    }
	
	public static USMJSONAPIException createJSONError(int errorCode, JSONException e) {
		JSONObject errorDetails = new JSONObject();
		try {
			errorDetails.put("JSONError", e.getMessage());
		} catch (JSONException ignored) {
			// Should never be possible, ignored to avoid endless error recursion
		}
		return new USMJSONAPIException(errorCode, ResponseStatusCode.INTERNAL_ERROR, "Error creating JSON data", null,
				errorDetails, e);
	}

	private final JSONObject _oxError;

	private final JSONObject _errorDetails;

	private final ResponseStatusCode _status;

	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message) {
		this(errorCode, status, message, null, null, null);
	}

	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message, JSONObject errorDetails) {
		this(errorCode, status, message, null, errorDetails, null);
	}

	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message, Throwable cause) {
		this(errorCode, status, message, null, null, cause);
	}


	public USMJSONAPIException(int errorCode, ResponseStatusCode status, String message, JSONObject oxError,
			JSONObject errorDetails, Throwable cause) {
		super(errorCode, message, cause);
		if (errorDetails == null && (cause instanceof OXCommunicationException))
			errorDetails = ((OXCommunicationException) cause).getOxErrorForJSONResponse();
		if (errorDetails == null && status == ResponseStatusCode.INTERNAL_ERROR && cause != null) {
			errorDetails = USMException.appendStacktrace(cause, new JSONObject());
		}
		_oxError = oxError;
		_errorDetails = errorDetails;
		_status = status;
	}

	@Override
	public JSONObject getOxErrorForJSONResponse() {
		return _oxError;
	}

	@Override
	public JSONObject getErrorDetailsForJSONResponse() {
		return _errorDetails;
	}

	public ResponseStatusCode getResponseStatusCode() {
		return _status;
	}
}
