/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.session;

import java.io.Serializable;
import java.util.BitSet;
import java.util.Comparator;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.locks.Lock;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.FolderNotFoundException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.api.session.assets.SyncResult;
import com.openexchange.usm.api.session.exceptions.SlowSyncRequiredException;

/**
 * <h3>Session uniquely associated with user and device</h3> A Session provides methods that retrieve cached information for sync states and
 * methods that perform both a slow and incremental synchronization. The Session also provides methods that can be used to configure how
 * synchronization is performed. Furthermore the Session also provides methods to directly read and store data from/on the OX server (e.g.
 * fields that will not be used in synchronization, like mail bodies or contact pictures).<br>
 * In addition to that, the Session acts as a central storage unit for both persistent fields (String information) that will be stored in
 * the database and custom data (e.g. an HttpClient that will be used for connections to the OX server) that will be lost if a Session
 * (temporarily) expires. <br>
 * Internal Session/Sync-Management:
 * <ul>
 * <li>keep Cache of DataObjects for completed updates
 * <li>On update, compare cached data with current information on server
 * <li>Determine conflicts between client changes and server changes, resolve according to resolution
 * <li>On successful resolve, store new state in cache and return necessary changes to client
 * </ul>
 * 
 * @author afe
 */
public interface Session extends Serializable {

    /**
     * Constant that can be used for all sync-calls that do not want to set any limit on the results
     */
    int NO_LIMIT = 0;

    String MAIL_HARDDELETE_PROPERTY = "ContentType.Mail.HardDelete";

    /**
     * @return user associated with Session
     */
    String getUser();

    /**
     * <br>
     * TODO Might be moved to internal interface
     * 
     * @return password for user
     */
    String getPassword();

    /**
     * @return device associated with Session
     */
    String getDevice();

    /**
     * @return protocol associated with Session
     */
    String getProtocol();

    /**
     * @return OX context id as determined from the OX JSON interface configuration
     */
    int getContextId();

    /**
     * @return internal USM session ID as determined on creation
     */
    int getSessionId();

    /**
     * @return OX user identifier as determined from the OX JSON interface configuration
     */
    int getUserIdentifier();

    /**
     * @return the default email address of the user as determined from the OX JSON configuration interface
     */
    String getDefaultEmailAddress();

    /**
     * This method is intended only as a temporary workaround until we can get real UTC times when communicating with the OX server.
     * Currently the OX server sends all time/date information converted to the TimeZone set for the user, so we have to convert it back
     * into UTC time. Note that the TimeZone is retrieved from the OX server only on Session (re)initialization, if the TimeZone is changed
     * during an active Session, USM will continue to use the old (wrong) TimeZone.
     * 
     * @return TimeZone stored in the OX configuration for the user
     */
    TimeZone getUserTimeZone();

    /**
     * This method returns a read-only view of the persistent fields that are currently stored in the Session.
     * 
     * @return
     */
    Map<String, String> getPersistentFields();

    /**
     * This method can be used to retrieve previously stored persistent values. The limit of the field name depends on the database
     * configuration, if at default, it may not exceed 18 characters. If the field is not set, an empty String will be returned.
     * 
     * @param name unique field name for which persistent data should be retrieved
     * @return
     */
    String getPersistentField(String name);

    /**
     * This method can be used to store custom fields in a persistent way. The limit of the field name and value depends on the database
     * configuration, if at default, the field name may not exceed 18 characters, the value is limited to 32 characters. An empty String or
     * null as value parameter indicates the removal of a previously stored value.
     * 
     * @param name
     * @param value
     */
    void setPersistentField(String name, String value) throws StorageAccessException, USMStorageException;

    /**
     * This method can be used to retrieve a previously stored custom property (which can be any Java Object) from the Session. The Property
     * is not persisted, so it will be stored only as long as the Session remains active in the USM.<br>
     * The custom property mechanism behaves similar to a Map, it will return null if the property is not set.
     * 
     * @param key
     * @return
     */
    Object getCustomProperty(String key);

    /**
     * This method can be used to store a custom property (which can be any Java Object) in the Session. The Property is not persisted, so
     * it will be stored only as long as the Session remains active in the USM.<br>
     * The custom property mechanism behaves similar to a Map. If a value of null is provided, the key-value association is removed.
     * 
     * @param key
     * @param value
     * @return previously stored Object
     */
    Object setCustomProperty(String key, Object value);

    /**
     * This method behaves as setCustomProperty(String, Object), except the value will be automatically removed after the given timeout.
     * This can be used for cache-like accesses, data that gets invalid after some time or for large data that should not be stored for too
     * long.
     * 
     * @param key
     * @param value
     * @param timeout timeout in milliseconds after which the stored value should be removed
     * @return previously stored Object
     */
    Object setCustomProperty(String key, Object value, long timeout);

    /**
     * Used for requests which need date range as parameter, some ContentTypes require a date range to be specified for their requests to
     * the OX server. Default value is 0l. <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @return start of the date range associated with Session.
     */
    long getStartDate();

    /**
     * Used for requests which need date range as parameter, some ContentTypes require a date range to be specified for their requests to
     * the OX server. Default value is Long.MAX_VALUE. <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @return end of the date range associated with Session.
     */
    long getEndDate();

    /**
     * Used for requests which need date range as parameter, some ContentTypes require a date range to be specified for their requests to
     * the OX server. <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @param date
     */
    void setStartDate(long date);
    

    /**
     * Used for requests which need date range as parameter, some ContentTypes require a date range to be specified for their requests to
     * the OX server. <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @param date
     */
    void setEndDate(long date);
    
    /**
     * Used for calendar specific requests which need date range as parameter.
     * This value overwrites the global value getStartDate().  
     * Default value is 0l.  <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @return start of the date range for calendar associated with Session.
     */
    long getCalendarStartDate();
    
    /**
     * Used for calendar specific requests which need date range as parameter.
     * This value overwrites the global value getEndDate().  
     * Default value is Long.MAX_VALUE.  <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @return start of the date range for calendar associated with Session.
     */
    long getCalendarEndDate();
    
    /**
     * Used for calendar specific requests which need date range as parameter.
     * This value overwrites the global value which can be set via setStartDate().  
     * Default value is 0l.  <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @return start of the date range for calendar associated with Session.
     */
    void setCalendarStartDate(long date);
    
    /**
     * Used for calendar specific requests which need date range as parameter.
     * This value overwrites the global value which can be set via setEndDate().  
     * Default value is Long.MAX_VALUE.  <br>
     * This field is not persistent, it must be reinitialized by the SessionInitializer.
     * 
     * @return start of the date range for calendar associated with Session.
     */
    void setCalendarEndDate(long date);

    /**
     * Sets a limit for the number of mails that will be reported when syncing a folder. If on the server more mails than the specified
     * limit are stored in a folder, only the newest ones (upto the limit) will be used for synchronization, older one will be treated as
     * not present. A value <= 0 specifies that no limit should be used and all emails should be synchronized.
     * 
     * @param limit
     */
    void setMailLimit(int limit);

    /**
     * Retrieves the currently set limit for mail synchronization.
     * 
     * @return
     */
    int getMailLimit();

    /**
     * Retrieves the configured limit for synchronized objects per folder. If on the server more objects than the specified limit are stored
     * in a folder, only the newest ones (upto the limit) will be used for synchronization, older one will be treated as not present. A
     * value <= 0 specifies that no limit should be used and all objects should be synchronized.
     * 
     * @return
     */
    int getObjectsLimit();

    /**
     * Returns the configured limit for the number of synchronized PIM attachments per object. If on the server more files are attached to 1
     * object, only the first ones (default: lowest OX ID) will be synchronized, all others will be treated as not present. A value < 0
     * specifies that no limit should be used and all PIM attachments should be synchronized. Limit is applied independently of
     * getPIMAttachmentsSizeLimit()
     * 
     * @return
     */
    int getPIMAttachmentsCountLimit();

    /**
     * Returns the configured size limit for the synchronized PIM attachments per object. If on the server the size of all attachments to 1
     * object exceeds this limit, only the first ones (default: lowest OX ID) will be synchronized, all others will be treated as not
     * present. A value < 0 specifies that no size limit should be used and all PIM attachments should be synchronized. Limit is applied
     * independently of getPIMAttachmentsCountLimit()
     * 
     * @return
     */
    long getPIMAttachmentsSizeLimit();

    // -- Listening / waiting for changes on the server --
    /**
     * This method can be used to wait for external changes on 1 or more elements in the specified folder(s). The method will return as soon
     * as at least 1 element change in one of the specified folders is detected or if the timeout has been reached, in which case it will
     * return an empty array.<br>
     * Two separate lists of folders can be provided, one to limit the notifications for folder updates, another to limit the notifications
     * for content updates.<br>
     * If a list is not provided (<code>null</code>), all changes of that type are reported, if a list is provided, changes of the
     * appropriate type are only reported if they occur within a specified parent folder<br>
     * Note: It is possible that this method returns immediately, e.g. if changes were made on the server while the Protocol itself was
     * working.<br>
     * Note: This method is not designed for multi-threaded access, if more than 1 thread calls this method simultaneously, only 1 will get
     * correctly notified of the changes. In a multi-threaded environment the addChangeListener() method should be used to get notifications
     * about folder (content) changes.
     * 
     * @param watchFolderChanges if set, changes to the folder structure will be reported, otherwise they do not stop the waiting
     * @param contentChangeParentIDsToWatch Array of folder IDs which will be monitored for changes
     * @param timeout milliseconds after which the method will return even if no changes have been detected
     * @return IDs of parent folders in which changes occurred, separated by folder structure and content change
     * @throws USMException if an unrecoverable error occurred
     * @throws InterruptedException if the wait is interrupted by another thread
     */
    ObjectChanges waitForChanges(boolean watchFolderChanges, String[] contentChangeParentIDsToWatch, long timeout) throws USMException, InterruptedException;

    /**
     * This method checks if a Folder requires synchronization, i.e. if there was a server change or if USM doesn't know, e.g. the session
     * was (re)initialized. It returns true if a sync may be necessary, false if there was no OX notification since the last
     * synchronization. If folderID is "" or null, the method checks if the folder hierarchy needs synchronization.
     * 
     * @param folderID folderID to check, "" or null to check for changes to the folder hierarchy
     * @return true if there may have been a server change, false if not
     */
    boolean needsSynchronization(String folderID);

    /**
     * Adds a listener to the session that will be called on changes to elements. Note that due to current restrictions in the OX event
     * system, listeners will not be called for changes to folders that contain Emails. This limitation will be lifted as soon as the OX
     * server sends Event notifications for changes to Email folders.
     * 
     * @param listener
     */
    void addChangeListener(SessionChangeListener listener);

    /**
     * Removes a previously added ChangeListener from the Session
     * 
     * @param listener
     */
    void removeChangeListener(SessionChangeListener listener);

    // -- Filtering based on ContentTypes and fields supported by client / protocol --

    /**
     * Not every protocol supports all ContentTypes. With this method a global filter can be specified so that Folders that contain unknown
     * ContentTypes will be filtered out automatically. If only some fields are of interest, the method setFieldFilter() can be used
     * directly to activate a ContentType with its selected fields. Note that this method does not reset previously set field filters for
     * ContentTypes that were in use and are also provided as parameters. <br>
     * These filter settings are persistent.
     * 
     * @param usedContentTypes Array of ContentTypes to use for this Session
     */
    void setContentTypeFilter(ContentType... usedContentTypes) throws StorageAccessException, USMStorageException;

    /**
     * Returns all supported ContentTypes. Content types can be filtered out with the method setContentTypeFilter.
     * 
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    ContentType[] getContentTypes() throws StorageAccessException, USMStorageException;

    /**
     * For each used ContentType a global filter can be specified that limits the fields to use for synchronization. Not all protocols
     * use/support all fields, so with this overhead for retrieving/storing unnecessary fields can be eliminated. Note that unless a filter
     * is set all fields of a ContentType will by retrieved/stored. If no fieldsOfInterest are specified, the ContentType will be ignored
     * completely. <br>
     * These filter settings are persistent.
     * 
     * @param contentType
     * @param fieldsOfInterest
     */
    void setFieldFilter(ContentType contentType, String... fieldsOfInterest) throws StorageAccessException, USMStorageException;

    /**
     * For each used ContentType a global filter can be specified that limits the fields to use for synchronization. Not all protocols
     * use/support all fields, so with this overhead for retrieving/storing unnecessary fields can be eliminated. Note that unless a filter
     * is set all fields of a ContentType will by retrieved/stored. If no fieldsOfInterest are specified, the ContentType will be ignored
     * completely. <br>
     * These filter settings are persistent.
     * 
     * @param contentType
     * @param fieldsOfInterest
     */
    void setFieldFilter(ContentType contentType, BitSet fieldsOfInterest) throws StorageAccessException, USMStorageException;

    /**
     * @param contentType
     * @return BitSet of used fields for given ContentType. If the ContentType itself is filtered out, returns null
     */
    BitSet getFieldFilter(ContentType contentType);

    // -- Configuration of Session behavior --

    /**
     * Set the method of conflict resolution on simultaneous changes from both client and server. <br>
     * This field is persistent.
     * 
     * @param resolution ConflictResolution to use for this Session
     */
    void setConflictResolution(ConflictResolution resolution) throws StorageAccessException, USMStorageException;

    /**
     * <br>
     * This field is persistent.
     * 
     * @return currently set ConflictResolution for this Session
     */
    ConflictResolution getConflictResolution();

    /**
     * By calling this method a client protocol may explicitly specify the root folders USM should use when synchronizing the folder
     * hierarchy. If this method is not called (or with no parameters), USM will synchronize the complete folder hierarchy. This information
     * is <b>not</b> stored persistently, so it has to be set by the SessionInitializer if only some folders should be used as root folders.
     * If the root folders are changed in a session with valid cache data, the next synchronization will report folders as deleted that are
     * no longer in the new hierarchy and created if they were previously not accessible but are now.
     * 
     * @param folderId
     */
    void setRootFolders(String... folderId);

    /**
     * This method synchronizes the content of a Folder between client and server. The provided elements represent the current state of all
     * DataObjects on the client (ChangeState is ignored and the DataObject is considered present). The method returns all DataObject
     * operations (i.e. only DataObjects with ChangeState MODIFIED, DELETED, CREATED) for that Folder and the timestamp that should be used
     * for future incremental synchronizations. All provided DataObjects are considered to have the provided folder as parent (their
     * ParentFolderId is ignored), in the result the parent may have changed (or DataObjects from other folder may have moved into the
     * folder).<br>
     * If the parameter limit is set to non-zero, it specifies the limit of changes/creations that are returned in the SyncResult. Note that
     * even with a limit, internally all changes have to be resolved in order to determine conflicts. The limit is reflected in the new
     * synchronized state, though, so that on the next synchronize the remaining updates will be reported. In the SyncResult the flag
     * incomplete will be set if not all server changes could be reported within the limit.<br>
     * Additional note: Changes that are reported due to ConflictResolution will always be present in the result. Therefore any given limit
     * may be exceeded if more elements are provided as parameter to the method.<br>
     * If the parameter storeResult is set to false, the timestamp of the result will be set to 0, to indicate that the result was not
     * stored in the cache
     * 
     * @param folderId
     * @param limit if set to != Session.NO_LIMIT, limits the number of resulting client changes that are reported
     * @param filter if set to != null, the provided filter is applied to the objects retrieved from the server before synchronization is
     *            performed, if set to null, the default filter (if any is set) is used
     * @param storeResult if true, the result of the synchronization will be stored in the database, if false, the result will be discarded
     *            (i.e. no incremental sync with the timestamp of the result will be possible), useful to compute estimates of the number of
     *            changes or to determine if a conflict currently exists
     * @param conflictResolution if set to != null, the provided ConflictResolution will be used instead of the configured one.
     * @param elements
     * @return
     * @throws USMException
     */
    SyncResult syncWithServer(String folderId, int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, DataObject... elements) throws USMException;

    /**
     * This method updates and synchronizes the content of a Folder with the server. The provided elements represent all changes from the
     * client since the given timestamp, i.e. it can contain elements with ChangeState MODIFIED, DELETED, CREATED. If the synchronization is
     * successful, the method returns an array of operations that need to be performed on the client in order to achieve synchronicity and
     * the timestamp that should be used for future incremental synchronizations. <br>
     * Note: For updates from the client to the server (represented in elementChanges) it is not required to provide the complete content,
     * only required fields (like id) and the modified fields (which must be marked modified) are necessary.<br>
     * If the parameter limit is set to non-zero, it specifies the limit of changes/deletions/creations that are returned in the SyncResult.
     * Note that even with a limit, internally all changes have to be resolved in order to determine conflicts. The limit is reflected in
     * the new synchronized state, though, so that on the next synchronize the remaining changes will be reported. In the SyncResult the
     * flag incomplete will be set if not all server changes could be reported within the limit.<br>
     * If the parameter storeResult is set to false, the timestamp of the result will be set to 0, to indicate that the result was not
     * stored in the cache
     * 
     * @param folderId
     * @param timestamp
     * @param limit if set to != Session.NO_LIMIT, limits the number of resulting client changes that are reported
     * @param filter if set to != null, the provided filter is applied to the objects retrieved from the server before synchronization is
     *            performed, if set to null, the default filter (if any is set) is used
     * @param storeResult if true, the result of the synchronization will be stored in the database, if false, the result will be discarded
     *            (i.e. no incremental sync with the timestamp of the result will be possible), useful to compute estimates of the number of
     *            changes or to determine if a conflict currently exists
     * @param conflictResolution if set to != null, the provided ConflictResolution will be used instead of the configured one.
     * @param elementChanges
     * @return
     * @throws USMException
     */
    SyncResult syncChangesWithServer(String folderId, long timestamp, int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, DataObject... elementChanges) throws USMException;

    /**
     * This method synchronizes the Folder hierarchy between client and server. The provided elements represent the current state of all
     * Folders and their subfolders on the client (ChangeState is ignored and the Folder is considered present). The method returns all
     * operations (i.e. only DataObjects with ChangeState MODIFIED, DELETED, CREATED) for all Folders that need updates (on their fields,
     * <b>not</b> their content!) and the timestamp that should be used for future incremental synchronizations.<br>
     * If the parameter limit is set to non-zero, it specifies the limit of changes/creations that are returned in the SyncResult. Note that
     * even with a limit, internally all changes have to be resolved in order to determine conflicts. The limit is reflected in the new
     * synchronized state, though, so that on the next synchronize the remaining updates will be reported. In the SyncResult the flag
     * incomplete will be set if not all server changes could be reported within the limit.<br>
     * Additional note: Changes that are reported due to ConflictResolution will always be present in the result. Therefore any given limit
     * may be exceeded if more elements are provided as parameter to the method.<br>
     * If the parameter storeResult is set to false, the timestamp of the result will be set to 0, to indicate that the result was not
     * stored in the cache
     * 
     * @param limit if set to != Session.NO_LIMIT, limits the number of resulting client changes that are reported
     * @param filter if set to != null, the provided filter is applied to the objects retrieved from the server before synchronization is
     *            performed, if set to null, the default filter (if any is set) is used
     * @param storeResult if true, the result of the synchronization will be stored in the database, if false, the result will be discarded
     *            (i.e. no incremental sync with the timestamp of the result will be possible), useful to compute estimates of the number of
     *            changes or to determine if a conflict currently exists
     * @param conflictResolution if set to != null, the provided ConflictResolution will be used instead of the configured one.
     * @param folders
     * @return
     * @throws USMException
     */
    SyncResult syncFoldersWithServer(int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, Folder... folders) throws USMException;

    /**
     * This method updates and synchronizes the content of a Folder with the server. The provided Folder objects represent all changes from
     * the client since the given timestamp, i.e. it can contain elements with ChangeState MODIFIED, DELETED, CREATED. If the
     * synchronization is successful, the method returns an array of operations that need to be performed on the client in order to achieve
     * synchronicity for the Folder hierarchy and the timestamp that should be used for future incremental synchronizations. <br>
     * Note: For updates from the client to the server (represented in folderChanges) it is not required to provide the complete content,
     * only required fields (like id) and the modified fields (which must be marked modified) are necessary.<br>
     * If the parameter limit is set to non-zero, it specifies the limit of changes/deletions/creations that are returned in the SyncResult.
     * Note that even with a limit, internally all changes have to be resolved in order to determine conflicts. The limit is reflected in
     * the new synchronized state, though, so that on the next synchronize the remaining changes will be reported. In the SyncResult the
     * flag incomplete will be set if not all server changes could be reported within the limit.<br>
     * If the parameter storeResult is set to false, the timestamp of the result will be set to 0, to indicate that the result was not
     * stored in the cache
     * 
     * @param timestamp
     * @param limit if set to != Session.NO_LIMIT, limits the number of resulting client changes that are reported
     * @param filter if set to != null, the provided filter is applied to the objects retrieved from the server before synchronization is
     *            performed, if set to null, the default filter (if any is set) is used
     * @param storeResult if true, the result of the synchronization will be stored in the database, if false, the result will be discarded
     *            (i.e. no incremental sync with the timestamp of the result will be possible), useful to compute estimates of the number of
     *            changes or to determine if a conflict currently exists
     * @param conflictResolution if set to != null, the provided ConflictResolution will be used instead of the configured one.
     * @param folderChanges
     * @return
     * @throws SlowSyncRequiredException if timestamp is invalid or cached information for given timestamp is no longer available
     * @throws USMException
     */
    SyncResult syncFolderChangesWithServer(long timestamp, int limit, DataObjectFilter filter, boolean storeResult, ConflictResolution conflictResolution, Folder... folderChanges) throws USMException;

    /**
     * This method can be used to store a custom sync state for a given timestamp. It can be used by protocols e.g. if not all changes could
     * be transmitted to the client so that in the next sync call differences between the sync state and the current server state will be
     * reported again as server changes. It can also be used if a sync action can not be performed in one step (e.g. some changes could
     * require a deletion and creation instead on the client) to reflect partial changes in the client. Note that all given DataObjects must
     * have the same parent folder ID as specified in the parameter.
     * 
     * @param timestampToKeep Timestamp that should not be deleted (e.g. original timestamp for sync call), set to 0L to not enforce keeping
     *            any timestamp
     * @param timestamp Timestamp under which the sync state is to be stored
     * @param folderID ID of the folder for which a custom SyncState shall be stored
     * @param objects DataObjects to store for that sync state, the current information of them will be stored
     * @return actual timestamp that is used for the modified sync state (may be greater than the specified timestamp)
     * @throws USMException
     * @throws IllegalArgumentException if the DataObjects are not all in the same folder
     */
    long storeSyncState(long timestampToKeep, long timestamp, String folderID, DataObject[] objects) throws USMException;

    /**
     * This method can be used to store a custom folder hierarchy sync state for a given timestamp. It can be used by protocols e.g. if not
     * all changes could be transmitted to the client so that in the next sync call differences between the sync state and the current
     * server state will be reported again as server changes. It can also be used if a sync action can not be performed in one step (e.g.
     * some changes could require a deletion and creation instead on the client) to reflect partial changes in the client. Note that all
     * given DataObjects must have the same parent folder ID as specified in the parameter.
     * 
     * @param timestampToKeep Timestamp that should not be deleted (e.g. original timestamp for sync call), set to 0L to not enforce keeping
     *            any timestamp
     * @param timestamp Timestamp under which the sync state is to be stored
     * @param objects DataObjects to store for that sync state, the current information of them will be stored
     * @return actual timestamp that is used for the modified sync state (may be greater than the specified timestamp)
     * @throws USMException
     * @throws IllegalArgumentException if the DataObjects are not folders
     */
    long storeSyncState(long timestampToKeep, long timestamp, DataObject[] objects) throws USMException;

    /**
     * This method can be used to store a custom sync state for a given timestamp. It can be used by protocols e.g. if not all changes could
     * be transmitted to the client so that in the next sync call differences between the sync state and the current server state will be
     * reported again as server changes. It can also be used if a sync action can not be performed in one step (e.g. some changes could
     * require a deletion and creation instead on the client) to reflect partial changes in the client. Note that all given DataObjects must
     * have the same parent folder ID as specified in the parameter.
     * 
     * @deprecated This method is deprecated, use the alternative version with the additional old timestamp to keep instead.
     * @param timestamp Timestamp under which the sync state is to be stored
     * @param folderID ID of the folder for which a custom SyncState shall be stored
     * @param objects DataObjects to store for that sync state, the current information of them will be stored
     * @return actual timestamp that is used for the modified sync state (may be greater than the specified timestamp)
     * @throws USMException
     * @throws IllegalArgumentException if the DataObjects are not all in the same folder
     */
    @Deprecated
    long storeSyncState(long timestamp, String folderID, DataObject[] objects) throws USMException;

    /**
     * Retrieves the cached folder hierarchy for a given synchronization timestamp, if available. If the data is not (invalid timestamp) or
     * no longer available, this method returns null.
     * 
     * @param timestamp
     * @return
     */
    Folder[] getCachedFolders(long timestamp) throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the cached folder for a given synchronization timestamp, if available. If the data is not (invalid timestamp) or no longer
     * available, this method returns null.
     * 
     * @param folderId
     * @param timestamp
     * @return
     */
    Folder getCachedFolder(String folderId, long timestamp) throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the cached elements of the given Folder for a given synchronization timestamp, if available. If the data is not (invalid
     * timestamp) or no longer available, this method returns null.<br>
     * Since in the cache only essential information about one DataObject is stored and the Folder object itself may not be available from
     * the cache, the caller has to provide the ContentType used by the elements in the folder. if an error occurs during restoration of the
     * cached state (probably caused due to an incorrect ContentType, but can also happen if the ContentType has changed its structure), an
     * IllegalArgumentException is thrown.
     * 
     * @param folderId
     * @param elementsType
     * @param timestamp
     * @return
     * @throws IllegalArgumentException
     */
    DataObject[] getCachedFolderElements(String folderId, ContentType elementsType, long timestamp) throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the cached elements of the given Folder for a given synchronization timestamp, if available. If the data is not (invalid
     * timestamp) or no longer available, this method returns null.<br>
     * Since in the cache only essential information about one DataObject is stored and the Folder object itself may not be available from
     * the cache, the caller has to provide the ContentType used by the elements in the folder. if an error occurs during restoration of the
     * cached state (probably caused due to an incorrect ContentType, but can also happen if the ContentType has changed its structure), an
     * IllegalArgumentException is thrown. Caution: This is a relatively time consuming call, if multiple elements of the same folder need
     * to be retrieved, use getCachedFolderElements instead, preferably storing them in a DataObjectSet for easy and fast access !
     * 
     * @param folderId
     * @param elementsType
     * @param objectId
     * @param timestamp
     * @return
     * @throws IllegalArgumentException
     */
    DataObject getCachedFolderElement(String folderId, ContentType elementsType, String objectId, long timestamp) throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the most recent cached folder hierarchy, if available. If the data is not (no sync) or no longer available, this method
     * returns null.
     * 
     * @param timestamp
     * @return
     */
    Folder[] getCachedFolders() throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the most recent cached folder, if available. If the data is not (no sync yet performed) or no longer available, this method
     * returns null.
     * 
     * @param folderId
     * @param timestamp
     * @return
     */
    Folder getCachedFolder(String folderId) throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the most recent cached elements of the given Folder, if available. If the data is not (no sync yet performed) or no longer
     * available, this method returns null.<br>
     * Since in the cache only essential information about one DataObject is stored and the Folder object itself may not be available from
     * the cache, the caller has to provide the ContentType used by the elements in the folder. if an error occurs during restoration of the
     * cached state (probably caused due to an incorrect ContentType, but can also happen if the ContentType has changed its structure), an
     * IllegalArgumentException is thrown.
     * 
     * @param folderId
     * @param elementsType
     * @param timestamp
     * @return
     * @throws IllegalArgumentException
     */
    DataObject[] getCachedFolderElements(String folderId, ContentType elementsType) throws StorageAccessException, USMStorageException;

    /**
     * Retrieves the most recent cached element of the given Folder, if available. If the data is not (no sync yet performed) or no longer
     * available, this method returns null.<br>
     * Since in the cache only essential information about one DataObject is stored and the Folder object itself may not be available from
     * the cache, the caller has to provide the ContentType used by the elements in the folder. if an error occurs during restoration of the
     * cached state (probably caused due to an incorrect ContentType, but can also happen if the ContentType has changed its structure), an
     * IllegalArgumentException is thrown. Caution: This is a relatively time consuming call, if multiple elements of the same folder need
     * to be retrieved, use getCachedFolderElements instead, preferably storing them in a DataObjectSet for easy and fast access !
     * 
     * @param folderId
     * @param elementsType
     * @param objectId
     * @return
     * @throws IllegalArgumentException
     */
    DataObject getCachedFolderElement(String folderId, ContentType elementsType, String objectId) throws StorageAccessException, USMStorageException;

    /**
     * Writes updates on 1 or more DataObjects to the Server. <br>
     * The method behaves similar to a syncChangesWithServer() or syncFolderChangesWithServer() in that client changes are reported to the
     * server, but no changes from the server are either reported or stored in the cache.<br>
     * If for the given timestamp of each provided DataObject cache information is available, the cache will be updated and a new timestamp
     * will be set. If no cache information is available, the timestamp will be set to 0 to indicate that a slow sync will be necessary on
     * the next synchronization.<br>
     * Note that there is no need for a third method that only reports changes from the server since this can be implemented by calling the
     * sync... methods with no client changes.<br>
     * The method returns the timestamps that can be used for future incremental synchronizations in the DataObjects that were given as
     * parameters. Note that DataObjects of different Folders will have independent timestamps.<br>
     * Note: This call does not support changes without ID, but with UUID. Use the new and extended sync(Folder)ChangesWithServer and a
     * negative limit for that.
     * 
     * @param timestamp
     * @param changedObjects
     * @return map of objects which update failed to the exceptions that caused the failures. An empty map is returned if no exceptions
     *         occurred
     * @throws USMException if an unrecoverable error occurred
     */
    Map<DataObject, USMException> updateDataObjects(DataObject... changedObjects) throws USMException;

    /**
     * This method returns the specified DataObject as currently stored on the Server. This method should be used to retrieve fields that
     * are not stored in the cache (e.g. if they are to big, like mail content). Its result will not be stored in the cache, so the protocol
     * has to account for possible duplicate (redundant) changes if fields are requested that are also requested during a normal
     * synchronize.
     * 
     * @param folderID
     * @param objectID
     * @param fields names of fields to retrieve
     * @return Current content for the specified DataObject
     * @throws USMException if an unrecoverable error occurred
     */
    DataObject readDataObject(String folderID, String objectID, String... fields) throws USMException;

    /**
     * This method returns the specified DataObject as currently stored on the Server. This method should be used to retrieve fields that
     * are not stored in the cache (e.g. if they are to big, like mail content). Its result will not be stored in the cache, so the protocol
     * has to account for possible duplicate (redundant) changes if fields are requested that are also requested during a normal
     * synchronize.
     * 
     * @param folderID
     * @param objectID
     * @param fields BitSet, each set bit will retrieve the corresponding field index
     * @return Current content for the specified DataObject
     * @throws USMException if an unrecoverable error occurred
     */
    DataObject readDataObject(String folderID, String objectID, BitSet fields) throws USMException;

    /**
     * This method can be used to set a default filter on all sync-operations. The filter will be used to restrict the DataObjects that are
     * visible to the sync mechanism (and therefore the client) on an individual basis (e.g. to limit the tasks to only the uncompleted
     * ones). The filter will be used on the next sync-call, all DataObjects that were previously filtered and are now visible will be
     * reported as creations, and all DataObjects that are no longer visible will be reported as deletions.<br>
     * Only one filter can be active at any time, i.e. a new one replaces the filter that was set previously. A previously set filter can be
     * removed by calling this method with a null parameter.<br>
     * Note that the filter will be called on all sync-operations (i.e. all ContentTypes, both folders and elements, slow and incremental
     * sync), the filter may have to check the ContentType of the provided DataObject for specific operations on a single ContentType.<br>
     * Also note that this field is note persistent, a SessionInitializer may have to explicitly set it.<br>
     * 
     * @param filter
     */
    void setSyncServerObjectFilter(DataObjectFilter filter);

    /**
     * Retrieves a previously set default DataObjectFilter. If no filter is set, it returns null.
     * 
     * @return currently set DataObjectFilter for server objects.
     */
    DataObjectFilter getSyncServerObjectFilter();

    /**
     * The toString() method of a Session must make sure that all information to identify the session is generated, e.g.
     * "CONTEXT:USER_IDENTIFIER:USER:PROTOCOL:DEVICE".
     * 
     * @return String that contains all information to uniquely identify the persistent session
     */
    @Override
    String toString();

    /**
     * Maps the OX Folder ID to internal USM ID (short id). This mapping is stored in a db, since 7.6.0 the folders UUID is used as a mapped ID for long folder IDs.
     * Same as getShortFolderID(folderID, null).
     * 
     * @param folderID
     * @return USM folder ID
     * @throws StorageAccessException if the database can not be accessed.
     * @throws USMStorageException if some problems occur while saving the folder id in the database or retrieving the next usm short id
     *             from the database.
     */
    String getShortFolderID(String folderID) throws StorageAccessException, USMStorageException;

    /**
     * Maps the OX folder ID to an internal USM ID (short id). If the folder id is long, no DB entry is found and the UUID is provided as second parameter, then that
     * UUID is used for the mapping, otherwise the folder hierarchy cache is searched for a matching folder.
     * 
     * @param folderID
     * @param folderUUID
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    String getShortFolderID(String folderID, UUID folderUUID) throws StorageAccessException, USMStorageException;
    
    /**
     * Gets the original OX Folder ID that is mapped to USM ID (shortFolderId).
     * 
     * @param shortFolderID
     * @return OX folder ID
     * @throws USMStorageException if some problems occur while retrieving the folder id from the database.
     */
    String getFolderID(String shortFolderID) throws USMStorageException;

    /**
     * Ends the synchronization between client and server. This means deleting all persistent and cached data for a session from the DB.
     * 
     * @throws USMStorageException if some problems occur while deleting data.
     * @throws StorageAccessException if the database can not be accessed
     */
    void endSyncronization() throws USMStorageException, StorageAccessException;

    /**
     * Locates a folder by its ID, either by searching it in the FolderHierarchy cache (if available), or by directly requesting the Folder
     * in its current state from the OX server. This method should never be used to perform operations on the Folder itself, only to
     * determine necessary information (like ContentType) to perform operations on elements in the Folder.
     * 
     * @param folderID
     * @return
     * @throws FolderNotFoundException if the folder was neither in the Cache nor found on the OX server
     * @throws USMException if an error occurred accessing the cache
     */
    Folder findFolder(String folderID) throws USMException;

    /**
     * Retrieves the OX Object ID based on a known UUID for a data object.
     * 
     * @param contentType
     * @param uuid
     * @param folderID
     * @return
     * @throws USMException
     */
    String getMappedObjectId(ContentType contentType, UUID uuid, String parentFolderID) throws USMException;

    /**
     * Returns UUID for the context. This value is saved in a table UUIDHeader.
     * 
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    UUID getContextUUID() throws StorageAccessException, USMStorageException;

    /**
     * Retrieves a UUID for the object identified with context, id and parent folder. The UUID is taken from the last known sync state, or
     * generated as new if not existing.
     * 
     * @param contentType
     * @param objectId
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    UUID getUUID(ContentType contentType, String objectId) throws StorageAccessException, USMStorageException;

    /**
     * Remaps the saved sync states for the object to the new object id.
     */
    void remapCachedData(String oldObjectID, String newObjectID) throws StorageAccessException, USMStorageException;

    /**
     * This method is deprecated, use getCachedFolder(contentType.getID()) instead
     * 
     * @param type
     * @return
     */
    @Deprecated
    Folder getDummyContentTypeFolder(ContentType type);

    /**
     * Retrieves the specified OX Ajax configuration element for the given user. The specified path must link to a JSONObject, elementary
     * final fields are not directly supported (you have to retrieve the containing JSONObject and check the field in the calling method).
     * This method always directly calls the OX server and performs no caching itself.
     * 
     * @param path
     * @return
     * @throws OXCommunicationException
     * @throws AuthenticationFailedException
     */
    JSONObject getOXUserConfiguration(String... path) throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Allows protocols to define a sorting order in which server operations are reported to the client. Note that the sync system will
     * always use a fixed sorting by type of action (first deletions, then changes, last creations). Within each action, a protocol can set
     * custom comparators that are used to sort the changes. If the client specified a limit (or any internal server limit is present or
     * reached) the elements with the highest priority (i.e. that are sorted to the beginning using the comparator) are reported first. Note
     * that as soon as a comparator is specified for a ContentType, the elements have always to be sorted due to the possibility of server
     * limits and dynamic limits imposed by a DataObjectFilter. Note: The sorter is only stored for an active session in memory, so it has
     * to be re-applied on (re)initialization of a Session.
     * 
     * @param contentType
     * @param comparator
     */
    void setContentTypeSorter(ContentType contentType, Comparator<? super DataObject> comparator);

    /**
     * Is used by the OXJSONAccess to store extra information required to access the OX server. Will be stored externally so that OX logout
     * can be performed after the session is removed.
     * 
     * @param data
     */
    void setOXConnectionData(OXConnectionInformation data);

    /**
     * Retrieves previously stored information from setOXConnectionData().
     * 
     * @return
     */
    OXConnectionInformation getOXConnectionData();

    /**
     * @return Lock that ensures that only 1 concurrent OX access is performed at a time.
     */
    Lock getOXConnectionLock();

    /**
     * Retrieves the folder id length limit for this session.
     * 
     * @return
     */
    int getFolderIdLengthLimit();

    /**
     * Sets the folder id length limit for this session.
     * 
     * @return
     */
    void setFolderIdLengthLimit(int limit);

    /**
     * Retrieves the current saved client ip address.
     * 
     * @return
     */
    String getClientIp();

    /**
     * Updates the current client ip address in the local and server session.
     * 
     * @param ip
     */
    void updateClientIp(String ip);

    Map<String, String> getXRequestHeaders();

    /**
     * Gets the newest known timestamp for the folder.
     * 
     * @param folderID
     * @return
     */
    long getNewestTimestamp(String folderID);

    /**
     * Invalidates any locally cached data for the current state in the OX server to make sure the next sync uses the data currently stored
     * in the OX server. Also marks the folder as modified, i.e. a synchronize is required to update client information.
     * 
     * @param folderID null or empty String for folder hierarchy, folder-id for the content of a given folder
     */
    void invalidateCachedData(String folderID);

    /**
     * Tries to acquire a lock for the given ID. If successful, returns null. If not successful, returns the String that was provided as
     * "acquirer" parameter by the instance that acquired the key and currently owns it. unlock() must be called to free the lock, or it
     * will be blocked indefinitely.
     * 
     * @param id lock to acquire
     * @param acquirer Text that will be stored and given as notification for anyone else that tries to acquire the key while it is locked.
     * @return null if successful, else "acquirer" parameter of locker.
     */
    String tryLock(String id, String acquirer);

    /**
     * Unlocks a (previously acquired) lock by its id. Must be calles after the synchronous action is finished or the lock will never be
     * freed.
     * 
     * @param id
     */
    void unlock(String id);

    /**
     * @return The configured mail folder id separators as stored in the Ajax-config "modules" / "mail" / "separators"
     */
    public Map<String, String> getMailFolderIdSeparators();
}
