/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.exceptions;

import org.json.JSONObject;

/**
 * RuntimeException that is generated if a severe error occurred during communication
 * with the OX server. Severe errors are:<ul>
 * <li>Server responses with http status code other than OK (200)
 * <li>An HttpException was caught during communication
 * <li>An IOExcaption was caught during communication
 * </ul>
 * The http status code can be retrieved using getHttpStatusCode(), it will
 * return -1 if no status code is available. The originating exception (if
 * present) can be retrieved using the getCause() method.<br>
 * Since this can happen on every synchronization call, it is implemented as a
 * RuntimeException to avoid adding it to each method signature.
 * 
 * @author afe
 *
 */
public class OXCommunicationException extends USMException {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final int _httpStatusCode;

	private final JSONObject _jsonError;

	private final String _serverResponse;

	public OXCommunicationException(int errorCode, int statusCode) {
		super(errorCode, "http status code " + statusCode + " received");
		_httpStatusCode = statusCode;
		_jsonError = null;
		_serverResponse = null;
	}

	public OXCommunicationException(int errorCode, Throwable cause) {
		super(errorCode, "Exception occurred during communication with OX server", cause);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = null;
	}

	public OXCommunicationException(int errorCode, JSONObject errorDetails) {
		super(errorCode, "OX server returned error");
		_httpStatusCode = -1;
		_jsonError = errorDetails;
		_serverResponse = null;
	}

	public OXCommunicationException(int errorCode, String message, JSONObject errorDetails) {
		super(errorCode, message);
		_httpStatusCode = -1;
		_jsonError = errorDetails;
		_serverResponse = null;
	}

	public OXCommunicationException(int errorCode, String message, Throwable cause) {
		super(errorCode, message, cause);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = null;
	}
	
	public OXCommunicationException(int errorCode, String message, Throwable cause, String serverResponse) {
		super(errorCode, message, cause);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = serverResponse;
	}

	public OXCommunicationException(int errorCode, String message, String serverResponse) {
		super(errorCode, message);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = serverResponse;
	}

	public OXCommunicationException(int errorCode, String serverResponse) {
		super(errorCode);
		_httpStatusCode = -1;
		_jsonError = null;
		_serverResponse = serverResponse;
	}

	public boolean isHttpStatusCodeSet() {
		return _httpStatusCode != -1;
	}

	public int getHttpStatusCode() {
		return _httpStatusCode;
	}

	public boolean isJSONErrorSet() {
		return _jsonError != null;
	}

	public JSONObject getJSONError() {
		return _jsonError;
	}

	public boolean isServerResponseSet() {
		return _serverResponse != null;
	}

	public String getServerResponse() {
		return _serverResponse;
	}

	@Override
	public JSONObject getOxErrorForJSONResponse() {
		return _jsonError;
	}

	@Override
	public String getErrorMessage() {
		String message = getMessage();
		if (message == null)
			message = _serverResponse;
		return message;
	}

	@Override
	protected void addStacktrace(Throwable t, JSONObject errorDetailsObject) {
		if (t != this)
			super.addStacktrace(t, errorDetailsObject);
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(super.toString());
		if (_httpStatusCode != -1)
			sb.append(", http-status-code=").append(_httpStatusCode);
		if (_jsonError != null)
			sb.append(", json-error=").append(_jsonError);
		if (_serverResponse != null)
			sb.append(", server-response=").append(_serverResponse);
		return sb.toString();
	}
}
