/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.UUID;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Object that stores all relevant information for a single attachment to an OX PIM object
 * 
 * @author afe
 *
 */
public class PIMAttachment implements Serializable {

	private static final String RTF_FLAG = "rtf_flag";

	private static final String FILE_MIMETYPE = "file_mimetype";

	private static final String FILE_SIZE = "file_size";

	private static final String FILENAME = "filename";

	private static final String MODULE = "module";

	private static final String ATTACHED = "attached";

	private static final String FOLDER = "folder";

	private static final String CREATION_DATE = "creation_date";

	private static final String CREATED_BY = "created_by";
	
	private static final String TEMPID = "tempid";

	public static final String ID = "id";

	private static final long serialVersionUID = 1L;

    private static boolean isEqual(Object a, Object b) {
        return (a == null) ? b == null : a.equals(b);
    }

	public static final LinkedHashMap<Integer, String> FIELDS = new LinkedHashMap<Integer, String>();

	static {
		FIELDS.put(1, ID); //id
		FIELDS.put(2, CREATED_BY); //created_by 
		FIELDS.put(4, CREATION_DATE); //creation_date 
		FIELDS.put(800, FOLDER); //folder; The ID of the first Folder in which the attached object resides.
		FIELDS.put(801, ATTACHED); //attached; The object id of the object this attachement is attached to. 
		FIELDS.put(802, MODULE); //module; The Module of this Object Possible Values:
		//		1 	Appointment
		//		4 	Task
		//		7 	Contact
		//		137 	Infostore 
		FIELDS.put(803, FILENAME); //filename; The filename of the attached file. 
		FIELDS.put(804, FILE_SIZE); //file_size; The file size (in bytes) of the attached file. 
		FIELDS.put(805, FILE_MIMETYPE); //file_mimetype; The MIME-Type of the attached file 
		FIELDS.put(806, RTF_FLAG); //rft_flag; If the attachment is a RTF Attachment of Outlook. (Outlook descriptions can be stored as RTF Documents). 
	}

	// Both oxId and uuid may be set in the constructor, but also later. However, they may be set only once,
	// afterwards they can not be changed any more. Only the uuid is stored in a generated JSONObject,
	// the oxId is only used for the internal mapping and stored in the serialized SyncState.
	private int _oxId;
	private UUID _uuid;

	private final transient String _createdBy;
	private final transient long _creationDate;
	private final transient String _fileName;
	private final transient String _mimeType;
	private final transient boolean _rtfFlag;
	private final transient long _fileSize;
	private final transient byte[] _data;
	private final transient String _tempid;

	public PIMAttachment(int oxId, UUID uuid) {
		_oxId = oxId;
		_uuid = uuid;
		_createdBy = null;
		_creationDate = 0L;
		_fileName = null;
		_mimeType = null;
		_rtfFlag = false;
		_fileSize = 0L;
		_data = null;
		_tempid = null;
	}

	public PIMAttachment(PIMAttachment source, UUID uuid) {
	    _uuid = uuid;
        _oxId = source._oxId;
	    _createdBy = source._createdBy;
	    _creationDate = source._creationDate;
	    _fileName = source._fileName;
	    _mimeType = source._mimeType;
	    _rtfFlag = source._rtfFlag;
	    _fileSize = source._fileSize;
	    _data = source._data;
        _tempid = source._tempid;
	}
	
	public PIMAttachment(UUID uuid, String fileName, String mimeType, long fileSize, String createdBy,
			long creationDate, boolean rtfFlag, byte[] data) {
		_uuid = uuid;
		_fileName = fileName;
		_mimeType = mimeType;
		_fileSize = fileSize;
		_createdBy = createdBy;
		_creationDate = creationDate;
		_rtfFlag = rtfFlag;
		_data = data;
        _tempid = null;
	}

	public PIMAttachment(UUID uuid, JSONObject jsonObject, byte[] data) {
		_uuid = uuid;
		_fileName = jsonObject.optString(FILENAME, null);
		_mimeType = jsonObject.optString(FILE_MIMETYPE, null);
		_fileSize = jsonObject.optLong(FILE_SIZE, 0L);
		_createdBy = jsonObject.optString(CREATED_BY, null);
		_creationDate = jsonObject.optLong(CREATION_DATE, 0L);
		_rtfFlag = jsonObject.optBoolean(RTF_FLAG, false);
		_oxId = jsonObject.optInt(ID);
        _tempid = jsonObject.optString(TEMPID, null);
		_data = data;
	}

	public PIMAttachment(UUID uuid, JSONArray jsonArray) {
		_uuid = uuid;
		//indexes in the JSONArray are expected as in the FIELDS map
		_oxId = jsonArray.optInt(0);
		_createdBy = jsonArray.optString(1, null);
		_creationDate = jsonArray.optLong(2, 0L);
		_fileName = jsonArray.optString(6, null);
		_fileSize = jsonArray.optLong(7, 0L);
		_mimeType = jsonArray.optString(8, null);
		_rtfFlag = jsonArray.optBoolean(9, false);
		_data = null;
        _tempid = null;
	}

	/**
	 * Creates JSONObject from all fields except the data and the uuid field. 
	 * @return
	 */
	public JSONObject toJSONObject() {
		JSONObject result = new JSONObject();
		try {
			result.put(FILENAME, _fileName);
			result.put(FILE_MIMETYPE, _mimeType);
			result.put(FILE_SIZE, _fileSize);
			result.put(CREATED_BY, _createdBy);
			result.put(CREATION_DATE, _creationDate);
			result.put(RTF_FLAG, _rtfFlag);
		} catch (JSONException ignored) {
			//do nothing
		}
		return result;
	}

	public int getOxId() {
		return _oxId;
	}

	public void setOxId(int oxId) {
		if (_oxId != 0)
			throw new IllegalStateException("Tried to change PIM Attachment OX ID after initial setting");
		_oxId = oxId;
	}

	public UUID getUUID() {
		return _uuid;
	}

	public void setUUID(UUID uuid) {
		_uuid = uuid;
	}

	public String getCreatedBy() {
		return _createdBy;
	}

	public long getCreationDate() {
		return _creationDate;
	}

	public String getFileName() {
		return _fileName;
	}

	public String getMimeType() {
		return _mimeType;
	}

	public boolean isRtfFlag() {
		return _rtfFlag;
	}

	public long getFileSize() {
		return _fileSize;
	}

	public byte[] getData() {
		return _data;
	}

	public String getTempId() {
	    return _tempid;
	}
	
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		PIMAttachment other = (PIMAttachment) obj;
		if (_oxId != other._oxId)
			return false;
		if (!isEqual(_createdBy, other._createdBy))
			return false;
		if (_creationDate != other._creationDate)
			return false;
		if (!isEqual(_fileName, other._fileName))
			return false;
		if (!isEqual(_mimeType, other._mimeType))
			return false;
		if (_rtfFlag != other._rtfFlag)
			return false;
		if (_fileSize != other._fileSize)
			return false;
		if (!isEqual(_uuid, other._uuid))
			return false;
		return true;
	}

	public boolean equalsNoUUIDAndServerFields(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		PIMAttachment other = (PIMAttachment) obj;

		if (!isEqual(_fileName, other._fileName))
			return false;
		if (!isEqual(_mimeType, other._mimeType))
			return false;
		if (_rtfFlag != other._rtfFlag)
			return false;
		if (_fileSize != other._fileSize)
			return false;
		return true;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + _oxId;
		result = prime * result + ((_createdBy == null) ? 0 : _createdBy.hashCode());
		result = (int) (prime * result + _creationDate);
		result = prime * result + ((_fileName == null) ? 0 : _fileName.hashCode());
		result = prime * result + ((_mimeType == null) ? 0 : _mimeType.hashCode());
		result = prime * result + (_rtfFlag ? 1 : 0);
		result = (int) (prime * result + _fileSize);
		result = prime * result + ((_uuid == null) ? 0 : _uuid.hashCode());
		return result;
	}

    @Override
    public String toString() {
        return _fileName + '(' + _uuid + '/' + _oxId + '/' + _fileSize + ')';
    }
}
