/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import javax.annotation.concurrent.NotThreadSafe;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.tools.StatusProperties;

/**
 * {@link ConnectionStatus}
 *
 * The ConnectionStatus class stores important state information about a
 * document connection. The content is used to be sent to connected clients
 * to update their internal state. The state includes the editor name and
 * id, the current server osn, the number of clients, the current selection
 * of every client etc.
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
@NotThreadSafe
public class ConnectionStatus implements Cloneable {
    public static final String EMPTY_USERID = "";
    public static final String EMPTY_USERNAME = "";
    public static final int NO_USERID = -1;

    static final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(ConnectionStatus.class);
    static final long INACTIVITY_TRESHOLD_TIME = 25 * 60; // 25 minutes, stored as seconds

    // - Members ---------------------------------------------------------------

    private boolean m_hasErrors = false;
    private boolean m_locked = false;
    private boolean m_finalLoad = false;
    private boolean m_syncLoad = false;
    private String m_currentEditingUserName = EMPTY_USERNAME;
    private String m_currentEditingUserId = EMPTY_USERID;
    private String m_wantsEditRightsUserId = EMPTY_USERID;
    private String m_wantsEditRightsUserName = EMPTY_USERNAME;
    private int m_operationStateNumber = 0;
    private String m_lockedByUser = "";
    private int m_lockedById = NO_USERID;
    private boolean m_failSafeSaveDone = false;
    private JSONArray m_activeUsers = new JSONArray();
    private Boolean m_optWriteProtected = null;

    // - Implementation --------------------------------------------------------

    /**
     * Initializes a new {@link ConnectionStatus}.
     */
    public ConnectionStatus() {
        super();
    }

    /**
     * Provides the error state.
     *
     * @return The m_hasErrors
     */
    public boolean isHasErrors() {
        return m_hasErrors;
    }

    /**
     * Sets the error state.
     *
     * @param hasErrors The m_hasErrors to set
     */
    public void setHasErrors(boolean hasErrors) {
        this.m_hasErrors = hasErrors;
    }

    /**
     * Sets a temporary write protected state.
     * ATTENTION: This must be a temporary state as the read/write
     * state depends on the client accessing the file. Don't use it
     * to modify the global connection status!!
     *
     * @param isWriteProtected
     */
    public void setWriteProtected(boolean isWriteProtected) {
        this.m_optWriteProtected = new Boolean(isWriteProtected);
    }

    /**
     * Get the lock state
     *
     * @return The current lock state
     */
    public boolean isLocked() {
        return m_locked;
    }

    /**
     * Sets the lock state
     *
     * @param locked The new lock state
     */
    public void setLockState(boolean locked) {
        this.m_locked = locked;
    }

    /**
     * Get user name who locked the file
     *
     * @return The name of the user who locked the file
     * or empty string if user is unknown or nobody set
     * a lock.
     */
    public String getLockedByUser() {
        return m_lockedByUser;
    }

    /**
     * Sets the user who locked the file
     *
     * @param lockedByUser The name of the user
     * or empty string if unknown or nobody set
     * a lock.
     */
    public void setLockedByUser(String lockedByUser) {
        this.m_lockedByUser = lockedByUser;
    }

    /**
     * Get the user id of the user who locked the file.
     *
     * @return The id of the user who locked the file. NO_USERID, if no
     * user locked the file.
     */
    public int getLockedByUserId() {
        return this.m_lockedById;
    }

    /**
     * Sets the user id of the user who locked the file.
     *
     * @param userId The id of the user who locked the file. NO_USERID, if no
     * user locked the file.
     */
    public void setLockedByUserId(int userId) {
        this.m_lockedById = userId;
    }

    /**
     * Gets the m_currentEditingUserName
     *
     * @return The m_currentEditingUserName
     */
    public void setCurrentEditingUser(String currentEditingUserId, String currentEditingUserName) {
        m_currentEditingUserId = currentEditingUserId;
        m_currentEditingUserName = currentEditingUserName;
    }

    /**
     * Gets the m_currentEditingUserName
     *
     * @return The m_currentEditingUserName
     */
    public String getCurrentEditingUserId() {
        return m_currentEditingUserId;
    }

    /**
     * Sets the m_currentEditingUserName
     *
     * @param currentEditingUserName The currentEditingUserName to set
     */
    public String getCurrentEditingUserName() {
        return m_currentEditingUserName;
    }

    /**
     * Gets the user id who wants to receive the edit rights
     */
    public String getWantsEditRightsUserId() {
    	return m_wantsEditRightsUserId;
    }

    /**
     * Gets the user name who wants to receive the edit rights
     * @return
     */
    public String getWantsEditRightsUserName() {
    	return m_wantsEditRightsUserName;
    }

    /**
     * Sets the user id who wants to receive the edit rights
     *
     * @param userId
     */
    public void setWantsEditRightsUser(String userId, String userName) {
    	m_wantsEditRightsUserId = userId;
    	m_wantsEditRightsUserName = userName;
    }

    /**
     * Provides the number of active clients.
     *
     * @return The number of active clients.
     */
    public int getActiveClients() {
        return (null != m_activeUsers) ? m_activeUsers.length() : 0;
    }

    /**
     * Gets the m_operationStateNumber
     *
     * @return The m_operationStateNumber
     */
    public int getOperationStateNumber() {
        return this.m_operationStateNumber;
    }

    /**
     * Sets the m_operationStateNumber
     *
     * @param operationStateNumber The operation state number to set
     */
    public void setOperationStateNumber(int operationStateNumber) {
        this.m_operationStateNumber = operationStateNumber;
    }

    /**
     * Provides information that a fail safe
     * save operation was done successfully.
     *
     * @return
     *  TRUE if there recently was a successful
     *  fail safe save operation, otherwise FALSE.
     */
    public boolean isFailSafeSaveDone() {
        return m_failSafeSaveDone;
    }

    /**
     * Sets the state of a fail safe save
     * operation.
     *
     * @param saveDone The state of the fail safe save
     */
    void setFailSafeSaveDone(boolean saveDone) {
        this.m_failSafeSaveDone = saveDone;
    }

    /**
     * Gets the activeUsers
     *
     * @return The activeUsers
     */
    public JSONArray getActiveUsers() {
        return m_activeUsers;
    }

    /**
     * Sets the activeUsers
     *
     * @param activeUsers The activeUsers to set
     */
    public void setActiveUsers(JSONArray activeUsers) {
        this.m_activeUsers = activeUsers;
    }

    /**
     * Add a user to the activeUsers map
     *
     * @param userId ID of the user
     * @param userDisplayName The UI name of the new user
     * @param guest Specifies, if the user is a guest or not
     * @param userData JSON Object containing user data, e.g. {userName:'Rocky Balboa', selection:{...}, ...}
     * @throws JSONException
     */
    public void addActiveUser(String userId, String userDisplayName, int id, boolean guest, JSONObject userData){
        JSONObject userObject = new JSONObject();
        try {
            userObject.put("userId", userId);
            userObject.put("userDisplayName", userDisplayName);
            userObject.put("guest", guest);
            userObject.put("userData", userData);
            userObject.put("active", true);
            userObject.put("id", id);
            userObject.put("durationOfInactivity", 0L);
        } catch (JSONException e) {
            LOG.error("Exception catched while adding new active user", e);
        }
        m_activeUsers.put(userObject);
    }

    /**
     * Remove a specified user from the activeUsers map
     *
     * @param userId ID from the user to be removed
     * @throws JSONException
     */
    public boolean removeActiveUser(String userId) throws JSONException {
        for (int i = 0; i < m_activeUsers.length();i++) {
            JSONObject user = m_activeUsers.getJSONObject(i);
            if (user.get("userId").equals(userId)) {
                m_activeUsers.remove(i);
                return true;
            }
        }
        return false;
    }

    /**
     * Update the userdata of a specific user
     *
     * @param userId
     *  ID of the user
     * @param userData
     *  JSON Object containing user data, e.g. {userName:'Rocky Balboa', selection:{...}, ...}
     * @return
     *  Returns true if an update is done.
     */
    public boolean updateUserData(String userId, JSONObject userData){
        // find the user and update his data
        JSONObject user;
        if (null != userId) {
            for (int i = 0; i < m_activeUsers.length();i++) {
                try {
                    user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        user.put("userData", userData);
                        user.put("active", true);
                        return true;
                    }
                } catch (JSONException e) {
                    LOG.error("Exception while setting updated user data", e);
                }
            }
        }

        return false;
    }

    /**
     * Disables an active user so her data is NOT sent to other clients.
     *
     * @param userId
     *  The ID of the user to be disabled.
     *
     * @return
     *  TRUE if the user is disabled, FALSE if the user couldn't be
     *  found in the list of active users.
     */
    public boolean disableActiveUser(String userId) {
        JSONObject user;

        if (null != userId) {
            for (int i = 0; i < m_activeUsers.length();i++) {
                try {
                    user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        user.put("active", false);
                        return true;
                    }
                } catch (JSONException e) {
                    LOG.error("Exception while setting updated user data", e);
                }
            }
        }

        return false;
    }

    /**
     * Disables an active user so her data is NOT sent to other clients.
     *
     * @param userId
     *  The ID of the user to be disabled.
     *
     * @param durationOfInactivity
     *  The duration of inactivity in seconds.
     *
     * @return
     *  TRUE if the user is disabled, FALSE if the user couldn't be
     *  found in the list of active users.
     */
    public boolean disableActiveUser(String userId, long durationOfInactivity) {

        if (null != userId) {
            for (int i = 0; i < m_activeUsers.length();i++) {
                try {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        int lastDurationOfInactivity = user.optInt("durationOfInactivity", 0);
                        user.put("active", false);
                        if (lastDurationOfInactivity < durationOfInactivity) {
                            // Don't set a lower inactivity time in case of
                            // disable active user. He/she must be activated
                            // before.
                            user.put("durationOfInactivity", durationOfInactivity);
                        }
                        return true;
                    }
                } catch (JSONException e) {
                    LOG.error("Exception while setting updated user data", e);
                }
            }
        }

        return false;
    }

    /**
     * Determine if the provided user is enabled or not.
     *
     * @param userId
     *  The ID of the user to find out the enabled state.
     *
     * @return
     *  TRUE if the user is enabled, FALSE if the user couldn't be
     *  found in the list or is not enabled.
     */
    public boolean isActiveUserEnabled(String userId) {

        if (null != userId) {
            try {
                for (int i = 0; i < m_activeUsers.length();i++) {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        return user.optBoolean("active", false);
                    }
                }
            } catch (JSONException e) {
                LOG.error("Exception while retrieving activity state of user", e);
            }
        }

        return false;
    }

    /**
     * Determine if the provided user is enabled or not.
     *
     * @param userId
     *  The ID of the user to find out the enabled state.
     *
     * @return
     *  The duration of inactivity, where 0 means no inactivity and
     *  -1 that the inactivity could not be determined (normally the
     *  user is unknown).
     */
    public long getDurationOfInactivity(String userId) {
        long durationOfInactivity = -1;

        if (null != userId) {
            try {
                for (int i = 0; i < m_activeUsers.length();i++) {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        return user.optLong("durationOfInactivity", 0);
                    }
                }
            } catch (JSONException e) {
                LOG.error("Exception while retrieving duration of inactivity for user", e);
            }
        }

        return durationOfInactivity;
    }

    /**
     * Get session user id of the user associated with the RT-id.
     *
     * @param userId
     *  The real-time user id.
     *
     * @return
     *  The session user id of the user associated width the RT-id or
     *  -1 if the RT-id is unknown.
     */
    public int getUserId(String userId) {
        int id = -1;

        if (null != userId) {
            try {
                for (int i = 0; i < m_activeUsers.length();i++) {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        return user.optInt("id", -1);
                    }
                }
            } catch (JSONException e) {
                LOG.error("Exception while retrieving duration of inactivity for user", e);
            }
        }

        return id;
    }

    /**
     * Enables an active user again so her data is sent to other clients.
     *
     * @param userId
     *  The ID of the user to be enabled again.
     * @return
     *  TRUE if the user is active again, FALSE if the user couldn't be
     *  found in the list of active users.
     *
     * @throws JSONException
     */
    public boolean enableActiveUser(String userId) {
        JSONObject user;
        for (int i = 0; i < m_activeUsers.length();i++) {
            try {
                user = m_activeUsers.getJSONObject(i);
                if (user.get("userId").equals(userId)) {
                    user.put("active", true);
                    user.put("durationOfInactivity", 0L);
                    return true;
                }
            } catch (JSONException e) {
                LOG.error("Exception while setting updated user data", e);
            }
        }
        return false;
    }

    /**
     * Determines if a user id is part of the active users.
     *
     * @param userId
     *  The ID of the user to determine that is a member of this connection.
     *
     * @return
     *  TRUE if the user is member or FALSE if not.
     */
    public boolean hasActiveUser(String userId) {
        JSONObject user;
        if (null != userId) {
	        for (int i = 0; i < m_activeUsers.length();i++) {
                user = m_activeUsers.optJSONObject(i);
                if ((null != user) && (userId.equals(user.optString("userId")))) {
                    return true;
                }
	        }
        }
        return false;
    }

    /**
     * ATTENTION: This method sets a TEMPORARY state. Don't use it on the global
     * connection state object.
     *
     * @param userId
     */
    public void setInactivityChanged(String userId) {
        JSONObject user;
        if (null != userId) {
        	try {
		        for (int i = 0; i < m_activeUsers.length();i++) {
	                user = m_activeUsers.optJSONObject(i);
	                if ((null != user) && (userId.equals(user.optString("userId")))) {
	                    user.put("inactivityChanged", true);
	                }
		        }
        	} catch (JSONException e) {
                LOG.error("Exception while setting updated user data", e);
        	}
        }
    }

    /**
     * ATTENTION: This method sets a TEMPORARY state. Don't use it on the global
     * connection state object.
     *
     * @param userId
     */
    public void setSelectionChanged(String userId) {
        JSONObject user;
        if (null != userId) {
        	try {
		        for (int i = 0; i < m_activeUsers.length();i++) {
	                user = m_activeUsers.optJSONObject(i);
	                if ((null != user) && (userId.equals(user.optString("userId")))) {
	                    user.put("selectionChanged", true);
	                }
		        }
        	} catch (JSONException e) {
                LOG.error("Exception while setting updated user data", e);
        	}
        }
    }

    /**
     * Sets the 'finalLoad' attribute, which notifies the client that the
     * asynchronous loading process is finished and the 'update' with
     * this attribute contains the document actions.
     *
     * @param finalLoad
     *  Sets the finalLoad attribute.
     */
    public void setFinalLoad(boolean set) {
    	m_finalLoad = set;
    }

    /**
     * Provides the finalLoad attribute of this connection status
     * object.
     *
     * @return
     *  TRUE if the finalLoad attribute is set otherwise FALSE.
     */
    public boolean getFinalLoad() {
    	return m_finalLoad;
    }

    /**
     * Sets the 'syncLoad' attribute which notifies the client that the
     * welcome message contains the full set of document actions. If
     * 'syncLoad' is not set the client must wait for the next 'update'
     * notification which has the 'finalLoad' attribute set.
     */
    public void setSyncLoad(boolean set) {
    	m_syncLoad = set;
    }

    /**
     * Provides the 'syncLoad' attribute of this connection status
     * object.
     *
     * @return
     *  TRUE if the syncLoad attribute is set otherwise FALSE.
     */
    public boolean getSyncLoad() {
    	return m_syncLoad;
    }

    /**
     * Creates a shallow clone of this connection status
     * object.
     *
     * @return
     *  A shallow clone.
     */
    @Override
    public ConnectionStatus clone() {
        ConnectionStatus clone = new ConnectionStatus();
        clone.setCurrentEditingUser(m_currentEditingUserId, m_currentEditingUserName);
        clone.setFailSafeSaveDone(m_failSafeSaveDone);
        clone.setHasErrors(m_hasErrors);
        clone.setLockedByUser(m_lockedByUser);
        clone.setLockState(m_locked);
        clone.setLockedByUserId(m_lockedById);
        clone.setOperationStateNumber(m_operationStateNumber);
        clone.setWantsEditRightsUser(m_wantsEditRightsUserId, m_wantsEditRightsUserName);
        JSONArray users = new JSONArray();
        try {
	        for (int i = 0; i < m_activeUsers.length(); i++) {
	        	JSONObject user = m_activeUsers.getJSONObject(i);
	        	if (null != user) {
	                users.put(new JSONObject(user.toString()));
	        	}
	        }
        } catch (JSONException e) {
        	LOG.debug("Exception while cloning ConnectionStatus", e);
        }
        clone.setActiveUsers(users);
        return clone;
    }

    /**
     * Creates the JSON representation of the current connection status
     * object. The JSON object should only provide properties that are
     * part of the client/server communication. Internal properties should
     * be omitted.
     *
     * @return
     *  A JSON object that contains the properties of the current connection
     *  status. The JSON object can be empty in case of an error.
     */
    public JSONObject toJSON() {
        final JSONObject ret = new JSONObject();

        try {
            ret.put(StatusProperties.KEY_HAS_ERRORS, m_hasErrors);
            ret.put(StatusProperties.KEY_EDIT_USERID, m_currentEditingUserId);
            ret.put(StatusProperties.KEY_EDIT_USER, m_currentEditingUserName);
            ret.put(StatusProperties.KEY_LOCKED, m_locked);
            ret.put(StatusProperties.KEY_LOCKED_BY_USER, m_lockedByUser);
            ret.put(StatusProperties.KEY_FAILSAFESAVEDONE, m_failSafeSaveDone);
            ret.put(StatusProperties.KEY_WANTSEDIT_USER, m_wantsEditRightsUserName);
            ret.put(StatusProperties.KEY_WANTSEDIT_USERID, m_wantsEditRightsUserId);
            ret.put(StatusProperties.KEY_SERVER_OSN, m_operationStateNumber);
            ret.put(StatusProperties.KEY_LOCKED_BY_ID, m_lockedById);
            if (null != m_optWriteProtected) {
                ret.put(StatusProperties.KEY_WRITE_PROTECTED, m_optWriteProtected);
            }
            if (m_finalLoad) {
                ret.put(StatusProperties.KEY_FINALLOADUPDATE, m_finalLoad);
            }
            if (m_syncLoad) {
                ret.put(StatusProperties.KEY_SYNCLOAD, m_syncLoad);
            }
            // send all collaboration user (client must filter/process list)
            ret.put(StatusProperties.KEY_ACTIVE_USERS, m_activeUsers);
        } catch (JSONException e) {
            LOG.error("Exception while creating JSON object of the current connection status", e);
        }

        return ret;
    }

    /**
     * Creates a new connection status object using the provided JSONObject
     * to set the state.
     *
     * @param jsonData
     *  The JSON object containing properties that are used to set the
     *  state of the new connection status object.
     *
     * @return
     *  The new connection status object.
     */
    static public ConnectionStatus createFrom(JSONObject jsonData) {
        ConnectionStatus queueStatus = new ConnectionStatus();
        boolean found = false;

        try {
            if (jsonData.has(StatusProperties.KEY_HAS_ERRORS)) {
                queueStatus.m_hasErrors = jsonData.getBoolean(StatusProperties.KEY_HAS_ERRORS);
                found = true;
            }

            if (jsonData.has(StatusProperties.KEY_EDIT_USERID)) {
                queueStatus.m_currentEditingUserId = jsonData.getString(StatusProperties.KEY_EDIT_USERID);
                found = true;
            }

            if (jsonData.has(StatusProperties.KEY_EDIT_USER)) {
                queueStatus.m_currentEditingUserName = jsonData.getString(StatusProperties.KEY_EDIT_USER);
                found = true;
            }

            if (jsonData.has(StatusProperties.KEY_LOCKED)) {
                queueStatus.m_locked = jsonData.getBoolean(StatusProperties.KEY_LOCKED);
                found = true;
            }

            if (jsonData.has(StatusProperties.KEY_LOCKED_BY_USER)) {
                queueStatus.m_lockedByUser = jsonData.getString(StatusProperties.KEY_LOCKED_BY_USER);
                found = true;
            }

            if (!jsonData.has(StatusProperties.KEY_FAILSAFESAVEDONE)) {
                queueStatus.m_failSafeSaveDone = jsonData.getBoolean(StatusProperties.KEY_FAILSAFESAVEDONE);
                found = true;
            }

            if (!jsonData.has(StatusProperties.KEY_WANTSEDIT_USER)) {
                queueStatus.m_wantsEditRightsUserName = jsonData.getString(StatusProperties.KEY_WANTSEDIT_USER);
                found = true;
            }

            if (!jsonData.has(StatusProperties.KEY_WANTSEDIT_USERID)) {
                queueStatus.m_wantsEditRightsUserId = jsonData.getString(StatusProperties.KEY_WANTSEDIT_USERID);
                found = true;
            }

            if (!jsonData.has(StatusProperties.KEY_ACTIVE_USERS)) {
                queueStatus.m_activeUsers = jsonData.getJSONArray(StatusProperties.KEY_ACTIVE_USERS);
                found = true;
            }

            if (!jsonData.has(StatusProperties.KEY_LOCKED_BY_ID)) {
                queueStatus.m_lockedById = jsonData.getInt(StatusProperties.KEY_LOCKED_BY_ID);
                found = true;
            }
        } catch (JSONException e) {
            LOG.error("Exception while creating new connection status object", e);
        }

        return (found ? queueStatus : null);
    }

    static public boolean isEmptyUser(final String user) {
        return ((user != null) && EMPTY_USERID.equals(user));
    }

    static public boolean isEmptyUserName(final String userName) {
        return ((userName != null) && EMPTY_USERNAME.equals(userName));
    }

    static public boolean isNotEmptyUser(final String user) {
        return ((user != null) && !(EMPTY_USERID.equals(user)));
    }

    static public boolean isNotEmptyUserName(final String userName) {
        return ((userName != null) && !(EMPTY_USERNAME.equals(userName)));
    }
}
