/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.office.realtime.osgi.Services;
import com.openexchange.realtime.Component;
import com.openexchange.realtime.ComponentHandle;
import com.openexchange.realtime.LoadFactorCalculator;
import com.openexchange.realtime.cleanup.AbstractRealtimeJanitor;
import com.openexchange.realtime.cleanup.GlobalRealtimeCleanup;
import com.openexchange.realtime.exception.RealtimeException;
import com.openexchange.realtime.packet.ID;
import com.openexchange.server.ServiceLookup;

/**
 * Abstract base class to control the life-time of office connections.
 *
 * {@link ConnectionComponent}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public abstract class ConnectionComponent extends AbstractRealtimeJanitor implements Component, IConnectionFactory, IConnectionPool {

    protected static final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(ConnectionComponent.class);

    private final ServiceLookup m_services;
    private final ConcurrentHashMap<ID, Connection> m_openConnections = new ConcurrentHashMap<ID, Connection>();
    private final String m_componentID;

    protected LoadFactorCalculator loadFactorCalculator;
    protected LoadFactorLimitManager limitLoadFactor;

    /**
     * Initializes a new {@link ConnectionComponent}.
     * @param services
     */
    protected ConnectionComponent(final ServiceLookup services, final String componentID) {
        super();
        m_services = services;
        m_componentID = componentID;
    }

    /**
     * Retrieves the service lookup used by this component instance.
     *
     * @return
     *  The service lookup instance used by this component instance.
     */
    public ServiceLookup getServiceLookup() {
        return m_services;
    }

    /**
     * Provides the name of the connection pool.
     *
     * @return
     *  The name of the connection pool.
     */
    @Override
    public String getName() {
        return m_componentID;
    }

    @Override
    public void setLoadFactorCalculator(LoadFactorCalculator loadFactorCalculator) {
        this.loadFactorCalculator = loadFactorCalculator;
        this.limitLoadFactor = new LoadFactorLimitManager(m_services, loadFactorCalculator, new WeakReference<Component>(this));
    }

    /**
     * Retrieves the connection as enumeration.
     *
     * @return
     *  A enumeration with all accessible connections as weak references.
     *  This ensures that connections could be freed while a processor
     *  makes long running tasks, which would keep the connection alive, due
     *  to a hard reference.
     */
    @Override
    public List<WeakReference<Connection>> getConnections() {
        final Enumeration<Connection> connections = m_openConnections.elements();
        final ArrayList<WeakReference<Connection>> list = new ArrayList<WeakReference<Connection>>();

        while (connections.hasMoreElements()) {
            list.add(new WeakReference<Connection>(connections.nextElement()));
        }
        return list;
    }

    /**
     * Returns the open connections controlled by this component instance.
     *
     * @return
     *  A map filled with the open connections of this component instance.
     */
    protected ConcurrentHashMap<ID, Connection> getOpenConnections() {
        return m_openConnections;
    }

    /**
     * Returns the number of connections for this component.
     * @return
     *  The number of connections for this component.
     */
    @Override
    public int getConnectionCount() {
        return m_openConnections.size();
    }

    /**
     * Returns the component ID.
     */
    @Override
    public String getId() {
        return m_componentID;
    }

    /**
     * Handler that is called by the RT-framework when a connection
     * should be created by the component.
     *
     * @param id
     *  The id of the connection created by the component.
     */
    @Override
    public ComponentHandle create(ID id) throws RealtimeException {
        // use the factory interface to create the concrete connection instance
        final Connection connection = createConnectionInstance(id);

        if (connection != null) {
            m_openConnections.put(id, connection);
            onConnectionCreated(id);

            LOG.debug("RT connection: New document created with ID: " + id.toString() + ", connection count: " + m_openConnections.size());
        }

        return connection;
    }

    /**
     * Returns the eviction policy of the component.
     */
    @Override
    public EvictionPolicy getEvictionPolicy() {
        // No automatic shut down of operations queue
        return com.openexchange.realtime.Component.NONE;
    }

    /**
     * Handler that is called by the RT-framework when a connection
     * should be removed from the component.
     *
     * @param id
     *  The id of the connection be removed from the component.
     */
    @Override
    public void cleanupForId(final ID id) {
        // We are only interested in IDs of office components for now
        if (m_componentID.equals(id.getComponent())) {
            final ConcurrentHashMap<ID, Connection> connections = getOpenConnections();
            final Connection removedConnection = connections.remove(id);
            if (removedConnection == null) {
                LOG.debug("RT connection: Couldn't remove connection for ID: " + id);
            } else {
                onConnectionDestroyed(id);
                LOG.debug("RT connection: Removed document with ID: " + id.toString() + ", connection count: " + connections.size());
            }
        }
    }

    /**
     * Handler to react specific when a new connection instance is
     * created.
     *
     * @param id The id of the new connection instance.
     */
    abstract protected void onConnectionCreated(final ID id);

    /**
     * Handler to react specific when a connection instance is
     * destroyed.
     *
     * @param id The id of the destroyed connection instance.
     */
    abstract protected void onConnectionDestroyed(final ID id);

    /**
     * Tries to shutdown the connection component gracefully, including saving
     * the current modified documents to OX Drive and to close timer/threads.
     */
    public void shutdown() {
        for(ID id: m_openConnections.keySet()) {
            final Connection connection = m_openConnections.get(id);
            if (null != connection) {
                connection.setShutdown();
            }
        }

        final GlobalRealtimeCleanup globalRealtimeCleanup = Services.optService(GlobalRealtimeCleanup.class);
        if(globalRealtimeCleanup == null) {
            LOG.error("Unable to issue cluster wide cleanup for Connection IDs due to missing GlobalRealtimeCleanup service.");
        }

        for(ID id: m_openConnections.keySet()) {
            final Connection connection = m_openConnections.get(id);
            if (null != connection) {
                connection.saveDocumentOnDispose();
            }

            if (globalRealtimeCleanup != null) {
                globalRealtimeCleanup.cleanForId(id);
            }
        }
    }
}
