/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.datatype.DatatypeConstants;
import javax.xml.datatype.XMLGregorianCalendar;

import org.apache.commons.io.IOUtils;
import org.docx4j.dml.Theme;
import org.docx4j.docProps.core.CoreProperties;
import org.docx4j.docProps.core.dc.elements.SimpleLiteral;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.contenttype.ContentTypeManager.NonPersistContent;
import org.docx4j.openpackaging.contenttype.ContentTypes;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.io3.Save;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.parts.DocPropsCorePart;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.docx4j.openpackaging.parts.WordprocessingML.DocumentSettingsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.wml.CTSettings;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.IResourceManager;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.memory.MemoryObserver;
import com.openexchange.office.tools.memory.MemoryObserver.MemoryListener;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

abstract public class OperationDocument {

	public class ChangeTrackAuthor {

		private final String author;

		private XMLGregorianCalendar date;

		public ChangeTrackAuthor(String _author, XMLGregorianCalendar _date) {
			author = _author;
			date = _date;
		}
		private void updateDate(XMLGregorianCalendar _date) {
			if(_date!=null) {
				if(date==null||_date.compare(date)==DatatypeConstants.LESSER) {
					date = _date;
				}
			}
		}
		@Override
        public boolean equals(Object p) {
		    boolean v = author.compareToIgnoreCase(((ChangeTrackAuthor)p).author) == 0;
		    if(v) {
		    	updateDate(((ChangeTrackAuthor)p).date);
		    }
		    return v;
		}
        @Override
        public int hashCode() {
            return 0;
        }
	}
	public static Comparator<ChangeTrackAuthor> ChangeTrackAuthorComparator = new Comparator<ChangeTrackAuthor>() {
		@Override
        public int compare(ChangeTrackAuthor a1, ChangeTrackAuthor a2) {

			if(a1.date==null) {
				return a2.date==null ? DatatypeConstants.EQUAL : DatatypeConstants.GREATER;
			}
			else {
				return a2.date==null ? DatatypeConstants.LESSER : a1.date.compare(a2.date);
			}
		}
	};

	final private static Logger log = LoggerFactory.getLogger(OperationDocument.class);

    private final IResourceManager      resourceManager;
    private final DocumentProperties    documentProperties;
    protected 	  InputStream           inputDocumentStream;
    private final ServiceLookup         services;
    private final Session				session;
    protected final boolean             saveDebugOperations;
    private boolean                     createFinalDocument = false;
    private Set<ChangeTrackAuthor> 		changeTrackAuthors = null;
    protected Part 						contextPart = null;
    private MemoryListener              memoryListener = null;
    protected static int 				successfulAppliedOperations = 0;

    static final int DEFAULT_BUFFER_SIZE = 8192;

    protected OperationDocument(Session session, ServiceLookup services, InputStream inputDocumentStream, IResourceManager resourceManager, DocumentProperties documentProperties) {
        this.services = services;
        this.session = session;
        this.inputDocumentStream = inputDocumentStream;
        this.resourceManager = resourceManager;
        this.documentProperties = documentProperties;
        saveDebugOperations = Boolean.parseBoolean(ConfigurationHelper.getOfficeConfigurationValue(services, session, "debugoperations", "false"));
        registerMemoryListener();
        Context.setLowMemoryAbort(false);
        successfulAppliedOperations = 0;
    };

    public void registerMemoryListener() {
        MemoryObserver memoryObserver = (MemoryObserver)documentProperties.get(DocumentProperties.PROP_MEMORYOBSERVER);
        if(memoryObserver!=null) {
            if(memoryListener==null) {
                memoryListener = new MemoryListener() {
                    @Override
                    public void memoryTresholdExceeded(long usedMemory, long maxMemory) {
                        notifyLowMemory();
                    }
                };
                memoryObserver.addListener(memoryListener);
            }
            if(MemoryObserver.isUsageThresholdExceeded()) {
                notifyLowMemory();
            }
        }
    }

    public void removeMemoryListener() {
        if(memoryListener!=null) {
            MemoryObserver memoryObserver = (MemoryObserver)documentProperties.get(DocumentProperties.PROP_MEMORYOBSERVER);
            if(memoryObserver!=null) {
                memoryObserver.removeListener(memoryListener);
                memoryListener = null;
            }
        }
    }

    private void notifyLowMemory() {
        log.info("OOXML Filter: Low memory notification received");
        Context.setLowMemoryAbort(true);
        final OpcPackage opcPackage = getPackage();
        if(opcPackage!=null) {
            opcPackage.setSourcePartStore(null);
            opcPackage.setContentTypeManager(null);
            setPackage(null);
        }
    }

    public InputStream createDocument(String applyOperations)
        throws FilterException {

        try {
            // applying some document properties...
            CoreProperties coreProperties = getPackage().getDocPropsCorePart(true).getJaxbElement();
            org.docx4j.docProps.core.dc.elements.ObjectFactory of = new org.docx4j.docProps.core.dc.elements.ObjectFactory();
            String userName = documentProperties.optString(DocumentProperties.PROP_LAST_MODIFIED_BY, "");
            coreProperties.setLastModifiedBy(userName);
            coreProperties.setRevision(documentProperties.optString(DocumentProperties.PROP_REVISION, null));
            coreProperties.setVersion(documentProperties.optString(DocumentProperties.PROP_VERSION, null));

            if(userName.length()>0) {
                SimpleLiteral creatorLiteral = coreProperties.getCreator();
                if(creatorLiteral==null) {
                    creatorLiteral = of.createSimpleLiteral();
                    creatorLiteral.getContent().add(userName);
                    coreProperties.setCreator(creatorLiteral);
                }
            }

            applyOperations(applyOperations);
            InputStream debugDocument = null;

            String uniqueDocumentIdentifier = documentProperties.optString(DocumentProperties.PROP_UNIQUE_DOCUMENT_IDENTIFIER, null);
            if (null != uniqueDocumentIdentifier) {
                putUniqueDocumentIdentifier(uniqueDocumentIdentifier);
            }

            if(saveDebugOperations&&inputDocumentStream instanceof ByteArrayInputStream) {
                try {
                    inputDocumentStream.reset();
                    debugDocument = inputDocumentStream;
                }
                catch(IOException e) {
                    //
                }
            }
            return debugSave(applyOperations, debugDocument, resourceManager);
        }
        catch(FilterException e) {
        	throw e;
        }
        catch(Throwable e) {
            log.error("OOXML Filter could not save the given document:", e);
            throw (e instanceof FilterException) ? (FilterException)e : new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    abstract public void applyOperations(String applyOperations)
        throws FilterException;

    abstract protected JSONObject getOperations()
        throws Exception;

    public InputStream save() throws FilterException {
        return debugSave(null, null, null);
    }

    public void setCreateFinalDocument(boolean createFinalDocument) {
        this.createFinalDocument = createFinalDocument;
    }

    public ServiceLookup getServiceLookup() {
    	return this.services;
	}

    public Session getSession() {
    	return this.session;
 	}

    protected void logMessage(final String msgClass, final String logMessage) {
        if ("fatal".equalsIgnoreCase(logMessage))
            log.error(logMessage);
        else if ("error".equalsIgnoreCase(msgClass))
            log.error(logMessage);
        else if ("warning".equalsIgnoreCase(msgClass))
            log.warn(logMessage);
        else if ("debug".equalsIgnoreCase(msgClass))
            log.debug(logMessage);
        else // "info"
            log.info(logMessage);
    }

    /**
     * Add the unique document identifier to the document.
     *
     * @param uniqueIdentifier
     *  The unique document identifier to be saved into the document stream.
     */
    private boolean putUniqueDocumentIdentifier(String uniqueIdentifier) {
        try {
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("uniqueDocumentIdentifier", uniqueIdentifier);

            getPackage().getContentTypeManager().addNonPersistContentFile("oxoffice/extendedData.json",
                jsonObject.toString().getBytes("UTF-8"),"json", "application/octet-stream");
        } catch (UnsupportedEncodingException e) {
            log.error("OOXML Filter could not save the unique document indentifier:", e);
            return false;
        } catch (JSONException e) {
            log.error("OOXML Filter could not create JSON object to store the unique document indentifier:", e);
            return false;
        }

        return true;
    }

    /* For debug purposes we create a folder (debug) within the destination document
     * that stores each information necessary to repeat the last save operation that was done.
     * (original document, the operations just applied and the content of the resource manager. */

    private InputStream debugSave(String debugOperations, java.io.InputStream debugDocument, IResourceManager debugResourceManager)
        throws FilterException {

        try {

    		boolean contentTypeChanged = true;
        	if (documentProperties.optBoolean(DocumentProperties.PROP_SAVE_TEMPLATE_DOCUMENT, false)) {
        		if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.WORDPROCESSINGML_DOCUMENT)!=null) {
        			getPackage().getContentTypeManager().addOverrideContentType(new PartName("/word/document.xml").getURI(), ContentTypes.WORDPROCESSINGML_TEMPLATE);
        		}
        		else if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.SPREADSHEETML_WORKBOOK)!=null) {
        			getPackage().getContentTypeManager().addOverrideContentType(new PartName("/xl/workbook.xml").getURI(), ContentTypes.SPREADSHEETML_TEMPLATE);
        		}
    			else if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.PRESENTATION)!=null) {
			    	getPackage().getContentTypeManager().addOverrideContentType(new PartName("/ppt/presentation.xml").getURI(), ContentTypes.PRESENTATIONML_TEMPLATE);
        		}
    			else {
    				contentTypeChanged = false;
    			}
        		if(contentTypeChanged) {
        			switchToTemplateDocument();
        		}
        	}
        	else {
        		if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.WORDPROCESSINGML_TEMPLATE)!=null) {
        			getPackage().getContentTypeManager().addOverrideContentType(new PartName("/word/document.xml").getURI(), ContentTypes.WORDPROCESSINGML_DOCUMENT);
        		}
        		else if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.SPREADSHEETML_TEMPLATE)!=null) {
        			getPackage().getContentTypeManager().addOverrideContentType(new PartName("/xl/workbook.xml").getURI(), ContentTypes.SPREADSHEETML_WORKBOOK);
        		}
    			else if(getPackage().getContentTypeManager().getPartNameOverridenByContentType(ContentTypes.PRESENTATIONML_TEMPLATE)!=null) {
			    	getPackage().getContentTypeManager().addOverrideContentType(new PartName("/ppt/presentation.xml").getURI(), ContentTypes.PRESENTATION);
        		}
    			else {
    				contentTypeChanged = false;
    			}
        		if(contentTypeChanged) {	// switching from document from template to non template -> we have to remove the author
        			switchToNonTemplateDocument();
        		}
        	}

        	// always removing thumbnails when saving
        	removePart(getPackage().getRelationshipsPart(), "/docProps/thumbnail.jpeg");

        	final Save saver = new Save(getPackage());
            if(saveDebugOperations) {
                if(debugOperations!=null) {
                    getPackage().getContentTypeManager().addNonPersistContentFile("debug/operationUpdates.dbg",
                        debugOperations.getBytes("UTF-8"),"dbg", "application/octet-stream");
                }
                if(debugDocument!=null) {
                    ZipInputStream zipInputStream = new ZipInputStream(debugDocument);
                    ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                    ZipOutputStream zipOutputStream = new ZipOutputStream(outStream);
                    ZipEntry zipEntry = null;
                    byte[] buffer = new byte[0x10000];  // 64kb buffer
                    while((zipEntry = zipInputStream.getNextEntry())!=null) {

                        // if the source document is containing a debug folder, we will
                        // skip the old debug information and not include it within our copy
                        if(!zipEntry.getName().startsWith("debug/")) {
                            ZipEntry newEntry = new ZipEntry(zipEntry.getName());
                            zipOutputStream.putNextEntry(newEntry);
                            int read;
                            while ((read = zipInputStream.read(buffer))>=0){
                                zipOutputStream.write(buffer,0,read);
                            }
                            zipOutputStream.closeEntry();
                        }
                        zipInputStream.closeEntry();
                    }
                    zipInputStream.close();
                    zipOutputStream.close();

                    getPackage().getContentTypeManager().addNonPersistContentFile("debug/documentStream.dbg",
                        outStream.toByteArray(), "dbg", "application/octet-stream");
                }
                if(debugResourceManager!=null) {

// TODO:  the resourceManager needs to be able to serialize the resources, so we can store these data also within or debug folder
//
//                  getPackage().getContentTypeManager().addNonPersistContentFile("debug/debugResources.dbg",
//                      debugResourceManager., "application/octet-stream");
                }
            }
            saver.setCheckUnusedRelations(createFinalDocument);
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            saver.save(out);
            return new ByteArrayInputStream(out.toByteArray());


        } catch (Throwable e) {
        	throw getRethrowException(e);
        }
    }

    public void writeFile(String path) throws FilterException {
        InputStream  inputStream = null;
        OutputStream outputStream = null;

        try {
            inputStream = save();

            if (null != inputStream) {
                IOUtils.copy(inputStream, outputStream = new FileOutputStream(new File(path)));
                outputStream.flush();
            }
        } catch (Throwable e) {
            log.error("OOXML Filter could not save the given document:", e);
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        } finally {
            IOUtils.closeQuietly(inputStream);
            IOUtils.closeQuietly(outputStream);
        }
    }

    abstract public OpcPackage getPackage();

    abstract public void setPackage(OpcPackage opcPackage);

    abstract public Theme getTheme()
    	throws FilterException;

    public IResourceManager getResourceManager() {
        return resourceManager;
    }

    public DocumentProperties getDocumentProperties() {
        return documentProperties;
    }

    public String getUserLanguage() {
        return documentProperties.optString(DocumentProperties.PROP_USER_LANGUAGE, "");
    }

    public static FilterException getRethrowException(Throwable e) {
    	FilterException ret;
        if(Context.isLowMemoryAbort()) {
            ret = new FilterException("FilterException...", ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED);
        }
        else if(e instanceof FilterException) {
    		final FilterException filterException = (FilterException)e;
    		if(filterException.getErrorcode()==ErrorCode.COMPLEXITY_TOO_HIGH) {
    			log.warn("could not load/save document, the document complexity is too high");
    		}
    		else if(filterException.getErrorcode()==ErrorCode.CRITICAL_ERROR) {
    			log.error(e.getMessage(), e);
    		}
    		ret = (FilterException)e;
    	}
    	else if (e instanceof OutOfMemoryError) {
    		ret = new FilterException(e, ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED);
    	}
    	else if(e instanceof Docx4JException) {
            ErrorCode errorCode = ErrorCode.CRITICAL_ERROR;
            if(e.getCause() instanceof java.security.InvalidKeyException) {
                errorCode = ErrorCode.WRONG_PASSWORD;
            }
            else if(e.getCause() instanceof org.apache.poi.EncryptedDocumentException) {
                errorCode = ErrorCode.UNSUPPORTED_ENCRYPTION_USED;
            }
            ret = new FilterException(e, errorCode);
    	}
    	else {
    		log.error(e.getMessage(), e);
    		ret = new FilterException(e, ErrorCode.CRITICAL_ERROR);
    	}
    	ret.setOperationCount(OperationDocument.successfulAppliedOperations);
    	return ret;
    }

    public long getFreeHeapSpace() {
        final MemoryMXBean memoryMXBean = ManagementFactory.getMemoryMXBean();
        final MemoryUsage heapMemoryUsage = memoryMXBean.getHeapMemoryUsage();
        final long max = heapMemoryUsage.getMax();
        final long used = heapMemoryUsage.getUsed();
        final long free = max - used;
        return free;
    }

    protected void switchToTemplateDocument() {

    	resetCoreProperties();
    	removePart(getPackage().getRelationshipsPart(), "/docProps/app.xml");

    	// removing extendedData
        final List<NonPersistContent> nonPersistContentList = getPackage().getContentTypeManager().getNonPersistContent();
        for(int i=0; i<nonPersistContentList.size(); i++) {
        	final NonPersistContent nonPersistContent = nonPersistContentList.get(i);
        	if(nonPersistContent.zipEntryName.equals("oxoffice/extendedData.json")) {
        		nonPersistContentList.remove(i);
        		break;
        	}
        }
    }

    protected void switchToNonTemplateDocument() {

    	resetCoreProperties();
    	removePart(getPackage().getRelationshipsPart(), "/docProps/app.xml");
    }

    private void resetCoreProperties() {
    	final DocPropsCorePart corePart = getPackage().getDocPropsCorePart(false);
    	if(corePart!=null) {
            final CoreProperties coreProperties = corePart.getJaxbElement();
            coreProperties.setCategory(removeEmptyElement(coreProperties.getCategory()));
            coreProperties.setContentStatus(removeEmptyElement(coreProperties.getContentStatus()));
            coreProperties.setContentType(removeEmptyElement(coreProperties.getContentType()));
            coreProperties.setCreated(null);
            coreProperties.setCreator(null);
            coreProperties.setDescription(removeEmptyElement(coreProperties.getDescription()));
            coreProperties.setIdentifier(removeEmptyElement(coreProperties.getIdentifier()));
            coreProperties.setKeywords(removeEmptyElement(coreProperties.getKeywords()));
            coreProperties.setLanguage(removeEmptyElement(coreProperties.getLanguage()));
            coreProperties.setLastModifiedBy(null);
            coreProperties.setLastPrinted(null);
            coreProperties.setModified(null);
            coreProperties.setRevision(null);
            coreProperties.setSubject(removeEmptyElement(coreProperties.getSubject()));
            coreProperties.setTitle(removeEmptyElement(coreProperties.getTitle()));
    	}
    }

    protected void removePart(final RelationshipsPart relationshipsPart, final String partName) {
		if(relationshipsPart!=null) {
			try {
				relationshipsPart.removePart(new PartName(partName));
			}
			catch(InvalidFormatException e) {
				//
			}
		}
    }

    private <T> T removeEmptyElement(T value) {
    	if(value instanceof String) {
    		if(((String) value).isEmpty()) {
    			return null;
    		}
    	}
    	else if(value instanceof SimpleLiteral) {
    		if((((SimpleLiteral)value).getLang()==null||((SimpleLiteral)value).getLang().isEmpty())&&
    			(((SimpleLiteral)value).getContent().isEmpty())) {
    				return null;
    		}
    	}
    	else if(value instanceof JAXBElement<?>) {
    		if(((JAXBElement<?>)value).getDeclaredType().equals("org.docx4j.docProps.core.dc.elements.SimpleLiteral")) {
        		final SimpleLiteral o = (SimpleLiteral)((JAXBElement<?>)value).getValue();
        		if((o.getLang()==null||o.getLang().isEmpty())&&
            		(o.getContent().isEmpty())) {
            		return null;
            	}
    		}
    	}
    	return value;
    }

    public void addChangeTrackAuthor(String author, XMLGregorianCalendar date) {
    	if(changeTrackAuthors==null) {
    		changeTrackAuthors = new HashSet<OperationDocument.ChangeTrackAuthor>();
    	}
    	changeTrackAuthors.add(new OperationDocument.ChangeTrackAuthor(author, date));
    }

    public JSONArray getChangeTrackAuthors() {
    	if(changeTrackAuthors==null||changeTrackAuthors.isEmpty())
    		return null;
    	final JSONArray authors = new JSONArray();

    	final ChangeTrackAuthor[] sortedAuthors = changeTrackAuthors.toArray(new ChangeTrackAuthor[changeTrackAuthors.size()]);
    	Arrays.sort(sortedAuthors, ChangeTrackAuthorComparator);

    	for(ChangeTrackAuthor changeTrackAuthor:sortedAuthors) {
    		authors.put(changeTrackAuthor.author);
    	}
    	return authors;
    }

    public abstract Part getContextPart();

    public void setContextPart(Part part) {
    	contextPart = part;
    }

    protected ThemePart getDefaultThemePart(PartName partName) {

        final String themeStr =

            "<a:theme xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\"" +
            "    name=\"${themeName}\">" +
            "    <a:themeElements>" +
            "        <a:clrScheme name=\"${themeName}\">" +
            "            <a:dk1>" + "${dk1}" + " </a:dk1>" +
            "            <a:lt1>" + "${lt1}" + " </a:lt1>" +
            "            <a:dk2>" + "${dk2}" + " </a:dk2>" +
            "            <a:lt2>" + "${lt2}" + " </a:lt2>" +
            "            <a:accent1>" + "${accent1}" + " </a:accent1>" +
            "            <a:accent2>" + "${accent2}" + " </a:accent2>" +
            "            <a:accent3>" + "${accent3}" + " </a:accent3>" +
            "            <a:accent4>" + "${accent4}" + " </a:accent4>" +
            "            <a:accent5>" + "${accent5}" + " </a:accent5>" +
            "            <a:accent6>" + "${accent6}" + " </a:accent6>" +
            "            <a:hlink>"   + "${hlink}"   + " </a:hlink>"   +
            "            <a:folHlink>"+ "${folHlink}"+ " </a:folHlink>"+
            "        </a:clrScheme>" +
            "        <a:fontScheme name=\"${themeName}\">" +
            "            <a:majorFont>" +
            "                <a:latin typeface=\"${majorFont}\" />" +
            "                <a:ea typeface=\"\" />" +
            "                <a:cs typeface=\"\" />" +
            "            </a:majorFont>" +
            "            <a:minorFont>" +
            "                <a:latin typeface=\"${minorFont}\" />" +
            "                <a:ea typeface=\"\" />" +
            "                <a:cs typeface=\"\" />" +
            "            </a:minorFont>" +
            "        </a:fontScheme>" +
            "        <a:fmtScheme name=\"${themeName}\">" +
            "            <a:fillStyleLst>" +
            "                <a:solidFill>" +
            "                    <a:schemeClr val=\"phClr\" />" +
            "                </a:solidFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"50000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"35000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"37000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"15000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:lin ang=\"16200000\" scaled=\"1\" />" +
            "                </a:gradFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"51000\" />" +
            "                                <a:satMod val=\"130000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"80000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"93000\" />" +
            "                                <a:satMod val=\"130000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"94000\" />" +
            "                                <a:satMod val=\"135000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:lin ang=\"16200000\" scaled=\"0\" />" +
            "                </a:gradFill>" +
            "            </a:fillStyleLst>" +
            "            <a:lnStyleLst>" +
            "                <a:ln w=\"9525\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\">" +
            "                            <a:shade val=\"95000\" />" +
            "                            <a:satMod val=\"105000\" />" +
            "                        </a:schemeClr>" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "                <a:ln w=\"25400\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\" />" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "                <a:ln w=\"38100\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\" />" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "            </a:lnStyleLst>" +
            "            <a:effectStyleLst>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"20000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"38000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                </a:effectStyle>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"23000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"35000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                </a:effectStyle>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"23000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"35000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                    <a:scene3d>" +
            "                        <a:camera prst=\"orthographicFront\">" +
            "                            <a:rot lat=\"0\" lon=\"0\" rev=\"0\" />" +
            "                        </a:camera>" +
            "                        <a:lightRig rig=\"threePt\" dir=\"t\">" +
            "                            <a:rot lat=\"0\" lon=\"0\" rev=\"1200000\" />" +
            "                        </a:lightRig>" +
            "                    </a:scene3d>" +
            "                    <a:sp3d>" +
            "                        <a:bevelT w=\"63500\" h=\"25400\" />" +
            "                    </a:sp3d>" +
            "                </a:effectStyle>" +
            "            </a:effectStyleLst>" +
            "            <a:bgFillStyleLst>" +
            "                <a:solidFill>" +
            "                    <a:schemeClr val=\"phClr\" />" +
            "                </a:solidFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"40000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"40000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"45000\" />" +
            "                                <a:shade val=\"99000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"20000\" />" +
            "                                <a:satMod val=\"255000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:path path=\"circle\">" +
            "                        <a:fillToRect l=\"50000\" t=\"-80000\" r=\"50000\" b=\"180000\" />" +
            "                    </a:path>" +
            "                </a:gradFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"80000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"30000\" />" +
            "                                <a:satMod val=\"200000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:path path=\"circle\">" +
            "                        <a:fillToRect l=\"50000\" t=\"50000\" r=\"50000\" b=\"50000\" />" +
            "                    </a:path>" +
            "                </a:gradFill>" +
            "            </a:bgFillStyleLst>" +
            "        </a:fmtScheme>" +
            "    </a:themeElements>" +
            "    <a:objectDefaults />" +
            "    <a:extraClrSchemeLst />" +
            "</a:theme>";

        final String defaultSchemeName = "Office";

        ThemePart themePart = null;
        try {
            themePart = new ThemePart(partName);
            java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();
            mappings.put("themeName", defaultSchemeName);
            mappings.put("dk1", "<a:sysClr val=\"windowText\" lastClr=\"000000\" />");
            mappings.put("lt1", "<a:sysClr val=\"window\" lastClr=\"FFFFFF\" />");
            mappings.put("dk2", "<a:srgbClr val=\"44546A\" />");
            mappings.put("lt2", "<a:srgbClr val=\"E7E6E6\" />");
            mappings.put("accent1", "<a:srgbClr val=\"5B9BD5\" />");
            mappings.put("accent2", "<a:srgbClr val=\"ED7D31\" />");
            mappings.put("accent3", "<a:srgbClr val=\"A5A5A5\" />");
            mappings.put("accent4", "<a:srgbClr val=\"FFC000\" />");
            mappings.put("accent5", "<a:srgbClr val=\"4472C4\" />");
            mappings.put("accent6", "<a:srgbClr val=\"70AD47\" />");
            mappings.put("hlink",   "<a:srgbClr val=\"0563C1\" />");
            mappings.put("folHlink","<a:srgbClr val=\"954F72\" />");
            mappings.put("majorFont", "Times New Roman");
            mappings.put("minorFont", "Arial");
            Theme theme = (Theme)org.docx4j.XmlUtils.unmarshallFromTemplate(themeStr, mappings);
            themePart.setJaxbElement(theme);
        }
        catch (JAXBException e) {
            e.printStackTrace();
        }
        catch (InvalidFormatException e) {
            e.printStackTrace();
        }
        return themePart;
    }
    
    public DocumentSettingsPart getDefaultSettingsPart(PartName partName) {
    	
    	final String documentSettingsStr =
    		"<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" + 
    		"<w:settings\n" + 
    		"	xmlns:mc=\"http://schemas.openxmlformats.org/markup-compatibility/2006\"\n" + 
    		"	xmlns:o=\"urn:schemas-microsoft-com:office:office\"\n" + 
    		"	xmlns:r=\"http://schemas.openxmlformats.org/officeDocument/2006/relationships\"\n" + 
    		"	xmlns:m=\"http://schemas.openxmlformats.org/officeDocument/2006/math\"\n" + 
    		"	xmlns:v=\"urn:schemas-microsoft-com:vml\" xmlns:w10=\"urn:schemas-microsoft-com:office:word\"\n" + 
    		"	xmlns:w=\"http://schemas.openxmlformats.org/wordprocessingml/2006/main\"\n" + 
    		"	xmlns:w14=\"http://schemas.microsoft.com/office/word/2010/wordml\"\n" + 
    		"	xmlns:w15=\"http://schemas.microsoft.com/office/word/2012/wordml\"\n" + 
    		"	xmlns:sl=\"http://schemas.openxmlformats.org/schemaLibrary/2006/main\"\n" + 
    		"	mc:Ignorable=\"w14 w15\">\n" + 
    		"	<w:zoom w:percent=\"100\" />\n" + 
    		"	<w:defaultTabStop w:val=\"720\" />\n" + 
    		"	<w:characterSpacingControl w:val=\"doNotCompress\" />\n" + 
    		"	<w:compat>\n" + 
    		"		<w:compatSetting w:name=\"compatibilityMode\"\n" + 
    		"			w:uri=\"http://schemas.microsoft.com/office/word\" w:val=\"15\" />\n" + 
    		"		<w:compatSetting w:name=\"overrideTableStyleFontSizeAndJustification\"\n" + 
    		"			w:uri=\"http://schemas.microsoft.com/office/word\" w:val=\"1\" />\n" + 
    		"		<w:compatSetting w:name=\"enableOpenTypeFeatures\"\n" + 
    		"			w:uri=\"http://schemas.microsoft.com/office/word\" w:val=\"1\" />\n" + 
    		"		<w:compatSetting w:name=\"doNotFlipMirrorIndents\"\n" + 
    		"			w:uri=\"http://schemas.microsoft.com/office/word\" w:val=\"1\" />\n" + 
    		"		<w:compatSetting w:name=\"differentiateMultirowTableHeaders\"\n" + 
    		"			w:uri=\"http://schemas.microsoft.com/office/word\" w:val=\"1\" />\n" + 
    		"	</w:compat>\n" + 
    		"	<w:rsids>\n" + 
    		"		<w:rsidRoot w:val=\"008463FB\" />\n" + 
    		"		<w:rsid w:val=\"002B1CD8\" />\n" + 
    		"		<w:rsid w:val=\"008463FB\" />\n" + 
    		"	</w:rsids>\n" + 
    		"	<m:mathPr>\n" + 
    		"		<m:mathFont m:val=\"Cambria Math\" />\n" + 
    		"		<m:brkBin m:val=\"before\" />\n" + 
    		"		<m:brkBinSub m:val=\"--\" />\n" + 
    		"		<m:smallFrac m:val=\"0\" />\n" + 
    		"		<m:dispDef />\n" + 
    		"		<m:lMargin m:val=\"0\" />\n" + 
    		"		<m:rMargin m:val=\"0\" />\n" + 
    		"		<m:defJc m:val=\"centerGroup\" />\n" + 
    		"		<m:wrapIndent m:val=\"1440\" />\n" + 
    		"		<m:intLim m:val=\"subSup\" />\n" + 
    		"		<m:naryLim m:val=\"undOvr\" />\n" + 
    		"	</m:mathPr>\n" + 
    		"	<w:themeFontLang w:val=\"en-US\" />\n" + 
    		"	<w:clrSchemeMapping w:bg1=\"light1\" w:t1=\"dark1\"\n" + 
    		"		w:bg2=\"light2\" w:t2=\"dark2\" w:accent1=\"accent1\" w:accent2=\"accent2\"\n" + 
    		"		w:accent3=\"accent3\" w:accent4=\"accent4\" w:accent5=\"accent5\" w:accent6=\"accent6\"\n" + 
    		"		w:hyperlink=\"hyperlink\" w:followedHyperlink=\"followedHyperlink\" />\n" + 
    		"	<w:shapeDefaults>\n" + 
    		"		<o:shapedefaults v:ext=\"edit\" spidmax=\"1026\" />\n" + 
    		"		<o:shapelayout v:ext=\"edit\">\n" + 
    		"			<o:idmap v:ext=\"edit\" data=\"1\" />\n" + 
    		"		</o:shapelayout>\n" + 
    		"	</w:shapeDefaults>\n" + 
    		"	<w:decimalSymbol w:val=\",\" />\n" + 
    		"	<w:listSeparator w:val=\";\" />\n" + 
    		"</w:settings>";

    	DocumentSettingsPart documentSettingsPart = null;
        try {
        	documentSettingsPart = new DocumentSettingsPart(partName);
            java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();

            final CTSettings documentSettings = (CTSettings)((JAXBElement<?>)org.docx4j.XmlUtils.unmarshallFromTemplate(documentSettingsStr, mappings)).getValue();
            documentSettingsPart.setJaxbElement(documentSettings);
        }
        catch (JAXBException e) {
            e.printStackTrace();
        }
        catch (InvalidFormatException e) {
            e.printStackTrace();
        }
        return documentSettingsPart;
    }
}
