/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang.mutable.MutableInt;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.SpreadsheetML.TablePart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.jaxb.Context;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTTable;
import org.xlsx4j.sml.CTTableColumn;
import org.xlsx4j.sml.CTTableColumns;
import org.xlsx4j.sml.CTTableFormula;
import org.xlsx4j.sml.CTTablePart;
import org.xlsx4j.sml.CTTableParts;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.STTotalsRowFunction;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.SmlUtils;
import org.xlsx4j.sml.SmlUtils.CellRef;
import org.xlsx4j.sml.SmlUtils.CellRefRange;
import org.xlsx4j.sml.Worksheet;

import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.operations.ApplyOperationHelper;

public class TableHelper {

    public static void createTableOperations(JSONArray operationsArray, WorksheetPart worksheetPart, int sheetIndex, CTTableParts tableParts)
        throws JSONException {

        if(tableParts==null) {
            return;
        }
    	final List<CTTablePart> tablePartList = tableParts.getTablePart();
    	for(CTTablePart tablePartId:tablePartList) {
    		final CTTable table = getTable(worksheetPart, tablePartId);
    		if(table!=null) {
    	        final SmlUtils.CellRefRange cellRefRange = table.getCellRefRange(false);
    			final String tableName = table.getDisplayName();
    	        if(cellRefRange!=null&&tableName!=null&&!tableName.isEmpty()) {
    	            final JSONObject attrs = new JSONObject(1);
    	            final JSONObject tableAttrs = new JSONObject(3);
    	            tableAttrs.put("headerRow", table.getHeaderRowCount()==1);
    	            tableAttrs.put("footerRow", table.getTotalsRowCount()==1);
    	            tableAttrs.put("filtered", false);
    	            attrs.put("table", tableAttrs);
    	            addInsertTableOperation(operationsArray, sheetIndex, cellRefRange, tableName, attrs);
    	        }
    		}
    	}
    }

    public static void addInsertTableOperation(JSONArray operationsArray, int sheetIndex, SmlUtils.CellRefRange range, String tableName, JSONObject attrs)
            throws JSONException {

        final JSONObject addInsertTableObject = new JSONObject(6);
        addInsertTableObject.put("name", "insertTable");
        addInsertTableObject.put("sheet", sheetIndex);
        addInsertTableObject.put("start", range.getStart().getJSONArray());
        addInsertTableObject.put("end", range.getEnd().getJSONArray());
        if(!attrs.isEmpty()) {
            addInsertTableObject.put("attrs", attrs);
        }
        addInsertTableObject.put("table", tableName);
        operationsArray.put(addInsertTableObject);
    }

    public static void changeOrInsertTable(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName, JSONArray start, JSONArray end, JSONObject attrs, boolean insert)
        throws JSONException {

    	final WorksheetPart worksheetPart = applyOperationHelper.getWorksheetPart(sheetIndex);
        if(insert) {
        	// TODO: 
        	return;
        }
        final CTTable table = getTable(worksheetPart, tableName);
        if(table!=null) {
        	final CellRefRange cellRefRange = table.getCellRefRange(true);
        	if(start!=null) {
        		final CellRef startRef = cellRefRange.getStart();
        		startRef.setColumn(start.getInt(0));
        		startRef.setRow(start.getInt(1));
        	}
        	if(end!=null) {
        		final CellRef endRef = cellRefRange.getEnd();
        		endRef.setColumn(end.getInt(0));
        		endRef.setRow(end.getInt(1));
        	}
        	if(attrs!=null) {
        		final JSONObject tableAttrs = attrs.optJSONObject("table");
        		if(tableAttrs!=null) {
        			final Object headerRow = tableAttrs.opt("headerRow");
        			if(headerRow instanceof Boolean) {
        				table.setHeaderRowCount(((Boolean)headerRow).booleanValue() ? null : Long.valueOf(0));
        			}
        			final Object footerRow = tableAttrs.opt("footerRow");
        			if(footerRow instanceof Boolean) {
        				table.setTotalsRowCount(((Boolean)footerRow).booleanValue() ? Long.valueOf(1) : null);
        			}
        		}
        	}
        }
    }

    public static void deleteTable(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName)
    	throws InvalidFormatException {

    	deleteTable(applyOperationHelper.getWorksheetPart(sheetIndex), tableName);
    }

    private static void deleteTable(WorksheetPart worksheetPart, String tableName) {
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        final RelationshipsPart worksheetRelationships = worksheetPart.getRelationshipsPart();
        if(worksheetRelationships!=null) {
        	final CTTableParts tableParts = worksheet.getTableParts();
        	final Iterator<CTTablePart> tablePartIter = tableParts.getTablePart().iterator();
        	while(tablePartIter.hasNext()) {
        		final CTTablePart tablePartId = tablePartIter.next();
        		final CTTable table = getTable(worksheetPart, tablePartId);
        		if(table!=null) {
        			if(tableName.equals(table.getDisplayName())) {
        				worksheetRelationships.removeRelationship(worksheetRelationships.getRelationshipByID(tablePartId.getId()));
        				tablePartIter.remove();
        				break;
        			}
        		}
        	}
        }
    }
    

    public static void changeTableColumn(ApplyOperationHelper applyOperationHelper, int sheetIndex, String tableName, long col, JSONObject attrs)
        throws JSONException {

    	// What to do ?
    }

    private static CTTable getTable(WorksheetPart worksheetPart, String tableName) {
        final Worksheet worksheet = worksheetPart.getJaxbElement();
        final CTTableParts tableParts = worksheet.getTableParts();
        if(tableParts!=null) {
	    	final List<CTTablePart> tablePartList = tableParts.getTablePart();
	    	for(CTTablePart tablePartId:tablePartList) {
	    		final CTTable table = getTable(worksheetPart, tablePartId);
	    		if(table!=null) {
	    	        final SmlUtils.CellRefRange cellRefRange = table.getCellRefRange(false);
	    	        if(tableName.equals(table.getDisplayName())&&cellRefRange!=null) {
	    	        	return table;
	    	        }
	    		}
	    	}
        }
    	return null;
    }

    private static CTTable getTable(WorksheetPart worksheetPart, CTTablePart tablePart) {
    	final String id = tablePart.getId();
    	if(id!=null&&!id.isEmpty()) {
	        final RelationshipsPart worksheetRelationships = worksheetPart.getRelationshipsPart();
	        if(worksheetRelationships!=null) {
	        	final Part part = worksheetRelationships.getPart(id);
	        	if(part.getRelationshipType().equals("http://schemas.openxmlformats.org/officeDocument/2006/relationships/table")&&part instanceof TablePart) {
	        		return ((TablePart)part).getJaxbElement();
	        	}
	        }
    	}
    	return null;
    }

    public static void insertRows(WorksheetPart worksheetPart, CTTablePart tablePart, int start, int count) {
    	final CTTable table = getTable(worksheetPart, tablePart);
    	final CellRefRange tableRef = SmlUtils.CellRefRange.insertRowRange(table.getCellRefRange(true), start, count, false);
    	final CellRefRange autofilterRef = table.getAutoFilter()!=null ? SmlUtils.CellRefRange.insertRowRange(table.getAutoFilter().getCellRefRange(true), start, count, false) : null;
    	applyTableRefChange(worksheetPart, table, tableRef, autofilterRef);
    }

    public static void deleteRows(WorksheetPart worksheetPart, CTTablePart tablePart, int start, int count) {
    	final CTTable table = getTable(worksheetPart, tablePart);
    	final CellRefRange tableRef = SmlUtils.CellRefRange.deleteRowRange(table.getCellRefRange(true), start, count);
    	final CellRefRange autofilterRef = table.getAutoFilter()!=null ? SmlUtils.CellRefRange.deleteRowRange(table.getAutoFilter().getCellRefRange(true), start, count) : null;
    	applyTableRefChange(worksheetPart, table, tableRef, autofilterRef);
    }

    public static void insertColumns(WorksheetPart worksheetPart, CTTablePart tablePart, int start, int count) {
    	final CTTable table = getTable(worksheetPart, tablePart);
    	final CellRefRange tableRef = SmlUtils.CellRefRange.insertColumnRange(table.getCellRefRange(true), start, count, false);
    	final CellRefRange autofilterRef = table.getAutoFilter()!=null ? SmlUtils.CellRefRange.insertColumnRange(table.getAutoFilter().getCellRefRange(true), start, count, false) : null;
    	applyTableRefChange(worksheetPart, table, tableRef, autofilterRef);
    }

    public static void deleteColumns(WorksheetPart worksheetPart, CTTablePart tablePart, int start, int count) {
    	final CTTable table = getTable(worksheetPart, tablePart);
    	final CellRefRange tableRef = SmlUtils.CellRefRange.deleteColumnRange(table.getCellRefRange(true), start, count);
    	final CellRefRange autofilterRef = table.getAutoFilter()!=null ? SmlUtils.CellRefRange.deleteColumnRange(table.getAutoFilter().getCellRefRange(true), start, count) : null;
    	applyTableRefChange(worksheetPart, table, tableRef, autofilterRef);
    }

    private static void applyTableRefChange(WorksheetPart worksheetPart, CTTable table, CellRefRange tableRef, CellRefRange autofilterRef) {
    	if(tableRef!=null) {
    		table.setCellRefRange(tableRef);
    		if(autofilterRef!=null) {
    			table.getAutoFilter().setCellRefRange(autofilterRef);
    		}
    		else {
    			table.setAutoFilter(null);
    		}
    	}
    	else {
    		deleteTable(worksheetPart, table.getDisplayName());
    	}
    }

    // this method is called after all operations were applied and the document is about to be saved.
    // we take care that column names matches the corresponding cell values of the worksheet and that each
    // table is getting its own unique id
    public static void finalizeTables(OperationDocument operationDocument, WorksheetPart worksheetPart, MutableInt uniqueTableId) {
    	final Worksheet worksheet = worksheetPart.getJaxbElement();
        final CTTableParts tableParts = worksheet.getTableParts();
        if(tableParts!=null) {
        	final SheetData sheetData = worksheet.getSheetData();
        	final List<CTTablePart> tablePartList = tableParts.getTablePart();
        	for(CTTablePart tablePart:tablePartList) {
        		final CTTable table = getTable(worksheetPart, tablePart);
        		if(table!=null) {
        			table.setId(uniqueTableId.longValue());
        			uniqueTableId.increment();

    				// ensure the correct number of column entries
        			final CellRefRange cellRefRange = table.getCellRefRange(true);
    				final int columns = (cellRefRange.getEnd().getColumn()-cellRefRange.getStart().getColumn())+1;
        			CTTableColumns tableColumns = table.getTableColumns();
        			if(tableColumns==null) {
        				tableColumns = Context.getsmlObjectFactory().createCTTableColumns();
        			}
        			tableColumns.setCount(Long.valueOf(columns));
        			final List<CTTableColumn> tableColumnList = tableColumns.getTableColumn();
        			while(tableColumnList.size()<columns) {
        				tableColumnList.add(Context.getsmlObjectFactory().createCTTableColumn());
        			}
        			while(tableColumnList.size()>columns) {
        				tableColumnList.remove(tableColumnList.size()-1);
        			}

        			// applying unique column id
        			long uniqueColumnId = 1;
        			for(CTTableColumn tableColumn:tableColumnList) {
        				tableColumn.setId(uniqueColumnId++);
        			}

        			final long headerRowCount = table.getHeaderRowCount();
        			if(headerRowCount==1) {
            			// take care that the column header matches the cell content
        				// and that the cell content is of type string and that the column name is unique
        				final HashSet<String> nameSet = new HashSet<String>(columns);
        				final Row row = sheetData.getRow(cellRefRange.getStart().getRow(), true);
        				int c1 = cellRefRange.getStart().getColumn();
        				while(c1<=cellRefRange.getEnd().getColumn()) {
        					final Cell cell = row.getCell(c1, true);
        					final STCellType cellType = cell.getT();

        					String preferedColumnName = null;
        					if(cellType==STCellType.S) {
        						preferedColumnName = operationDocument.getSharedString(cell.getV());
        					}
        					else if(cellType==STCellType.STR) {
        						preferedColumnName = cell.getV();
        					}
        					final String uniqueName = getCheckUniqueName(nameSet, preferedColumnName, (c1-cellRefRange.getStart().getColumn())+1);
        					tableColumnList.get(c1-cellRefRange.getStart().getColumn()).setName(uniqueName);
        					cell.setT(STCellType.STR);
        					cell.setV(uniqueName);
        					cell.setF(null);
        					c1++;
        				}
        			}
        			else {
        				// just take care of unique column entries.
        				final HashSet<String> nameSet = new HashSet<String>(columns);
        				for(CTTableColumn tableColumn:tableColumnList) {
        					final String preferedColumnName = tableColumn.getName();
        					tableColumn.setName(getCheckUniqueName(nameSet, preferedColumnName, table.getId()));
        				}
        			}

        			final long totalsRowCount = table.getTotalsRowCount();
        			if(totalsRowCount==1) {
        				final Row row = sheetData.getRow(cellRefRange.getEnd().getRow(), true);
        				int c1 = cellRefRange.getStart().getColumn();
        				while(c1<=cellRefRange.getEnd().getColumn()) {
        					final Cell cell = row.getCell(c1, false);
    						final CTTableColumn tableColumn = tableColumnList.get(c1-cellRefRange.getStart().getColumn());
    						tableColumn.setTotalsRowFormula(null);
    						tableColumn.setTotalsRowFunction(null);
    						tableColumn.setTotalsRowLabel(null);
        					if(cell!=null) {
        						final STCellType cellType = cell.getT()!=null?cell.getT():STCellType.N;
        						final CTCellFormula cellFormula = cell.getF();
        						final boolean isFormula = cellFormula!=null&&(cellFormula.getT()==STCellFormulaType.NORMAL||cellFormula.getT()==STCellFormulaType.SHARED);
        						if(isFormula) {
        							final CTTableFormula tableFormula = Context.getsmlObjectFactory().createCTTableFormula();
        							tableFormula.setValue(cellFormula.getValue());
        							tableColumn.setTotalsRowFunction(STTotalsRowFunction.CUSTOM);
        							tableColumn.setTotalsRowFormula(tableFormula);
        						}
        						else if(cellType==STCellType.S) {
        							tableColumn.setTotalsRowLabel(operationDocument.getSharedString(cell.getV()));
        						}
        						else if(cellType==STCellType.STR) {
        							tableColumn.setTotalsRowLabel(cell.getV());
        						}
        					}
        					c1++;
        				}
        			}
        		}
        	}
        }
    }

    private static String getCheckUniqueName(HashSet<String> nameSet, String preferedName, long column) {
    	if(preferedName==null||preferedName.isEmpty()) {
    		preferedName = "column" + Long.toString(column);
    	}
    	while(nameSet.contains(preferedName)) {
    		preferedName = preferedName + "0";
    	}
    	nameSet.add(preferedName);
    	return preferedName;
    }
}
